#include <SPAD/AC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/WQ.H>
#include <SYS/UTSNAME.H>
#include <SPAD/IOCTL.H>
#include <ARCH/BSF.H>
#include <LIMITS.H>
#include <STRING.H>
#include <SPAD/CONSOLE.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <ARCH/PAGE.H>
#include <SPAD/TTY.H>
#include <SYS/KD.H>

static void *lnte, *dlrq;

static int unload(void *p, void **release, const char * const argv[]);

static WQ_DECL(block_wait, "NULL$BLOCK_WAIT");

static void *syscalls_lookup(HANDLE *h, char *str, int open_flags);

static void *block_lookup(HANDLE *h, char *str, int open_flags);
static void *block_create(HANDLE *handle, char *filename, int open_flags);
static void *block_delete(HANDLE *h, IORQ *rq, int open_flags, HANDLE *hp);
static void *block_rename(HANDLE *h, char *str, HANDLE *fh, IORQ *rq);
static void block_lookup_io(HANDLE *h, char *str, IORQ *rq, int open_flags);
static void init_root(HANDLE *h, void *data);
static void block_detach(HANDLE *h);
static IO_STUB block_read;
static IO_STUB block_write;
static IO_STUB block_aread;
static IO_STUB block_awrite;
static IO_STUB block_ioctl;

static void *hole_lookup(HANDLE *h, char *str, int open_flags);
static void *hole_create(HANDLE *handle, char *filename, int open_flags);
static void *hole_delete(HANDLE *h, IORQ *rq, int open_flags, HANDLE *hp);
static IO_STUB hole_write;
static IO_STUB hole_awrite;

static PAGE *zero_get_page(HANDLE *h, __v_off idx, int wr);
static IO_STUB zero_read;
static IO_STUB zero_aread;
static IO_STUB zero_ioctl;

static void *zpage;
static PAGE *zp;

static const HANDLE_OPERATIONS syscalls_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	syscalls_lookup,	/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$SUCCESS,		/* READ */
	hole_write,		/* WRITE */
	KERNEL$SUCCESS,		/* AREAD */
	hole_awrite,		/* AWRITE */
	KERNEL$NO_OPERATION,	/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static const HANDLE_OPERATIONS block_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	block_lookup,		/* lookup */
	block_create,		/* create */
	block_delete,		/* delete */
	block_rename,		/* rename */
	block_lookup_io,	/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	block_detach,		/* detach */
	NULL,			/* close */
	block_read,		/* READ */
	block_write,		/* WRITE */
	block_aread,		/* AREAD */
	block_awrite,		/* AWRITE */
	block_ioctl,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static const HANDLE_OPERATIONS null_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	hole_lookup,		/* lookup */
	hole_create,		/* create */
	hole_delete,		/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$SUCCESS,		/* READ */
	hole_write,		/* WRITE */
	KERNEL$SUCCESS,		/* AREAD */
	hole_awrite,		/* AWRITE */
	KERNEL$NO_OPERATION,	/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static const HANDLE_OPERATIONS nulltty_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	hole_lookup,		/* lookup */
	hole_create,		/* create */
	hole_delete,		/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	block_detach,		/* detach */
	NULL,			/* close */
	block_read,		/* READ */
	hole_write,		/* WRITE */
	block_aread,		/* AREAD */
	hole_awrite,		/* AWRITE */
	block_ioctl,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static const HANDLE_OPERATIONS zero_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	zero_get_page,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	hole_lookup,		/* lookup */
	hole_create,		/* create */
	hole_delete,		/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	zero_read,		/* READ */
	hole_write,		/* WRITE */
	zero_aread,		/* AREAD */
	hole_awrite,		/* AWRITE */
	zero_ioctl,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};


/* partial duplicate in TTY.C */
static int tty_lookup(HANDLE *h, char *str)
{
	long num;
	char *e = strchr(str, '=');
	if (__unlikely(!e)) {
		if (!_strcasecmp(str, "^NONBLOCK")) {
			h->flags |= TTYF_NONBLOCK;
		} else if (__unlikely(!_strcasecmp(str, "^MOUSESELECT"))) {
			h->flags |= TTYF_MOUSESELECT;
		} else return 1;
		return 0;
	}
	if (!__strcasexcmp("^ICANON", str, e)) {
		if (e[1] != '0' && __unlikely(e[1] != '1')) return 1;
		if (__unlikely(e[2])) return 1;
		h->flags = (h->flags & ~TTYF_RAW) | (e[1] == '0' ? TTYF_RAW : 0);
	} else if (!__strcasexcmp("^ECHO", str, e)) {
		if (e[1] != '0' && __unlikely(e[1] != '1')) return 1;
		if (__unlikely(e[2])) return 1;
		h->flags = (h->flags & ~TTYF_NOECHO) | (e[1] == '0' ? TTYF_NOECHO : 0);
	} else if (!__strcasexcmp("^WINSIZE", str, e)) {
		/* ignore it */
	} else if (!__strcasexcmp("^ONLCR", str, e)) {
		if (e[1] != '0' && __unlikely(e[1] != '1')) return 1;
		if (__unlikely(e[2])) return 1;
		h->flags = (h->flags & ~TTYF_O_NL_CRNL) | (e[1] == '0' ? 0 : TTYF_O_NL_CRNL);
	} else if (!__strcasexcmp("^MOUSE", str, e)) {
		int mode;
		if (!_strcasecmp(e + 1, "COPYPASTE")) mode = TTYF_MOUSE_COPYPASTE;
		else if (!_strcasecmp(e + 1, "CHAR")) mode = TTYF_MOUSE_CHAR;
		else if (!_strcasecmp(e + 1, "RAW")) mode = TTYF_MOUSE_RAW;
		else return 1;
		h->flags = (h->flags & ~TTYF_MOUSE_MODE) | mode;
	} else if (!__strcasexcmp("^KBDRAW", str, e)) {
		if (__unlikely(e[1] < '0') || __unlikely(e[1] > '2')) return 1;
		if (__unlikely(e[2])) return 1;
		h->flags = (h->flags & ~TTYF_KBDMODE) | (e[1] - '0');
	} else if (!__strcasexcmp("^KBDLED", str, e)) {
		int mode;
		if (__unlikely(__get_number(e + 1, e + strlen(e), 1, &num))) return 1;
		if (__likely((unsigned long)num <= TTYF_LEDS >> __BSF_CONST(TTYF_LEDS))) mode = num << __BSF_CONST(TTYF_LEDS);
		else mode = TTYF_DEFAULT_LEDS;
		h->flags = (h->flags & ~(TTYF_LEDS | TTYF_DEFAULT_LEDS)) | mode;
	} else if (!__strcasexcmp("^KBD_DELAY", str, e) ||
		   !__strcasexcmp("^KBD_RATE", str, e) ||
		   !__strcasexcmp("^KBD_REPEAT_MODE", str, e)) {
	} else return 1;
	return 0;
}

static void *syscalls_lookup(HANDLE *h, char *str, int open_flags)
{
	if (__unlikely(!tty_lookup(h, str))) return NULL;
	if (!_strcasecmp(str, "BLOCK")) h->op = &block_operations;
	else if (!_strcasecmp(str, "NULL")) h->op = &null_operations;
	else if (!_strcasecmp(str, "NULLTTY")) h->op = &nulltty_operations;
	else if (!_strcasecmp(str, "ZERO")) h->op = &zero_operations;
	else return __ERR_PTR(-ENOENT);
	return NULL;
}

static void *block_lookup(HANDLE *h, char *str, int open_flags)
{
	if (__unlikely(!tty_lookup(h, str))) return NULL;
	return (void *)1;
}

static void *block_create(HANDLE *handle, char *filename, int open_flags)
{
	return &block_wait;
}

static void *block_delete(HANDLE *h, IORQ *rq, int open_flags, HANDLE *hp)
{
	WQ_WAIT_F(&block_wait, rq);
	return (void *)1;
}

static void *block_rename(HANDLE *h, char *str, HANDLE *fh, IORQ *rq)
{
	return &block_wait;
}

static void block_lookup_io(HANDLE *h, char *str, IORQ *rq, int open_flags)
{
	WQ_WAIT_F(&block_wait, rq);
}

static void init_root(HANDLE *h, void *data)
{
	h->op = &syscalls_operations;
	h->flags = K_XLATE | TTYF_O_NL_CRNL | TTYF_DEFAULT_LEDS;
}

static void block_detach(HANDLE *h)
{
	WQ_WAKE_ALL(&block_wait);
}

static DECL_IOCALL(block_read, SPL_DEV, SIORQ)
{
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op != &block_operations) && __unlikely(h->op != &nulltty_operations))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	if (__unlikely(!RQ->v.len)) {
		if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
		else RQ->status = -EOVERFLOW;
		RETURN_AST(RQ);
	}
	WQ_WAIT_F(&block_wait, RQ);
	RETURN;
}

static DECL_IOCALL(block_write, SPL_DEV, SIORQ)
{
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op != &block_operations))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	WQ_WAIT_F(&block_wait, RQ);
	RETURN;
}

static DECL_IOCALL(block_aread, SPL_DEV, AIORQ)
{
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op != &block_operations) && __unlikely(h->op != &nulltty_operations))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_AREAD);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_AREAD;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	if (__unlikely(!RQ->v.len)) {
		if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
		else RQ->status = -EOVERFLOW;
		RETURN_AST(RQ);
	}
	WQ_WAIT_F(&block_wait, RQ);
	RETURN;
}

static DECL_IOCALL(block_awrite, SPL_DEV, AIORQ)
{
	HANDLE *h = RQ->handle;
	if (__unlikely(h->op != &block_operations))
		RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_AWRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_AWRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	WQ_WAIT_F(&block_wait, RQ);
	RETURN;
}

static DECL_IOCALL(block_ioctl, SPL_DEV, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	int nulltty;
	union {
		struct tty_window_size winsize;
		struct mouse_info m;
	} u;
	if (__likely(h->op == &block_operations)) nulltty = 0;
	else if (__likely(h->op == &nulltty_operations)) nulltty = 1;
	else RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	switch (RQ->ioctl) {
		case IOCTL_NOP:
		case IOCTL_FSYNC:
		case IOCTL_NREAD: {
			ret_z:
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_STAT:
		case IOCTL_STATFS:
			/* perl lseeks on stdin */
		case IOCTL_LSEEK:
		case IOCTL_POLL:
			/* bash calls getpeername on handle 0 */
		case IOCTL_GETPEERNAME: {
			enoop:
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
		case IOCTL_SELECT_WRITE: {
			if (nulltty) {
				goto ret_z;
			}
			/* fall through */
		}
		case IOCTL_SELECT_READ: {
			if (RQ->param) goto wait;
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_TERM_FLAGS: {
			RQ->status = h->flags;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_FLUSH:
		case IOCTL_TTY_PEEK:
		case IOCTL_TTY_NTAB:
		case IOCTL_TTY_INCREASE_DETACH_SEQ:
		case IOCTL_TTY_ERASE_CLIPBOARD:
		{
			goto ret_z;
		}
		case IOCTL_TTY_GET_SPEED: {
			RQ->status = -EOPNOTSUPP;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_WINSIZE: {
			int r;
			memset(&u.winsize, 0, sizeof u.winsize);
			u.winsize.ws_col = 80;
			u.winsize.ws_row = 25;
			u.winsize.ws_xpixel = u.winsize.ws_col * 8;
			u.winsize.ws_ypixel = u.winsize.ws_row * 16;
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.winsize, sizeof u.winsize)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_MOUSE_INFO: {
			int r;
			u.m.buttons = 2;
			u.m.wheels = 0;
			u.m.axes = 2;
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.m, sizeof u.m)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_CONTROL:
		case IOCTL_TTY_WAIT_CONTROL:
		case IOCTL_TTY_WAIT_WINSIZE:
		case IOCTL_TTY_GETVIDEOMODE:
		case IOCTL_TTY_WAITACTIVE:
		case IOCTL_TTY_SETPALETTE:
		case IOCTL_TTY_SETDISPLAYSTART:
		case IOCTL_TTY_WAITRETRACE_SETDISPLAYSTART:
		case IOCTL_TTY_GET_PIXEL_CLOCK:
		case IOCTL_TTY_AVAIL_ACCEL:
		case IOCTL_TTY_GET_FONT_MODE:
		{
			goto wait;
		}
		case IOCTL_TTY_GETKBDMODE:
		{
			RQ->status = h->flags & TTYF_KBDMODE;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GETLED:
		{
			if (!(h->flags & TTYF_DEFAULT_LEDS)) {
				RQ->status = (h->flags & TTYF_LEDS) >> __BSF_CONST(TTYF_LEDS);
			} else {
				RQ->status = 0;
			}
			RETURN_AST(RQ);
		}
	}
	if (nulltty) goto enoop;
	wait:
	/*__debug_printf("block (%d,%d): %x\n", h->handle_num & 0xffffff, nulltty, RQ->ioctl);*/
	WQ_WAIT_F(&block_wait, RQ);
	RETURN;
}


static void *hole_lookup(HANDLE *h, char *str, int open_flags)
{
	if (__unlikely(!tty_lookup(h, str))) return NULL;
	return NULL;
}

static void *hole_create(HANDLE *handle, char *filename, int open_flags)
{
	return NULL;
}

static void *hole_delete(HANDLE *h, IORQ *rq, int open_flags, HANDLE *hp)
{
	return NULL;
}

static DECL_IOCALL(hole_write, SPL_DEV, SIORQ)
{
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	RQ->v.ptr += RQ->v.len;
	RQ->progress += RQ->v.len;
	RQ->v.len = 0;
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static DECL_IOCALL(hole_awrite, SPL_DEV, AIORQ)
{
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_AWRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	if (RQ->pos != -(_u_off_t)1) RQ->pos += RQ->v.len;
	RQ->v.ptr += RQ->v.len;
	RQ->progress += RQ->v.len;
	RQ->v.len = 0;
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static DECL_IOCALL(zero_read, SPL_DEV, SIORQ)
{
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	if (RQ->v.len) {
		VBUF vbuf;
		long s;
		again:
		vbuf.ptr = zpage;
		vbuf.len = __PAGE_CLUSTER_SIZE;
		vbuf.spl = SPL_X(SPL_DEV);
		RAISE_SPL(SPL_VSPACE);
		if (__unlikely(!(s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf)))) {
			DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
		}
		RQ->progress += s;
		SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
		TEST_LOCKUP_LOOP(RQ, RETURN);
		if (RQ->v.len) goto again;
	}
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static DECL_IOCALL(zero_aread, SPL_DEV, AIORQ)
{
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_AREAD;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	if (RQ->v.len) {
		VBUF vbuf;
		long s;
		again:
		vbuf.ptr = zpage;
		vbuf.len = __PAGE_CLUSTER_SIZE;
		vbuf.spl = SPL_X(SPL_DEV);
		RAISE_SPL(SPL_VSPACE);
		if (__unlikely(!(s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf)))) {
			DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
		}
		RQ->progress += s;
		SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
		TEST_LOCKUP_LOOP(RQ, RETURN);
		if (RQ->v.len) goto again;
	}
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

static DECL_IOCALL(zero_ioctl, SPL_DEV, IOCTLRQ)
{
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	switch (RQ->ioctl) {
		case IOCTL_CAN_MMAP:
			RQ->status = __likely(!RQ->param) ? 0 : -EPERM;
			RETURN_AST(RQ);
		default:
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
	}
}

static PAGE *zero_get_page(HANDLE *h, __v_off idx, int wr)
{
	if (__unlikely(wr & ~PF_RW)) return __ERR_PTR(-EINVAL);
	if (__unlikely(wr & PF_WRITE)) return __ERR_PTR(-EPERM);
	return zp;
}

#define DEVICE_NAME	"SYS$NULL"

int main(int argc, char *argv[])
{
	int r;
	if (argc > 1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NULL: SYNTAX ERROR");
		return -EBADSYN;
	}
	retry_alloc:
	zpage = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
	if (__unlikely(!zpage)) {
		int r = WQ_WAIT_SYNC_CANCELABLE(&KERNEL$FREEMEM_WAIT);
		if (__unlikely(r)) {
			if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "NULL: CAN'T ALLOCATE ZERO PAGE");
			return r;
		}
		goto retry_alloc;
	}
	zp = KERNEL$PHYS_2_PAGE(KERNEL$VIRT_2_PHYS(zpage));
	memset(zpage, 0, __PAGE_CLUSTER_SIZE);
	KERNEL$VM_PREPARE_PAGE_FOR_MMAP(zp);

	r = KERNEL$REGISTER_DEVICE(DEVICE_NAME, "NULL.SYS", LNTE_PUBLIC, NULL, init_root, NULL, NULL, NULL, unload, &lnte, NULL);
	if (r < 0) {
		KERNEL$FREE_KERNEL_PAGE(zpage, VM_TYPE_WIRED_MAPPED);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, DEVICE_NAME ": COULD NOT REGISTER DEVICE: %s", strerror(-r));
		return r;
	}
	strcpy(KERNEL$ERROR_MSG(), DEVICE_NAME);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;
}

static int unload(void *p, void **release, const char * const argv[])
{
	int r;
	WQ *wq;
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	WQ_WAKE_ALL(&block_wait);
	while (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(zp)) != NULL)) {
		WQ_WAIT_SYNC(wq);
		LOWER_SPL(SPL_ZERO);
	}
	LOWER_SPL(SPL_ZERO);
	KERNEL$FREE_KERNEL_PAGE(zpage, VM_TYPE_WIRED_MAPPED);
	*release = dlrq;
	return 0;
}

