#include <SPAD/AC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <SPAD/WQ.H>
#include <SPAD/THREAD.H>
#include <SPAD/TIMER.H>
#include <SYS/UTSNAME.H>
#include <SPAD/IOCTL.H>
#include <STRING.H>
#include <STDLIB.H>
#include <LIMITS.H>
#include <SPAD/CONSOLE.H>

static void *lnte, *dlrq;

static int outstanding;

static int unload(void *p, void **release, const char * const argv[]);

static void *syscalls_lookup(HANDLE *h, char *str, int open_flags);
static IO_STUB syscalls_ioctl;
static long syscall_thread(void *null);
static AST_STUB syscall_done;

static IO_STUB dmesg_read;

static const HANDLE_OPERATIONS syscalls_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	syscalls_lookup,	/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* READ */
	KERNEL$NO_OPERATION,	/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	syscalls_ioctl,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static const HANDLE_OPERATIONS dmesg_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	NULL,			/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	dmesg_read,		/* READ */
	KERNEL$NO_OPERATION,	/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	KERNEL$NO_OPERATION,	/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static void *syscalls_lookup(HANDLE *h, char *str, int open_flags)
{
	if (__likely(!_strcasecmp(str, "DMESG"))) h->op = &dmesg_operations;
	else return __ERR_PTR(-ENOENT);
	return NULL;
}

static void init_root(HANDLE *h, void *data)
{
	h->op = &syscalls_operations;
}

typedef struct {
	THREAD_RQ th;
	IOCTLRQ *rq;
	unsigned size;
	unsigned ioctl;
	unsigned long param;
	union {
		struct utsname u;
		unsigned long mem[KQM_N_ENTRIES];
	} u;
} SYSCALL;

static DECL_IOCALL(syscalls_ioctl, SPL_DEV, IOCTLRQ)
{
	unsigned size;
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	switch (RQ->ioctl) {
		case IOCTL_SYSCALLS_UNAME: {
			struct utsname u;
			int r;
			size = sizeof u;
			if (__unlikely(FEATURE_TEST(FEATURE_USERSPACE))) goto need_thread;
			if (__unlikely(r = uname(&u))) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u, sizeof u)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_SYSCALLS_QUERY_MEMORY: {
			unsigned long mem[KQM_N_ENTRIES];
			int r;
			size = sizeof mem;
			if (__unlikely(FEATURE_TEST(FEATURE_USERSPACE))) goto need_thread;
			if (__unlikely(r = KERNEL$QUERY_MEMORY(mem))) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &mem, sizeof mem)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
	if (0) need_thread: {
		SYSCALL *syscall = malloc(sizeof(SYSCALL));
		if (__unlikely(!syscall)) {
			KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$WAKE_IOCTL, sizeof(SYSCALL));
			RETURN;
		}
		syscall->th.fn = syscall_done;
		syscall->th.thread_main = syscall_thread;
		syscall->th.p = syscall;
		syscall->th.error = NULL;
		syscall->th.cwd = NULL;
		syscall->th.std_in = -1;
		syscall->th.std_out = -1;
		syscall->th.std_err = -1;
		syscall->th.dlrq = NULL;
		syscall->th.thread = NULL;
		syscall->th.spawned = 0;
		syscall->rq = RQ;
		syscall->size = size;
		syscall->ioctl = RQ->ioctl;
		syscall->param = RQ->param;
		outstanding++;
		RETURN_IORQ_CANCELABLE(&syscall->th, KERNEL$THREAD, RQ);
	}
}

static long syscall_thread(void *syscall_)
{
	SYSCALL *syscall = syscall_;
	int r;
	switch (syscall->ioctl) {
		case IOCTL_SYSCALLS_UNAME:
			r = uname(&syscall->u.u);
			break;
		case IOCTL_SYSCALLS_QUERY_MEMORY:
			r = KERNEL$QUERY_MEMORY(syscall->u.mem);
			break;
		default:
			KERNEL$SUICIDE("syscall_thread: INVALID IOCTL %X", syscall->ioctl);
	}
	return r;
}

static DECL_AST(syscall_done, SPL_DEV, THREAD_RQ)
{
	SYSCALL *syscall = GET_STRUCT(RQ, SYSCALL, th);
	IOCTLRQ *rq;
	int r;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), syscall->rq);
	rq = syscall->rq;
	outstanding--;
	if (__likely(!(r = syscall->th.status))) {
		if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(rq, &syscall->u, syscall->size)) == 1)) {
			free(syscall);
			DO_PAGEIN(rq, &rq->v, PF_WRITE);
		}
	}
	free(syscall);
	rq->status = r;
	RETURN_AST(rq);
}

static DECL_IOCALL(dmesg_read, SPL_DEV, SIORQ)
{
	vspace_unmap_t *funmap;
	VBUF vbuf;
	FFILE *file;
	__u64 pos;
	unsigned long l, s;

	static char buffer[1024];

	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	file = KERNEL$MAP_FILE_STRUCT(RQ->handle, (IORQ *)RQ, &funmap);
	if (__unlikely(!file)) RETURN;

	if (!RQ->v.len) goto ret;

	again:

	pos = (__u64)file->pos;
	l = KERNEL$CONSOLE_READ(buffer, sizeof buffer, &pos);
	if (!l) goto ret;

	vbuf.ptr = buffer;
	vbuf.len = l;
	vbuf.spl = SPL_X(SPL_DEV);
	RAISE_SPL(SPL_VSPACE);
	if (__unlikely(!(s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf)))) {
		funmap(file);
		DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	}

	RQ->progress += s;
	pos -= vbuf.len - s;
	file->pos = pos;

	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_DEV));
	TEST_LOCKUP_LOOP(RQ, {
		funmap(file);
		RETURN;
	});
	if (RQ->v.len) goto again;
	ret:
	funmap(file);
	if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
	else RQ->status = -EOVERFLOW;
	RETURN_AST(RQ);
}

#define DEVICE_NAME	"SYS$SYSCALLS"

int main(int argc, char *argv[])
{
	int r;
	if (argc > 1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SYSCALLS: SYNTAX ERROR");
		return -EBADSYN;
	}
	outstanding = 0;

	r = KERNEL$REGISTER_DEVICE(DEVICE_NAME, "SYSCALLS.SYS", LNTE_PUBLIC, NULL, init_root, NULL, NULL, NULL, unload, &lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, DEVICE_NAME ": COULD NOT REGISTER DEVICE: %s", strerror(-r));
		return r;
	}
	strcpy(KERNEL$ERROR_MSG(), DEVICE_NAME);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;
}

static int unload(void *p, void **release, const char * const argv[])
{
	int r;
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(lnte, argv))) return r;
	while (__unlikely(outstanding)) KERNEL$SLEEP(1);
	*release = dlrq;
	return 0;
}

