#include <SPAD/AC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/TTY.H>
#include <SYS/TYPES.H>
#include <VALUES.H>
#include <SPAD/LIBC.H>
#include <STRING.H>
#include <SPAD/IOCTL.H>
#include <TERMIOS.H>
#include <STDLIB.H>

#define IN_BUF		16

#define OUT_BUF		2048

struct tty_area {
	TTY tty;
	int kbd_h;
	int scr_h;
	SIORQ kbd_rq;
	SIORQ scr_rq;
	int state;
	char in_buf[IN_BUF];
	char dev_name[__MAX_STR_LEN];
	void *lnte;
	void *dlrq;
};

#define TTY_KBD_PENDING		1
#define TTY_SCR_PENDING		2

static int unload(void *p, void **release, const char * const argv[]);
static void TTY_WRITEOUT(TTY *tty);

static AST_STUB TTY_IN;
static AST_STUB TTY_OUT;

int main(int argc, const char * const argv[])
{
	int lines = -1, columns = -1;
	int xpixels = 0, ypixels = 0;
	char *kbd = NULL, *scr = NULL, *kbdscr = NULL, *name = NULL;
	const char * const *arg;
	int state;
	struct tty_window_size ws;
	struct tty_area *tty;
	union {
		IOCTLRQ ioctlrq;
		MALLOC_REQUEST mrq;
		OPENRQ openrq;
	} u;
	int r;
	char *xname, *xx;
	static const struct __param_table params[8] = {
		"KBD", __PARAM_STRING, 1, __MAX_STR_LEN,
		"SCR", __PARAM_STRING, 1, __MAX_STR_LEN,
		"KBDSCR", __PARAM_STRING, 1, __MAX_STR_LEN,
		"TTY", __PARAM_STRING, 1, __MAX_STR_LEN,
		"BUFSIZE", __PARAM_INT, __MAX_STR_LEN, MAXINT,
		"LINES", __PARAM_INT, 1, 1024 + 1,
		"COLUMNS", __PARAM_INT, 1, 1024 + 1,
		NULL, 0, 0, 0,
	};
	void *vars[8];
	u.mrq.size = sizeof(struct tty_area);
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		r = u.mrq.status;
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT ALLOCATE TTY STRUCTURE: %s", strerror(-u.mrq.status));
		goto err0;
	}
	tty = u.mrq.ptr;
	memset(tty, 0, sizeof(struct tty_area));
	tty->tty.buf_size = OUT_BUF;
	vars[0] = &kbd;
	vars[1] = &scr;
	vars[2] = &kbdscr;
	vars[3] = &name;
	vars[4] = &tty->tty.buf_size;
	vars[5] = &lines;
	vars[6] = &columns;
	vars[7] = NULL;
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: SYNTAX ERROR");
		r = -EBADSYN;
		goto err1;
	}
	if (kbdscr) {
		if (kbd || scr) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: CAN'T SET BOTH /KBDSCR AND /KBD OR /SCR");
			r = -EBADSYN;
			goto err1;
		}
		kbd = scr = kbdscr;
	}
	if (!kbd || !scr || !name) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: /KBD, /SCR AND /TTY MUST BE SET");
		r = -EBADSYN;
		goto err1;
	}
	strcpy(tty->dev_name, name);
	if (tty->tty.buf_size < __MAX_STR_LEN || tty->tty.buf_size > __PAGE_CLUSTER_SIZE || (tty->tty.buf_size & (tty->tty.buf_size - 1))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: BAD BUFFER SIZE (ALLOWED POWER OF 2 FROM: %u TO %u)", __MAX_STR_LEN, (unsigned)__PAGE_CLUSTER_SIZE);
		r = -EINVAL;
		goto err1;
	}
	u.mrq.size = tty->tty.buf_size;
	SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
	if (u.mrq.status < 0) {
		if (u.mrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT ALLOCATE TTY BUFFER: %s", strerror(u.mrq.status));
		r = u.mrq.status;
		goto err1;
	}
	tty->tty.buf = u.mrq.ptr;
	tty->tty.writeout = TTY_WRITEOUT;
	tty->tty.dtty = NULL;
	if ((r = TTY$CREATE(&tty->tty))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT CREATE TTY: %s", strerror(-r));
		goto err2;
	}
	u.openrq.flags = O_RDONLY;
	u.openrq.path = kbd;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT OPEN %s: %s", u.openrq.path, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto err2;
	}
	tty->kbd_h = u.openrq.status;
	u.openrq.flags = O_WRONLY;
	u.openrq.path = scr;
	u.openrq.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&u.openrq, KERNEL$OPEN);
	if (u.openrq.status < 0) {
		if (u.openrq.status != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT OPEN %s: %s", u.openrq.path, strerror(-u.openrq.status));
		r = u.openrq.status;
		goto err3;
	}
	tty->scr_h = u.openrq.status;
	if (lines == -1 || columns == -1) {
		u.ioctlrq.h = tty->scr_h;
		u.ioctlrq.ioctl = IOCTL_TTY_GET_WINSIZE;
		u.ioctlrq.param = 0;
		u.ioctlrq.v.ptr = (unsigned long)&ws;
		u.ioctlrq.v.len = sizeof ws;
		u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&u.ioctlrq, KERNEL$IOCTL);
		if (u.ioctlrq.status >= 0) {
			if (lines == -1) lines = ws.ws_row, ypixels = ws.ws_ypixel;
			if (columns == -1) columns = ws.ws_col, xpixels = ws.ws_xpixel;
		}
	}
	if (columns != -1) tty->tty.xsize = columns;
	if (lines != -1) tty->tty.ysize = lines;
	if (columns != -1 && lines != -1) tty->tty.bits |= TTY_LOCK_WINDOW_SIZE;
	if (xpixels) tty->tty.xpixels = xpixels;
	if (ypixels) tty->tty.ypixels = ypixels;

	u.ioctlrq.h = tty->kbd_h;
	u.ioctlrq.ioctl = IOCTL_TTY_GET_SPEED;
	u.ioctlrq.param = PARAM_ISPEED;
	u.ioctlrq.v.ptr = 0;
	u.ioctlrq.v.len = 0;
	u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&u.ioctlrq, KERNEL$IOCTL);
	if (u.ioctlrq.status >= 0) tty->tty.ispeed = u.ioctlrq.status;

	u.ioctlrq.h = tty->scr_h;
	u.ioctlrq.ioctl = IOCTL_TTY_GET_SPEED;
	u.ioctlrq.param = PARAM_OSPEED;
	u.ioctlrq.v.ptr = 0;
	u.ioctlrq.v.len = 0;
	u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&u.ioctlrq, KERNEL$IOCTL);
	if (u.ioctlrq.status >= 0) tty->tty.ospeed = u.ioctlrq.status;

	u.ioctlrq.h = tty->scr_h;
	u.ioctlrq.ioctl = IOCTL_CONSOLE_VC;
	u.ioctlrq.param = 0;
	u.ioctlrq.v.ptr = 0;
	u.ioctlrq.v.len = 0;
	u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&u.ioctlrq, KERNEL$IOCTL);
	if (u.ioctlrq.status < 0) goto skip_dctl;
	tty->tty.vc = u.ioctlrq.status;

	xname = alloca(strlen(scr) + 1);
	strcpy(xname, scr);
	xx = strchr(xname, ':');
	if (__unlikely(!xx)) goto skip_dctl;
	*xx = 0;
	if ((r = KERNEL$DCALL(xname, "CONSOLE", CONSOLE_CMD_GET + tty->tty.vc, &tty->tty.vmode))) goto skip_dctl;

	skip_dctl:

	u.ioctlrq.h = tty->scr_h;
	u.ioctlrq.ioctl = IOCTL_CONSOLE_VC;
	u.ioctlrq.param = 0;
	u.ioctlrq.v.ptr = 0;
	u.ioctlrq.v.len = 0;
	u.ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&u.ioctlrq, KERNEL$IOCTL);
	if (u.ioctlrq.status < 0) goto skip_kdctl;
	tty->tty.kc = u.ioctlrq.status;

	xname = alloca(strlen(kbd) + 1);
	strcpy(xname, kbd);
	xx = strchr(xname, ':');
	if (__unlikely(!xx)) goto skip_kdctl;
	*xx = 0;
	if ((r = KERNEL$DCALL(xname, "KEYBOARD", KEYBOARD_CMD_GET + tty->tty.vc, &tty->tty.kmode))) goto skip_kdctl;

	skip_kdctl:

	r = KERNEL$REGISTER_DEVICE(tty->dev_name, "CTTY.SYS", 0, &tty->tty, TTY$INIT_ROOT, NULL, NULL, NULL, unload, &tty->lnte, KERNEL$HANDLE_PATH(tty->kbd_h), KERNEL$HANDLE_PATH(tty->scr_h), NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CTTY: COULD NOT REGISTER DEVICE %s: %s", tty->dev_name, strerror(-r));
		if (tty->tty.vmode) tty->tty.vmode(VIDEOMODE_RELEASE, tty->tty.vc);
		if (tty->tty.kmode) tty->tty.kmode(KBD_RELEASE, tty->tty.kc);
		goto err4;
	}


	tty->scr_rq.fn = &TTY_OUT;
	tty->scr_rq.h = tty->scr_h;
	tty->scr_rq.v.vspace = &KERNEL$VIRTUAL;

	tty->kbd_rq.fn = &TTY_IN;
	tty->kbd_rq.h = tty->kbd_h;
	tty->kbd_rq.v.ptr = (unsigned long)tty->in_buf;
	tty->kbd_rq.v.vspace = &KERNEL$VIRTUAL;
	tty->kbd_rq.v.len = IN_BUF;
	tty->kbd_rq.progress = 0;
	tty->state |= TTY_KBD_PENDING;
	CALL_IORQ(&tty->kbd_rq, KERNEL$READ);

	strcpy(KERNEL$ERROR_MSG(), tty->dev_name);
	tty->dlrq = KERNEL$TSR_IMAGE();
	return 0;

	err4:
	KERNEL$FAST_CLOSE(tty->scr_h);

	err3:
	KERNEL$FAST_CLOSE(tty->kbd_h);

	err2:
	free(tty->tty.buf);

	err1:
	free(tty);

	err0:
	return r;
}

static void TTY_WRITEOUT(TTY *ttyp)
{
	struct tty_area *tty = GET_STRUCT(ttyp, struct tty_area, tty);
	tty->scr_rq.v.ptr = (unsigned long)ttyp->buffer_to_write;
	tty->scr_rq.v.len = (unsigned long)ttyp->buffer_to_write_len;
	if (__unlikely(!ttyp->buffer_to_write_len))
		KERNEL$SUICIDE("TTY_WRITEOUT: ZERO BYTES");
	tty->scr_rq.progress = 0;
	if (__unlikely(tty->state & TTY_SCR_PENDING))
		KERNEL$SUICIDE("TTY_WRITEOUT: CALLED TWICE");
	tty->state |= TTY_SCR_PENDING;
	CALL_IORQ(&tty->scr_rq, KERNEL$WRITE);
	return;
}

static DECL_AST(TTY_OUT, SPL_TTY, SIORQ)
{
	struct tty_area *tty = GET_STRUCT(RQ, struct tty_area, scr_rq);
	if (__likely(RQ->status > 0) && __unlikely(tty->scr_rq.v.len)) {
		tty->scr_rq.progress = 0;
		RETURN_IORQ(&tty->scr_rq, KERNEL$WRITE);
	}
	tty->state &= ~TTY_SCR_PENDING;
	TTY$WRITE_DONE(&tty->tty);
	RETURN;
}

static DECL_AST(TTY_IN, SPL_TTY, SIORQ)
{
	int r;
	struct tty_area *tty = GET_STRUCT(RQ, struct tty_area, kbd_rq);
	if (__unlikely(RQ->status < 0)) {
		tty->state &= ~TTY_KBD_PENDING;
		RETURN;
	}
	tty_in_again:
	r = TTY$IN(&tty->tty, (char *)(unsigned long)tty->kbd_rq.v.ptr - tty->kbd_rq.progress, tty->kbd_rq.progress);
	if (__unlikely(!r)) {
#if 0
		if (tty->tty.mode & TTYF_RAW) {
/* without this, pasting into a slow application would be corrupted */
			tty->kbd_rq.status = RQS_PROCESSING;
			WQ_WAIT(&tty->tty.in_wait, RQ, KERNEL$SUCCESS);
			RETURN;
		}
#endif
/* ... and with that, we could miss control characters. Comment it out until
   a better solution is found */
	} else if (__unlikely(r < tty->kbd_rq.progress)) {
		tty->kbd_rq.progress -= r;
		goto tty_in_again;
	}
	tty->kbd_rq.v.ptr = (unsigned long)tty->in_buf;
	tty->kbd_rq.v.len = IN_BUF;
	tty->kbd_rq.progress = 0;
	RETURN_IORQ(&tty->kbd_rq, KERNEL$READ);
}

static int unload(void *p, void **release, const char * const argv[])
{
	int r;
	struct tty_area *tty = p;
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(tty->lnte, argv))) return r;
	RAISE_SPL(SPL_TTY);
	while (tty->state & TTY_KBD_PENDING) {
		KERNEL$CIO((IORQ *)(void *)&tty->kbd_rq);
		KERNEL$SLEEP(1);
	}
	while (tty->state & TTY_SCR_PENDING) {
		KERNEL$CIO((IORQ *)(void *)&tty->scr_rq);
		KERNEL$SLEEP(1);
	}
	TTY$DESTROY(&tty->tty);
	LOWER_SPL(SPL_ZERO);
	if (tty->tty.vmode) tty->tty.vmode(VIDEOMODE_RELEASE, tty->tty.vc);
	if (tty->tty.kmode) tty->tty.vmode(KBD_RELEASE, tty->tty.kc);
	KERNEL$FAST_CLOSE(tty->kbd_h);
	KERNEL$FAST_CLOSE(tty->scr_h);
	*release = tty->dlrq;
	free(tty->tty.buf);
	free(tty);
	return 0;
}

