#include <SYS/TYPES.H>
#include <SPAD/LIBC.H>
#include <SPAD/VM.H>
#include <SPAD/IOCTL.H>
#include <SYS/IOCTL.H>
#include <STRING.H>
#include <UNISTD.H>
#include <STDLIB.H>
#include <VALUES.H>
#include <SPAD/SLAB.H>
#include <SPAD/ALLOC.H>
#include <SPAD/TIMER.H>
#include <SYS/TIME.H>
#include <MD5.H>
#include <SPAD/DL.H>

#include <SYS/KD.H>
#include <SPAD/TTY.H>

#define TTY_UPDATE_TIME		(JIFFIES_PER_SECOND / 4)
#define TTY_WRITE_TIME		1

#define TTYIN_NORMAL	0
#define TTYIN_NEXTRAW	1
#define TTYIN_ESC	2
#define TTYIN_SQUARE	3
#define TTYIN_SQUARE2	4

struct blob_ref {
	PROC *proc;
	FBLOB *blob;
	size_t blob_size;
	char set_in_this_handle;
};

struct tty_extra_params {
	struct tty_videomode_params vm;
	struct tty_keyboard_params kbd;
	struct blob_ref font;
	struct blob_ref keymap;
};

static /*const*/ struct tty_extra_params default_params = { 0 };
static struct __slhead tty_extra_params[2];

/* return 0 - ok, 1 - more chars needed.
   It MUST NEVER return 1 when end - ptr >= __MAX_BYTES_IN_CHAR
*/

#define ABSOLUTE_POS	0x10000
#define INCREASE_POS	0x20000
#define DECREASE_POS	0x40000

static int get_char_len(const char *ptr, const char *end, int *bytes, int *columns, int input)
{
	const char *p;
	if (__unlikely(ptr == end)) return 1;
	if (__unlikely(!*ptr)) {
		*bytes = 1;
		*columns = 0;
		return 0;
	}
	if (input && __unlikely((unsigned char)*ptr < 32)) {
		*bytes = 1;
		*columns = 2;
		return 0;
	}
	if (__likely(*ptr != 27)) {
		/* use charset DLL function here when non-1byte characters will be implemented */
		*bytes = *columns = 1;
		return 0;
	}
	p = ptr + 1;
	if (__unlikely(p == end)) return 1;
	if (__likely(*p == '[')) {
		unsigned int par[2] = { 0, 0 };
		unsigned int npar = 0;
		int ques = 0;
		if (__likely(p + 1 != end) && p[1] == '?') ques = 1, p++;
		while (1) {
			p++;
			if (p + 1 - ptr > __MAX_BYTES_IN_CHAR) {
				*bytes = 1;
				goto c;
			}
			if (p == end) return 1;
			if (*p >= '0' && *p <= '9') {
				if (__likely(npar < 2)) {
					par[npar] = par[npar] * 10 + *p - '0';
				}
				continue;
			}
			if (*p == ';') {
				if (__likely(npar < 2)) npar++;
				continue;
			}
			break;
		}
		*bytes = p + 1 - ptr;
		if (__likely(!ques)) {
			if (__likely(*p == 'H') || __unlikely(*p == 'f')) {
				abs_p1:
				if (__likely(par[1])) {
					par[1]--;
					if (__unlikely(par[1] > 0x7ffe)) par[1] = 0x7fff;
				}
				*columns = ABSOLUTE_POS | par[1];
				return 0;
			}
			if (__unlikely(*p == 'G') || __unlikely(*p == '`')) {
				par[1] = par[0];
				goto abs_p1;
			}
			if (*p == 'C' || *p == 'D' || __unlikely(*p == 'a')) {
				if (!par[0]) par[0] = 1;
				if (__unlikely(par[0] > 0x7ffe)) par[0] = 0x7fff;
				*columns = (*p == 'D' ? DECREASE_POS : INCREASE_POS) | par[0];
				return 0;
			}
			if (__unlikely(*p == 'E') || __unlikely(*p == 'F') || __unlikely(*p == 'r')) goto abs0;
		} else {
			if (__likely(*p == 'h') || __likely(*p == 'l')) goto abs0;
		}
		goto c;
	}
	*bytes = 2;
	if (*p == 'c') {
		abs0:
		*columns = ABSOLUTE_POS | 0;
		return 0;
	}
	c:
	*columns = 0;
	return 0;
}

static void get_char_len_buf(TTY *tty, int buf_ptr, int xpos, int *bytes, int *columns, char **ptr, int *strl)
{
	if (__unlikely(buf_ptr == tty->buf_end)) goto nul;
	if (__unlikely((unsigned char)tty->buf[buf_ptr] < 32)) goto ctrl;
	if (!get_char_len(*ptr = tty->buf + buf_ptr, tty->buf + tty->buf_end, bytes, columns, 1)) {
		*strl = *bytes;
		return;
	}
	*bytes = *columns = *strl = tty->buf_end - buf_ptr;
	return;
	nul:
	*bytes = *columns = *strl = 0;
	*ptr = NULL;
	return;
	ctrl:
	*bytes = 1;
	if (tty->buf[buf_ptr] == 10) {
		*ptr = NULL;
		*columns = xpos >= 0 ? 10000 : 0;
		*strl = 0;
		return;
	} else if (tty->buf[buf_ptr] == 9) {
		char *t = "        ";
		if (xpos == -1) xpos = 0;
		*ptr = t;
		*columns = *strl = 8 - (xpos & 7);
		return;
	} else {
		static char t[2] = "^ ";
		t[1] = tty->buf[buf_ptr] + '@';
		*ptr = t;
		*columns = *strl = 2;
		return;
	}
}

static __finline__ void check_redzone(TTY *tty, const char *str)
{
	if (__likely(tty->buffer_len) && __unlikely(tty->buffer[tty->buffer_len] != 'R')) KERNEL$SUICIDE("TTY CODE IS BROKEN -- REDZONE DAMAGED WITH %d AT %s", (int)(unsigned char)tty->buffer[tty->buffer_len], str);
	/*else __critical_printf("OK: %s\n", str);*/
}

static __finline__ void tty_alloc_buffer(TTY *tty)
{
	char *p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED);
	if (__likely(p != NULL)) {
		p[__PAGE_CLUSTER_SIZE_MINUS_1] = 'R';	/* red zone */
		tty->buffer = p;
		tty->buffer_len = __PAGE_CLUSTER_SIZE_MINUS_1;
		tty->buffer_ptr = 0;
	} else if (tty->buffer_to_write != tty->nomem_buffer) {
		tty->nomem_buffer[__TTY_NOMEM_BUFFER_SIZE] = 'R'; /* red zone */
		tty->buffer = tty->nomem_buffer;
		tty->buffer_len = __TTY_NOMEM_BUFFER_SIZE;
		tty->buffer_ptr = 0;
	}
	KERNEL$DEL_TIMER(&tty->write_timer);
	KERNEL$SET_TIMER(TTY_WRITE_TIME, &tty->write_timer);
}

static void tty_force_writeout(TTY *tty)
{
	if (!tty->buffer_to_write && __likely(tty->buffer_ptr)) {
		check_redzone(tty, "tty_force_writeout");
		tty->buffer_to_write = tty->buffer;
		tty->buffer_to_write_len = tty->buffer_ptr;
		tty->buffer = NULL;
		tty->buffer_len = 0;
		tty->buffer_ptr = 0;
		tty->bits |= TTY_IN_WRITEOUT;
		tty->writeout(tty);
		tty->bits &= ~TTY_IN_WRITEOUT;
	}
}

static int tty_check_space_failed(TTY *tty, int space)
{
	tty_force_writeout(tty);
	if (__likely(!tty->buffer)) {
		tty_alloc_buffer(tty);
	}
	return tty->buffer_ptr + space > tty->buffer_len;
}

#define tty_check_space(tty, space)					\
	(__unlikely((tty)->buffer_ptr + (space) > (tty)->buffer_len) ?	\
		tty_check_space_failed(tty, space) :			\
		0)

static void find_position(TTY *tty, int start, int pos, int xoff, int *xp, int *yp, int (*fn)(int p, int x, int y), int e)
{
	int x = xoff;
	int y = 0;
	/*__debug_printf("find_position(%d, %d, end %d)", start, pos, tty->buf_end);*/
	while (start <= pos - e) {
		int by, co, strl;
		char *ptr;
		get_char_len_buf(tty, start, x, &by, &co, &ptr, &strl);
		if (__unlikely(x + co >= tty->xsize)) {
			if (!strl) {
				if (fn && fn(start, x, y)) return;
				if (start == pos) break;
			}
			get_char_len_buf(tty, start, -1, &by, &co, &ptr, &strl);
			x = 0, y++;
		}
		if (fn && strl && fn(start, x, y)) return;
		if (start == pos) break;
		x += co;
		start += by;
		if (x == tty->xsize - 1) x = 0, y++;
	}
	if (fn && fn(start, x, y)) return;
	*xp = x;
	*yp = y;
	/*__debug_printf("done\n");*/
}

static __finline__ void tty_forward_arrow(TTY *tty)
{
	int by, co, strl;
	char *ptr;
	get_char_len_buf(tty, tty->buf_cursor, 0, &by, &co, &ptr, &strl);
	tty->buf_cursor = tty->buf_cursor + by;
}

static void tty_back_arrow(TTY *tty)
{
	int o_cur = tty->buf_cursor;
	tty->buf_cursor = tty->buf_line;
	while (1) {
		int ccur = tty->buf_cursor;
		tty_forward_arrow(tty);
		if (tty->buf_cursor >= o_cur || tty->buf_cursor == ccur) {
			tty->buf_cursor = ccur;
			return;
		}
	}
}

static void tty_home(TTY *tty)
{
	tty->buf_cursor = tty->buf_line;
}

static void tty_end(TTY *tty)
{
	tty->buf_cursor = tty->buf_end;
}

static int u_cx, u_cy;
static int u_bp;
static int u_bd;

static int up_fn(int pos, int x, int y)
{
	if (y == u_cy) {
		int d = x > u_cx ? x - u_cx : u_cx - x;
		if (u_bp == -1 || d < u_bd) u_bp = pos, u_bd = d;
	}
	return y > u_cy;
}

static void tty_up(TTY *tty)
{
	find_position(tty, tty->buf_line, tty->buf_cursor, tty->xstart, &u_cx, &u_cy, NULL, 0);
	u_cy--, u_bp = -1;
	find_position(tty, tty->buf_line, tty->buf_cursor, tty->xstart, &u_cx, &u_cy, up_fn, 0);
	if (u_bp != -1) tty->buf_cursor = u_bp;
}

static void tty_down(TTY *tty)
{
	find_position(tty, tty->buf_line, tty->buf_cursor, tty->xstart, &u_cx, &u_cy, NULL, 0);
	u_cy++, u_bp = -1;
	find_position(tty, tty->buf_line, tty->buf_end, tty->xstart, &u_cx, &u_cy, up_fn, 0);
	if (u_bp != -1) tty->buf_cursor = u_bp;
}

static void tty_erase_buffer(TTY *tty)
{
	tty->buf_start = tty->buf_end = tty->buf_line = tty->buf_updateline = tty->buf_cursor = tty->buf_needupdate = 0;
	tty->bits &= ~TTY_PUTBACK;
	tty->text_ok = 0;
	tty->in_state = TTYIN_NORMAL;
	WQ_WAKE_ALL(&tty->in_wait);
}

static void tty_flush(TTY *tty)
{
	tty_erase_buffer(tty);
	tty->bits &= ~TTY_EOF;
}

static __finline__ int buffer_room(TTY *tty)
{
	return tty->buf_size - (tty->buf_end - tty->buf_start);
}

static void insert_in_buffer(TTY *tty, const char *str, int len)
{
	if (__unlikely(len > buffer_room(tty))) KERNEL$SUICIDE("OUT OF TTY BUFFER");
	if (tty->buf_cursor < tty->buf_needupdate) tty->buf_needupdate = tty->buf_cursor;
	if (__unlikely(tty->buf_end + len > tty->buf_size)) goto mv;
	b:
	memmove(tty->buf + tty->buf_cursor + len, tty->buf + tty->buf_cursor, tty->buf_end - tty->buf_cursor);
	memcpy(tty->buf + tty->buf_cursor, str, len);
	tty->buf_cursor += len;
	tty->buf_end += len;
	tty->text_ok = 0;
	return;
	mv:
	memmove(tty->buf, tty->buf + tty->buf_start, tty->buf_end - tty->buf_start);
	tty->buf_updateline -= tty->buf_start;
	tty->buf_needupdate -= tty->buf_start;
	tty->buf_line -= tty->buf_start;
	tty->buf_cursor -= tty->buf_start;
	tty->buf_end -= tty->buf_start;
	tty->buf_start = 0;
	goto b;
}

static __finline__ void del_from_buffer(TTY *tty, int ch)
{
	if (tty->buf_cursor < tty->buf_needupdate) tty->buf_needupdate = tty->buf_cursor;
	memmove(tty->buf + tty->buf_cursor, tty->buf + tty->buf_cursor + ch, tty->buf_end - tty->buf_cursor - ch);
	tty->buf_end -= ch;
	tty->text_ok = 0;
}

static void tty_del_char(TTY *tty)
{
	int by, co, strl;
	char *str;
	get_char_len_buf(tty, tty->buf_cursor, 0, &by, &co, &str, &strl);
	del_from_buffer(tty, by);
}

static int set_pos(TTY *tty, int x, int y, int can_bs)
{
	char *buf;
	int buflen;
	int xp;
	int t;
	int r;
	tty_check_space(tty, __TTY_NOMEM_BUFFER_SIZE);
	buf = tty->buffer + tty->buffer_ptr;
	buflen = tty->buffer_len - tty->buffer_ptr;
	xp = tty->xpos;
	/*__debug_printf("(%d->%d)", tty->xpos, x);*/
	if (y < tty->ypos) {
		if (y + 1 == tty->ypos) r = _snprintf(buf, buflen, "\033[A");
		else r = _snprintf(buf, buflen, "\033[%dA", tty->ypos - y);
	} else if (y > tty->ypos) {
		if (y - 1 == tty->ypos && x && x == xp) r = _snprintf(buf, buflen, "\033[B");
		else if (y - 1 == tty->ypos) r = _snprintf(buf, buflen, "\r\n"), __likely(r != -1) ? (xp = 0) : 0;
		else r = _snprintf(buf, buflen, "\033[%dB", y - tty->ypos);
	} else r = 0;
	if (__unlikely(r < 0)) {
		return -1;
	}
	t = r;
	buf += r;
	buflen -= r;
	if (x < xp) {
		if (x + 1 == xp) {
			if (can_bs) r = _snprintf(buf, buflen, "\010");
			else r = _snprintf(buf, buflen, "\033[D");
		}
		else if (can_bs && x + 2 == xp) r = _snprintf(buf, buflen, "\010\010");
		else if (can_bs && x + 3 == xp) r = _snprintf(buf, buflen, "\010\010\010");
		else r = _snprintf(buf, buflen, "\033[%dD", xp - x);
	} else if (x > xp) {
		if (x - 1 == xp) r = _snprintf(buf, buflen, "\033[C");
		else r = _snprintf(buf, buflen, "\033[%dC", x - xp);
	} else r = 0;
	if (__unlikely(r < 0)) {
		return -1;
	}
	tty->xpos = x;
	tty->ypos = y;
	tty->buffer_ptr += t + r;
	return 0;
}

static int tty_erase_to_xmax(TTY *tty)
{
	int x = tty->xpos, y = tty->ypos, xx = x, yy = y;
	while (tty->max_ypos > y) {
		if (x < tty->xsize - 2) {
			if (tty_check_space(tty, 6)) goto xxx;
			memcpy(tty->buffer + tty->buffer_ptr, "\033[0K\r\n", 6);
			tty->buffer_ptr += 6;
		} else if (x == tty->xsize - 2) {
			if (tty_check_space(tty, 3)) goto xxx;
			tty->buffer[tty->buffer_ptr++] = ' ';
			tty->buffer[tty->buffer_ptr++] = '\r';
			tty->buffer[tty->buffer_ptr++] = '\n';
		} else {
			if (tty_check_space(tty, 2)) goto xxx;
			tty->buffer[tty->buffer_ptr++] = '\r';
			tty->buffer[tty->buffer_ptr++] = '\n';
		}
		x = 0;
		y++;
		tty->xpos = x, tty->ypos = y;
	}
	if (y > tty->max_ypos) tty->max_xpos = x;
	if (tty->max_xpos > x) {
		if (tty->max_xpos == x + 1) {
			if (tty_check_space(tty, 2)) goto xxx;
			tty->buffer[tty->buffer_ptr++] = ' ';
			tty->buffer[tty->buffer_ptr++] = 8;
		} else if (tty->max_xpos == x + 2) {
			if (tty_check_space(tty, 4)) goto xxx;
			memcpy(tty->buffer + tty->buffer_ptr, "  \010\010", 4);
			tty->buffer_ptr += 4;
		} else {
			if (tty_check_space(tty, 4)) goto xxx;
			memcpy(tty->buffer + tty->buffer_ptr, "\033[0K", 4);
			tty->buffer_ptr += 4;
		}
	}
	tty->max_xpos = xx;
	tty->max_ypos = yy;
	return 0;

	xxx:
	return 1;
}

static void tty_erase_input(TTY *tty)
{
	if (!tty->input_displayed) return;
	if (tty->input_displayed == 1) {
		if (set_pos(tty, tty->xstart, 0, 1)) goto xxx;
	}
	tty->input_displayed = 2;
	tty->text_ok = 0;
	if (tty->buf_needupdate > tty->buf_line) tty->buf_needupdate = tty->buf_line;
	if (tty_erase_to_xmax(tty)) goto xxx;
	if (set_pos(tty, tty->xstart, 0, 1)) goto xxx;
	tty->input_displayed = 0;
	xxx:;
}

#define RSVD	10	/* ESC [ 0 K cr lf cr lf cr lf */

static void tty_update(TTY *tty)
{
	int x, y;
	if (__unlikely(!WQ_EMPTY(&tty->write_wait))) {
		KERNEL$DEL_TIMER(&tty->update_timer);
		KERNEL$SET_TIMER(TTY_UPDATE_TIME, &tty->update_timer);
		return;
	}
	if (__unlikely(tty->bits & TTY_PERMANENT_RAW)) goto real_raw;
	if (tty->mode & TTYF_NOECHO) {
		tty_erase_input(tty);
		real_raw:
		tty->buf_updateline = tty->buf_line;
		WQ_WAKE_ALL_PL(&tty->read_wait);
		return;
	}
	if (__unlikely(tty->bits & TTY_DELAY)) {
		if (tty_check_space(tty, 1)) goto x;
		tty->buffer[tty->buffer_ptr++] = 0;
		tty->bits &= ~TTY_DELAY;
	}
	if (tty->input_displayed == 2) {
		tty->input_displayed = 1;
	}
	if (!tty->input_displayed) {
		tty->input_displayed = 1;
		tty->xpos = tty->xstart;
		tty->ypos = 0;
		tty->buf_needupdate = tty->buf_updateline;
	}
	if (tty->buf_updateline < tty->buf_line && tty->buf_needupdate >= tty->buf_line) tty->buf_needupdate = tty->buf_updateline;
	if (tty->buf_needupdate < tty->buf_end) {
		tty->text_ok = 0;
		find_position(tty, tty->buf_updateline, tty->buf_needupdate, tty->xstart, &x, &y, NULL, 1);
		if (set_pos(tty, x, y, 1)) goto x;
		if (tty_check_space(tty, RSVD)) goto x;
		while (tty->buf_needupdate != tty->buf_end) {
			int by, co, strl;
			char *str;
			get_char_len_buf(tty, tty->buf_needupdate, x, &by, &co, &str, &strl);
			if (__unlikely(x + co >= tty->xsize)) {
				get_char_len_buf(tty, tty->buf_needupdate, -1, &by, &co, &str, &strl);
				if (x == tty->xsize - 1 || y > tty->max_ypos || (y == tty->max_ypos && x >= tty->max_xpos)) tty->buffer[tty->buffer_ptr++] = '\r', tty->buffer[tty->buffer_ptr++] = '\n';
				else memcpy(tty->buffer + tty->buffer_ptr, "\033[0K\r\n", 6), tty->buffer_ptr += 6;
				x = 0;
				y++;
			}
			if (tty_check_space(tty, 2 * strl + RSVD) || !by) {
				tty->xpos = x, tty->ypos = y;
				goto x;
			}
			if (__likely(strl == 1)) {
				if (__unlikely(str[0] == '\n')) tty->buffer[tty->buffer_ptr++] = '\r';
				tty->buffer[tty->buffer_ptr++] = str[0];
			} else {
				int i;
				for (i = 0; i < strl; i++) {
					if (__unlikely(str[i] == '\n')) tty->buffer[tty->buffer_ptr++] = '\r';
					tty->buffer[tty->buffer_ptr++] = str[i];
				}
			}
			if (__unlikely((x += co) == tty->xsize - 1)) {
				tty->buffer[tty->buffer_ptr++] = '\r';
				tty->buffer[tty->buffer_ptr++] = '\n';
				x = 0;
				y++;
			}
			if (tty->buf_needupdate < tty->buf_line && tty->buf_needupdate + by >= tty->buf_line) {
				find_position(tty, tty->buf_updateline, tty->buf_line, tty->xstart, &tty->xstart, (void *)&KERNEL$LIST_END, NULL, 0);
				tty->buf_updateline = tty->buf_line;
				WQ_WAKE_ALL_PL(&tty->read_wait);
				/*if (x) tty->buffer[tty->buffer_ptr++] = '\n', x = 0;*/
				tty->max_ypos -= y;
				y = 0;
				/*tty->xstart = 0;*/
			}
			tty->buf_needupdate += by;
			if (tty->buf_needupdate > tty->buf_end || !by) KERNEL$SUICIDE("TTY_UPDATE: %d > %d", tty->buf_needupdate, tty->buf_end);
		}
		tty->buf_needupdate = tty->buf_end;
		tty->xpos = x, tty->ypos = y;
	}
	if (tty->text_ok == 1) goto erase_to_end;
	if (tty->text_ok) goto update_cursor;
	find_position(tty, tty->buf_updateline, tty->buf_end, tty->xstart, &x, &y, NULL, 0);
	if (set_pos(tty, x, y, 1)) goto x;
	tty->text_ok = 1;
	erase_to_end:
	if (tty_erase_to_xmax(tty)) goto x;
	tty->text_ok = 2;
	update_cursor:
	find_position(tty, tty->buf_updateline, tty->buf_cursor, tty->xstart, &x, &y, NULL, 0);
	if (set_pos(tty, x, y, 0)) goto x;
	tty->text_ok = 3;
	x:
	check_redzone(tty, "tty_update");
}

static void tty_update_timer(TIMER *t)
{
	TTY *tty;
	LOWER_SPL(SPL_TTY);
	SET_TIMER_NEVER(t);
	tty = LIST_STRUCT(t, TTY, update_timer);
	if (!tty->input_displayed && WQ_EMPTY(&tty->write_wait) && tty->text_ok != 3) {
		tty_update(tty);
	}
}

static int is_input_break_char(TTY *tty, int i)
{
	if (__unlikely(tty->buf[i] == '\n') || __unlikely(tty->buf[i] == '\t')) return 1;
	if (__unlikely(tty->buf[i] == '\e') && __likely(i + 2 < tty->buf_end) && __likely(tty->buf[i + 1] == '[') && (__likely(tty->buf[i + 2] == 'A') || __likely(tty->buf[i + 2] == 'B'))) {
		return 3;
	}
	return 0;
}

static int tty_do_updatekeyboardmode(TTY *tty, struct tty_extra_params *ex, IORQ *rq)
{
	if (__unlikely(memcmp(&ex->kbd, &tty->current_keyboard_params, sizeof(struct tty_keyboard_params)))) {
		if (__likely(tty->kmode != NULL)) {
			tty->kmode(KBD_SET_KEYBOARD_RATE, tty->kc, &ex->kbd);
		}
		memcpy(&tty->current_keyboard_params, &ex->kbd, sizeof(struct tty_keyboard_params));
	}
	if (__unlikely(ex->keymap.proc != tty->keymap_proc) || __unlikely(ex->keymap.blob != tty->keymap_blob)) {
		if (__likely(tty->kmode != NULL)) {
			WQ *r;
			if (__unlikely((r = (WQ *)tty->kmode(KBD_SET_KEYMAP, tty->vc, ex->keymap.proc, ex->keymap.blob, ex->keymap.blob_size, rq)) != NULL)) {
				if (__IS_ERR(r)) {
					rq->status = __PTR_ERR(r);
					CALL_AST(rq);
				} else if (r != (void *)1) WQ_WAIT_F(r, rq);
				LOWER_SPL(SPL_TTY);
				return 1;
			}
		}
		tty->keymap_proc = ex->keymap.proc;
		tty->keymap_blob = ex->keymap.blob;
	}
	return 0;
}

static __finline__ int tty_updatekeyboardmode(HANDLE *h, TTY *tty, IORQ *rq)
{
	struct tty_extra_params *ex = (struct tty_extra_params *)h->flags2;
	if (__unlikely(memcmp(&ex->kbd, &tty->current_keyboard_params, sizeof(struct tty_keyboard_params))) || __unlikely(ex->keymap.proc != tty->keymap_proc) || __unlikely(ex->keymap.blob != tty->keymap_blob)) {
		return tty_do_updatekeyboardmode(tty, ex, rq);
	}
	return 0;
}

static void tty_do_updatemode(TTY *tty, int mode)
{
	if (__unlikely((mode & TTYF_KBDMODE) != (tty->mode & (TTYF_KBDMODE | TTYF_NEW)))) {
		if (__likely((mode & TTYF_KBDMODE) != (tty->mode & TTYF_KBDMODE)) && __likely(!(tty->bits & TTY_PERMANENT_RAW))) tty_flush(tty);
		if (__likely(tty->kmode != NULL)) {
			tty->kmode(KBD_CLEAR_BUFFER, tty->kc, mode & TTYF_KBDMODE);
			tty->kmode(KBD_SET_MODE, tty->kc, mode & TTYF_KBDMODE);
		}
	}
	if ((mode & (TTYF_LEDS | TTYF_DEFAULT_LEDS)) != (tty->mode & (TTYF_LEDS | TTYF_DEFAULT_LEDS | TTYF_NEW))) {
		if (__likely(tty->kmode != NULL)) {
			tty->kmode(KBD_SET_LEDS, tty->kc, !(mode & TTYF_DEFAULT_LEDS) ? (mode & TTYF_LEDS) >> __BSF_CONST(TTYF_LEDS) : -1);
		}
	}
	if (__unlikely((mode & TTYF_MOUSE_MODE) != (tty->mode & (TTYF_MOUSE_MODE | TTYF_NEW)))) {
		if (__likely(tty->kmode != NULL)) tty->kmode(KBD_SET_MOUSE_MODE, tty->kc, mode & TTYF_MOUSE_MODE);
	}
	tty->mode = mode;
	if (tty->mode & TTYF_RAW || tty->bits & TTY_PERMANENT_RAW) {
		tty->buf_line = tty->buf_cursor = tty->buf_end;
		tty_update(tty);
	} else {
		int i, j;
		for (j = i = tty->buf_start; i < tty->buf_end; i++) {
			int r;
			if (__unlikely(tty->buf[i] == '\r')) tty->buf[i] = '\n';
			if (__unlikely(r = is_input_break_char(tty, i))) j = i + r;
		}
		if (__unlikely(j < tty->buf_updateline)) {
			tty->buf_updateline = j;
		}
		if (__unlikely(j < tty->buf_line)) {
			tty->buf_line = j;
			if (j < tty->buf_needupdate) tty->buf_needupdate = j;
			tty_update(tty);
		}
	}
}

static __finline__ int tty_updatemode(HANDLE *h, TTY *tty, IORQ *rq)
{
	if (__unlikely((h->flags & TTYF_MODE_MASK) != tty->mode)) {
		tty_do_updatemode(tty, h->flags & TTYF_MODE_MASK);
	}
	return tty_updatekeyboardmode(h, tty, rq);
}

static void tty_wake_state(TTY *tty)
{
	if (tty->vmode != NULL) tty->vmode(VIDEOMODE_WAITACTIVE_WAKE, tty->vc);
	WQ_WAKE_ALL(&tty->state_wait);
}

void TTY$WRITE_DONE(TTY *tty)
{
	char *p;

	if (__unlikely(KERNEL$SPL != SPL_X(SPL_TTY)))
		KERNEL$SUICIDE("TTY$WRITE_DONE AT SPL %08X", KERNEL$SPL);
	if (__unlikely(!tty->buffer_to_write))
		KERNEL$SUICIDE("TTY$WRITE_DONE: UNEXPECTED CALL");
	if (__unlikely(tty->bits & TTY_IN_WRITEOUT))
		KERNEL$SUICIDE("TTY$WRITE_DONE: CALLED DIRECTLY FROM writeout");

	p = tty->buffer_to_write;
	tty->buffer_to_write = NULL;

	if (tty->buffer_ptr > tty->buffer_len >> 1) {
		check_redzone(tty, "TTY_WRITE_DONE_AST");
		tty->buffer_to_write = tty->buffer;
		tty->buffer_to_write_len = tty->buffer_ptr;
		tty->buffer = NULL;
		tty->buffer_len = 0;
		tty->buffer_ptr = 0;
		tty->bits |= TTY_IN_WRITEOUT;
		tty->writeout(tty);
		tty->bits &= ~TTY_IN_WRITEOUT;
	}
	if (__likely(p != tty->nomem_buffer)) {
		if (!tty->buffer) {
			tty->buffer = p;
			tty->buffer_len = __PAGE_CLUSTER_SIZE_MINUS_1;
		} else {
			KERNEL$FREE_KERNEL_PAGE(p, VM_TYPE_WIRED_MAPPED);
		}
	}
	KERNEL$DEL_TIMER(&tty->write_timer);
	KERNEL$SET_TIMER(TTY_WRITE_TIME, &tty->write_timer);

	if (!WQ_EMPTY(&tty->write_wait)) {
		if (tty->input_displayed) {
			tty_erase_input(tty);
		}
	}

	if (!tty->input_displayed) {
		WQ_WAKE_ALL(&tty->write_wait);
		KERNEL$DEL_TIMER(&tty->update_timer);
		if (tty->buf_updateline < tty->buf_end) {
			KERNEL$SET_TIMER(TTY_UPDATE_TIME, &tty->update_timer);
		} else {
			SET_TIMER_NEVER(&tty->update_timer);
		}
	} else {
		if (tty->text_ok != 3) tty_update(tty);
		WQ_WAKE_ALL(&tty->write_wait);
	}
}

static void tty_write_timer(TIMER *t)
{
	TTY *tty;
	LOWER_SPL(SPL_TTY);
	SET_TIMER_NEVER(t);
	tty = LIST_STRUCT(t, TTY, write_timer);
	if (tty->buffer_ptr) {
		tty_force_writeout(tty);
	} else {
		if (tty->buffer && tty->buffer != tty->nomem_buffer)
			KERNEL$FREE_KERNEL_PAGE(tty->buffer, VM_TYPE_WIRED_MAPPED);
		tty->buffer = NULL;
		tty->buffer_len = 0;
	}
}

int TTY$IN(TTY *tty, char *str, int str_len)
{
	int c, ret_val;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_TTY)))
		KERNEL$SUICIDE("TTY$IN AT SPL %08X", KERNEL$SPL);
	if (!str_len) return 0;
	if (__unlikely(tty->bits & TTY_PERMANENT_RAW)) {
		c = buffer_room(tty);
		if (c > str_len) c = str_len;
		tty->buf_cursor = tty->buf_end;
		insert_in_buffer(tty, str, c);
		tty->buf_line = tty->buf_end;
		tty_update(tty);
		ret_val = c;
		goto done;
	}
	ret_val = str_len;
	for (c = 0; c < str_len; c++) {
		unsigned char s;
		int skip;
		test_only_control:
		s = str[c];
		skip = ((tty->mode & TTYF_KBDMODE) - K_XLATE) | (ret_val - str_len);
		if (__unlikely(s < 32)) {
			if (__unlikely(tty->last_char == 27)) {
				/*__debug_printf("escape %d\n", s + 32);*/
				tty->control_seq[s + 32]++;
				skip |= WQ_WAKE_ALL(&tty->control_wait[s + 32]);
			}
			tty->control_seq[s]++;
			skip |= WQ_WAKE_ALL(&tty->control_wait[s]);
			if (s != 8 && s != 9 && s != 10 && s != 13) {
				tty->bits &= ~TTY_AUTH_POS;
				tty->bits |= TTY_AUTH_NEG;
				tty_wake_state(tty);
			}
		}
		tty->last_char = s;
		if (__unlikely(skip)) continue;
		if (__unlikely(tty->mode & TTYF_RAW)) {
			tty->buf_cursor = tty->buf_end;
			goto raw;
		}
		if (__likely(tty->in_state == TTYIN_NORMAL)) {
			if (__unlikely(s == '\t')) {
				if (__unlikely(tty->bits & TTY_TAB)) tty->bits |= TTY_2TAB;
				tty->bits |= TTY_TAB | TTY_DELAY;
				if (__unlikely(buffer_room(tty) < 2)) break;
				insert_in_buffer(tty, "\t", 1);
				if (__likely(tty->buf_updateline == tty->buf_line)) {
					tty->buf_updateline = tty->buf_cursor;
					tty->buf_needupdate = tty->buf_cursor;
					WQ_WAKE_ALL_PL(&tty->read_wait);
				}
				tty->buf_line = tty->buf_cursor;
				continue;
			}
			tty->bits &= ~(TTY_TAB | TTY_2TAB);
			if (__likely(s >= ' ') && __likely(s != 127)) goto raw;
			if (s == 'A' - 64) {
				tty_home(tty);
				continue;
			}
			if (s == 'D' - 64) {
				if (tty->buf_cursor != tty->buf_end) tty_del_char(tty);
				else {
					if (tty->buf_line == tty->buf_end) tty->bits |= TTY_EOF;
					else tty->buf_line = tty->buf_end;
					WQ_WAKE_ALL_PL(&tty->read_wait);
				}
				continue;
			}
			if (s == 'E' - 64) {
				tty_end(tty);
				continue;
			}
			if (s == 'U' - 64) {
				int x = tty->buf_cursor - tty->buf_line;
				tty->buf_cursor = tty->buf_line;
				del_from_buffer(tty, x);
				continue;
			}
			if (s == 'V' - 64) {
				tty->in_state = TTYIN_NEXTRAW;
				continue;
			}
			if (s == 10 || s == 13) {
				if (__unlikely(!buffer_room(tty))) break;
				tty->buf_cursor = tty->buf_end;
				insert_in_buffer(tty, "\n", 1);
				tty->buf_line = tty->buf_end;
				continue;
			}
			if (s == 8 || s == 127) {
				if (tty->buf_cursor > tty->buf_line) {
					tty_back_arrow(tty);
					tty_del_char(tty);
				}
				continue;
			}
			if (s != 27) {
				int i, b;
				raw:
				for (i = c + 1; i < str_len && (unsigned char)str[i] >= ' ' && str[i] != 127; i++) ;
				if (__unlikely((i -= c) >= (b = buffer_room(tty)))) {
					if (b <= 1) {
						tty->in_state = TTYIN_NORMAL;
						break;
					}
					i = b - 1;
				}
				insert_in_buffer(tty, str + c, i);
				tty->in_state = TTYIN_NORMAL;
				if (__unlikely(tty->mode & TTYF_RAW)) {
					tty->buf_line = tty->buf_end;
				}
				c += i - 1;	/* 1 will be added at loop */
				tty->last_char = str[c];
				continue;
			}
			tty->in_state = TTYIN_ESC;
			continue;
		} else if (tty->in_state == TTYIN_NEXTRAW) {
			goto raw;
		} else if (tty->in_state == TTYIN_ESC) {
			if (s == '[' || s == 'O') {
				tty->in_state = TTYIN_SQUARE;
				tty->in_number = 0;
				continue;
			} else if (s == 127) {
				goto delete;
			} else {
				char e[2];
				e[0] = 27;
				e[1] = s;
				if (buffer_room(tty) < 3) {
					tty->in_state = TTYIN_NORMAL;
					break;
				}
				insert_in_buffer(tty, e, 2);
				goto next;
			}
			continue;
		} else if (tty->in_state == TTYIN_SQUARE || tty->in_state == TTYIN_SQUARE2) {
			if (s == '[' && tty->in_state == TTYIN_SQUARE && !tty->in_number) {
				tty->in_state = TTYIN_SQUARE2;
				continue;
			}
			if (s >= '0' && s <= '9') {
				tty->in_number = tty->in_number * 10 + s - '0';
				if (tty->in_number >= 10000) tty->in_number = 9999;
				continue;
			}
			if (s == ';') {
				tty->in_number = 0;
				continue;
			}
			if (tty->in_state == TTYIN_SQUARE) {
				if (s == 'A' || s == 'B') {
					if (__unlikely(buffer_room(tty) < 4)) goto next;
					tty->bits |= TTY_DELAY;
					tty->buf_cursor = tty->buf_end;
					insert_in_buffer(tty, "\e[", 2);
					insert_in_buffer(tty, (char *)&s, 1);
					if (__likely(tty->buf_updateline == tty->buf_line)) {
						tty->buf_updateline = tty->buf_end;
						tty->buf_needupdate = tty->buf_end;
						WQ_WAKE_ALL_PL(&tty->read_wait);
					}
					tty->buf_line = tty->buf_end;
					goto next;
				}
				if (s == 'C') {
					tty_forward_arrow(tty);
					goto next;
				}
				if (s == 'D') {
					tty_back_arrow(tty);
					goto next;
				}
				if (s == '@') goto insert;
				if (s == 'F' || s == 'K' || s == 'e') goto end;
				if (s == 'H' || s == 0) goto home;
				if (s == 'I' || s == 'V') goto page_up;
				if (s == 'G' || s == 'U') goto page_down;
				if (s == '~') {
					switch (tty->in_number) {
						home:
						case 1:
							tty_home(tty);
							goto next;
						insert:
						case 2:
							/* insert */
							goto next;
						delete:
						case 3:
							tty_del_char(tty);
							goto next;
						end:
						case 4:
							tty_end(tty);
							goto next;
						page_up:
						case 5:
							tty_up(tty);
							goto next;
						page_down:
						case 6:
							tty_down(tty);
							goto next;
						case 7:
							goto home;
						case 8:
							goto end;
					}
				}
				if (s == 'z') {
/* This is probably SUN keyboard or something, copied from Links */
					switch (tty->in_number) {
						case 247: goto insert;
						case 214: goto home;
						case 220: goto end;
						case 216: goto page_up;
						case 222: goto page_down;
						case 249: goto delete;
					}
				}
			}
#if 0
			{
				char e[9]; /*= "\e[[9999x"*/
				if (tty->in_number) _snprintf(e, 9, "\e[%s%d%c", tty->in_state == TTYIN_SQUARE2 ? "[" : "", tty->in_number, s);
				else _snprintf(e, sizeof e, "\e[%s%c", tty->in_state == TTYIN_SQUARE2 ? "[" : "", s);
				if (__unlikely(buffer_room(tty) < strlen(e) + 1)) {
					tty->in_state = TTYIN_NORMAL;
					break;
				}
				insert_in_buffer(tty, e, strlen(e));
			}
#endif
			next:
			tty->in_state = TTYIN_NORMAL;
		} else KERNEL$SUICIDE("TTY$IN: in_state == %d", tty->in_state);
	}
	if (c != str_len) {
		ret_val = c;
		goto test_only_control;
	}
	/*__debug_printf("accepted: %d\n", c);*/
	tty_update(tty);
	done:
	return ret_val;
}

static __finline__ int refill_raw(TTY *tty)
{
	if (__unlikely((tty->mode & TTYF_KBDMODE) != K_XLATE)) {
		long r;
		if (__unlikely(!tty->kmode)) return 0;
		tty_erase_buffer(tty);
		if (__unlikely((r = tty->kmode(KBD_READ, tty->kc, tty->buf, (int)tty->buf_size, (int)(tty->mode & TTYF_KBDMODE))) < 0)) return 0;
		if (r) {
			tty->buf_updateline = tty->buf_needupdate = tty->buf_line = tty->buf_cursor = tty->buf_end = r;
			return 1;
		}
	}
	return 0;
}

static __finline__ void read_wait(TTY *tty, IORQ *rq)
{
	if (__likely((tty->mode & TTYF_KBDMODE) == K_XLATE) || __unlikely(!tty->kmode)) {
		WQ_WAIT_F(&tty->read_wait, rq);
	} else {
		WQ_WAIT_F((WQ *)tty->kmode(KBD_WAIT, tty->kc, (int)(tty->mode & TTYF_KBDMODE)), rq);
		LOWER_SPL(SPL_TTY);
	}
}

static const HANDLE_OPERATIONS tty_operations;

static DECL_IOCALL(TTY_READ, SPL_TTY, SIORQ)
{
	VBUF vbuf;
	long s;
	int i;
	int p;
	HANDLE *h = RQ->handle;
	TTY *tty;
	if (__unlikely(h->op != &tty_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_READ);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_READ;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_TTY));
	tty = h->fnode;
	if (__unlikely((h->flags & TTYF_MOUSESELECT) != 0)) {
		if (tty->kmode) tty->kmode(KBD_PULL_MOUSE_STATE, tty->kc);
		goto ret;
	}

	again:
	if (__unlikely(tty_updatemode(h, tty, (IORQ *)RQ))) {
		RETURN;
	}
	if (__unlikely(!RQ->v.len)) goto ret;
	if (tty->buf_start == tty->buf_updateline) {
		if (__unlikely(refill_raw(tty))) goto again;
		if (__unlikely(tty->bits & (TTY_EOF | TTY_PERMANENT_EOF))) {
			tty->bits &= ~TTY_EOF;
			goto ret;
		}
		if (RQ->progress) goto ret;
		if (__unlikely((h->flags & TTYF_NONBLOCK) != 0)) {
			RQ->status = -EWOULDBLOCK;
			RETURN_AST(RQ);
		}
		read_wait(tty, (IORQ *)RQ);
		RETURN;
	}

	if (tty->mode & TTYF_RAW || tty->bits & TTY_PERMANENT_RAW || (tty->mode & TTYF_KBDMODE) != K_XLATE) {
		i = tty->buf_updateline;
		goto t;
	}
	for (i = tty->buf_start + 1; i < tty->buf_updateline && tty->buf[i - 1] != '\n'; i++) ;
	t:
	p = tty->buf_start;
	if (i - p > RQ->v.len) i = p + RQ->v.len;
	vbuf.ptr = tty->buf + p;
	vbuf.len = i - p;
	vbuf.spl = SPL_X(SPL_TTY);
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_put(&RQ->v, &vbuf);
	if (__unlikely(!s)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
	RQ->progress += s;
	tty->buf_start += s;
	if (tty->buf_start == tty->buf_end) tty_erase_buffer(tty);
	else {
		if (tty->buf_start > tty->buf_updateline) tty->buf_updateline = tty->buf_start;
		if (tty->buf_start > tty->buf_needupdate) tty->buf_needupdate = tty->buf_start;
		if (tty->buf_start > tty->buf_line) tty->buf_line = tty->buf_start;
		if (tty->buf_start > tty->buf_cursor) tty->buf_cursor = tty->buf_start;
	}
	WQ_WAKE_ALL(&tty->in_wait);
	if (__likely(p + s == i)) {
		ret:
		if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
		else RQ->status = -EOVERFLOW;
		RETURN_AST(RQ);
	}
	TEST_LOCKUP_LOOP(RQ, RETURN);
	goto again;
}

static int tty_do_updatevideomode(TTY *tty, struct tty_extra_params *ex, IORQ *rq)
{
	WQ *r;
	if (memcmp(&ex->vm, &tty->current_videomode, sizeof(struct tty_videomode_params))) {
		if (__unlikely(tty->vmode == NULL)) {
			r = __ERR_PTR(-ENXIO);
			err:
			if (__IS_ERR(r)) {
				if (rq) {
					rq->status = __PTR_ERR(r);
					CALL_AST(rq);
				}
				return __PTR_ERR(r);
			} else {
				if (rq && r != (void *)1) WQ_WAIT_F(r, rq);
				return 1;
			}
		}
		if (__unlikely((r = (WQ *)tty->vmode(VIDEOMODE_SET_MODE, tty->vc, &ex->vm)) != NULL)) goto err;
		memcpy(&tty->current_videomode, &ex->vm, sizeof(struct tty_videomode_params));
	}
	if (__unlikely(ex->font.proc != tty->font_proc) || __unlikely(ex->font.blob != tty->font_blob)) {
		if (tty->vmode)
			if (__unlikely((r = (WQ *)tty->vmode(VIDEOMODE_SET_FONT, tty->vc, ex->font.proc, ex->font.blob, ex->font.blob_size, rq)) != NULL)) goto err;
		tty->font_proc = ex->font.proc;
		tty->font_blob = ex->font.blob;
	}
	return 0;
}

static __finline__ int tty_updatevideomode(HANDLE *h, TTY *tty, IORQ *rq)
{
	struct tty_extra_params *ex = (struct tty_extra_params *)h->flags2;
	if (__unlikely(memcmp(&ex->vm, &tty->current_videomode, sizeof(struct tty_videomode_params))) || __unlikely(ex->font.proc != tty->font_proc) || __unlikely(ex->font.blob != tty->font_blob)) {
		return tty_do_updatevideomode(tty, ex, rq);
	}
	return 0;
}

static DECL_IOCALL(TTY_WRITE, SPL_TTY, SIORQ)
{
	VBUF v;
	int s, i;
	int nl_crnl;
	HANDLE *h = RQ->handle;
	TTY *tty;
	if (__unlikely(h->op != &tty_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_WRITE);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_WRITE;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_TTY));
	tty = h->fnode;
	if (__unlikely((h->flags & TTYF_MOUSESELECT) != 0)) {
		goto ret;
	}
	if (__unlikely(tty_updatevideomode(h, tty, (IORQ *)RQ))) {
		RETURN;
	}
	if (__unlikely(!RQ->v.len)) goto ret;
	again:
	if (__unlikely(tty->input_displayed)) {
		if (__unlikely(RQ->v.len == 1)) {
			char c;
			v.ptr = &c;
			v.len = 1;
			v.spl = SPL_X(SPL_TTY);
			RAISE_SPL(SPL_VSPACE);
			s = RQ->v.vspace->op->vspace_get(&RQ->v, &v);
			RQ->v.ptr -= s;
			RQ->v.len += s;
				/* allow beep to go through */
			if (s == 1 && c == 7) goto no_erase;
		}
		tty_erase_input(tty);
		if (__unlikely(tty->input_displayed)) {
			wa:
			if (__unlikely((h->flags & TTYF_NONBLOCK) != 0)) {
				RQ->status = -EWOULDBLOCK;
				RETURN_AST(RQ);
			}
			WQ_WAIT_F(&tty->write_wait, RQ);
			RETURN;
		}
		no_erase:;
	}

	tty_check_space(tty, RQ->v.len);
	v.ptr = tty->buffer + tty->buffer_ptr;
	v.len = tty->buffer_len - tty->buffer_ptr;
	nl_crnl = 0;
	if (h->flags & TTYF_O_NL_CRNL && __likely(!(tty->bits & TTY_PERMANENT_RAW))) v.len >>= 1, nl_crnl = 1;
	v.spl = SPL_X(SPL_TTY);
	if (__unlikely(!v.len)) goto wa;

#define vptr ((char *)v.ptr)
	RAISE_SPL(SPL_VSPACE);
	s = RQ->v.vspace->op->vspace_get(&RQ->v, &v);
	RQ->progress += s;
	if (__likely(!(tty->bits & TTY_PERMANENT_RAW))) for (i = 0; i < s; ) {
		unsigned char c = vptr[i];
		switch (c) {
			int by, co;

			case 11:
				if (nl_crnl) {
					if (!tty->xstart) vptr[i] = '\r';
					else vptr[i] = '\n';
				}
			case 10:
				if (nl_crnl) {
			case 13:
					tty->xstart = 0;
				}
			case 0:
			case 7:
			case 12:
			case 14:
			case 15:
			/*case 24:*/
			/*case 26:*/
				i++;
				continue;
			case 8:
				if (tty->xstart) tty->xstart--;
				i++;
				continue;
			case 9:
				tty->xstart = (tty->xstart + 8) & ~7;
				if (tty->xstart >= tty->xsize) tty->xstart = 0;
				i++;
				continue;
			default:
				if (__unlikely(tty->out_state_buffer_size)) {
					int l = s - i;
					if (l > __MAX_BYTES_IN_CHAR - tty->out_state_buffer_size)
						l = __MAX_BYTES_IN_CHAR - tty->out_state_buffer_size;
					memcpy(tty->out_state_buffer + tty->out_state_buffer_size, vptr + i, l);
					tty->out_state_buffer_size += l;
					if (get_char_len(tty->out_state_buffer, tty->out_state_buffer + tty->out_state_buffer_size, &by, &co, 0)) goto brk;
					by -= tty->out_state_buffer_size - l;
					tty->out_state_buffer_size = 0;
				} else if (__unlikely(get_char_len(vptr + i, vptr + s, &by, &co, 0))) {
					if (__unlikely(i + __MAX_BYTES_IN_CHAR < s))
						KERNEL$SUICIDE("get_char_len: too long string");
					memcpy(tty->out_state_buffer, vptr + i, s - i);
					tty->out_state_buffer_size = s - i;
					goto brk;
				}
				i += by;
				if (__unlikely(co & (ABSOLUTE_POS | INCREASE_POS | DECREASE_POS))) {
					if (co & ABSOLUTE_POS) {
						if ((co & ~ABSOLUTE_POS) < tty->xsize) tty->xstart = co & ~ABSOLUTE_POS;
						else tty->xstart = tty->xsize - 1;
					} else if (co & INCREASE_POS) {
						if (tty->xstart + (co & ~INCREASE_POS) < tty->xsize) tty->xstart += co & ~INCREASE_POS;
						else tty->xstart = tty->xsize - 1;
					} else if (co & DECREASE_POS) {
						if ((co & ~DECREASE_POS) < tty->xstart) tty->xstart -= co & ~DECREASE_POS;
						else tty->xstart = 0;
					}
				} else {
					if ((tty->xstart += co) >= tty->xsize) tty->xstart = 0;
				}
		}
	}
	brk:
	if (nl_crnl) {
		int j, k = 0;
		for (j = 0; j < s; j++) if (__unlikely(vptr[j] == '\n')) k++;
		if (__likely(k)) {
			for (j = s - 1, s += k; j >= 0; j--) {
				if (__unlikely((vptr[j + k] = vptr[j]) == '\n')) vptr[j + --k] = '\r';
			}
		}
	}
	tty->buffer_ptr += s;
#undef vptr

	if (__likely(!RQ->v.len)) {
		ret:
		if (__likely(RQ->progress >= 0)) RQ->status = RQ->progress;
		else RQ->status = -EOVERFLOW;
		RETURN_AST(RQ);
	}
	if (s) {
		TEST_LOCKUP_LOOP(RQ, RETURN);
		goto again;
	}
	DO_PAGEIN(RQ, &RQ->v, PF_READ);
}

static __finline__ unsigned TTY_DEPTH(HANDLE *h)
{
	unsigned depth = KERNEL$PROC_DEPTH(h->name_addrspace);
	if (__unlikely(depth >= MAX_PROC_DEPTH)) return MAX_PROC_DEPTH - 1;
	return depth;
}

static __finline__ int TTY_XPIXELS(TTY *tty)
{
	if (tty->xpixels) return tty->xpixels;
	else return tty->xsize * 8;
}

static __finline__ int TTY_YPIXELS(TTY *tty)
{
	if (tty->ypixels) return tty->ypixels;
	else return tty->ysize * 16;
}

static __u8 palette[256 * 4];

static DECL_IOCALL(TTY_IOCTL, SPL_TTY, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	TTY *tty;
	int r;
	union {
		union accel_param ap;
		struct tty_control_state control_state;
		struct tty_window_size winsize;
		struct {
			VDESC vdesc;
			vspace_unmap_t *unmap;
			size_t string_len, string_offset;
			char *data;
		} v;
		struct tty_videomode tty_videomode;
		struct mouse_state mouse_state;
		struct mouse_info mouse_info;
		struct tty_detach_seq detach_seq;
	} u;
	if (__unlikely(h->op != &tty_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_TTY));
	tty = h->fnode;
	if (__likely(RQ->ioctl == IOCTL_TTY_DO_ACCEL)) {	/* fast path */
		if (__unlikely(tty_updatevideomode(h, tty, (IORQ *)RQ))) {
			RETURN;
		}
		if (__unlikely(!tty->vmode)) {
			RQ->status = -ENXIO;
			RETURN_AST(RQ);
		}
		if (__likely(RQ->v.len != 0)) {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &u.ap, sizeof u.ap))) {
				if (__likely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
		} else {
			memset(&u.ap, 0, sizeof u.ap);
		}
		if (__unlikely((r = tty->vmode(VIDEOMODE_DO_ACCEL, tty->vc, (unsigned)RQ->param, &u.ap, RQ)) > 0)) {
			RETURN;
		}
		RQ->status = r;
		RETURN_AST(RQ);
	}
	switch (RQ->ioctl) {
		case IOCTL_CAN_MMAP:
		case IOCTL_TTY_WAITACTIVE:
		case IOCTL_TTY_SETPALETTE:
		case IOCTL_TTY_SETDISPLAYSTART:
		case IOCTL_TTY_WAITRETRACE_SETDISPLAYSTART:
		case IOCTL_TTY_WAITRETRACE:
		case IOCTL_TTY_GET_PIXEL_CLOCK:
		case IOCTL_TTY_AVAIL_ACCEL:
			if (__unlikely(tty_updatevideomode(h, tty, (IORQ *)RQ))) {
				RETURN;
			}
	}
	switch (RQ->ioctl) {
		case IOCTL_NOP: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_FSYNC: {
			tty_force_writeout(tty);
			if (__likely(!(RQ->param & PARAM_FSYNC_ASYNC)) && (tty->buffer_ptr || tty->buffer_to_write)) {
				fsync_wait:
				WQ_WAIT_F(&tty->write_wait, RQ);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_FPATHCONF: {
			switch (RQ->param) {
				case _PC_MAX_CANON:
				case _PC_MAX_INPUT:
					RQ->status = tty->buf_size;
					RETURN_AST(RQ);
				default:
					RQ->status = -EINVAL;
					RETURN_AST(RQ);
			}
		}
		case IOCTL_SELECT_READ: {
			if (__unlikely((h->flags & TTYF_MOUSESELECT) != 0)) {
				int r;
				if (__unlikely(!(tty->kmode))) {
					if (RQ->param) {
						WQ_WAIT_F(&KERNEL$LOGICAL_WAIT, RQ);
						RETURN;
					}
					RQ->status = -EWOULDBLOCK;
					RETURN_AST(RQ);
				}
				r = tty->kmode(KBD_WAIT_MOUSE, tty->kc, (IORQ *)RQ, (int)RQ->param);
				if (r > 0) RETURN;
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(tty_updatemode(h, tty, (IORQ *)RQ))) {
				RETURN;
			}
			sr_again:
			if (tty->buf_start == tty->buf_updateline && __likely(!(tty->bits & (TTY_EOF | TTY_PERMANENT_EOF)))) {
				if (__unlikely(refill_raw(tty))) goto sr_again;
				if (RQ->param) {
					read_wait(tty, (IORQ *)RQ);
					RETURN;
				}
				RQ->status = -EWOULDBLOCK;
			} else RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_SELECT_WRITE: {
			if (__likely(!(h->flags & TTYF_MOUSESELECT)) && tty_check_space(tty, 2)) {
				if (RQ->param) {
					WQ_WAIT_F(&tty->write_wait, RQ);
					RETURN;
				}
				RQ->status = -EWOULDBLOCK;
			} else {
				RQ->status = 0;
			}
			RETURN_AST(RQ);
		}
		case IOCTL_NREAD: {
			if (__unlikely(tty_updatemode(h, tty, (IORQ *)RQ))) {
				RETURN;
			}
			if (tty->buf_start == tty->buf_updateline) {
				refill_raw(tty);
			}
			RQ->status = tty->buf_updateline - tty->buf_start;
			RETURN_AST(RQ);
		}
		case IOCTL_RWV_NONBLOCK: {
			RQ->status = RQ->param != PARAM_RWV_WRITE;
			RETURN_AST(RQ);
		}
		case IOCTL_CAN_MMAP: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			RQ->status = tty->vmode(VIDEOMODE_CAN_MMAP, tty->vc);
			RETURN_AST(RQ);
		}

		case IOCTL_TTY_GET_TERM_FLAGS: {
			if (__unlikely(tty->bits & TTY_PERMANENT_RAW)) {
				RQ->status = -ENOTTY;
				RETURN_AST(RQ);
			}
			RQ->status = h->flags;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_FLUSH: {
			if (__unlikely((RQ->param & (FREAD | ((_FPUTBACK < TTY_PUTBACK ? tty->bits >> __BSF_CONST(TTY_PUTBACK / _FPUTBACK) : tty->bits << __BSF_CONST(_FPUTBACK / TTY_PUTBACK)) & _FPUTBACK))) != 0)) {
				tty_flush(tty);
				tty_update(tty);
			}
			/* discarding data to send is not supported */
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_CONTROL: {
			memset(&u.control_state, 0, sizeof u.control_state);
			if (__likely(RQ->param != (int)-1)) {
				if (__unlikely(RQ->param >= N_CONTROLS)) {
					RQ->status = -EINVAL;
					RETURN_AST(RQ);
				}
				u.control_state.control_seq = tty->control_seq[RQ->param];
			}
			u.control_state.tty_seq = tty->tty_seq;
			u.control_state.detach_seq = tty->detach_seq[TTY_DEPTH(h)];
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.control_state, sizeof u.control_state)) == 1)) {
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_WAIT_CONTROL: {
			if (__unlikely(RQ->param >= N_CONTROLS) && __unlikely(RQ->param != (int)-1)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &u.control_state, sizeof u.control_state))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__likely(u.control_state.tty_seq == tty->tty_seq) &&
			    __likely(u.control_state.detach_seq == tty->detach_seq[TTY_DEPTH(h)])) {
				if (RQ->param == (int)-1) {
					WQ_WAIT_F(&tty->control_wait[0], RQ);
					RETURN;
				}
				if (__likely(u.control_state.control_seq == tty->control_seq[RQ->param])) {
					WQ_WAIT_F(&tty->control_wait[RQ->param], RQ);
					RETURN;
				}
				tty_flush(tty);
				tty_update(tty);
				/*__debug_printf("wait control succeeded\n");*/
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			RQ->status = -ETTYCHG;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_WINSIZE: {
			memset(&u.winsize, 0, sizeof u.winsize);
			u.winsize.tty_seq = tty->tty_seq;
			u.winsize.detach_seq = tty->detach_seq[TTY_DEPTH(h)];
			u.winsize.ws_col = tty->xsize;
			u.winsize.ws_row = tty->ysize;
			u.winsize.ws_xpixel = TTY_XPIXELS(tty);
			u.winsize.ws_ypixel = TTY_YPIXELS(tty);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.winsize, sizeof u.winsize)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_WAIT_WINSIZE: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &u.winsize, sizeof u.winsize))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__likely(u.winsize.tty_seq == tty->tty_seq) &&
			    __likely(u.winsize.detach_seq == tty->detach_seq[TTY_DEPTH(h)])) {
				if (__likely(u.winsize.ws_col == tty->xsize) &&
				    __likely(u.winsize.ws_row == tty->ysize) &&
				    __likely(u.winsize.ws_xpixel == TTY_XPIXELS(tty)) &&
				    __likely(u.winsize.ws_ypixel == TTY_YPIXELS(tty))) {
					WQ_WAIT_F(&tty->state_wait, RQ);
					RETURN;
				}
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			RQ->status = -ETTYCHG;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_INCREASE_DETACH_SEQ: {
			unsigned depth = KERNEL$PROC_DEPTH(h->name_addrspace) + 1;
			while (depth < MAX_PROC_DEPTH) tty->detach_seq[depth++]++;
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_PEEK: {
			long l = RQ->v.len;
			if (__likely(RQ->v.len > tty->buf_line - tty->buf_start)) RQ->v.len = tty->buf_line - tty->buf_start;
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, tty->buf + tty->buf_start, RQ->v.len)) == 1)) {
				RQ->v.len = l;
				DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			}
			if (__likely(!r)) r = RQ->v.len;
			RQ->v.len = l;
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_NTAB: {
			if (__unlikely(!(tty->bits & TTY_TAB))) RQ->status = 0;
			else if (__likely(!(tty->bits & TTY_2TAB))) RQ->status = 1;
			else RQ->status = 2;
			if (RQ->param) tty->bits &= ~(TTY_TAB | TTY_2TAB);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_SPEED: {
			long status;
			switch (RQ->param) {
				case PARAM_ISPEED:
					status = tty->ispeed;
					break;
				case PARAM_OSPEED:
					status = tty->ospeed;
					break;
				default:
					RQ->status = -EINVAL;
					RETURN_AST(RQ);
			}
			if (__likely(status <= 0)) status = -EOPNOTSUPP;
			RQ->status = status;
			RETURN_AST(RQ);
		}

		case IOCTL_TTY_DTTY: {
			int i;
			tty_force_writeout(tty);
			if (__likely(!(RQ->param & PARAM_FSYNC_ASYNC)) && (tty->buffer_ptr || tty->buffer_to_write)) {
				goto fsync_wait;
			}
			for (i = 0; i < N_CONTROLS; i++) {
				tty->control_seq[i]++;
				WQ_WAKE_ALL(&tty->control_wait[i]);
			}
			tty_wake_state(tty);
			if (tty->dtty) tty->dtty(tty);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_PUTBACK: {
			if (__unlikely(RQ->v.len > buffer_room(tty))) {
				RQ->status = -EMSGSIZE;
				RETURN_AST(RQ);
			}
			if (__unlikely(RQ->v.len > tty->buf_start)) {
				int shift = tty->buf_size - tty->buf_end;
				memmove(tty->buf + tty->buf_start + shift, tty->buf + tty->buf_start, tty->buf_end - tty->buf_start);
				tty->buf_start += shift;
				tty->buf_updateline += shift;
				tty->buf_needupdate += shift;
				tty->buf_line += shift;
				tty->buf_cursor += shift;
				tty->buf_end += shift;
			}
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, tty->buf + tty->buf_start - RQ->v.len, RQ->v.len))) {
				if (__likely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			tty->buf_start -= RQ->v.len;
			/* If this is from kernel space, do not set TTY_PUTBACK,
			   this prevents flushing the console typeahead buffer
			   on login */
			if (__unlikely(RQ->v.vspace != &KERNEL$VIRTUAL))
				tty->bits |= TTY_PUTBACK;
			WQ_WAKE_ALL(&tty->read_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_PUTBACK_EDIT: {
			int i;
			for (i = tty->buf_start; i < tty->buf_end; i++) {
				if (__unlikely(is_input_break_char(tty, i))) {
					RQ->status = 0;
					RETURN_AST(RQ);
				}
			}
			tty->buf_updateline = tty->buf_start;
			tty->buf_needupdate = tty->buf_start;
			tty->buf_line = tty->buf_start;
			tty->text_ok = 0;
			tty_update(tty);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_AUTH: {
			if (__likely(!RQ->param)) tty->bits &= ~TTY_AUTH_NEG, tty->bits |= TTY_AUTH_POS;
			else tty->bits &= ~TTY_AUTH_POS, tty->bits |= TTY_AUTH_NEG;
			tty_wake_state(tty);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GETAUTH: {
			if (tty->bits & TTY_AUTH_POS) {
				tty->bits &= ~TTY_AUTH_POS;
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			if (__unlikely(tty->bits & TTY_AUTH_NEG)) {
				tty->bits &= ~TTY_AUTH_NEG;
				RQ->status = 1;
				RETURN_AST(RQ);
			}
			if (RQ->param) {
				RQ->status = -EWOULDBLOCK;
				RETURN_AST(RQ);
			}
			WQ_WAIT_F(&tty->state_wait, RQ);
			RETURN;
		}
		case IOCTL_TTY_ERASE_CLIPBOARD: {
			if (tty->kmode) {
				tty->kmode(KBD_ERASE_CLIPBOARD, tty->kc);
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}

		case IOCTL_TTY_GETVIDEOMODE: {
			r = -ENXIO;
			if (__unlikely(!tty->vmode) || __unlikely(r = tty->vmode(VIDEOMODE_GET_PARAM, tty->vc, &((struct tty_extra_params *)h->flags2)->vm, &u.tty_videomode))) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.tty_videomode, sizeof u.tty_videomode)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GETKBDMODE: {
			RQ->status = h->flags & TTYF_KBDMODE;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_MOUSE_EVENT: {
			if (__unlikely(tty_updatemode(h, tty, (IORQ *)RQ))) {
				RETURN;
			}
			if (__unlikely(!tty->kmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			tty->kmode(KBD_GET_MOUSE_STATE, tty->kc, &u.mouse_state);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.mouse_state, sizeof u.mouse_state)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			tty->kmode(KBD_PULL_MOUSE_STATE, tty->kc);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GETMOUSEMODE: {
			RQ->status = h->flags & TTYF_MOUSE_MODE;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_MOUSE_INFO: {
			if (__unlikely(tty_updatemode(h, tty, (IORQ *)RQ))) {
				RETURN;
			}
			if (__unlikely(!tty->kmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			tty->kmode(KBD_GET_MOUSE_INFO, tty->kc, &u.mouse_info);
			if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.mouse_info, sizeof u.mouse_info)) == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_WAITACTIVE: {
			if ((RQ->param == PARAM_WAIT_FOR_ACTIVE) || (RQ->param == PARAM_WAIT_FOR_ACTIVE_AND_LOCK)) {
				memset(&u.detach_seq, 0, sizeof u.detach_seq);
				u.detach_seq.tty_seq = tty->tty_seq;
				u.detach_seq.detach_seq = tty->detach_seq[TTY_DEPTH(h)];
				if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(RQ, &u.detach_seq, sizeof u.detach_seq))) {
					if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
					RQ->status = r;
					RETURN_AST(RQ);
				}
			} else {
				if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &u.detach_seq, sizeof u.detach_seq))) {
					if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
					RQ->status = r;
					RETURN_AST(RQ);
				}
				if (__unlikely(u.detach_seq.tty_seq != tty->tty_seq) ||
				    __unlikely(u.detach_seq.detach_seq != tty->detach_seq[TTY_DEPTH(h)])) {
					RQ->status = -ETTYCHG;
					RETURN_AST(RQ);
				}
			}
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			r = tty->vmode(VIDEOMODE_WAITACTIVE, tty->vc, (int)RQ->param, (IORQ *)RQ);
			if (r <= 0) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			RETURN;
		}
		case IOCTL_TTY_SETPALETTE: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			if (__unlikely(RQ->param + (RQ->v.len >> 2) > 256) || __unlikely(RQ->param >= 256) || __unlikely(RQ->v.len > 256 * 4) || __unlikely((int)RQ->v.len & 3)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, palette, RQ->v.len))) {
				if (__likely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			RQ->status = tty->vmode(VIDEOMODE_SET_PALETTE, tty->vc, palette, (int)RQ->param, (int)RQ->v.len >> 2);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_SETDISPLAYSTART: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			RQ->status = tty->vmode(VIDEOMODE_SET_DISPLAY_START, tty->vc, (int)RQ->param);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_WAITRETRACE_SETDISPLAYSTART: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			RQ->status = tty->vmode(VIDEOMODE_WAITRETRACE_SET_DISPLAY_START, tty->vc, (int)RQ->param);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_WAITRETRACE: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			tty->vmode(VIDEOMODE_WAITRETRACE, tty->vc, RQ);
			RETURN;
		}
		case IOCTL_TTY_GET_PIXEL_CLOCK: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			RQ->status = tty->vmode(VIDEOMODE_GET_PIXEL_CLOCK, tty->vc, &((struct tty_extra_params *)h->flags2)->vm, (unsigned long)RQ->param);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_AVAIL_ACCEL: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			RQ->status = tty->vmode(VIDEOMODE_AVAIL_ACCEL, tty->vc, (int)RQ->param);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_FONT_MODE: {
			if (__unlikely(!tty->vmode)) {
				RQ->status = RESULT_GET_FONT_MODE_NONE;
				RETURN_AST(RQ);
			}
			RQ->status = tty->vmode(VIDEOMODE_GET_FONT_MODE, tty->vc);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_CHARSET: {
			struct tty_extra_params *ex;
			u.v.string_len = sizeof(((struct font_blob *)u.v.data)->charset);
			u.v.string_offset = __offsetof(struct font_blob, charset);
			get_fontblob_data:
			ex = (struct tty_extra_params *)h->flags2;
			if (ex->font.proc && ex->font.blob_size >= sizeof(struct font_blob)) {
				unsigned long l, sl;
				u.v.vdesc.ptr = (unsigned long)ex->font.blob;
				u.v.vdesc.len = ex->font.blob_size;
				u.v.vdesc.vspace = KERNEL$PROC_VSPACE(ex->font.proc);
				get_string:
				RAISE_SPL(SPL_VSPACE);
				u.v.data = u.v.vdesc.vspace->op->vspace_map(&u.v.vdesc, PF_READ, SPL_X(SPL_TTY), &u.v.unmap);
				if (__unlikely(!u.v.data)) {
					DO_PAGEIN(RQ, &u.v.vdesc, PF_READ);
				}
				if (__unlikely(__IS_ERR(u.v.data))) {
					RQ->status = __PTR_ERR(u.v.data);
					RETURN_AST(RQ);
				}
				put_string_name:
				if (__unlikely((sl = strnlen(u.v.data + u.v.string_offset, u.v.string_len)) >= u.v.string_len)) {
					r = -EINVAL;
					goto ret_unmap_data;
				}
				if (__unlikely(sl >= RQ->v.len)) {
					r = -EMSGSIZE;
					goto ret_unmap_data;
				}
				l = RQ->v.len;
				RQ->v.len = sl + 1;
				r = KERNEL$PUT_IOCTL_STRUCT(RQ, u.v.data + u.v.string_offset, RQ->v.len);
				RQ->v.len = l;
				ret_unmap_data:
				u.v.unmap(u.v.data);
				if (__unlikely(r == 1)) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (u.v.string_offset == __offsetof(struct font_blob, charset)) {
				u.v.data = __ERR_PTR(-ENXIO);
				if (__unlikely(!tty->vmode) || __unlikely(__IS_ERR(u.v.data = (char *)tty->vmode(VIDEOMODE_GET_DEFAULT_CHARSET, tty->vc)))) {
					RQ->status = __PTR_ERR(u.v.data);
					RETURN_AST(RQ);
				}
			} else {
				RQ->status = __PTR_ERR(-ENOENT);
				RETURN_AST(RQ);
			}
			u.v.string_offset = 0;
			u.v.string_len = strlen(u.v.data) + 1;
			u.v.unmap = KERNEL$NULL_VSPACE_UNMAP;
			goto put_string_name;
		}
		case IOCTL_TTY_GET_FONT_FILE: {
			u.v.string_len = sizeof(((struct font_blob *)u.v.data)->fontfile);
			u.v.string_offset = __offsetof(struct font_blob, fontfile);
			goto get_fontblob_data;
		}
		case IOCTL_TTY_SET_KEYBOARD: {
			if (__unlikely(!tty->kmode)) {
				RQ->status = -ENXIO;
				RETURN_AST(RQ);
			}
			RQ->status = tty->kmode(KBD_SWITCH_KEYBOARD, tty->kc, (unsigned)RQ->param);
			RETURN_AST(RQ);
		}
		case IOCTL_TTY_GET_KEYBOARD_NAME: {
			struct tty_extra_params *ex = (struct tty_extra_params *)h->flags2;
			u.v.string_len = sizeof(((struct keymap_blob *)u.v.data)->keymap);
			u.v.string_offset = __offsetof(struct keymap_blob, keymap);
			if (ex->keymap.proc && ex->keymap.blob_size >= sizeof(struct keymap_blob)) {
				u.v.vdesc.ptr = (unsigned long)ex->keymap.blob;
				u.v.vdesc.len = ex->keymap.blob_size;
				u.v.vdesc.vspace = KERNEL$PROC_VSPACE(ex->keymap.proc);
				goto get_string;
			}
			u.v.data = "";
			u.v.string_offset = 0;
			u.v.string_len = 1;
			u.v.unmap = KERNEL$NULL_VSPACE_UNMAP;
			goto put_string_name;
		}
		case IOCTL_TTY_GETLED: {
			if (__unlikely(!tty->kmode)) {
				RQ->status = -ENXIO;
			} else if (!(h->flags & TTYF_DEFAULT_LEDS)) {
				RQ->status = (h->flags & TTYF_LEDS) >> __BSF_CONST(TTYF_LEDS);
			} else {
				RQ->status = tty->kmode(KBD_GET_LEDS, tty->kc);
			}
			RETURN_AST(RQ);
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static void *tty_clone(HANDLE *from, HANDLE *to, int open_flags)
{
	struct tty_extra_params *m_from = (struct tty_extra_params *)from->flags2, *m_to;
	if (__likely(m_from == &default_params)) to->flags2 = (unsigned long)&default_params;
	else {
		m_to = __slalloc(tty_extra_params + __likely(to->name_addrspace == &KERNEL$PROC_KERNEL));
		if (__unlikely(!m_to)) {
			if (__unlikely(KERNEL$OOM(VM_TYPE_WIRED_MAPPED))) return __ERR_PTR(-ENOMEM);
			return &KERNEL$FREEMEM_WAIT;
		}
		to->flags2 = (unsigned long)m_to;
		memcpy(m_to, m_from, sizeof(struct tty_extra_params));
		m_to->font.set_in_this_handle = 0;
		m_to->keymap.set_in_this_handle = 0;
	}
	to->op = from->op;
	to->fnode = from->fnode;
	to->flags = from->flags;
	return NULL;
}

static void *get_params(HANDLE *h)
{
	if (((struct tty_extra_params *)h->flags2) == &default_params) {
		struct tty_extra_params *m = calloc(1, sizeof(struct tty_extra_params));
		if (__unlikely(!m)) {
			if (__unlikely(KERNEL$OOM(VM_TYPE_WIRED_MAPPED))) return __ERR_PTR(-ENOMEM);
			return &KERNEL$FREEMEM_WAIT;
		}
		h->flags2 = (unsigned long)m;
	}
	return NULL;
}

/* partial duplicate in NULL.C */
static void *tty_lookup(HANDLE *h, char *str, int open_flags)
{
	TTY *tty;
	long num;
	void *w;
	char *e = strchr(str, '=');
	if (__unlikely(!e)) {
		if (!_strcasecmp(str, "^NONBLOCK")) {
			h->flags |= TTYF_NONBLOCK;
		} else if (__unlikely(!_strcasecmp(str, "^MOUSESELECT"))) {
			h->flags |= TTYF_MOUSESELECT;
		} else return __ERR_PTR(-EBADMOD);
		return NULL;
	}
	if (!__strcasexcmp("^ICANON", str, e)) {
		if (e[1] != '0' && __unlikely(e[1] != '1')) return __ERR_PTR(-EBADMOD);
		if (__unlikely(e[2])) return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~TTYF_RAW) | (e[1] == '0' ? TTYF_RAW : 0);
	} else if (!__strcasexcmp("^ECHO", str, e)) {
		if (e[1] != '0' && __unlikely(e[1] != '1')) return __ERR_PTR(-EBADMOD);
		if (__unlikely(e[2])) return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~TTYF_NOECHO) | (e[1] == '0' ? TTYF_NOECHO : 0);
	} else if (!__strcasexcmp("^ONLCR", str, e)) {
		if (e[1] != '0' && __unlikely(e[1] != '1')) return __ERR_PTR(-EBADMOD);
		if (__unlikely(e[2])) return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~TTYF_O_NL_CRNL) | (e[1] == '0' ? 0 : TTYF_O_NL_CRNL);
	} else if (!__strcasexcmp("^WINSIZE", str, e)) {
		/* copied in SWAPTTY.C */
		long x, y, cx, cy;
		char *d1, *d2, *d3;
		d1 = strchr(e + 1, ',');
		if (__unlikely(!d1)) return __ERR_PTR(-EBADMOD);
		d2 = strchr(d1 + 1, ',');
		if (__unlikely(!d2)) return __ERR_PTR(-EBADMOD);
		d3 = strchr(d2 + 1, ',');
		if (__unlikely(!d3)) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(e + 1, d1, 1, &x))) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(d1 + 1, d2, 1, &y))) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(d2 + 1, d3, 1, &cx))) return __ERR_PTR(-EBADMOD);
		if (__unlikely(__get_number(d3 + 1, d3 + strlen(d3), 1, &cy))) return __ERR_PTR(-EBADMOD);
		tty = h->fnode;
		if (!(tty->bits & TTY_LOCK_WINDOW_SIZE)) {
			tty->xsize = x;
			tty->ysize = y;
			tty->xpixels = cx;
			tty->ypixels = cy;
			WQ_WAKE_ALL(&tty->state_wait);
		}
	} else if (!__strcasexcmp("^VIDEOMODE", str, e)) {
		long vm;
		tty = h->fnode;
		if (__unlikely(!tty->vmode)) return __ERR_PTR(-ENXIO);
		if (__unlikely((w = get_params(h)) != NULL)) return w;
		vm = tty->vmode(VIDEOMODE_GET_MODE, tty->vc, e + 1);
		if (__unlikely(vm < 0)) return __ERR_PTR(vm);
		((struct tty_extra_params *)h->flags2)->vm.vm = vm;
	} else if (!__strcasexcmp("^VIDEOTIMING", str, e)) {
		static long pixel_clock, hx, h_sync_start, h_sync_end, h_total, vx, v_sync_start, v_sync_end, v_total, flags;
		static long * const params[] = { &pixel_clock, &hx, &h_sync_start, &h_sync_end, &h_total, &vx, &v_sync_start, &v_sync_end, &v_total, &flags, NULL };
		struct tty_videomode_params *m;
		char *c;
		long * const *ptr = params;
		e++;
		while (1) {
			if (__unlikely(!(c = strchr(e, ',')))) c = e + strlen(e);
			if (__unlikely(__get_number(e, c, 0, *ptr))) return __ERR_PTR(-EBADMOD);
			if (__unlikely(**ptr > MAXINT)) return __ERR_PTR(-EBADMOD);
			ptr++;
			if (__unlikely(!*c)) {
				if (__likely(!*ptr)) break;
				return __ERR_PTR(-EBADMOD);
			}
			e = c + 1;
		}
		if (__unlikely((w = get_params(h)) != NULL)) return w;
		m = &((struct tty_extra_params *)h->flags2)->vm;
		m->pixel_clock = pixel_clock;
		m->h = hx;
		m->h_sync_start = h_sync_start;
		m->h_sync_end = h_sync_end;
		m->h_total = h_total;
		m->v = vx;
		m->v_sync_start = v_sync_start;
		m->v_sync_end = v_sync_end;
		m->v_total = v_total;
		m->flags = flags;
	} else if (!__strcasexcmp("^KBDRAW", str, e)) {
		tty = h->fnode;
		if (__unlikely(e[1] < '0') || __unlikely(e[1] > '2')) return __ERR_PTR(-EBADMOD);
		if (__unlikely(e[2])) return __ERR_PTR(-EBADMOD);
		if (__unlikely(!tty->kmode)) return __ERR_PTR(-ENXIO);
		h->flags = (h->flags & ~TTYF_KBDMODE) | (e[1] - '0');
	} else if (!__strcasexcmp("^KBDLED", str, e)) {
		int mode;
		tty = h->fnode;
		if (__unlikely(__get_number(e + 1, e + strlen(e), 1, &num))) return __ERR_PTR(-EBADMOD);
		if (__unlikely(!tty->kmode)) return __ERR_PTR(-ENXIO);
		if (__likely((unsigned long)num <= TTYF_LEDS >> __BSF_CONST(TTYF_LEDS))) mode = num << __BSF_CONST(TTYF_LEDS);
		else mode = TTYF_DEFAULT_LEDS;
		h->flags = (h->flags & ~(TTYF_LEDS | TTYF_DEFAULT_LEDS)) | mode;
	} else if (!__strcasexcmp("^MOUSE", str, e)) {
		int mode;
		if (!_strcasecmp(e + 1, "COPYPASTE")) mode = TTYF_MOUSE_COPYPASTE;
		else if (!_strcasecmp(e + 1, "CHAR")) mode = TTYF_MOUSE_CHAR;
		else if (!_strcasecmp(e + 1, "RAW")) mode = TTYF_MOUSE_RAW;
		else return __ERR_PTR(-EBADMOD);
		h->flags = (h->flags & ~TTYF_MOUSE_MODE) | mode;
	} else if (!__strcasexcmp("^KBD_REPEAT_DELAY", str, e) ||
		   !__strcasexcmp("^KBD_REPEAT_RATE", str, e)) {
		if (__unlikely(__get_number(e + 1, e + strlen(e), 1, &num))) return __ERR_PTR(-EBADMOD);
		if (__likely(num != -1) && __unlikely((unsigned long)num > MAXINT)) return __ERR_PTR(-EBADMOD);
		if (__unlikely((w = get_params(h)) != NULL)) return w;
		if (!__strcasexcmp("^KBD_REPEAT_DELAY", str, e))
			((struct tty_extra_params *)h->flags2)->kbd.repeat_delay = num;
		else
			((struct tty_extra_params *)h->flags2)->kbd.repeat_rate = num;
	} else if (!__strcasexcmp("^KBD_REPEAT_MODE", str, e)) {
		char mode;
		if (__likely(!_strcasecmp(e + 1, "SOFT"))) mode = 1;
		else if (__likely(!_strcasecmp(e + 1, "HARD"))) mode = 0;
		else return __ERR_PTR(-EBADMOD);
		if (__unlikely((w = get_params(h)) != NULL)) return w;
		((struct tty_extra_params *)h->flags2)->kbd.soft = mode;
	} else return __ERR_PTR(-EBADMOD);
	return NULL;
}

static void *tty_instantiate(HANDLE *h, IORQ *rq, int flags)
{
	TTY *tty;
	if (__likely(rq != NULL)) {
		size_t blob_size;
		FBLOB *blob = KERNEL$FIND_FILE_BLOB(h, rq, BLOB_TTY_FONT, &blob_size);
		if (__unlikely(blob != (void *)1)) {
			struct tty_extra_params *ext;
			WQ *wq;
			if (__unlikely(!blob)) return (void *)3;
			if (__unlikely((wq = get_params(h)) != NULL)) return wq;
			ext = (struct tty_extra_params *)h->flags2;
			ext->font.proc = h->name_addrspace;
			ext->font.blob = blob;
			ext->font.blob_size = blob_size;
			ext->font.set_in_this_handle = 1;
		}
		blob = KERNEL$FIND_FILE_BLOB(h, rq, BLOB_TTY_KEYMAP, &blob_size);
		if (__unlikely(blob != (void *)1)) {
			struct tty_extra_params *ext;
			WQ *wq;
			if (__unlikely(!blob)) return (void *)3;
			if (__unlikely((wq = get_params(h)) != NULL)) return wq;
			ext = (struct tty_extra_params *)h->flags2;
			ext->keymap.proc = h->name_addrspace;
			ext->keymap.blob = blob;
			ext->keymap.blob_size = blob_size;
			ext->keymap.set_in_this_handle = 1;
		}
	}
	tty = h->fnode;
	ADD_TO_LIST(&tty->handles, &h->fnode_entry);
	return NULL;
}

static void tty_leave(HANDLE *h)
{
	if (__unlikely((struct tty_extra_params *)h->flags2 != &default_params))
		__slfree((struct tty_extra_params *)h->flags2);
}

static void tty_detach(HANDLE *h)
{
	int i;
	TTY *tty = h->fnode;
	WQ_WAKE_ALL(&tty->write_wait);
	WQ_WAKE_ALL(&tty->read_wait);
	WQ_WAKE_ALL(&tty->in_wait);
	tty_wake_state(tty);
	for (i = 0; i < N_CONTROLS; i++)
		WQ_WAKE_ALL(&tty->control_wait[i]);
	DEL_FROM_LIST(&h->fnode_entry);
	if (__unlikely((struct tty_extra_params *)h->flags2 != &default_params)) {
		struct tty_extra_params *ex = (struct tty_extra_params *)h->flags2;
		if (__unlikely(ex->font.set_in_this_handle)) {
			tty->font_proc = (void *)1;
			tty->font_blob = (void *)1;
		}
		if (__unlikely(ex->keymap.set_in_this_handle)) {
			tty->keymap_proc = (void *)1;
			tty->keymap_blob = (void *)1;
		}
		__slfree(ex);
	}
}

static PAGE *tty_get_page(HANDLE *h, __v_off idx, int wr)
{
	int r;
	TTY *tty = h->fnode;
	if (__unlikely(wr & ~PF_RW)) return __ERR_PTR(-EINVAL);
	if (__unlikely(!tty->vmode)) return __ERR_PTR(-ENXIO);
	if (__unlikely(r = tty_updatevideomode(h, tty, NULL))) {
		if (r < 0) return __ERR_PTR(r);
		LOWER_SPL(SPL_TTY);
		return NULL;
	}
	return (PAGE *)tty->vmode(VIDEOMODE_GET_PAGE, tty->vc, idx);
}

static IORQ * volatile pagein_rq = NULL;
static VDESC pagein_vdesc;
static IORQ pagein_io;
static WQ_DECL(pagein_wait, "TTY$PAGEIN_WAIT");
static IO_STUB tty_pagein_cont;

static IORQ *tty_getpagein_rq(VDESC *desc, IORQ *rq, int wr)
{
	if (__unlikely(pagein_rq != NULL)) {
		WQ_WAIT_F(&pagein_wait, rq);
		return NULL;
	}
	pagein_rq = rq;
	memcpy(&pagein_vdesc, desc, sizeof(VDESC));
	pagein_io.tmp1 = (unsigned long)tty_pagein_cont;
	return &pagein_io;
}

static DECL_IOCALL(tty_pagein_cont, SPL_TTY, IORQ)
{
	int r;
	TTY *tty;
	HANDLE *h = pagein_vdesc.vspace;
	if (__unlikely(h->op != &tty_operations)) {
		repost:
		CALL_IORQ_LSTAT_EXPR(pagein_rq, (IO_STUB *)pagein_rq->tmp1);
		goto end;
	}
	tty = h->fnode;
	if (__unlikely(!tty->vmode)) {
		pagein_rq->status = -ENXIO;
		CALL_AST(pagein_rq);
		goto end;
	}
	if (__likely(tty_updatevideomode(h, tty, pagein_rq))) {
		goto end;
	}
	if (__unlikely(r = tty->vmode(VIDEOMODE_UNMAP_PAGES, tty->vc, pagein_rq))) {
		if (r < 0) {
			pagein_rq->status = r;
			CALL_AST(pagein_rq);
		}
		goto end;
	}
	goto repost;
	end:
	RAISE_SPL(SPL_VSPACE);
	pagein_rq = NULL;
	WQ_WAKE_ALL(&pagein_wait);
	LOWER_SPL(SPL_TTY);
	RETURN;
}

static const HANDLE_OPERATIONS tty_operations = {
	SPL_X(SPL_TTY),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	tty_getpagein_rq,
	tty_get_page,
	KERNEL$NO_VSPACE_SWAP_OP,
	tty_clone,		/* clone */
	tty_lookup,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	tty_instantiate,	/* instantiate */
	tty_leave,		/* leave */
	tty_detach,		/* detach */
	NULL,			/* close */
	TTY_READ,
	TTY_WRITE,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
	TTY_IOCTL,

	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
};

void TTY$INIT_ROOT(HANDLE *ttyh, void *tty)
{
	ttyh->op = &tty_operations;
	ttyh->fnode = tty;
	ttyh->flags = K_XLATE | TTYF_O_NL_CRNL | TTYF_DEFAULT_LEDS;
	ttyh->flags2 = (unsigned long)&default_params;
}

int TTY$CREATE(TTY *tty)
{
	int i;
	MD5_CTX md5;
	__u64 digest[2];
	__u64 tim;
	struct timeval tv;
	static unsigned seq;
	if (!tty->buf_size || (tty->buf_size & (tty->buf_size - 1)))
		KERNEL$SUICIDE("TTY$CREATE: INVALID BUFFER SIZE (%d)", tty->buf_size);

	INIT_LIST(&tty->handles);

	tty->buffer = NULL;
	tty->buffer_len = 0;
	tty->buffer_ptr = 0;
	tty->buffer_to_write = NULL;
	
	tty->input_displayed = 0;

	tty->update_timer.fn = &tty_update_timer;
	INIT_TIMER(&tty->update_timer);
	SET_TIMER_NEVER(&tty->update_timer);
	tty->write_timer.fn = &tty_write_timer;
	INIT_TIMER(&tty->write_timer);
	SET_TIMER_NEVER(&tty->write_timer);

	WQ_INIT(&tty->read_wait, "TTY$READ_WAIT");
	WQ_INIT(&tty->write_wait, "TTY$WRITE_WAIT");
	WQ_INIT(&tty->in_wait, "TTY$IN_WAIT");

	tty->buf_start = tty->buf_end = tty->buf_line = tty->buf_updateline = tty->buf_cursor = tty->buf_needupdate = 0;

	tty->bits = 0;

	tty->mode = K_XLATE | TTYF_DEFAULT_LEDS | TTYF_NEW;
	tty->in_state = TTYIN_NORMAL;
	tty->in_number = 0;

	tty->xsize = 80;
	tty->ysize = 24;
	tty->xpixels = 0;
	tty->ypixels = 0;

	tty->xstart = 0;
	tty->xpos = tty->ypos = 0;
	tty->max_xpos = tty->max_ypos = 0;
	tty->text_ok = 0;

	tty->out_state_buffer_size = 0;

	tty->last_char = -1;
	WQ_INIT(&tty->state_wait, "TTY$STATE_WAIT");
	for (i = 0; i < N_CONTROLS; i++) {
		WQ_INIT(&tty->control_wait[i], "TTY$CONTROL_WAIT");
	}

	tty->vmode = NULL;
	tty->vc = 0;
	memcpy(&tty->current_videomode, &default_params.vm, sizeof(struct tty_videomode_params));
	memcpy(&tty->current_keyboard_params, &default_params.kbd, sizeof(struct tty_keyboard_params));
	
	tty->font_proc = NULL;
	tty->font_blob = NULL;
	tty->keymap_proc = NULL;
	tty->keymap_blob = NULL;

	tty->kmode = NULL;
	tty->kc = 0;

	tty->ispeed = -1;
	tty->ospeed = -1;

	/* the result does not have to be cryptographically unpredictable,
	   it just must be prone to random collisions */
	MD5Init(&md5);
	tim = TIMER_RANDOMNESS();
	MD5Update(&md5, (unsigned char *)&tim, sizeof tim);
	gettimeofday(&tv, NULL);
	MD5Update(&md5, (unsigned char *)&tv, sizeof tv);
	seq++;
	MD5Update(&md5, (unsigned char *)&seq, sizeof seq);
	MD5Final((unsigned char *)&digest, &md5);
	
	tty->tty_seq = digest[0];
	for (i = 0; i < N_CONTROLS; ) {
		MD5Init(&md5);
		MD5Update(&md5, (unsigned char *)digest, sizeof digest);
		MD5Final((unsigned char *)&digest, &md5);
		tty->control_seq[i++] = digest[0];
		if (__unlikely(i >= N_CONTROLS)) break;
		tty->control_seq[i++] = digest[1];
	}
	for (i = 0; i < MAX_PROC_DEPTH; ) {
		MD5Init(&md5);
		MD5Update(&md5, (unsigned char *)digest, sizeof digest);
		MD5Final((unsigned char *)&digest, &md5);
		tty->detach_seq[i++] = digest[0];
		if (__unlikely(i >= MAX_PROC_DEPTH)) break;
		tty->detach_seq[i++] = digest[1];
	}

	return 0;
}

void TTY$DESTROY(TTY *tty)
{
	int i;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_TTY)))
		KERNEL$SUICIDE("TTY$DESTROY AT SPL %08X", KERNEL$SPL);
	if (tty->buffer && tty->buffer != tty->nomem_buffer)
		KERNEL$FREE_KERNEL_PAGE(tty->buffer, VM_TYPE_WIRED_MAPPED);
	if (__unlikely(tty->buffer_to_write != NULL) && tty->buffer_to_write != tty->nomem_buffer)
		KERNEL$FREE_KERNEL_PAGE(tty->buffer_to_write, VM_TYPE_WIRED_MAPPED);
	KERNEL$DEL_TIMER(&tty->update_timer);
	KERNEL$DEL_TIMER(&tty->write_timer);
	WQ_WAKE_ALL(&tty->write_wait);
	WQ_WAKE_ALL(&tty->read_wait);
	WQ_WAKE_ALL(&tty->in_wait);
	tty_wake_state(tty);
	for (i = 0; i < N_CONTROLS; i++)
		WQ_WAKE_ALL(&tty->control_wait[i]);
}

DECL_IOCALL(DLL$LOAD, SPL_FS, DLINITRQ)
{
	KERNEL$SLAB_INIT(&tty_extra_params[0], sizeof(struct tty_extra_params), 0, VM_TYPE_CACHED_MAPPED, NULL, NULL, "TTY$EXTRA_PARAMS_USER");
	KERNEL$SLAB_INIT(&tty_extra_params[1], sizeof(struct tty_extra_params), 0, VM_TYPE_WIRED_MAPPED, NULL, NULL, "TTY$EXTRA_PARAMS_KERNEL");
	RQ->status = 0;
	RETURN_AST(RQ);
}

DECL_IOCALL(DLL$UNLOAD, SPL_FS, DLINITRQ)
{
	KERNEL$SLAB_DESTROY(&tty_extra_params[0]);
	KERNEL$SLAB_DESTROY(&tty_extra_params[1]);
	RQ->status = 0;
	RETURN_AST(RQ);
}

