#include <SPAD/AC.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SYNC.H>
#include <STRING.H>
#include <SPAD/WQ.H>
#include <SPAD/HASH.H>
#include <SPAD/TTY.H>
#include <SPAD/IOCTL.H>
#include <SPAD/THREAD.H>
#include <STDLIB.H>
#include <VALUES.H>

#define TTY_HASH_SIZE		4096

#define TTY_PREFIX		"TTY$"
#define TTY_PREFIX_LEN		4

#define TTY_BUFFER_SIZE		8192
#define CREATE_BUFFER_SIZE	(TTY_BUFFER_SIZE + TTYSTR_LEN)
#define READ_BUFFER_SIZE	1024

static void *lnte, *dlrq;

typedef struct {
	LIST_ENTRY hash;
	TTY tty;
	long tclass;
	char name[TTYSTR_LEN];
	int state;
	int writeout_pos;
	WQ read_wait;
	void *lnte;
} TTYS;

#define TTYS_WRITEOUT_READY	1
#define TTYS_DTTY		2

static XLIST_HEAD tty_hash[TTY_HASH_SIZE];

static char read_buffer[READ_BUFFER_SIZE];

static MTX_DECL(CREATE_MTX, "TTYSRVR$CREATE_MTX");
static THREAD_RQ trq;
static IOCTLRQ *irq;
static TTYS *ittys;
static char tty_name[TTY_PREFIX_LEN + 4 + TTYSTR_LEN + 1];
static char create_buffer[CREATE_BUFFER_SIZE];

static int unload(void *p, void **release, const char * const argv[]);
static void *ttysrvr_lookup(HANDLE *h, char *str, int open_flags);
static IO_STUB ttysrvr_ioctl;
static long register_device_thread(void *p);
static AST_STUB tty_registered;

static const HANDLE_OPERATIONS ttysrvr_operations = {
	SPL_X(SPL_DEV),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,			/* clone */
	ttysrvr_lookup,		/* lookup */
	NULL,			/* create */
	NULL,			/* delete */
	NULL,			/* rename */
	NULL,			/* lookup_io */
	NULL,			/* instantiate */
	NULL,			/* leave */
	NULL,			/* detach */
	NULL,			/* close */
	KERNEL$NO_OPERATION,	/* READ */
	KERNEL$NO_OPERATION,	/* WRITE */
	KERNEL$NO_OPERATION,	/* AREAD */
	KERNEL$NO_OPERATION,	/* AWRITE */
	ttysrvr_ioctl,		/* IOCTL */
	KERNEL$NO_OPERATION,	/* BIO */
	KERNEL$NO_OPERATION,	/* PKTIO */
};

static void init_root(HANDLE *h, void *data)
{
	h->op = &ttysrvr_operations;
	h->flags2 = 0;
}

static void *ttysrvr_lookup(HANDLE *h, char *str, int open_flags)
{
	if (__unlikely(str[0] == '^')) return __ERR_PTR(-EBADMOD);
	if (__unlikely(h->flags2 != 0)) return __ERR_PTR(-ENOENT);
	if (__unlikely(strlen(str) > 4)) return __ERR_PTR(-ENOENT);
	strncpy((char *)&h->flags2, str, 4);
	return NULL;
}

static void tty_destroy(TTYS *ttys)
{
	DEL_FROM_LIST(&ttys->hash);
	KERNEL$UNREGISTER_DEVICE(ttys->lnte, LN_DONT_WAKE_ALL);
	RAISE_SPL(SPL_TTY);
	TTY$DESTROY(&ttys->tty);
	LOWER_SPL(SPL_DEV);
	WQ_WAKE_ALL(&ttys->read_wait);
	free(ttys);
}

static __finline__ int tty_mkhash(long tclass, char *name)
{
	int hash = tclass;
	char *n = name;
	quickcasehash(n, n < name + TTYSTR_LEN, hash);
	return hash & (TTY_HASH_SIZE - 1);
}

static __finline__ TTYS *tty_lookup(long tclass, char *name)
{
	TTYS *ttys;
	int hash = tty_mkhash(tclass, name);
	XLIST_FOR_EACH(ttys, &tty_hash[hash], TTYS, hash)
		if (__likely(ttys->tclass == tclass && !memcmp(ttys->name, name, TTYSTR_LEN)))
			return ttys;
	return NULL;
}

static void clear_ttys(long tclass)
{
	int i;
	TTYS *t, *tt;
	for (i = 0; i < TTY_HASH_SIZE; i++) XLIST_FOR_EACH_UNLIKELY(t, &tty_hash[i], TTYS, hash) {
		if (t->tclass == tclass || !tclass) {
			tt = LIST_STRUCT(t->hash.prev, TTYS, hash);
			tty_destroy(t);
			t = tt;
		}
	}
}

static void tty_writeout(TTY *tty)
{
	TTYS *ttys = GET_STRUCT(tty, TTYS, tty);
	if (__unlikely(ttys->state & TTYS_WRITEOUT_READY))
		KERNEL$SUICIDE("tty_writeout: CALLED TWICE");
	ttys->state |= TTYS_WRITEOUT_READY;
	ttys->writeout_pos = 0;
	WQ_WAKE_ALL_PL(&ttys->read_wait);
}

static void tty_dtty(TTY *tty)
{
	TTYS *ttys = GET_STRUCT(tty, TTYS, tty);
	ttys->state |= TTYS_DTTY;
	WQ_WAKE_ALL_PL(&ttys->read_wait);
}

static DECL_IOCALL(ttysrvr_ioctl, SPL_DEV, IOCTLRQ)
{
	HANDLE *h = RQ->handle;
	int r;
	TTYS *ttys;
	if (__unlikely(h->op != &ttysrvr_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(h->name_addrspace, SPL_X(SPL_DEV));
	switch (RQ->ioctl) {
		case IOCTL_TTYSRVR_CLEAR: {
			clear_ttys(h->flags2);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTYSRVR_MAKETTY: {
			int k;
			if (__unlikely(!h->flags2)) {
				RQ->status = -EINVAL;
				RETURN_AST(RQ);
			}
			if (__unlikely(RQ->v.len < TTYSTR_LEN) || __unlikely(RQ->v.len > CREATE_BUFFER_SIZE)) {
				RQ->status = -EMSGSIZE;
				RETURN_AST(RQ);
			}
			MTX_LOCK(&CREATE_MTX, RQ, KERNEL$WAKE_IOCTL, RETURN);
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, create_buffer, RQ->v.len))) {
				MTX_UNLOCK(&CREATE_MTX);
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely((ttys = tty_lookup(h->flags2, create_buffer)) != NULL)) {
				tty_destroy(ttys);
			}
			if (__unlikely(!(ttys = malloc(sizeof(TTYS))))) {
				KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$WAKE_IOCTL, sizeof(TTYS));
				MTX_UNLOCK(&CREATE_MTX);
				RETURN;
			}
			ttys->tclass = h->flags2;
			memcpy(ttys->name, create_buffer, TTYSTR_LEN);
			if (__unlikely(!(ttys->tty.buf = malloc(TTY_BUFFER_SIZE)))) {
				free(ttys);
				KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$WAKE_IOCTL, TTY_BUFFER_SIZE);
				MTX_UNLOCK(&CREATE_MTX);
				RETURN;
			}
			ttys->state = 0;
			WQ_INIT(&ttys->read_wait, "TTYSRVR$READ_WAIT");
			ttys->tty.buf_size = TTY_BUFFER_SIZE;
			ttys->tty.writeout = tty_writeout;
			ttys->tty.dtty = tty_dtty;
			if (__unlikely(r = TTY$CREATE(&ttys->tty))) {
				free(ttys->tty.buf);
				free(ttys);
				MTX_UNLOCK(&CREATE_MTX);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			ttys->tty.mode |= TTYF_NOECHO;
			memcpy(tty_name, TTY_PREFIX, TTY_PREFIX_LEN);
			memcpy(tty_name + TTY_PREFIX_LEN, &ttys->tclass, 4);
			k = strnlen(tty_name + TTY_PREFIX_LEN, 4);
			memcpy(tty_name + TTY_PREFIX_LEN + k, ttys->name, TTYSTR_LEN);
			tty_name[TTY_PREFIX_LEN + k + TTYSTR_LEN] = 0;
			irq = RQ;
			ittys = ttys;

			trq.fn = tty_registered;
			trq.thread_main = register_device_thread;
			trq.p = NULL;
			trq.error = NULL;
			trq.cwd = NULL;
			trq.std_in = -1;
			trq.std_out = -1;
			trq.std_err = -1;
			trq.dlrq = NULL;
			trq.thread = NULL;
			trq.spawned = 0;
			RETURN_IORQ_CANCELABLE(&trq, KERNEL$THREAD, RQ);
		}
		case IOCTL_TTYSRVR_SENDCHR: {
			unsigned long rl;
			if (__unlikely(RQ->v.len <= TTYSTR_LEN)) {
				if (__likely(RQ->v.len == TTYSTR_LEN)) RQ->status = 0;
				else RQ->status = -EMSGSIZE;
				RETURN_AST(RQ);
			}
			rl = RQ->v.len;
			if (__unlikely(RQ->v.len > READ_BUFFER_SIZE)) {
				RQ->v.len = READ_BUFFER_SIZE;
			}
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, read_buffer, RQ->v.len))) {
				RQ->v.len = rl;
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(ttys = tty_lookup(h->flags2, read_buffer)))) {
				RQ->v.len = rl;
				RQ->status = -ENOENT;
				RETURN_AST(RQ);
			}
			RAISE_SPL(SPL_TTY);
			r = TTY$IN(&ttys->tty, read_buffer + TTYSTR_LEN, RQ->v.len - TTYSTR_LEN);
			/* when pasting into a raw terminal, block */
			if (__unlikely(!(ttys->tty.bits & TTY_PERMANENT_RAW)) &&
			    __unlikely(!(ttys->tty.mode & TTYF_RAW)))
				r = RQ->v.len - TTYSTR_LEN;
			RQ->v.len = rl;
			if (__unlikely(!r) && __likely(!RQ->param)) {
				WQ_WAIT_F(&ttys->tty.in_wait, RQ);
				LOWER_SPL(SPL_DEV);
				RETURN;
			}
			LOWER_SPL(SPL_DEV);
			RQ->status = r;
			RETURN_AST(RQ);
		}
		case IOCTL_TTYSRVR_RCVCHR: {
			long ll;
			long l = RQ->v.len;
			if (__unlikely(l <= TTYSTR_LEN)) {
				RQ->status = -EMSGSIZE;
				RETURN_AST(RQ);
			}
			RQ->v.len = TTYSTR_LEN;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, read_buffer, TTYSTR_LEN))) {
				RQ->v.len = l;
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			RQ->v.len = l;
			if (__unlikely(!(ttys = tty_lookup(h->flags2, read_buffer)))) {
				RQ->status = -ENOENT;
				RETURN_AST(RQ);
			}
			if (__unlikely(ttys->state & TTYS_DTTY)) {
				tty_destroy(ttys);
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			if (!(ttys->state & TTYS_WRITEOUT_READY)) {
				WQ_WAIT_F(&ttys->read_wait, RQ);
				RETURN;
			}
			ll = ttys->tty.buffer_to_write_len - ttys->writeout_pos;
			if (ll > l - TTYSTR_LEN) ll = l - TTYSTR_LEN;
			RQ->v.ptr += TTYSTR_LEN;
			RQ->v.len = ll;
			if (__unlikely(r = KERNEL$PUT_IOCTL_STRUCT(RQ, ttys->tty.buffer_to_write + ttys->writeout_pos, ll))) {
				RQ->v.ptr -= TTYSTR_LEN;
				RQ->v.len = l;
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			RQ->v.ptr -= TTYSTR_LEN;
			RQ->v.len = l;
			if ((ttys->writeout_pos += ll) == ttys->tty.buffer_to_write_len) {
				ttys->state &= ~TTYS_WRITEOUT_READY;
				RAISE_SPL(SPL_TTY);
				TTY$WRITE_DONE(&ttys->tty);
				LOWER_SPL(SPL_DEV);
			}
			RQ->status = ll;
			RETURN_AST(RQ);
		}
		case IOCTL_TTYSRVR_CLOSETTY: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, read_buffer, TTYSTR_LEN))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(ttys = tty_lookup(h->flags2, read_buffer)))) {
				RQ->status = -ENOENT;
				RETURN_AST(RQ);
			}
			tty_destroy(ttys);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTYSRVR_EOF: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, read_buffer, TTYSTR_LEN))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(ttys = tty_lookup(h->flags2, read_buffer)))) {
				RQ->status = -ENOENT;
				RETURN_AST(RQ);
			}
			RAISE_SPL(SPL_TTY);
			ttys->tty.bits |= TTY_PERMANENT_EOF;
			WQ_WAKE_ALL_PL(&ttys->tty.read_wait);
			LOWER_SPL(SPL_DEV);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTYSRVR_RAW: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, read_buffer, TTYSTR_LEN))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(ttys = tty_lookup(h->flags2, read_buffer)))) {
				RQ->status = -ENOENT;
				RETURN_AST(RQ);
			}
			RAISE_SPL(SPL_TTY);
			ttys->tty.bits |= TTY_PERMANENT_RAW;
			WQ_WAKE_ALL_PL(&ttys->tty.read_wait);
			LOWER_SPL(SPL_DEV);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTYSRVR_WINSIZE: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, read_buffer, TTYSTR_LEN))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(ttys = tty_lookup(h->flags2, read_buffer)))) {
				RQ->status = -ENOENT;
				RETURN_AST(RQ);
			}
			ttys->tty.xsize = (RQ->param >> 16) & 0xffff;
			ttys->tty.ysize = RQ->param & 0xffff;
			ttys->tty.bits |= TTY_LOCK_WINDOW_SIZE;
			WQ_WAKE_ALL(&ttys->tty.state_wait);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
		case IOCTL_TTYSRVR_GETAUTH: {
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, read_buffer, TTYSTR_LEN))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(!(ttys = tty_lookup(h->flags2, read_buffer)))) {
				RQ->status = -ENOENT;
				RETURN_AST(RQ);
			}
			RAISE_SPL(SPL_TTY);
			if (ttys->tty.bits & TTY_AUTH_POS) {
				ttys->tty.bits &= ~TTY_AUTH_POS;
				RQ->status = 0;
				RETURN_AST(RQ);
			}
			if (__unlikely(ttys->tty.bits & TTY_AUTH_NEG)) {
				ttys->tty.bits &= ~TTY_AUTH_NEG;
				RQ->status = 1;
				RETURN_AST(RQ);
			}
			if (RQ->param) {
				RQ->status = -EWOULDBLOCK;
				RETURN_AST(RQ);
			}
			WQ_WAIT_F(&ttys->tty.state_wait, RQ);
			RETURN;
		}
		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static long register_device_thread(void *p)
{
	return KERNEL$REGISTER_DEVICE(tty_name, "TTYSRVR.SYS", LN_DONT_WAKE_ALL, &ittys->tty, TTY$INIT_ROOT, NULL, NULL, NULL, NULL, &ittys->lnte, ":DMAN_SKIP", NULL);
}

static DECL_AST(tty_registered, SPL_DEV, THREAD_RQ)
{
	if (__unlikely(RQ->status < 0)) {
		irq->status = RQ->status;
		free(ittys->tty.buf);
		free(ittys);
		MTX_UNLOCK(&CREATE_MTX);
		RETURN_AST(irq);
	}
	RAISE_SPL(SPL_TTY);
	TTY$IN(&ittys->tty, create_buffer + TTYSTR_LEN, irq->v.len - TTYSTR_LEN);
	LOWER_SPL(SPL_DEV);
	ADD_TO_XLIST(&tty_hash[tty_mkhash(ittys->tclass, ittys->name)], &ittys->hash);
	MTX_UNLOCK(&CREATE_MTX);
	irq->status = 0;
	RETURN_AST(irq);
}

#define DEVICE_NAME	"SYS$TTYSRVR"

int main(int argc, char *argv[])
{
	int i, r;
	if (argc > 1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "TTYSRVR: SYNTAX ERROR");
		return -EBADSYN;
	}
	for (i = 0; i < TTY_HASH_SIZE; i++) INIT_XLIST(&tty_hash[i]);

	r = KERNEL$REGISTER_DEVICE(DEVICE_NAME, "TTYSRVR.SYS", 0, NULL, init_root, NULL, NULL, NULL, unload, &lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, DEVICE_NAME ": COULD NOT REGISTER DEVICE: %s", strerror(-r));
		return r;
	}
	strcpy(KERNEL$ERROR_MSG(), DEVICE_NAME);
	dlrq = KERNEL$TSR_IMAGE();
	return 0;
}

static int unload(void *p, void **release, const char * const argv[])
{
	int r;
	MTX_LOCK_SYNC(&CREATE_MTX);
	if (__unlikely(r = KERNEL$DEVICE_UNLOAD(lnte, argv))) {
		MTX_UNLOCK(&CREATE_MTX);
		return r;
	}
	RAISE_SPL(SPL_DEV);
	clear_ttys(0);
	LOWER_SPL(SPL_ZERO);
	MTX_UNLOCK(&CREATE_MTX);
	*release = dlrq;
	return 0;
}
