#include <SPAD/DEV_KRNL.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/VM.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/TIMER.H>
#include <ARCH/BSF.H>
#include <ARCH/BITOPS.H>
#include <ARCH/IO.H>
#include <SPAD/IRQ.H>
#include <SPAD/PCI.H>
#include <STDLIB.H>
#include <VALUES.H>

#include "EHCIBUST.H"

#include <SPAD/USBHC.H>
#include <SPAD/USBHUB.H>

#include "UHCIREG.H"

/*
 * UHCI sucks. I wrote this driver several times.
 *
 * The last problem to solve was that UHCI on VIA 586 mainboard corrupts
 * its state when the driver adds something to TD lists while the controller
 * reads it. I don't know why ... maybe they read it byte-by-byte?
 * UHCI on Intel or on VIA Athlon chipset behaves correctly in that case.
 *
 * So adding descriptors as described in the UHCI documentation doesn't really
 * work and I have to use the same trick as the Linux kernel --- append one
 * dummy TD that is never executed to every queue. And never modify link
 * pointer of an active TD.
 *
 * I'm wondering if modifying QH list while the controller reads it is ok...
 */

#define QHS_INACTIVE		1
#define QHS_ACTIVE		2
#define QHS_UNLINKING		3
#define QHS_UNLINKING_FREEING	4

typedef struct {
	io_t io;
	AST irq_ast;
	IRQ_HANDLE *irq_ctrl;
	unsigned flags;
	char errorlevel;
	__u32 *last_head_link;
	XLIST_HEAD intr_qh;
	TIMER fsbr_timer;
	u_jiffies_lo_t fsbr_time;
	unsigned n_ports;
	TIMER hub_timer;
	XLIST_HEAD unlinking;
	USB_ENDPOINT root_ctrl;
	USB_ENDPOINT root_intr;
	USB_REQUEST *rh_intr;
	u_jiffies_t last_port_change;
	XLIST_HEAD endpoints;
	__u32 area_dmaaddr;
	vspace_dmaunlock_t *area_dmaunlock;
	char dev_name[__MAX_STR_LEN];
	pci_id_t id;
	void *lnte;
	void *dlrq;
} UHCI;

#define FLAG_REVERT_OC				1
#define FLAG_DEAD				2

typedef struct __bounce_buffer BOUNCE_BUFFER;
typedef struct __data_page DATA_PAGE;
typedef struct __ed ED;

struct __bounce_buffer {
	BOUNCE_BUFFER *list;
	ED *ed;
	VPHYS vphys1;
	vspace_physunlock_t *physunlock1;
	VPHYS vphys2;
	vspace_physunlock_t *physunlock2;
	__u32 prev_dma_addr;
	vspace_dmaunlock_t *prev_dma_unlock;
	__u32 data_dma_addr;
	vspace_dmaunlock_t *data_dma_unlock;
};

typedef struct {
	HW_TD td;
	LIST_ENTRY list;
	USB_REQUEST *brq;
	__u32 data_dma_addr;
	__u32 desc_dma_addr;
	vspace_dmaunlock_t *data_dma_unlock;
	vspace_dmaunlock_t *desc_dma_unlock;
} TD;

struct __data_page {
	DATA_PAGE *next;
};

typedef struct {
	HW_QH qh;
	LIST_HEAD tds;
	UHCI *uhci;
	__u32 desc_dma_addr;
	char state;
	char type;
	unsigned short stride;
	__u16 cleanup_frame;
	u_jiffies_lo_t unlink_start;
	LIST_ENTRY list;

	TD *td_freelist;
	unsigned total_tds;
	unsigned required_tds;
	BOUNCE_BUFFER *bounce_freelist;
	unsigned total_bounces;
	unsigned required_bounces;
	DATA_PAGE *data_pages;
	void *free_space_ptr;

	vspace_dmaunlock_t *desc_dma_unlock;
} QH;

struct __ed {
	QH qh_default;
	unsigned n_bounce_requests;
	unsigned bounce_request_size;
	unsigned required_tds;
	unsigned required_bounces;
	char addr;
	char pipe;
	char type;
	QH *qh;
	unsigned pkt_size;
	__u32 td_flags_1_le;
	__u32 td_flags_2_le;
	__u32 d_le;
	USB_ENDPOINT ep;
	LIST_ENTRY endpoints_list;
};

typedef struct {
	FRAME_LIST fl;
	HW_TD sof_irq;
	HW_QH ctrl_start;
/* VIA must have queue heads aligned to 16 bytes. Intel accepts 8 bytes. */
	char pad1[8];
	HW_QH loop_start;
	char pad2[8];
	QH low_ctrl;
} AREA;

#define UHCI_SOF_IRQ_Q_LE(uhci)		__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, sof_irq))
#define UHCI_CTRL_START_Q_LE(uhci)	__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, ctrl_start) + UHCI_ADDR_Q)
#define UHCI_LOOP_START_Q_LE(uhci)	__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, loop_start) + UHCI_ADDR_Q)
#define UHCI_LOW_CTRL_Q_LE(uhci)	__32CPU2LE((uhci)->area_dmaaddr + __offsetof(AREA, low_ctrl.qh) + UHCI_ADDR_Q)

#define UHCI2USB(u)		((USB *)((char *)(u) + sizeof(UHCI)))
#define USB2UHCI(u)		((UHCI *)((char *)(u) - sizeof(UHCI)))
#define UHCI2AREA(u)		((AREA *)((char *)(u) - sizeof(AREA)))

#define UHCI_RESET_STEP		10
#define UHCI_RESET_TIMEOUT	100000
#define UHCI_HUB_CHECK_TIME	(JIFFIES_PER_SECOND / 2)
#define UHCI_HUB_RESET_TIME	((JIFFIES_PER_SECOND + 19) / 20)
#define UHCI_PORTCHG_QUIET	(JIFFIES_PER_SECOND / 4)
#define QH_CLEANUP_TIMEOUT	(JIFFIES_PER_SECOND / 10 + 1)
#define DEFAULT_FSBR_TIME	20	/* ms */
		/* 63 --- bytes per low-speed control frame */
#define MAX_PERIODIC_BANDWIDTH	(USB_HIGH_SPEED / USB_FPS * 19 / 20 - 63 * (USB_HIGH_SPEED / USB_LOW_SPEED))

#define UHCI_READ_8(u, i)		io_inb((u)->io + (i))
#define UHCI_READ_16(u, i)		io_inw((u)->io + (i))
#define UHCI_READ_32(u, i)		io_inl((u)->io + (i))
#define UHCI_WRITE_8(u, i, v)		io_outb((u)->io + (i), (v))
#define UHCI_WRITE_16(u, i, v)		io_outw((u)->io + (i), (v))
#define UHCI_WRITE_32(u, i, v)		io_outl((u)->io + (i), (v))
#define UHCI_READ_PORT(u, p)		UHCI_READ_16(u, UHCI_PORTSC + (unsigned)(p) * UHCI_PORTSC_NEXT)
#define UHCI_WRITE_PORT(u, p, v)	UHCI_WRITE_16(u, UHCI_PORTSC + (unsigned)(p) * UHCI_PORTSC_NEXT, (v))
#define UHCI_READ_FRNUM(u)		(UHCI_READ_16(u, UHCI_FRNUM) & UHCI_FRNUM_MASK)

static __finline__ void UHCI_SET_PORT_BIT(UHCI *uhci, unsigned port, unsigned bit)
{
	__u16 val = UHCI_READ_PORT(uhci, port);
	val &= ~(UHCI_PORTSC_RWC | UHCI_PORTSC_WZ);
	val |= bit;
	UHCI_WRITE_PORT(uhci, port, val);
}

static __finline__ void UHCI_CLEAR_PORT_BIT(UHCI *uhci, unsigned port, unsigned bit)
{
	__u16 val = UHCI_READ_PORT(uhci, port);
	val &= ~(UHCI_PORTSC_RWC | UHCI_PORTSC_WZ);
	val &= ~bit;
	val |= bit & UHCI_PORTSC_RWC;
	UHCI_WRITE_PORT(uhci, port, val);
}

#define CONTIG_SIZE	(sizeof(AREA) + sizeof(UHCI) + USB$SIZEOF_USB)
#define CONTIG_ALIGN	16

static USB_ENDPOINT *UHCI_ALLOC_ENDPOINT(USB *usb, unsigned speed, int type, int addr, int pipe, unsigned pkt_size, int interval, unsigned n_rq, unsigned rq_size);
static void UHCI_DISABLE_ENDPOINT(USB_ENDPOINT *endpt);
static void UHCI_FREE_ENDPOINT(USB_ENDPOINT *endpt);

static const struct pci_id_s uhci_pci[] = {
	{ PCI_VENDOR_ID_VIA, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, (PCI_CLASS_SERIAL_USB << 16) | (0x00 << 8), 0xFFFFFF00U, "UHCI (VIA)", FLAG_REVERT_OC },
	{ PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID, (PCI_CLASS_SERIAL_USB << 16) | (0x00 << 8), 0xFFFFFF00U, "UHCI (INTEL)", 0 },
	{ 0 },
};

static const USB_HC_OPERATIONS UHCI_HC_OPERATIONS = {
	0,
	UHCI_ALLOC_ENDPOINT,
	UHCI_DISABLE_ENDPOINT,
	UHCI_FREE_ENDPOINT,
};

static __finline__ void UHCI_ENABLE_SOF_IRQ(UHCI *uhci)
{
	UHCI2AREA(uhci)->ctrl_start.element_link = UHCI_SOF_IRQ_Q_LE(uhci);
}

static __finline__ void UHCI_DISABLE_SOF_IRQ(UHCI *uhci)
{
	UHCI2AREA(uhci)->ctrl_start.element_link = __32CPU2LE(UHCI_ADDR_T);
}

static __finline__ void UHCI_SET_FSBR(UHCI *uhci)
{
	*uhci->last_head_link = UHCI2AREA(uhci)->loop_start.head_link;
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	KERNEL$SET_TIMER(uhci->fsbr_time, &uhci->fsbr_timer);
}

static __finline__ void UHCI_CLEAR_FSBR(UHCI *uhci)
{
	*uhci->last_head_link = __32CPU2LE(UHCI_ADDR_T);
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	SET_TIMER_NEVER(&uhci->fsbr_timer);
}

static __finline__ TD *UHCI_GET_TD(QH *qh)
{
	TD *td = qh->td_freelist;
#if __DEBUG >= 1
	if (__unlikely(!td))
		KERNEL$SUICIDE("UHCI_GET_TD: NO FREE TDS");
#endif
	qh->td_freelist = (TD *)td->brq;
	return td;
}

static __finline__ void UHCI_FREE_TD(QH *qh, TD *td)
{
	td->brq = (void *)qh->td_freelist;
	qh->td_freelist = td;
}

static BOUNCE_BUFFER *UHCI_GET_BB(QH *qh)
{
	BOUNCE_BUFFER *bb = qh->bounce_freelist;
#if __DEBUG >= 1
	if (__unlikely(!bb))
		KERNEL$SUICIDE("UHCI_GET_BB: NO FREE BOUNCE BUFFERS");
#endif
	qh->bounce_freelist = bb->list;
	return bb;
}

static __finline__ void UHCI_FREE_BB(QH *qh, BOUNCE_BUFFER *bb)
{
	bb->list = qh->bounce_freelist;
	qh->bounce_freelist = bb;
}

static __finline__ void UHCI_SET_DUMMY_TD(TD *td)
{
	td->td.link = __32CPU2LE(0);
	td->td.flags1 = __32CPU2LE(0);
	td->td.flags2 = __32CPU2LE(TD_FLAGS2_MAXLEN | TD_FLAGS2_PID_OUT);
	td->td.ptr = __32CPU2LE(0);
	td->brq = NULL;
}

static void UHCI_UNBOUNCE_COMMON(BOUNCE_BUFFER *bb);

static void UHCI_UNBOUNCE_COPY(__u32 dma)
{
	void *p1;
	BOUNCE_BUFFER *bb = (BOUNCE_BUFFER *)KERNEL$DMA_2_VIRT(dma) - 1;
	bb->prev_dma_unlock(bb->prev_dma_addr);
	p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys1.ptr);
	memcpy(p1, bb + 1, bb->vphys1.len);
	KERNEL$UNMAP_PHYSICAL_BANK(p1);
	p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys2.ptr);
	memcpy(p1, (char *)(bb + 1) + bb->vphys1.len, bb->vphys2.len);
	KERNEL$UNMAP_PHYSICAL_BANK(p1);
	UHCI_UNBOUNCE_COMMON(bb);
}

static void UHCI_UNBOUNCE(__u32 dma)
{
	BOUNCE_BUFFER *bb = (BOUNCE_BUFFER *)KERNEL$DMA_2_VIRT(dma) - 1;
	bb->prev_dma_unlock(bb->prev_dma_addr);
	UHCI_UNBOUNCE_COMMON(bb);
}

static void UHCI_UNBOUNCE_COMMON(BOUNCE_BUFFER *bb)
{
	bb->physunlock1(bb->vphys1.ptr);
	bb->physunlock2(bb->vphys2.ptr);
	UHCI_FREE_BB(bb->ed->qh, bb);
}

static void UHCI_FREE_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq);
static void UHCI_FREE_BULK_INTR_CTRL_REQUEST_LIST(USB_REQUEST *brq, LIST_HEAD *new_list);
static void UHCI_SET_VERTICAL(USB_REQUEST *brq);

static __finline__ void LINK_TD(LIST_HEAD *tds, TD *td)
{
	if (tds->prev != (LIST_ENTRY *)tds)
		LIST_STRUCT(tds->prev, TD, list)->td.link = __32CPU2LE(td->desc_dma_addr);
	ADD_TO_LIST_END(tds, &td->list);
}

static int UHCI_PREPARE_BULK_INTR_REQUEST(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	ED *ed = GET_STRUCT(endpt, ED, ep);
	TD *td;
	VDMA vdma;
	__u32 data_dma_addr;
	vspace_dmaunlock_t *data_dma_unlock;
	int r;
	DECL_LIST(tds);
	brq->internal1 = (unsigned long)ed;
	HC_CHECK_SPL("UHCI_PREPARE_BULK_INTR_REQUEST");
	vdma.spl = SPL_X(SPL_USB);
	if (__unlikely(brq->flags & USBRQ_TERMINATE_SHORT)) if (__likely(!(ed->pkt_size & (ed->pkt_size - 1))) ? brq->v.len & (ed->pkt_size - 1) : brq->v.len % ed->pkt_size) brq->flags &= ~USBRQ_TERMINATE_SHORT;

	brq->status = brq->v.len;
	if (__unlikely(!brq->v.len)) goto zero_packet;
	next_run:
	RAISE_SPL(SPL_VSPACE);
	data_dma_unlock = brq->v.vspace->op->vspace_dmalock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &vdma);
	if (__unlikely(!data_dma_unlock)) {
		r = -EVSPACEFAULT;
		ret_err:
		if (!LIST_EMPTY(&tds)) {
			UHCI_FREE_BULK_INTR_CTRL_REQUEST_LIST(brq, &tds);
		}
		if (__unlikely(r != -EVSPACEFAULT)) brq->v.len = 0;
		return r;
	}
	data_dma_addr = vdma.ptr;
	brq->v.ptr += vdma.len;
	brq->v.len -= vdma.len;

	td = NULL;	/* warning, go away */

		/* change to > to test bounce */
	while (__likely(vdma.len >= ed->pkt_size)) {
		vdma.len -= ed->pkt_size;
		td = UHCI_GET_TD(ed->qh);
		LINK_TD(&tds, td);
		td->td.ptr = __32CPU2LE(vdma.ptr);
		vdma.ptr += ed->pkt_size;
		td->td.flags1 = ed->td_flags_1_le;
		td->td.flags2 = ed->td_flags_2_le;
		ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
		td->data_dma_unlock = KERNEL$NULL_VSPACE_DMAUNLOCK;
		td->brq = brq;
	}
	if (__unlikely(vdma.len)) {
		if (__likely(!brq->v.len)) {
			td = UHCI_GET_TD(ed->qh);
			LINK_TD(&tds, td);
			td->td.ptr = __32CPU2LE(vdma.ptr);
			td->td.flags1 = ed->td_flags_1_le;
			td->td.flags2 = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_MAXLEN)) | __32CPU2LE((vdma.len - 1) << __BSF_CONST(TD_FLAGS2_MAXLEN));
			ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
			td->brq = brq;
			goto set_dma_unlock;
		} else {
			unsigned long orig_len;
			BOUNCE_BUFFER *bb = UHCI_GET_BB(ed->qh);
			bb->prev_dma_unlock = data_dma_unlock;
			bb->prev_dma_addr = data_dma_addr;
			brq->v.ptr -= vdma.len;
			brq->v.len += vdma.len;
			orig_len = brq->v.len;
			brq->v.len = vdma.len;
			bb->vphys1.spl = SPL_X(SPL_USB);
			RAISE_SPL(SPL_VSPACE);
			bb->physunlock1 = brq->v.vspace->op->vspace_physlock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &bb->vphys1);
			if (__unlikely(bb->vphys1.len != brq->v.len)) {
				if (!bb->vphys1.len) {
					r = -EVSPACEFAULT;
				} else {
					r = -EINVAL;
					bb_ret_unl:
					bb->physunlock1(bb->vphys1.ptr);
				}
				brq->v.len = orig_len;
				td->data_dma_unlock = bb->prev_dma_unlock;
				td->data_dma_addr = bb->prev_dma_addr;
				UHCI_FREE_BB(ed->qh, bb);
				goto ret_err;
			}
			brq->v.ptr += brq->v.len;
			orig_len -= brq->v.len;
			brq->v.len = orig_len > ed->pkt_size - brq->v.len ? ed->pkt_size - brq->v.len : orig_len;
			bb->vphys2.spl = SPL_X(SPL_USB);
			RAISE_SPL(SPL_VSPACE);
			bb->physunlock2 = brq->v.vspace->op->vspace_physlock(&brq->v, __likely(USB_IS_IN(ed->type)) ? PF_WRITE : PF_READ, &bb->vphys2);
			if (__unlikely(bb->vphys2.len != brq->v.len)) {
				if (!bb->vphys2.len) {
					r = -EVSPACEFAULT;
				} else {
					r = -EINVAL;
					bb->physunlock2(bb->vphys2.ptr);
				}
				goto bb_ret_unl;
			}
			brq->v.ptr += brq->v.len;
			orig_len -= brq->v.len;
			brq->v.len = orig_len;
			if (__unlikely(!USB_IS_IN(ed->type))) {
				void *p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys1.ptr);
				memcpy(bb + 1, p1, bb->vphys1.len);
				KERNEL$UNMAP_PHYSICAL_BANK(p1);
				p1 = KERNEL$MAP_PHYSICAL_BANK(bb->vphys2.ptr);
				memcpy((char *)(bb + 1) + bb->vphys1.len, p1, bb->vphys2.len);
				KERNEL$UNMAP_PHYSICAL_BANK(p1);
			}

			td = UHCI_GET_TD(ed->qh);
			LINK_TD(&tds, td);
			td->td.ptr = __32CPU2LE(bb->data_dma_addr);
			td->td.flags1 = ed->td_flags_1_le;
			td->td.flags2 = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_MAXLEN)) | __32CPU2LE((bb->vphys1.len + bb->vphys2.len - 1) << __BSF_CONST(TD_FLAGS2_MAXLEN));
			ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
			td->brq = brq;
			td->data_dma_unlock = __likely(USB_IS_IN(ed->type)) ? UHCI_UNBOUNCE_COPY : UHCI_UNBOUNCE;
			td->data_dma_addr = bb->data_dma_addr;
		}
	} else {
		set_dma_unlock:
		td->data_dma_unlock = data_dma_unlock;
		td->data_dma_addr = data_dma_addr;
	}
	if (brq->v.len) goto next_run;
	if (__unlikely(brq->flags & USBRQ_TERMINATE_SHORT)) {
		zero_packet:
		td = UHCI_GET_TD(ed->qh);
		LINK_TD(&tds, td);
		td->td.ptr = __32CPU2LE(0);
		td->td.flags1 = ed->td_flags_1_le;
		td->td.flags2 = ed->td_flags_2_le | __32CPU2LE(TD_FLAGS2_MAXLEN);
		ed->td_flags_2_le ^= __32CPU2LE(TD_FLAGS2_D);
		td->brq = brq;
		td->data_dma_unlock = KERNEL$NULL_VSPACE_DMAUNLOCK;
	}
	if (__likely(!(brq->flags & USBRQ_DELAY_INTERRUPT)))
		td->td.flags1 |= __32CPU2LE(TD_FLAGS1_IOC);
	tds.next->prev = tds.prev->next = (LIST_ENTRY *)(void *)&brq->v.ptr;
	memcpy(&brq->v.ptr, &tds, sizeof(LIST_HEAD));
	if (__unlikely(!ed->qh_default.uhci->fsbr_time)) {
		brq->flags |= USBRQ_NO_FSBR;
	}
	if (__unlikely(brq->flags & USBRQ_NO_FSBR) && __likely(USB_IS_BULK(ed->type))) UHCI_SET_VERTICAL(brq);
	return 0;
}

static void UHCI_CHANGE_TOGGLE(USB_REQUEST *brq);
static void UHCI_ACTIVATE_QH(QH *qh, USB_REQUEST *brq);

static void UHCI_POST_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
#define new_list	((LIST_HEAD *)(void *)&brq->v.ptr)
	TD *first_new_td, *second_new_td, *last_new_td, *last_old_td;
	ED *ed;
	QH *qh;
	__u32 flags1;
	HC_CHECK_SPL("UHCI_POST_BULK_INTR_CTRL_REQUEST");
	ed = (ED *)brq->internal1;
	first_new_td = LIST_STRUCT(new_list->next, TD, list);
	if (__unlikely((first_new_td->td.flags2 ^ ed->d_le) & __32CPU2LE(TD_FLAGS2_D)) && !USB_IS_CTRL(ed->type)) {
		UHCI_CHANGE_TOGGLE(brq);
		return;
	}
	qh = ed->qh;
	last_new_td = LIST_STRUCT(new_list->prev, TD, list);
	ed->d_le = (~last_new_td->td.flags2) & __32CPU2LE(TD_FLAGS2_D);
	last_old_td = LIST_STRUCT(qh->tds.prev, TD, list);
	last_old_td->td.flags2 = first_new_td->td.flags2;
	last_old_td->td.ptr = first_new_td->td.ptr;
	last_old_td->brq = brq;
	last_old_td->data_dma_addr = first_new_td->data_dma_addr;
	last_old_td->data_dma_unlock = first_new_td->data_dma_unlock;
	/*__debug_printf("", "new list 1 %p, %p\n", new_list->next, new_list->prev);*/
	DEL_FROM_LIST(&first_new_td->list);
	/*__debug_printf("", "new list 2 %p, %p\n", new_list->next, new_list->prev);*/
	ADD_TO_LIST_END(new_list, &first_new_td->list);
	second_new_td = LIST_STRUCT(new_list->next, TD, list);
	last_old_td->td.link = __32CPU2LE(second_new_td->desc_dma_addr);
	last_old_td->list.next = &second_new_td->list;
	second_new_td->list.prev = &last_old_td->list;
	LIST_STRUCT(first_new_td->list.prev, TD, list)->td.link = __32CPU2LE(first_new_td->desc_dma_addr);
	first_new_td->list.next = (LIST_ENTRY *)(void *)&qh->tds;
	qh->tds.prev = &first_new_td->list;
#if __DEBUG >= 2
	{
		LIST_ENTRY *e = (LIST_ENTRY *)(void *)&qh->tds;
		do {
			if (__unlikely(e->next->prev != e) || __unlikely(e->prev->next != e))
				KERNEL$SUICIDE("UHCI_POST_BULK_INTR_CTRL_REQUEST: LIST CORRUPTED: &qh->tds %p, e %p, e->next %p, e->next->prev %p, e->prev %p, &qh->tds, e->prev->next %p, last_old_td %p, first_new_td %p, last_new_td %p, second_new_td %p", &qh->tds, e, e->next, e->next->prev, e->prev, e->prev->next, last_old_td, first_new_td, last_new_td, second_new_td);
			if (__likely(e != (LIST_ENTRY *)(void *)&qh->tds) && __likely(e->next != (LIST_ENTRY *)(void *)&qh->tds)) {
				if (__unlikely(__32LE2CPU(LIST_STRUCT(e, TD, list)->td.link & ~UHCI_ADDR_VF) != LIST_STRUCT(e->next, TD, list)->desc_dma_addr))
					KERNEL$SUICIDE("UHCI_POST_BULK_INTR_CTRL_REQUEST: DMA LINK CORRUPTED: e->td.link %08X, e->next->dma_desc_addr %08X, &qh->tds %p, e %p, e->next %p, e->next->prev %p, e->prev %p, e->prev->next %p, last_old_td %p, first_new_td %p, last_new_td %p, second_new_td %p", __32LE2CPU(LIST_STRUCT(e, TD, list)->td.link), LIST_STRUCT(e->next, TD, list)->desc_dma_addr, &qh->tds, e, e->next, e->next->prev, e->prev, e->prev->next, last_old_td, first_new_td, last_new_td, second_new_td);
			}
		} while ((e = e->next) != (LIST_ENTRY *)(void *)&qh->tds);
	}
#endif
	flags1 = first_new_td->td.flags1;
	UHCI_SET_DUMMY_TD(first_new_td);
	__write_barrier();
	last_old_td->td.flags1 = flags1;
	if (__unlikely(qh->state == QHS_INACTIVE)) UHCI_ACTIVATE_QH(qh, brq);
	else if (__likely(!(brq->flags & USBRQ_NO_FSBR)) && __likely(USB_IS_BULK(qh->type))) UHCI_SET_FSBR(qh->uhci);
	brq->v.len = brq->status;
	brq->status = 0;
#undef new_list
}

static UHCI *UHCI_PROCESS_SCHEDULE(UHCI *uhci);
static UHCI *UHCI_SPECIAL_IRQ(UHCI *uhci, __u16 sts);

static DECL_IRQ_AST(UHCI_IRQ, SPL_USB, AST)
{
	UHCI *uhci = GET_STRUCT(RQ, UHCI, irq_ast);
	__u16 sts = UHCI_READ_16(uhci, UHCI_USBSTS);
	/*__debug_printf("irq, sts %04x\n", sts);*/
	if (__likely(!sts)) goto eoi;
	UHCI_WRITE_16(uhci, UHCI_USBSTS, sts);
	if (__unlikely(sts & (UHCI_USBSTS_HC_HALTED | UHCI_USBSTS_HC_PROCESS_ERROR | UHCI_USBSTS_HC_SYSTEM_ERROR | UHCI_USBSTS_RESUME_DETECT))) {
		uhci = UHCI_SPECIAL_IRQ(uhci, sts);
		goto eoi;
	}
	uhci = UHCI_PROCESS_SCHEDULE(uhci);
	eoi:
	KERNEL$UNMASK_IRQ(uhci->irq_ctrl);
	RETURN;
}

static void UHCI_ENDPOINT_STALL(QH *qh);
static int UHCI_PROCESS_QUEUE_HEAD_NOINLINE(QH *qh);
static void UHCI_PROCESS_UNLINKING_ENTRIES(UHCI *uhci, int force);

#define UHCI_QUEUE_IS_EMPTY(qh)		\
	((qh)->tds.next == (qh)->tds.prev)

#define UHCI_PROCESS_QUEUE_HEAD(qh)	\
	(__unlikely(!UHCI_QUEUE_IS_EMPTY(qh)) ? UHCI_PROCESS_QUEUE_HEAD_NOINLINE(qh) : 0)

static UHCI *UHCI_PROCESS_SCHEDULE(UHCI *uhci)
{
	__u32 addr_le, end_ctrl_le;
	QH *qh;

	UHCI_DISABLE_SOF_IRQ(uhci);

	XLIST_FOR_EACH(qh, &uhci->intr_qh, QH, list) {
		UHCI_PROCESS_QUEUE_HEAD(qh);
	}

	addr_le = UHCI2AREA(uhci)->ctrl_start.head_link;
	end_ctrl_le = UHCI_LOOP_START_Q_LE(uhci);
	while (__unlikely(addr_le != end_ctrl_le)) {
		qh = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le) & ~UHCI_ADDR_Q), QH, qh);
		addr_le = qh->qh.head_link;
		UHCI_PROCESS_QUEUE_HEAD(qh);
	}
	addr_le = UHCI2AREA(uhci)->loop_start.head_link;
	if (!(addr_le & __32CPU2LE(UHCI_ADDR_T))) {
		int fsbr_advance = 0;
		while (1) {
			int brk;
			qh = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le) & ~UHCI_ADDR_Q), QH, qh);
			brk = uhci->last_head_link == &qh->qh.head_link;
			addr_le = qh->qh.head_link;
			fsbr_advance |= UHCI_PROCESS_QUEUE_HEAD(qh);
			if (__unlikely(brk)) break;
		}
		if (__likely(fsbr_advance)) {
			UHCI_SET_FSBR(uhci);
		}
	}

	if (__unlikely(!XLIST_EMPTY(&uhci->unlinking))) UHCI_PROCESS_UNLINKING_ENTRIES(uhci, 0);

	return uhci;
}

#define TD_LEN_DIFFER(t)	(((__32LE2CPU((t)->td.flags2) >> __BSF_CONST(TD_FLAGS2_MAXLEN)) - (__32LE2CPU((t)->td.flags1) >> __BSF_CONST(TD_FLAGS1_ACTLEN))) & (TD_FLAGS1_ACTLEN >> __BSF_CONST(TD_FLAGS1_ACTLEN)))
#define TD_PKTLEN(t)		(((__32LE2CPU((t)->td.flags2) + (1 << __BSF_CONST(TD_FLAGS2_MAXLEN))) & TD_FLAGS2_MAXLEN) >> __BSF_CONST(TD_FLAGS2_MAXLEN))
#define TD_XFERLEN(t)		(((__32LE2CPU((t)->td.flags1) + (1 << __BSF_CONST(TD_FLAGS1_ACTLEN))) & TD_FLAGS1_ACTLEN) >> __BSF_CONST(TD_FLAGS1_ACTLEN))

static int UHCI_PROCESS_QUEUE_HEAD_NOINLINE(QH *qh)
{
	int fsbr_advance = 0;
	/* assume that there is at least one entry to process */
	TD *td = LIST_STRUCT(qh->tds.next, TD, list);
	do {
		USB_REQUEST *brq;
		TD *next_td;
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_ACTIVE))
			break;
		/* Intel PIIX errata: bitstuff error may be set without stall */
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_STALLED | TD_FLAGS1_STATUS_BITSTUFF_ERROR) || TD_LEN_DIFFER(td)) {
			UHCI_ENDPOINT_STALL(qh);
			break;
		}
		if (__unlikely(td->data_dma_unlock != KERNEL$NULL_VSPACE_DMAUNLOCK)) {
			td->data_dma_unlock(td->data_dma_addr);
		}
		brq = td->brq;
		if (__likely(!(brq->flags & USBRQ_NO_FSBR))) fsbr_advance = 1;
		next_td = LIST_STRUCT(td->list.next, TD, list);
		DEL_FROM_LIST(&td->list);
		if (__unlikely(brq != next_td->brq)) {
			brq->internal1 = 0;
			CALL_AST(brq);
		}
		UHCI_FREE_TD(qh, td);
	} while ((td = LIST_STRUCT(qh->tds.next, TD, list)) != LIST_STRUCT(qh->tds.prev, TD, list));
	return fsbr_advance;
}

static void UHCI_FSBR_TIMER(TIMER *t)
{
	int settime;
	UHCI *uhci;
	__u32 *addr_le;
	LOWER_SPL(SPL_USB);
	uhci = GET_STRUCT(t, UHCI, fsbr_timer);
	SET_TIMER_NEVER(&uhci->fsbr_timer);
	addr_le = &UHCI2AREA(uhci)->loop_start.head_link;
	settime = 0;
	while (addr_le != uhci->last_head_link) {
		QH *x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
		TD *first_td;
		if (UHCI_QUEUE_IS_EMPTY(x)) goto cont;
		first_td = LIST_STRUCT(x->tds.next, TD, list);
		if (__unlikely(first_td->brq->flags & USBRQ_NO_FSBR)) goto cont;
		__LOCK_OR32(&first_td->td.flags1, __32CPU2LE(TD_FLAGS1_IOC));
		if (!(first_td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_ACTIVE)))
			goto fsbr_en;
		cont:
		addr_le = &x->qh.head_link;
	}
	*uhci->last_head_link = __32CPU2LE(UHCI_ADDR_T);
	return;

	fsbr_en:
	*uhci->last_head_link = UHCI2AREA(uhci)->loop_start.head_link;
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	KERNEL$SET_TIMER(uhci->fsbr_time, &uhci->fsbr_timer);
}

static void UHCI_REPORT_ERROR(TD *td, int level, const char *msg, int only_stall);
static void UHCI_START_UNLINK_QH(QH *qh);
static void UHCI_POST_STALLED(USB_REQUEST *brq);
static int UHCI_TEST_PORT_CHANGE(UHCI *uhci);

__COLD_ATTR__ static int TD_SUBTR(USB_REQUEST *brq, TD *td, int diff)
{
	if ((unsigned long)diff > (unsigned long)brq->v.len) {
		UHCI_REPORT_ERROR(td, 0, "LEFT DATA LARGER THAN WHOLE TRANSFER LENGTH", 0);
		brq->v.len = 0;
		return -1;
	}
	brq->v.len -= diff;
	return 0;
}

__COLD_ATTR__ static void UHCI_ED_SET_ERROR(ED *ed, int error)
{
	TD *td;
	QH *qh = ed->qh;
	LIST_FOR_EACH(td, &qh->tds, TD, list)
		if (td->brq && __likely((ED *)td->brq->internal1 == ed) && td->brq->status >= 0)
			td->brq->status = error;
}

__COLD_ATTR__ static void UHCI_ENDPOINT_STALL(QH *qh)
{
	TD *td = LIST_STRUCT(qh->tds.next, TD, list);
	USB_REQUEST *brq = td->brq;
	int r;
	int zap_one = 0;
		/* Intel PIIX errata: bitstuff error may be set without stall */
	if (!(td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_STALLED | TD_FLAGS1_STATUS_BITSTUFF_ERROR))) {
		if (__unlikely(!(brq->flags & USBRQ_ALLOW_SHORT))) {
			UHCI_REPORT_ERROR(td, 1, "SHORT PACKET", 0), r = -ENODATA;
		} else if (__unlikely(brq->status < 0)) {
			r = brq->status;
			zap_one = 1;
		} else if (qh->state == QHS_ACTIVE) {
			goto do_unlink;
		} else {
			zap_one = 1;
			r = brq->status;
		}
	} else {
		r = 0;
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_BITSTUFF_ERROR)) {
			if (!UHCI_TEST_PORT_CHANGE(qh->uhci))
				UHCI_REPORT_ERROR(td, 2, "BIT STUFFING ERROR", 0);
			r = -EIO;
		}
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_CRC_TIMEOUT)) {
			if (!UHCI_TEST_PORT_CHANGE(qh->uhci))
				UHCI_REPORT_ERROR(td, 2, "CRC ERROR OR DEVICE NOT RESPONDING", 0);
			r = -EIO;
		}
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_BABBLE)) {
			if (!UHCI_TEST_PORT_CHANGE(qh->uhci))
				UHCI_REPORT_ERROR(td, 1, "BABBLE", 0);
			r = -EPROTO;
		}
		if (td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_BUFFER_ERROR)) {
			UHCI_REPORT_ERROR(td, 2, (__32LE2CPU(td->td.flags2) & TD_FLAGS2_PID) == TD_FLAGS2_PID_IN ? "BUFFER UNDERRUN" : "BUFFER OVERRUN", 0);
			r = -ENOBUFS;
		}
		if (!r) {
			UHCI_REPORT_ERROR(td, 1, "STALL", 1);
			r = -EPIPE;
		}
	}
	if (USB_IS_CTRL(qh->type) || USB_IS_INTR(qh->type)) zap_one = 1;
	brq->status = r;
	if (!zap_one) {
		UHCI_ED_SET_ERROR(GET_STRUCT(qh, ED, qh_default), -EPIPE);
		GET_STRUCT(qh, ED, qh_default)->ep.post = UHCI_POST_STALLED;
	}
	if (__likely(qh->state == QHS_ACTIVE)) do_unlink: UHCI_START_UNLINK_QH(qh);
}

__COLD_ATTR__ static void UHCI_REPORT_ERROR(TD *td, int level, const char *msg, int only_stall)
{
	USB_REQUEST *brq = td->brq;
	ED *ed;
	UHCI *uhci;
	int errorlevel;
	if (__likely(only_stall) && __likely(brq->flags & (USBRQ_QUIET_STALL | USBRQ_QUIET_ERROR))) return;
	if (brq->status < 0) return;
	ed = (ED *)brq->internal1;
	uhci = ed->qh_default.uhci;
	errorlevel = brq->flags & USBRQ_QUIET_ERROR ? 0 : uhci->errorlevel;
	if (__unlikely(level <= errorlevel)) {
		int lclass = level == 2 ? __SYSLOG_NET_WARNING : level == 1 ? __SYSLOG_NET_ERROR : __SYSLOG_HW_BUG;
		KERNEL$SYSLOG(lclass, uhci->dev_name, "DEVICE %u, %s PIPE %u: %s", (unsigned)(__32LE2CPU(td->td.flags2) & TD_FLAGS2_DEVICE_ADDRESS) >> __BSF_CONST(TD_FLAGS2_DEVICE_ADDRESS), USB$ENDPOINT_TYPE_STRING(ed->type), (unsigned)(__32LE2CPU(td->td.flags2) & TD_FLAGS2_ENDPT) >> __BSF_CONST(TD_FLAGS2_ENDPT), msg);
	}
}

static void UHCI_PROCESS_UNLINKING_QUEUE_HEAD(QH *qh);

__COLD_ATTR__ static void UHCI_PROCESS_UNLINKING_ENTRIES(UHCI *uhci, int force)
{
	QH *qh;
	u_jiffies_lo_t jj = KERNEL$GET_JIFFIES_LO();
	__u16 frame = UHCI_READ_FRNUM(uhci);
	XLIST_FOR_EACH(qh, &uhci->unlinking, QH, list) {
		QH *qhh;
		if (__unlikely(qh->cleanup_frame == frame) && !force) {
			if (__unlikely(jj - qh->unlink_start > TIMEOUT_JIFFIES(QH_CLEANUP_TIMEOUT))) {
				KERNEL$SYSLOG(__SYSLOG_HW_ERROR, uhci->dev_name, "FRAME NUMBER DIDN'T ADVANCE TO CLEAN UP ENDPOINTS, FORCING CLEANUP");
				goto force_cleanup;
			}
			UHCI_ENABLE_SOF_IRQ(uhci);
			continue;
		}
		force_cleanup:
		qhh = LIST_STRUCT(qh->list.prev, QH, list);
		DEL_FROM_LIST(&qh->list);
		UHCI_PROCESS_UNLINKING_QUEUE_HEAD(qh);
		qh = qhh;
	}
}

__COLD_ATTR__ static void UHCI_PROCESS_UNLINKING_QUEUE_HEAD(QH *qh)
{
	TD *td;
	USB_REQUEST *brq;
	__u32 d_le;
	__u32 *link_le;
	UHCI_PROCESS_QUEUE_HEAD(qh);
	if (UHCI_QUEUE_IS_EMPTY(qh)) {
		if (__unlikely(qh->state == QHS_UNLINKING_FREEING)) {
			qh->state = QHS_INACTIVE;
			UHCI_FREE_ENDPOINT(&GET_STRUCT(qh, ED, qh_default)->ep);
			return;
		} else {
			qh->state = QHS_INACTIVE;
			return;
		}
	}
	/* process requests that have been correctly finished while unlinking */
	td = LIST_STRUCT(qh->tds.next, TD, list);
	brq = td->brq;
	d_le = td->td.flags2 & __32CPU2LE(TD_FLAGS2_D);
	if (__unlikely(!(td->td.flags1 & __32CPU2LE(TD_FLAGS1_STATUS_ACTIVE))) && brq->status >= 0) {
		int diff = TD_LEN_DIFFER(td);
		if (diff) {
			if (__unlikely(TD_SUBTR(brq, td, diff))) {
				brq->status = -EFTYPE;
				goto next_td;
			}
			d_le ^= __32CPU2LE(TD_FLAGS2_D);
			goto post_current_td;
		}
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, qh->uhci->dev_name, "INACTIVE NON-ERRORNEOUS TD ON UNLINKED QUEUE: FLAGS1: %08X, FLAGS2: %08X, STATUS: %ld", (unsigned)__32LE2CPU(td->td.flags1), (unsigned)__32LE2CPU(td->td.flags2), brq->status);
		brq->status = -EFTYPE;
	}
	next_td:
	brq = td->brq;
	if (__unlikely(brq->status < 0)) {
		while (td->brq == brq) {
			TD *next_td;
			/* for short control transfers, we need to leave status phase descriptor */
			if (__likely(brq->status >= 0) && __unlikely(USB_IS_CTRL(qh->type))) {
				TD *td_next = LIST_STRUCT(td->list.next, TD, list);
				if (td_next->brq != brq) goto skip_this_td;
			}
			if (__unlikely(TD_SUBTR(brq, td, TD_PKTLEN(td))))
				brq->status = -EFTYPE;

			post_current_td:
			td->data_dma_unlock(td->data_dma_addr);
			next_td = LIST_STRUCT(td->list.next, TD, list);
			DEL_FROM_LIST(&td->list);
			UHCI_FREE_TD(((ED *)brq->internal1)->qh, td);
			td = next_td;
		}
		brq->internal1 = 0;
		CALL_AST(brq);
		if (td->brq) goto next_td;
	} else {
		skip_this_td:
		if (!USB_IS_CTRL(qh->type)) {
			td->td.flags2 = (td->td.flags2 & ~__32CPU2LE(TD_FLAGS2_D)) | d_le;
			d_le ^= __32CPU2LE(TD_FLAGS2_D);
		}
		td = LIST_STRUCT(td->list.next, TD, list);
		if (td->brq) goto next_td;
	}
	if (!USB_IS_CTRL(qh->type)) {
		GET_STRUCT(qh, ED, qh_default)->d_le = d_le;
		GET_STRUCT(qh, ED, qh_default)->td_flags_2_le = (GET_STRUCT(qh, ED, qh_default)->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_D)) | d_le;
	}

	link_le = &qh->qh.element_link;
	LIST_FOR_EACH(td, &qh->tds, TD, list) {
		*link_le = __32CPU2LE(td->desc_dma_addr);
		link_le = &td->td.link;
	}

	qh->state = QHS_INACTIVE;
	if (!UHCI_QUEUE_IS_EMPTY(qh))
		UHCI_ACTIVATE_QH(qh, LIST_STRUCT(qh->tds.next, TD, list)->brq);
}

__COLD_ATTR__ static void UHCI_START_UNLINK_QH(QH *qh)
{
	__u32 *addr_le;
	if (__likely(USB_IS_BULK(qh->type))) {
		addr_le = &UHCI2AREA(qh->uhci)->loop_start.head_link;
		if (__unlikely(*addr_le & __32CPU2LE(UHCI_ADDR_T)))
			KERNEL$SUICIDE("UHCI_START_UNLINK_QH: BULK LIST IS EMPTY");
		while ((__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q) != qh->desc_dma_addr) {
			QH *x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
			addr_le = &x->qh.head_link;
			if (__unlikely(*addr_le & __32CPU2LE(UHCI_ADDR_T)))
				KERNEL$SUICIDE("UHCI_START_UNLINK_QH: BULK LIST TERMINATED AND QH NOT FOUND");
			if (__unlikely(addr_le == qh->uhci->last_head_link))
				KERNEL$SUICIDE("UHCI_START_UNLINK_QH: BULK LIST LOOPS AND QH NOT FOUNT");
		}
		*addr_le = qh->qh.head_link;
		if (__32LE2CPU(UHCI2AREA(qh->uhci)->loop_start.head_link) == (qh->desc_dma_addr | UHCI_ADDR_Q)) {
			UHCI2AREA(qh->uhci)->loop_start.head_link = __32CPU2LE(UHCI_ADDR_T);
			KERNEL$DEL_TIMER(&qh->uhci->fsbr_timer);
			SET_TIMER_NEVER(&qh->uhci->fsbr_timer);
		}
		if (qh->uhci->last_head_link == &qh->qh.head_link) qh->uhci->last_head_link = addr_le;
		if (!(*qh->uhci->last_head_link & __32CPU2LE(UHCI_ADDR_T))) *qh->uhci->last_head_link = UHCI2AREA(qh->uhci)->loop_start.head_link;
	} else if (USB_IS_CTRL(qh->type) && qh != &UHCI2AREA(qh->uhci)->low_ctrl) {
		addr_le = &UHCI2AREA(qh->uhci)->ctrl_start.head_link;
		while ((__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q) != qh->desc_dma_addr) {
			QH *x;
			if (__unlikely(*addr_le == UHCI_LOOP_START_Q_LE(qh->uhci)))
				KERNEL$SUICIDE("UHCI_START_UNLINK_QH: QH NOT FOUND IN CTRL LIST");
			x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
			addr_le = &x->qh.head_link;
		}
		*addr_le = qh->qh.head_link;
	} else {
		int set = 0;
		unsigned j;
		for (j = 0; j < N_FRAME_LIST; j++) {
			addr_le = &UHCI2AREA(qh->uhci)->fl.addr[j];
			while (*addr_le != UHCI_CTRL_START_Q_LE(qh->uhci)) {
				QH *x;
				if (__unlikely(!(*addr_le & __32CPU2LE(UHCI_ADDR_Q)))) {
					TD *is;
					is = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le)), TD, td);
					addr_le = &is->td.link;
					continue;
				}
				if ((__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q) == qh->desc_dma_addr) {
					*addr_le = qh->qh.head_link;
					set = 1;
					break;
				}
				x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
				if (j >= x->stride) break;
				addr_le = &x->qh.head_link;
			}
		}
		if (__unlikely(!set))
			KERNEL$SUICIDE("UHCI_START_UNLINK_QH: QH NOT FOUND IN INTERRUPT TREE");
		DEL_FROM_LIST(&qh->list);
	}
	__barrier();
	qh->state = QHS_UNLINKING;
	qh->cleanup_frame = UHCI_READ_FRNUM(qh->uhci);
	qh->unlink_start = KERNEL$GET_JIFFIES_LO();
	ADD_TO_XLIST(&qh->uhci->unlinking, &qh->list);
	UHCI_ENABLE_SOF_IRQ(qh->uhci);
}

static void UHCI_ACTIVATE_INTR(QH *qh);

__COLD_ATTR__ static void UHCI_ACTIVATE_QH(QH *qh, USB_REQUEST *brq)
{
	if (__likely(USB_IS_BULK(qh->type))) {
		__u32 link = UHCI2AREA(qh->uhci)->loop_start.head_link;
		if (link & __32CPU2LE(UHCI_ADDR_T)) {
			qh->uhci->last_head_link = &qh->qh.head_link;
		}
		qh->qh.head_link = link;
		__write_barrier();
		UHCI2AREA(qh->uhci)->loop_start.head_link = __32CPU2LE(qh->desc_dma_addr | UHCI_ADDR_Q);
		if (__likely(!(brq->flags & USBRQ_NO_FSBR)) || !(*qh->uhci->last_head_link & __32CPU2LE(UHCI_ADDR_T))) {
			UHCI_SET_FSBR(qh->uhci);
		}
		qh->state = QHS_ACTIVE;
	} else if (__likely(USB_IS_CTRL(qh->type)) && qh != &UHCI2AREA(qh->uhci)->low_ctrl) {
		qh->qh.head_link = UHCI2AREA(qh->uhci)->ctrl_start.head_link;
		__write_barrier();
		UHCI2AREA(qh->uhci)->ctrl_start.head_link = __32CPU2LE(qh->desc_dma_addr | UHCI_ADDR_Q);
		qh->state = QHS_ACTIVE;
	} else {
		UHCI_ACTIVATE_INTR(qh);
	}
}

static unsigned QH_TIME(QH *qh);

__COLD_ATTR__ static void UHCI_ACTIVATE_INTR(QH *qh)
{
	unsigned i, j;
	unsigned mintim, mini;
#if __DEBUG >= 1
	if (__unlikely((unsigned)(qh->stride - 1) >= N_FRAME_LIST) || __unlikely(qh->stride & (qh->stride - 1)))
		KERNEL$SUICIDE("UHCI_ACTIVATE_INTR: INVALID STRIDE %u", (unsigned)qh->stride);
#endif
	mintim = MAXINT, mini = 0;
	/*__debug_printf("activate intr: stride %d\n", qh->stride);*/
	if (__unlikely(qh == &UHCI2AREA(qh->uhci)->low_ctrl)) goto no_bw_check;
	/*__debug_printf("check bw\n");*/
	for (i = 0; i < qh->stride; i++) {
		unsigned maxtim = 0;
		for (j = i; j < N_FRAME_LIST; j += qh->stride) {
			__u32 addr_le = UHCI2AREA(qh->uhci)->fl.addr[j];
			unsigned tim = 0;
			while (addr_le != UHCI_CTRL_START_Q_LE(qh->uhci)) {
				QH *x;
				if (__unlikely(!(addr_le & __32CPU2LE(UHCI_ADDR_Q)))) {
					TD *is;
					is = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le)), TD, td);
					addr_le = is->td.link;
					continue;
				}
				x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(addr_le) & ~UHCI_ADDR_Q), QH, qh);
				if (x != &UHCI2AREA(qh->uhci)->low_ctrl)
					tim += QH_TIME(x);
				addr_le = x->qh.head_link;
			}
			if (__likely(tim >= maxtim)) maxtim = tim;
		}
		if (maxtim < mintim) mintim = maxtim, mini = i;
	}
	if (__unlikely(mintim + QH_TIME(qh) > MAX_PERIODIC_BANDWIDTH)) {
		if (qh->uhci->errorlevel >= 2)
			KERNEL$SYSLOG(__SYSLOG_SW_WARNING, qh->uhci->dev_name, "CAN'T ALLOCATE PERIODIC BANDWIDTH");
		UHCI_ED_SET_ERROR(GET_STRUCT(qh, ED, qh_default), -ENOSR);
		UHCI_PROCESS_UNLINKING_QUEUE_HEAD(qh);
		return;
	}
	no_bw_check:
	for (j = mini; j < N_FRAME_LIST; j += qh->stride) {
		__u32 *addr_le = &UHCI2AREA(qh->uhci)->fl.addr[j];
		while (*addr_le != UHCI_CTRL_START_Q_LE(qh->uhci)) {
			QH *x;
			if (__unlikely(!(*addr_le & __32CPU2LE(UHCI_ADDR_Q)))) {
				TD *is;
				is = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le)), TD, td);
				addr_le = &is->td.link;
				continue;
			}
			x = GET_STRUCT(KERNEL$DMA_2_VIRT(__32LE2CPU(*addr_le) & ~UHCI_ADDR_Q), QH, qh);
			if (x == qh) goto already_have;
			if (x->stride < qh->stride) break;
			addr_le = &x->qh.head_link;
		}
		if (__unlikely(j == mini)) qh->qh.head_link = *addr_le;
		else if (__unlikely(qh->qh.head_link != *addr_le))
			KERNEL$SUICIDE("UHCI_ACTIVATE_INTR: INTERRUPT TREE CORRUPT: %08X != %08X", __32CPU2LE(qh->qh.head_link), __32CPU2LE(*addr_le));
		__write_barrier();
		*addr_le = __32CPU2LE(qh->desc_dma_addr | UHCI_ADDR_Q);
		already_have:;
	}
	qh->state = QHS_ACTIVE;
	ADD_TO_XLIST(&qh->uhci->intr_qh, &qh->list);
}

__COLD_ATTR__ static unsigned QH_TIME(QH *qh)
{
	ED *ed = GET_STRUCT(qh, ED, qh_default);
	int bytes = USB$HC_ENDPOINT_BANDWIDTH(ed->type, ed->td_flags_1_le & __32CPU2LE(TD_FLAGS1_LS) ? USB_LOW_SPEED : USB_FULL_SPEED, ed->pkt_size);
	if (__unlikely(bytes < 0)) KERNEL$SUICIDE("QH_TIME: COULD NOT DETERMINE TIME FOR ALLOCATED ENDPOINT: %d", bytes);
	return bytes;
}

static void UHCI_CHANGE_TOGGLE(USB_REQUEST *brq)
{
#define new_list	((LIST_HEAD *)(void *)&brq->v.ptr)
	ED *ed;
	TD *td;
	__u32 flags2 = 0;
	LIST_FOR_EACH(td, new_list, TD, list) {
		flags2 = td->td.flags2 ^= __32CPU2LE(TD_FLAGS2_D);
	}
	ed = (ED *)brq->internal1;
	ed->td_flags_2_le = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_D)) | ((~flags2) & __32CPU2LE(TD_FLAGS2_D));
	UHCI_POST_BULK_INTR_CTRL_REQUEST(brq);
#undef new_list
}

__COLD_ATTR__ static void UHCI_POST_DISABLED(USB_REQUEST *brq)
{
	HC_CHECK_SPL("UHCI_POST_DISABLED");
	UHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
	brq->status = -ECONNRESET;
	CALL_AST(brq);
}

__COLD_ATTR__ static void UHCI_POST_STALLED(USB_REQUEST *brq)
{
	HC_CHECK_SPL("UHCI_POST_STALLED");
	UHCI_FREE_BULK_INTR_CTRL_REQUEST(brq);
	brq->status = -EPIPE;
	CALL_AST(brq);
}

__COLD_ATTR__ static void UHCI_FREE_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	return UHCI_FREE_BULK_INTR_CTRL_REQUEST_LIST(brq, (LIST_HEAD *)(void *)&brq->v.ptr);
}

__COLD_ATTR__ static void UHCI_FREE_BULK_INTR_CTRL_REQUEST_LIST(USB_REQUEST *brq, LIST_HEAD *new_list)
{
	ED *ed = (ED *)brq->internal1;
	HC_CHECK_SPL("UHCI_FREE_BULK_INTR_CTRL_REQUEST");
	do {
		TD *td = LIST_STRUCT(new_list->next, TD, list);
		DEL_FROM_LIST(&td->list);
		td->data_dma_unlock(td->data_dma_addr);
		UHCI_FREE_TD(ed->qh, td);
	} while (!LIST_EMPTY(new_list));
	brq->internal1 = 0;
	brq->v.len = 0;
}

__COLD_ATTR__ static void UHCI_CLEAR_STALL(USB_ENDPOINT *endpt)
{
	ED *ed = GET_STRUCT(endpt, ED, ep);
	HC_CHECK_SPL("UHCI_CLEAR_STALL");
	ed->d_le = __32CPU2LE(0);
	ed->td_flags_2_le &= ~__32CPU2LE(TD_FLAGS2_D);
	if (__likely(ed->ep.post == UHCI_POST_STALLED))
		ed->ep.post = UHCI_POST_BULK_INTR_CTRL_REQUEST;
	if (__unlikely(ed->qh != &ed->qh_default)) return;
	if (__unlikely(!UHCI_QUEUE_IS_EMPTY(&ed->qh_default)))
		KERNEL$SUICIDE("UHCI_CLEAR_STALL: CLEAR STALL ON ACTIVE ENDPOINT");
}

__COLD_ATTR__ static void UHCI_CANCEL_BULK_INTR_CTRL_REQUEST(USB_REQUEST *brq)
{
	ED *ed;
	QH *qh;
	HC_CHECK_SPL("UHCI_CANCEL_BULK_INTR_CTRL_REQUEST");
	if (!brq->internal1) return;
	if (__likely(brq->status >= 0)) brq->status = -EINTR;
	ed = (ED *)brq->internal1;
	qh = ed->qh;
	if (__unlikely(qh->state == QHS_ACTIVE)) UHCI_START_UNLINK_QH(qh);
}

static void UHCI_SET_VERTICAL(USB_REQUEST *brq)
{
#define new_list	((LIST_HEAD *)(void *)&brq->v.ptr)
	TD *td;
	LIST_FOR_EACH(td, new_list, TD, list) {
		td->td.link |= __32CPU2LE(UHCI_ADDR_VF);
	}
#undef new_list
}

static int UHCI_PREPARE_CTRL_REQUEST(USB_ENDPOINT *endpt, USB_REQUEST *rq)
{
	USB_CTRL_REQUEST *brq = (USB_CTRL_REQUEST *)rq;
#define new_list	((LIST_HEAD *)(void *)&brq->v.ptr)
#define setup_td_list	((LIST_HEAD *)(void *)&setup.v.ptr)
	int r;
	int was_data;
	unsigned ioc;
	ED *ed = GET_STRUCT(endpt, ED, ep);
	USB_REQUEST setup;
	TD *setup_td, *status_td;
	unsigned total_size;
	HC_CHECK_SPL("UHCI_PREPARE_CTRL_REQUEST");

	total_size = brq->v.len + sizeof brq->setup;
	ioc = !(brq->flags & USBRQ_DELAY_INTERRUPT);
	brq->flags |= USBRQ_DELAY_INTERRUPT;
	ed->td_flags_2_le |= __32CPU2LE(TD_FLAGS2_D);
	was_data = 0;
	if (brq->v.len) {
		was_data = 1;
		if (brq->setup.request & __16CPU2LE(USB_CTRL_DIR_IN)) ed->type = 0;
		ed->td_flags_2_le = (ed->td_flags_2_le & ~__32CPU2LE(TD_FLAGS2_PID)) | (USB_IS_IN(ed->type) ? __32CPU2LE(TD_FLAGS2_PID_IN) : __32CPU2LE(TD_FLAGS2_PID_OUT));
		r = UHCI_PREPARE_BULK_INTR_REQUEST(&ed->ep, (USB_REQUEST *)brq);
		ed->type = USB_EP_CTRL;
		if (__unlikely(r)) return r;
	} else {
		brq->internal1 = (unsigned long)ed;
		INIT_LIST(new_list);
	}

	setup.flags = USBRQ_DELAY_INTERRUPT;
	setup.v.ptr = (unsigned long)&brq->setup;
	setup.v.len = sizeof brq->setup;
	setup.v.vspace = &KERNEL$VIRTUAL;
	if (__unlikely(r = UHCI_PREPARE_BULK_INTR_REQUEST(&ed->ep, &setup)))
		KERNEL$SUICIDE("UHCI_PREPARE_CTRL_REQUEST: UHCI_PREPARE_BULK_INTR_REQUEST RETURNED ERROR %d", r);
	if (__unlikely(LIST_EMPTY(setup_td_list)))
		KERNEL$SUICIDE("UHCI_PREPARE_CTRL_REQUEST: SETUP PACKET IS EMPTY, REQEUST %p", brq);
	if (__unlikely(setup_td_list->next != setup_td_list->prev))
		KERNEL$SUICIDE("UHCI_PREPARE_CTRL_REQUEST: SETUP PACKET IS SPLIT, REQEUST %p", brq);
	setup_td = LIST_STRUCT(setup_td_list->next, TD, list);
	setup_td->brq = (USB_REQUEST *)brq;
	setup_td->td.flags2 = (setup_td->td.flags2 & ~__32CPU2LE(TD_FLAGS2_D | TD_FLAGS2_PID)) | __32CPU2LE(TD_FLAGS2_PID_SETUP);

	status_td = UHCI_GET_TD(ed->qh);
	status_td->data_dma_unlock = KERNEL$NULL_VSPACE_DMAUNLOCK;
	status_td->brq = (USB_REQUEST *)brq;
	status_td->td.flags1 = ed->td_flags_1_le |
		(ioc * __32CPU2LE(TD_FLAGS1_IOC));
	status_td->td.flags2 = __32CPU2LE(TD_FLAGS2_MAXLEN | TD_FLAGS2_D) |
		(ed->td_flags_2_le & __32CPU2LE(TD_FLAGS2_ENDPT | TD_FLAGS2_DEVICE_ADDRESS)) |
		(brq->setup.request & __16CPU2LE(USB_CTRL_DIR_IN) && was_data ? __32CPU2LE(TD_FLAGS2_PID_OUT) : __32CPU2LE(TD_FLAGS2_PID_IN));
	status_td->td.ptr = __32CPU2LE(0);

	LINK_TD(new_list, status_td);
	setup_td->td.link = __32CPU2LE(LIST_STRUCT(new_list->next, TD, list)->desc_dma_addr);
	ADD_TO_LIST(new_list, &setup_td->list);

	brq->status = total_size;

	if (__likely(!(setup_td->td.flags1 & __32CPU2LE(TD_FLAGS1_LS)))) UHCI_SET_VERTICAL((USB_REQUEST *)brq);
	return 0;
#undef new_list
}

__COLD_ATTR__ static void *UHCI_ALLOC_DATA(QH *qh, unsigned size, unsigned align)
{
	void *ret;
	unsigned align_add;
	unsigned long ptr = (unsigned long)qh->free_space_ptr;
	get_again:
	ptr = (unsigned long)qh->free_space_ptr;
	if (__unlikely(!ptr))
		goto new_page;
	align_add = (-ptr) & (align - 1);
	if (__unlikely((ptr & (__PAGE_CLUSTER_SIZE_MINUS_1)) + align_add + size > __PAGE_CLUSTER_SIZE)) {
		DATA_PAGE *pg;
		new_page:
		/*__debug_printf(".");*/
		pg = KERNEL$ALLOC_CONTIG_AREA(__PAGE_CLUSTER_SIZE, AREA_DATA | AREA_PCIDMA | AREA_PHYSCONTIG | AREA_ALIGN, (unsigned long)__PAGE_CLUSTER_SIZE);
		if (__unlikely(__IS_ERR(pg))) return (void *)pg;

		if (__unlikely(ptr != (unsigned long)qh->free_space_ptr)) {
			KERNEL$FREE_CONTIG_AREA(pg, __PAGE_CLUSTER_SIZE);
			goto get_again;
		}

		pg->next = qh->data_pages;
		qh->data_pages = pg;
		qh->free_space_ptr = pg + 1;
		goto get_again;
	}
	ptr += align_add;
	ret = (void *)ptr;
	ptr += size;
	if (!(ptr & (__PAGE_CLUSTER_SIZE_MINUS_1))) ptr = 0;
	qh->free_space_ptr = (void *)ptr;
	return ret;
}

__COLD_ATTR__ static TD *UHCI_ALLOC_TD(QH *qh)
{
	VDESC v;
	VDMA vdma;
	TD *td = UHCI_ALLOC_DATA(qh, sizeof(TD), sizeof(HW_TD));
	if (__unlikely(__IS_ERR(td))) {
		return td;
	}
	v.ptr = (unsigned long)&td->td;
	v.len = sizeof(HW_TD);
	v.vspace = &KERNEL$VIRTUAL;
	vdma.spl = SPL_X(SPL_USB);
	RAISE_SPL(SPL_VSPACE);
	td->desc_dma_unlock = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &vdma);
	if (__unlikely(vdma.len != v.len))
		KERNEL$SUICIDE("UHCI_ALLOC_TD: CAN'T DMALOCK TD");
	td->desc_dma_addr = vdma.ptr;

	return td;
}

__COLD_ATTR__ static int UHCI_INIT_QH(UHCI *uhci, QH *qh, char type, unsigned short stride, void *next_ptr)
{
	VDESC v;
	VDMA vdma;
	TD *td;

	memset(qh, 0, sizeof(QH));
	INIT_LIST(&qh->tds);
	qh->uhci = uhci;
	qh->state = QHS_INACTIVE;
	qh->type = type;
	if (__unlikely(stride > N_FRAME_LIST)) stride = N_FRAME_LIST;
	if (__unlikely(stride <= 0)) stride = 1;
	stride = 1 << __BSR(stride);
	qh->stride = stride;

	qh->free_space_ptr = next_ptr;

	td = UHCI_ALLOC_TD(qh);
	if (__unlikely(__IS_ERR(td))) {
		return __PTR_ERR(td);
	}

	UHCI_SET_DUMMY_TD(td);
	qh->qh.element_link = __32CPU2LE(td->desc_dma_addr);
	ADD_TO_LIST_END(&qh->tds, &td->list);

	v.ptr = (unsigned long)&qh->qh;
	v.len = sizeof(HW_QH);
	v.vspace = &KERNEL$VIRTUAL;
	vdma.spl = SPL_X(SPL_USB);
	RAISE_SPL(SPL_VSPACE);
	qh->desc_dma_unlock = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &vdma);
	if (__unlikely(vdma.len != v.len))
		KERNEL$SUICIDE("UHCI_INIT_QH: CAN'T DMALOCK QH");
	qh->desc_dma_addr = vdma.ptr;

	return 0;
}

__COLD_ATTR__ static void UHCI_FREE_QH(QH *qh)
{
	DATA_PAGE *pg;

	qh->desc_dma_unlock(qh->desc_dma_addr);

	if (__likely(!LIST_EMPTY(&qh->tds))) {
		TD *td;
		if (__unlikely(!UHCI_QUEUE_IS_EMPTY(qh)))
			KERNEL$SUICIDE("UHCI_FREE_QH: QUEUE NOT EMPTY");
		td = LIST_STRUCT(qh->tds.next, TD, list);
		td->desc_dma_unlock(td->desc_dma_addr);
	}

	while (qh->total_tds--) {
		TD *td = UHCI_GET_TD(qh);
		td->desc_dma_unlock(td->desc_dma_addr);
	}
	if (__unlikely(qh->td_freelist))
		KERNEL$SUICIDE("UHCI_FREE_QH: SOME TDS LEFT");

	while (qh->total_bounces--) {
		BOUNCE_BUFFER *bb = UHCI_GET_BB(qh);
		bb->data_dma_unlock(bb->data_dma_addr);
	}

	if (__unlikely(qh->bounce_freelist))
		KERNEL$SUICIDE("UHCI_FREE_QH: SOME BOUNCES LEFT");

	while ((pg = qh->data_pages)) {
		qh->data_pages = pg->next;
		KERNEL$FREE_CONTIG_AREA(pg, __PAGE_CLUSTER_SIZE);
	}
}

__COLD_ATTR__ static USB_ENDPOINT *UHCI_ALLOC_ENDPOINT(USB *usb, unsigned speed, int type, int addr, int pipe, unsigned pkt_size, int interval, unsigned n_rq, unsigned rq_size)
{
	UHCI *uhci = USB2UHCI(usb);
	ED *ed;
	int error = 0;
	VDESC v;
	VDMA vdma;
	unsigned required_tds, required_bounces;
	/*__debug_printf("alloc ep: pkt %u, n_rq %u, rq_size %u\n", pkt_size, n_rq, rq_size);*/
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT AT SPL %08X", KERNEL$SPL);
	if (__unlikely(!n_rq) || __unlikely(!rq_size))
		KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT: REQUEST_RESERVATION NOT SET (N_RQ %u, RQ_SIZE %u", n_rq, rq_size);
	if (__unlikely(uhci->flags & FLAG_DEAD)) return __ERR_PTR(-EIO);
	if (__unlikely(addr == USB_ROOT_HUB_ADDR)) {
		switch (pipe) {
			case 0:
				if (type != USB_EP_CTRL) goto invl;
				uhci->root_ctrl.usb = usb;
				return &uhci->root_ctrl;
			case 1:
				if (type != USB_EP_INTR_IN) goto invl;
				uhci->root_intr.usb = usb;
				return &uhci->root_intr;
			default:
			invl:
				KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT: INVALID ROOT HUB ENDPOINT: TYPE %d, PIPE %d", type, pipe);
		}
	}
	ed = KERNEL$ALLOC_CONTIG_AREA(__PAGE_CLUSTER_SIZE, AREA_DATA | AREA_PCIDMA | AREA_PHYSCONTIG | AREA_ALIGN, (unsigned long)__PAGE_CLUSTER_SIZE);
	if (__unlikely(__IS_ERR(ed))) return (void *)ed;
	ADD_TO_XLIST(&uhci->endpoints, &ed->endpoints_list);
	ed->pkt_size = pkt_size;
	ed->type = type;
	ed->td_flags_1_le = __32CPU2LE(TD_FLAGS1_SPD | TD_FLAGS1_C_ERR_3 | ((speed == USB_LOW_SPEED) ? TD_FLAGS1_LS : 0) | (USB_IS_ISO(type) ? TD_FLAGS1_ISO : 0) | TD_FLAGS1_STATUS_ACTIVE | TD_FLAGS1_ACTLEN);
	ed->td_flags_2_le = __32CPU2LE(((pkt_size - 1) << __BSF_CONST(TD_FLAGS2_MAXLEN)) | (pipe << __BSF_CONST(TD_FLAGS2_ENDPT)) | (addr << __BSF_CONST(TD_FLAGS2_DEVICE_ADDRESS)) | (USB_IS_IN(type) ? TD_FLAGS2_PID_IN : TD_FLAGS2_PID_OUT));
	ed->d_le = __32CPU2LE(0);
	ed->addr = addr;
	ed->pipe = pipe;
	ed->required_tds = 0;
	ed->required_bounces = 0;
	/*__debug_printf("%d %d %d\n", rq_size, pkt_size, n_rq);*/
	required_tds = ((rq_size + pkt_size - 1) / pkt_size + (USB_IS_CTRL(type)) * 2) * n_rq;
	required_bounces = (((rq_size + __PAGE_CLUSTER_SIZE - 2) >> __PAGE_CLUSTER_BITS) + USB_IS_CTRL(type)) * n_rq;

	if (USB_IS_CTRL(type) && __unlikely(speed == USB_LOW_SPEED)) {
		ed->qh = &UHCI2AREA(uhci)->low_ctrl;
		ed->qh_default.uhci = uhci;
		if (__unlikely(pkt_size != 8)) {
			error = -EINVAL;
			goto alloc_skip;
		}
	} else {
		ed->qh = &ed->qh_default;
		error = UHCI_INIT_QH(uhci, &ed->qh_default, type, interval, (void *)(ed + 1));
		if (__unlikely(error)) {
			goto alloc_skip;
		}
	}

	/*__debug_printf("%d + %d < %d\n", ed->qh->required_tds, required_tds, ed->qh->total_tds);*/
	/*__debug_printf("%d + %d < %d\n", ed->qh->required_bounces, required_bounces, ed->qh->total_bounces);*/
	alloc_next:
	if (__likely(ed->qh->required_tds + required_tds > ed->qh->total_tds)) {
		TD *td = UHCI_ALLOC_TD(ed->qh);
		if (__unlikely(__IS_ERR(td))) {
			error = __PTR_ERR(td);
			goto alloc_skip;
		}
		UHCI_FREE_TD(ed->qh, td);
		ed->qh->total_tds++;
		goto alloc_next;
	}
	if (__likely(ed->qh->required_bounces + required_bounces > ed->qh->total_bounces)) {
		BOUNCE_BUFFER *bb = UHCI_ALLOC_DATA(ed->qh, sizeof(BOUNCE_BUFFER) + pkt_size, 8);
		if (__unlikely(__IS_ERR(bb))) {
			error = __PTR_ERR(bb);
			goto alloc_skip;
		}
		v.ptr = (unsigned long)(bb + 1);
		v.len = pkt_size;
		v.vspace = &KERNEL$VIRTUAL;
		vdma.spl = SPL_X(SPL_USB);
		RAISE_SPL(SPL_VSPACE);
		bb->data_dma_unlock = KERNEL$VIRTUAL.op->vspace_dmalock(&v, PF_READ | PF_WRITE, &vdma);
		if (__unlikely(vdma.len != v.len))
			KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT: CAN'T DMALOCK TD");
		bb->data_dma_addr = vdma.ptr;
		bb->ed = ed;
		UHCI_FREE_BB(ed->qh, bb);
		ed->qh->total_bounces++;
		goto alloc_next;
	}
	ed->required_tds = required_tds;
	ed->qh->required_tds += required_tds;
	ed->required_bounces = required_bounces;
	ed->qh->required_bounces += required_bounces;

	alloc_skip:

	switch (type) {
		case USB_EP_CTRL:
			ed->ep.prepare = UHCI_PREPARE_CTRL_REQUEST;
			ed->ep.post = UHCI_POST_BULK_INTR_CTRL_REQUEST;
			ed->ep.free = UHCI_FREE_BULK_INTR_CTRL_REQUEST;
			ed->ep.cancel = UHCI_CANCEL_BULK_INTR_CTRL_REQUEST;
			ed->ep.clear_stall = UHCI_CLEAR_STALL;
			break;
		case USB_EP_BULK_IN:
		case USB_EP_BULK_OUT:
		case USB_EP_INTR_IN:
		case USB_EP_INTR_OUT:
			ed->ep.prepare = UHCI_PREPARE_BULK_INTR_REQUEST;
			ed->ep.post = UHCI_POST_BULK_INTR_CTRL_REQUEST;
			ed->ep.free = UHCI_FREE_BULK_INTR_CTRL_REQUEST;
			ed->ep.cancel = UHCI_CANCEL_BULK_INTR_CTRL_REQUEST;
			ed->ep.clear_stall = UHCI_CLEAR_STALL;
			break;
		case USB_EP_ISO_IN:
		case USB_EP_ISO_OUT:
			/* unimplemented */
			error = -ENOSYS;
			break;
		default:
			KERNEL$SUICIDE("UHCI_ALLOC_ENDPOINT: TYPE %d", type);
	}
	ed->ep.usb = usb;

	ed->n_bounce_requests = n_rq + (type == USB_EP_CTRL);
	ed->bounce_request_size = rq_size;
	if (__unlikely(error) || __unlikely(error = KERNEL$RESERVE_BOUNCE(uhci->dev_name, ed->n_bounce_requests, ed->bounce_request_size, 0))) {
		ed->n_bounce_requests = 0;
		ed->bounce_request_size = 0;
	}

	if (__unlikely(error)) {
		UHCI_FREE_ENDPOINT(&ed->ep);
		return __ERR_PTR(error);
	}
	return &ed->ep;
}

__COLD_ATTR__ static void UHCI_DISABLE_ENDPOINT(USB_ENDPOINT *endpt)
{
	UHCI *uhci;
	ED *ed;
	QH *qh;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_DISABLE_ENDPOINT AT SPL %08X", KERNEL$SPL);
	uhci = USB2UHCI(endpt->usb);
	if (__unlikely(endpt == &uhci->root_ctrl) || __unlikely(endpt == &uhci->root_intr)) return;
	ed = GET_STRUCT(endpt, ED, ep);
	ed->ep.post = UHCI_POST_DISABLED;
	UHCI_ED_SET_ERROR(ed, -ECONNRESET);
	qh = ed->qh;
	if (__likely(qh->state == QHS_ACTIVE)) UHCI_START_UNLINK_QH(qh);
}

__COLD_ATTR__ static void UHCI_FREE_ENDPOINT(USB_ENDPOINT *endpt)
{
	UHCI *uhci;
	ED *ed;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_FREE_ENDPOINT AT SPL %08X", KERNEL$SPL);
	uhci = USB2UHCI(endpt->usb);
	if (__unlikely(endpt == &uhci->root_ctrl) || __unlikely(endpt == &uhci->root_intr)) return;
	ed = GET_STRUCT(endpt, ED, ep);

	if (__unlikely(ed->qh->required_tds < ed->required_tds))
		KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: REQUIRED_TDS MISCOUNTED: %u < %u", ed->qh->required_tds, ed->required_tds);
	ed->qh->required_tds -= ed->required_tds;
	ed->required_tds = 0;

	if (__unlikely(ed->qh->required_bounces < ed->required_bounces))
		KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: REQUIRED_BOUNCES MISCOUNTED: %u < %u", ed->qh->required_bounces, ed->required_bounces);
	ed->qh->required_bounces -= ed->required_bounces;
	ed->required_bounces = 0;

	if (__likely(ed->qh == &ed->qh_default)) {
		if (!UHCI_QUEUE_IS_EMPTY(&ed->qh_default))
			KERNEL$SUICIDE("UHCI_FREE_ENDPOINT: FREEING %s ENDPOINT %u:%u WHILE SOME REQUESTS ARE PENDING", USB$ENDPOINT_TYPE_STRING(ed->type), ed->addr, ed->pipe);
		if (ed->qh_default.state != QHS_INACTIVE) {
			if (ed->qh_default.state == QHS_ACTIVE) UHCI_START_UNLINK_QH(&ed->qh_default);
			ed->qh_default.state = QHS_UNLINKING_FREEING;
			return;
		}
		UHCI_FREE_QH(&ed->qh_default);
	}
	DEL_FROM_LIST(&ed->endpoints_list);
	if (__likely(ed->bounce_request_size != 0)) {
		KERNEL$UNRESERVE_BOUNCE(uhci->dev_name, ed->n_bounce_requests, ed->bounce_request_size, 0);
	}
	KERNEL$FREE_CONTIG_AREA(ed, __PAGE_CLUSTER_SIZE);
}

__COLD_ATTR__ static int UHCI_ROOT_PREPARE_CTRL(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	UHCI *uhci = GET_STRUCT(endpt, UHCI, root_ctrl);
	HC_CHECK_SPL("UHCI_ROOT_PREPARE_CTRL");
	brq->tmp1 = (unsigned long)uhci;
	if (__unlikely(brq->v.vspace != &KERNEL$VIRTUAL)) {
		KERNEL$SUICIDE("UHCI_ROOT_PREPARE_CTRL: INVALID VSPACE");
	}
	return 0;
}

__COLD_ATTR__ static void UHCI_ROOT_FREE(USB_REQUEST *rq)
{
	HC_CHECK_SPL("UHCI_ROOT_FREE");
}

__COLD_ATTR__ static void UHCI_ROOT_POST_CTRL(USB_REQUEST *rq)
{
	USB_CTRL_REQUEST *urq = (USB_CTRL_REQUEST *)rq;
	UHCI *uhci = (UHCI *)urq->tmp1;
	HC_CHECK_SPL("UHCI_ROOT_POST_CTRL");
	urq->status = 0;
	switch (__16LE2CPU(urq->setup.request)) {
		case USBHUB_REQ_SET_PORT_FEATURE: {
			unsigned idx;
			if (__unlikely(urq->v.len != 0)) {
				inval:
				urq->status = -EINVAL;
				break;
			}
			idx = __16LE2CPU(urq->setup.w_index) - 1;
			if (__unlikely(idx >= uhci->n_ports)) goto inval;
			switch (__16LE2CPU(urq->setup.w_value)) {
				case USB_PORT_FEAT_SUSPEND:
					UHCI_SET_PORT_BIT(uhci, idx, UHCI_PORTSC_SUSPEND);
					break;
				case USB_PORT_FEAT_RESET:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE);
					UHCI_SET_PORT_BIT(uhci, idx, UHCI_PORTSC_RESET);
					KERNEL$UDELAY(10);
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE_CHANGE);
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_CONNECT_CHANGE);
					KERNEL$DEL_TIMER(&uhci->hub_timer);
					KERNEL$SET_TIMER(UHCI_HUB_RESET_TIME, &uhci->hub_timer);
					break;
				default:
					goto dflt;
			}
			break;
		}
		case USBHUB_REQ_CLEAR_PORT_FEATURE: {
			unsigned idx;
			if (__unlikely(urq->v.len != 0)) goto inval;
			idx = __16LE2CPU(urq->setup.w_index) - 1;
			if (__unlikely(idx >= uhci->n_ports)) goto inval;
			switch (__16LE2CPU(urq->setup.w_value)) {
				case USB_PORT_FEAT_ENABLE:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE);
					break;
				case USB_PORT_FEAT_C_ENABLE:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_ENABLE_CHANGE);
					break;
				/* unimplemented 
				case USB_PORT_FEAT_SUSPEND:
				case USB_PORT_FEAT_C_SUSPEND:
				*/
				case USB_PORT_FEAT_C_CONNECTION:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_CONNECT_CHANGE);
					break;
				case USB_PORT_FEAT_C_OVER_CURRENT:
					UHCI_CLEAR_PORT_BIT(uhci, idx, UHCI_PORTSC_OC_CHANGE);
					break;
				default:
					goto dflt;
			}
			break;
		}
		case USBHUB_REQ_GET_HUB_STATUS: {
			if (__unlikely(urq->v.len != sizeof(__u32))) goto inval;
			*(__u32 *)(unsigned long)urq->v.ptr = __32CPU2LE(0);
			break;
		}
		case USBHUB_REQ_GET_PORT_STATUS: {
			__u32 stat;
			__u16 ps;
			unsigned idx;
			if (__unlikely(urq->v.len != sizeof(__u32))) goto inval;
			idx = __16LE2CPU(urq->setup.w_index) - 1;
			if (__unlikely(idx >= uhci->n_ports)) goto inval;
			stat = 0;
			ps = UHCI_READ_PORT(uhci, idx);
			if (uhci->flags & FLAG_REVERT_OC) ps ^= UHCI_PORTSC_OC;
			if (ps & UHCI_PORTSC_CONNECT_CHANGE) stat |= USB_PORT_STAT_C_CONNECTION;
			if (ps & UHCI_PORTSC_ENABLE_CHANGE) stat |= USB_PORT_STAT_C_ENABLE;
			if (ps & UHCI_PORTSC_OC_CHANGE) stat |= USB_PORT_STAT_C_OVER_CURRENT;
			stat |= USB_PORT_STAT_POWER;
			if (ps & UHCI_PORTSC_CONNECT) stat |= USB_PORT_STAT_CONNECTION;
			if (ps & UHCI_PORTSC_ENABLE) stat |= USB_PORT_STAT_ENABLE;
			if (ps & (UHCI_PORTSC_SUSPEND | UHCI_PORTSC_RESUME_DETECT)) stat |= USB_PORT_STAT_SUSPEND;
			if (ps & UHCI_PORTSC_OC) stat |= USB_PORT_STAT_OVER_CURRENT;
			if (ps & UHCI_PORTSC_RESET) stat |= USB_PORT_STAT_RESET;
			if (ps & UHCI_PORTSC_LOW_SPEED) stat |= USB_PORT_STAT_LOWSPEED;
			*(__u32 *)(unsigned long)urq->v.ptr = __32CPU2LE(stat);
			/*__debug_printf("PORT STATUS: %04X -> %08X\n", ps, stat);*/
			break;
		}
		case USBHUB_REQ_GET_HUB_DESCRIPTOR: {
			USB_DESCRIPTOR_HUB *desc;
			if (__unlikely(urq->v.len != USB_DESCRIPTOR_HUB_SIZE)) goto inval;
			desc = (USB_DESCRIPTOR_HUB *)(unsigned long)urq->v.ptr;
			memset(desc, 0xff, USB_DESCRIPTOR_HUB_SIZE);
			desc->bDescLength = 7 + 2 * ((uhci->n_ports + 1 + 7) / 8);
			desc->bDescriptorType = USB_DT_HUB;
			desc->bNbrPorts = uhci->n_ports;
			desc->wHubCharacteristics = USB_DESCRIPTOR_HUB_CHAR_OC_INDIVIDUAL | USB_DESCRIPTOR_HUB_CHAR_POWER_NONE;
			desc->wHubCharacteristics2 = 0;
			desc->bPwrOn2PwrGood = 1;
			desc->bHubContrCurrent = 0;
			desc->DeviceRemovable[0] = 0;
			urq->v.len = desc->bDescLength;
			break;
		}
		default:
		dflt: {
			USB$HC_GENERIC_ROOT_CTRL(UHCI2USB(uhci), urq);
			return;
		}
	}
	if (__likely(urq->status >= 0))
		urq->v.len += sizeof(USB_CTRL_SETUP);
	else
		urq->v.len = 0;
	CALL_AST(urq);
}

__COLD_ATTR__ static void UHCI_ROOT_CANCEL_CTRL(USB_REQUEST *rq)
{
	HC_CHECK_SPL("UHCI_ROOT_CANCEL_CTRL");
}

__COLD_ATTR__ static int UHCI_ROOT_PREPARE_INTR(USB_ENDPOINT *endpt, USB_REQUEST *brq)
{
	UHCI *uhci = GET_STRUCT(endpt, UHCI, root_intr);
	HC_CHECK_SPL("UHCI_ROOT_PREPARE_INTR");
	brq->tmp1 = (unsigned long)uhci;
	if (__unlikely(brq->v.vspace != &KERNEL$VIRTUAL)) {
		KERNEL$SUICIDE("UHCI_ROOT_PREPARE_INTR: INVALID VSPACE");
	}
	return 0;
}

__COLD_ATTR__ static void UHCI_ROOT_POST_INTR(USB_REQUEST *urq)
{
	UHCI *uhci = (UHCI *)urq->tmp1;
	HC_CHECK_SPL("UHCI_ROOT_POST_INTR");
	if (__unlikely(uhci->rh_intr != NULL))
		KERNEL$SUICIDE("UHCI_ROOT_POST_INTR: ANOTHER REQUEST ALREADY PENDING");
	if (__unlikely(urq->v.len < ((uhci->n_ports + 1 + 7) >> 3))) {
		urq->status = -EINVAL;
		CALL_AST(urq);
		return;
	}
	urq->v.len = (uhci->n_ports + 1 + 7) >> 3;
	memset((void *)(unsigned long)urq->v.ptr, 0, urq->v.len);
	urq->status = RQS_CANCELABLE;
	uhci->rh_intr = urq;
	return;
}

__COLD_ATTR__ static void UHCI_ROOT_CANCEL_INTR(USB_REQUEST *urq)
{
	UHCI *uhci = (UHCI *)urq->tmp1;
	HC_CHECK_SPL("UHCI_ROOT_CANCEL_INTR");
	if (__unlikely(urq->status != RQS_CANCELABLE)) return;
	uhci = (UHCI *)urq->tmp1;
	if (__likely(urq == uhci->rh_intr)) {
		uhci->rh_intr = NULL;
		urq->status = -EINTR;
		CALL_AST(urq);
	}
}

__COLD_ATTR__ static void UHCI_ROOT_CLEAR_STALL(USB_ENDPOINT *endpt)
{
	HC_CHECK_SPL("UHCI_ROOT_CLEAR_STALL");
}

static void UHCI_HUB_SCAN(UHCI *uhci);

static void UHCI_HUB_TIMER(TIMER *t)
{
	UHCI *uhci;
	LOWER_SPL(SPL_USB);
	uhci = GET_STRUCT(t, UHCI, hub_timer);
	KERNEL$SET_TIMER(UHCI_HUB_CHECK_TIME, &uhci->hub_timer);
	UHCI_PROCESS_SCHEDULE(uhci);
	UHCI_HUB_SCAN(uhci);
}

static void UHCI_HUB_SCAN(UHCI *uhci)
{
	unsigned i;
	int set = 0;
	for (i = 0; i < uhci->n_ports; i++) {
		__u16 status = UHCI_READ_PORT(uhci, i);
		/*if (!i) {
			static __u16 old = 0;
			if (status != old)
				__debug_printf("%s: port %d, status %04x\n", uhci->dev_name, i, status);
			old = status;
		}*/
		if (__unlikely(status & UHCI_PORTSC_RESET)) {
			UHCI_CLEAR_PORT_BIT(uhci, i, UHCI_PORTSC_RESET);
			KERNEL$UDELAY(10);
			UHCI_CLEAR_PORT_BIT(uhci, i, UHCI_PORTSC_ENABLE_CHANGE);
			UHCI_CLEAR_PORT_BIT(uhci, i, UHCI_PORTSC_CONNECT_CHANGE);
			UHCI_SET_PORT_BIT(uhci, i, UHCI_PORTSC_ENABLE);
			status = UHCI_READ_PORT(uhci, i);
		}
		if (__unlikely(status & (UHCI_PORTSC_CONNECT_CHANGE | UHCI_PORTSC_ENABLE_CHANGE | UHCI_PORTSC_OC_CHANGE))) {
			uhci->last_port_change = KERNEL$GET_JIFFIES();
			if (__likely(uhci->rh_intr != NULL)) {
				__u8 *data = (__u8 *)(unsigned long)uhci->rh_intr->v.ptr;
				unsigned off = i + 1;
				data[off / 8] |= 1 << (off & 7);
				set = 1;
			}
		}
	}
	if (__unlikely(set)) {
		USB_REQUEST *intr = uhci->rh_intr;
		intr->status = 0;
		uhci->rh_intr = NULL;
		CALL_AST(intr);
	}
}

__COLD_ATTR__ static int UHCI_TEST_PORT_CHANGE(UHCI *uhci)
{
	unsigned i;
	for (i = 0; i < uhci->n_ports; i++) {
		__u16 status = UHCI_READ_PORT(uhci, i);
		if (__unlikely(status & (UHCI_PORTSC_CONNECT_CHANGE | UHCI_PORTSC_ENABLE_CHANGE | UHCI_PORTSC_OC_CHANGE))) {
			UHCI_HUB_SCAN(uhci);
			return 1;
		}
	}
	return (KERNEL$GET_JIFFIES() - uhci->last_port_change) <= UHCI_PORTCHG_QUIET;
}

__COLD_ATTR__ static int UHCI_DETECT_PORTS(UHCI *uhci)
{
	unsigned port;
	for (port = 0; port <= UHCI_MAX_PORTS; port++) {
		__u16 status = UHCI_READ_PORT(uhci, port);
		/*__debug_printf("%d: %x, %d\n", port, status, uhci->flags);*/
		if (!(status & UHCI_PORTSC_RESERVED_ONE) || status == 0xffff) break;
	}
	if (port > UHCI_MAX_PORTS) port = UHCI_DEFAULT_PORTS;
	return port;
}

__COLD_ATTR__ static void UHCI_TAKEOVER(pci_id_t id, int grab_it)
{
	__u16 legsup;
	__u16 usbsts;
	__u16 usbcmd;
	io_t io;
	if (grab_it) {
		if (__unlikely(PCI$FIND_DEVICE(uhci_pci, id, ~(pci_id_t)0, 0, PCI$TEST_LIST, NULL, NULL, NULL, 0))) return;
		/*__debug_printf("id: %08x\n", id);*/
	}
	io = PCI$READ_IO_RESOURCE(id, 4);
	if (__unlikely(!io)) goto ret;
	usbcmd = io_inw(io + UHCI_USBCMD);
	usbcmd &= ~(UHCI_USBCMD_RS | UHCI_USBCMD_CF);
	io_outw(io + UHCI_USBCMD, usbcmd);
	KERNEL$UDELAY(1500);
	usbsts = io_inw(io + UHCI_USBSTS);
	io_outw(io + UHCI_USBSTS, usbsts);
	legsup = PCI$READ_CONFIG_WORD(id, UHCI_PCI_LEGSUP);
	if (legsup & UHCI_PCI_LEGSUP_HAS_BIOS) {
		PCI$WRITE_CONFIG_WORD(id, UHCI_PCI_LEGSUP, legsup & ~UHCI_PCI_LEGSUP_USBSMIEN);
		/*__debug_printf("takeover...\n");*/
	}
	/*else __debug_printf("no takeover...\n");*/
	PCI$WRITE_CONFIG_WORD(id, UHCI_PCI_LEGSUP, UHCI_PCI_LEGSUP_DEFAULT);
	ret:
	if (grab_it) {
		PCI$FREE_DEVICE(id);
	}
}

__COLD_ATTR__ static int UHCI_RESET(UHCI *uhci)
{
	unsigned i;
	__u8 sofmod;
	sofmod = UHCI_READ_8(uhci, UHCI_SOFMOD);
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_HCRESET);
	for (i = 0; i < UHCI_RESET_TIMEOUT; i += UHCI_RESET_STEP) {
		KERNEL$UDELAY(UHCI_RESET_STEP);
		if (!(UHCI_READ_16(uhci, UHCI_USBCMD) & UHCI_USBCMD_HCRESET)) goto done_reset;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "RESET FAILED");
	return -EIO;
	done_reset:
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	UHCI_WRITE_16(uhci, UHCI_USBSTS, UHCI_READ_16(uhci, UHCI_USBSTS));
	UHCI_WRITE_16(uhci, UHCI_USBCMD, 0);
	for (i = 0; i < uhci->n_ports; i++) {
		UHCI_WRITE_PORT(uhci, i, 0);
	}
	UHCI_WRITE_8(uhci, UHCI_SOFMOD, sofmod);
	UHCI_WRITE_32(uhci, UHCI_FLBASEADD, uhci->area_dmaaddr + __offsetof(AREA, fl));
	uhci->last_head_link = &UHCI2AREA(uhci)->loop_start.head_link;
	INIT_XLIST(&uhci->intr_qh);
	for (i = 0; i < N_FRAME_LIST; i++) {
		UHCI2AREA(uhci)->fl.addr[i] = UHCI_CTRL_START_Q_LE(uhci);
	}
	UHCI2AREA(uhci)->sof_irq.link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->sof_irq.flags1 = __32CPU2LE(TD_FLAGS1_IOC | TD_FLAGS1_ACTLEN);
	UHCI2AREA(uhci)->sof_irq.flags2 = __32CPU2LE(TD_FLAGS2_MAXLEN | TD_FLAGS2_DEVICE_ADDRESS | TD_FLAGS2_PID_SETUP);
	UHCI2AREA(uhci)->sof_irq.ptr = __32CPU2LE(0);
	UHCI2AREA(uhci)->ctrl_start.head_link = UHCI_LOOP_START_Q_LE(uhci);
	UHCI2AREA(uhci)->ctrl_start.element_link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->loop_start.head_link = __32CPU2LE(UHCI_ADDR_T);
	UHCI2AREA(uhci)->loop_start.element_link = __32CPU2LE(UHCI_ADDR_T);
	return 0;
}

__COLD_ATTR__ static void UHCI_CONTINUE_RESET(UHCI *uhci)
{
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_RS | UHCI_USBCMD_MAXP);
	UHCI_WRITE_16(uhci, UHCI_USBINTR, UHCI_USBINTR_TIMEOUT_CRC | UHCI_USBINTR_RESUME | UHCI_USBINTR_IOC | UHCI_USBINTR_SHORT_PACKET);
}

static void UHCI_HARD_ERROR(UHCI *uhci);

__COLD_ATTR__ static UHCI *UHCI_SPECIAL_IRQ(UHCI *uhci, __u16 sts)
{
	if (uhci->flags & FLAG_DEAD) goto h_e;
	if (sts == 0xffff) goto h_e;
	if (sts & UHCI_USBSTS_HC_PROCESS_ERROR)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "UHCI PROCESSING ERROR");
	if (sts & UHCI_USBSTS_HC_SYSTEM_ERROR)
		KERNEL$SYSLOG(__SYSLOG_HW_ERROR, uhci->dev_name, "UHCI PCI SYSTEM ERROR");
	if (sts & UHCI_USBSTS_RESUME_DETECT)
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "UNEXPECTED RESUME INTERRUPT");
	if (sts & ~(UHCI_USBSTS_HC_PROCESS_ERROR | UHCI_USBSTS_HC_SYSTEM_ERROR | UHCI_USBSTS_RESUME_DETECT)) {
		if ((sts & ~UHCI_USBSTS_RESERVED) == UHCI_USBSTS_HC_HALTED)
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "CONTROLLER UNEXPECTEDLY HALTED");
		else
			KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "UNKNOWN STATUS REGISTER: %04X", sts);
	}
	h_e:
	UHCI_HARD_ERROR(uhci);
	return uhci;
}

__COLD_ATTR__ static int UHCI_STOP(UHCI *uhci)
{
	unsigned i;
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	UHCI_WRITE_16(uhci, UHCI_USBSTS, UHCI_READ_16(uhci, UHCI_USBSTS));
	UHCI_WRITE_16(uhci, UHCI_USBCMD, 0);
	for (i = 0; i < UHCI_RESET_TIMEOUT; i += UHCI_RESET_STEP) {
		KERNEL$UDELAY(UHCI_RESET_STEP);
		if (UHCI_READ_16(uhci, UHCI_USBSTS) & UHCI_USBSTS_HC_HALTED) goto done_stop;
	}
	if (uhci->errorlevel >= 2)
		KERNEL$SYSLOG(__SYSLOG_HW_WARNING, uhci->dev_name, "STOP FAILED, RESETTING");
	done_stop:
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_HCRESET);
	for (i = 0; i < UHCI_RESET_TIMEOUT; i += UHCI_RESET_STEP) {
		KERNEL$UDELAY(UHCI_RESET_STEP);
		if (!(UHCI_READ_16(uhci, UHCI_USBCMD) & UHCI_USBCMD_HCRESET)) goto done_reset;
	}
	KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "RESET FAILED, TRYING HARD RESET");
	UHCI_WRITE_16(uhci, UHCI_USBCMD, UHCI_USBCMD_GRESET);
	return -EIO;
	done_reset:
	return 0;
}

static void UHCI_DIE(UHCI *uhci);

__COLD_ATTR__ static void UHCI_HARD_ERROR(UHCI *uhci)
{
	ED *ed;
	QH *qh;
	if (__unlikely(UHCI_STOP(uhci))) {
		UHCI_DIE(uhci);
	}
	XLIST_FOR_EACH(ed, &uhci->endpoints, ED, endpoints_list)
		UHCI_DISABLE_ENDPOINT(&ed->ep);
	XLIST_FOR_EACH(qh, &uhci->intr_qh, QH, list)	/* needed only to unlink low-speed control queue */
		UHCI_START_UNLINK_QH(qh);
	UHCI_PROCESS_UNLINKING_ENTRIES(uhci, 1);
	if (__unlikely(!XLIST_EMPTY(&uhci->intr_qh)))
		KERNEL$SUICIDE("UHCI_HARD_ERROR: INTERRUPT LIST NOT EMPTY");
	if (__unlikely(UHCI2AREA(uhci)->ctrl_start.head_link != UHCI_LOOP_START_Q_LE(uhci)))
		KERNEL$SUICIDE("UHCI_HARD_ERROR: CONTROL LIST NOT EMPTY");
	if (__unlikely(UHCI2AREA(uhci)->loop_start.head_link != __32CPU2LE(UHCI_ADDR_T)))
		KERNEL$SUICIDE("UHCI_HARD_ERROR: BULK LIST NOT EMPTY");
	if (__unlikely(UHCI_RESET(uhci)))
		UHCI_DIE(uhci);
}

__COLD_ATTR__ static void UHCI_DIE(UHCI *uhci)
{
	if (!(uhci->flags & FLAG_DEAD))
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, uhci->dev_name, "CONTROLLER DIED");
	uhci->flags |= FLAG_DEAD;
}

__COLD_ATTR__ static void UHCI_WAIT_FOR_LISTS(UHCI *uhci)
{
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS AT SPL %08X", KERNEL$SPL);
	if (UHCI2AREA(uhci)->low_ctrl.state == QHS_ACTIVE) UHCI_START_UNLINK_QH(&UHCI2AREA(uhci)->low_ctrl);
	/* this won't loop indefinitely because of unlink_start */
	while (1) {
		if (__unlikely(!XLIST_EMPTY(&uhci->intr_qh)))
			KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: LEAKED INTERRUPT OR LOW-SPEED CONTROL ENDPOINT");
		if (__unlikely(UHCI2AREA(uhci)->ctrl_start.head_link != UHCI_LOOP_START_Q_LE(uhci)))
			KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: LEAKED CONTROL ENDPOINT");
		if (__unlikely(UHCI2AREA(uhci)->loop_start.head_link != __32CPU2LE(UHCI_ADDR_T)))
			KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: LEAKED BULK ENDPOINT");
		if (__likely(XLIST_EMPTY(&uhci->unlinking))) break;
		KERNEL$SLEEP(1);
		UHCI_PROCESS_SCHEDULE(uhci);	/* in case interrupt was blocked by some other driver */
	}
	if (__unlikely(!XLIST_EMPTY(&uhci->endpoints)))
		KERNEL$SUICIDE("UHCI_WAIT_FOR_LISTS: ENDPOINTS LEAKED");
}

static int UHCI_UNLOAD(void *p, void **release, const char * const argv[]);

__COLD_ATTR__ int main(int argc, const char *const argv[])
{
	unsigned fsbr_time = DEFAULT_FSBR_TIME;
	const char * const *arg = argv;
	int state = 0;
	pci_id_t id = 0, id_mask = 0;
	int order = 0;
	char dev_name[__MAX_STR_LEN];
	char gstr[__MAX_STR_LEN];
	const char *opt, *optend, *str;
	int r, rr, reg = 0;
	unsigned long flags;
	char *chip_name;
	io_t io;
	int irq;
	UHCI *uhci;
	VDESC desc;
	VDMA vdma;
	int n_ports;
	int errorlevel = 0;
	char *plug = USB$HC_DEFAULT_PLUG_SCRIPT;
	static const struct __param_table params[5] = {
		"LOG_ERRORS", __PARAM_BOOL, ~0, 1,
		"LOG_WARNINGS", __PARAM_BOOL, ~0, 2,
		"PLUG", __PARAM_STRING, 0, __MAX_STR_LEN,
		"FSBR_TIME", __PARAM_UNSIGNED_INT, 0, (MAXINT - 999) / JIFFIES_PER_SECOND,
		NULL, 0, 0, 0,
	};
	void *vars[5];

	vars[0] = &errorlevel;
	vars[1] = &errorlevel;
	vars[2] = &plug;
	vars[3] = &fsbr_time;
	vars[4] = NULL;
	while (__parse_params(&arg, &state, params, vars, &opt, &optend, &str)) {
		if (PCI$PARSE_PARAMS(opt, optend, str, &id, &id_mask, &order)) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UHCI: SYNTAX ERROR");
			r = -EBADSYN;
			goto ret0;
		}
	}
	if (PCI$FIND_DEVICE(uhci_pci, id, id_mask, order, PCI$TEST_LIST, &id, &chip_name, &flags, 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UHCI: NO PCI DEVICE FOUND");
		r = -ENODEV;
		goto ret0;
	}

	BUST_EHCI();

	_snprintf(dev_name, sizeof(dev_name), "USB$UHCI@" PCI_ID_FORMAT, id);
	PCI$ENABLE_DEVICE(id, PCI_COMMAND_IO | PCI_COMMAND_MASTER);

	io = PCI$READ_IO_RESOURCE(id, 4);
	if (!io) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO IO RESOURCE");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO IO RESOURCE", dev_name);
		r = -ENXIO;
		goto ret1;
	}
	irq = PCI$READ_INTERRUPT_LINE(id);
	/* Documentation says that function 2 holds the LEGSUP register, however
	   ICH6 will lock up if takeover is not performed on function 0 (and
	   doesn't care about other functions). Rather takeover both. */
	RAISE_SPL(SPL_DEV);
	UHCI_TAKEOVER((id & ~(pci_id_t)PCI_ID_T_FUNCTION) | __SET_FIELD((pci_id_t)0, PCI_ID_T_FUNCTION), 1);
	UHCI_TAKEOVER((id & ~(pci_id_t)PCI_ID_T_FUNCTION) | __SET_FIELD((pci_id_t)2, PCI_ID_T_FUNCTION), 1);
	LOWER_SPL(SPL_ZERO);
	uhci = KERNEL$ALLOC_CONTIG_AREA(CONTIG_SIZE, AREA_DATA | AREA_PCIDMA | AREA_PHYSCONTIG | AREA_ALIGN, (unsigned long)CONTIG_ALIGN);
	if (__IS_ERR(uhci) < 0) {
		r = __PTR_ERR(uhci);
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE MEMORY: %s", dev_name, strerror(-r));
		goto ret1;
	}
	memset(uhci, 0, sizeof(AREA) + sizeof(UHCI));
	uhci = (UHCI *)((char *)uhci + sizeof(AREA));
	strcpy(uhci->dev_name, dev_name);
	uhci->id = id;
	uhci->io = io;
	uhci->flags = flags;
	uhci->errorlevel = errorlevel;
	uhci->fsbr_time = (fsbr_time * JIFFIES_PER_SECOND + 999) / 1000;
	uhci->fsbr_timer.fn = UHCI_FSBR_TIMER;
	INIT_TIMER(&uhci->fsbr_timer);
	SET_TIMER_NEVER(&uhci->fsbr_timer);
	INIT_XLIST(&uhci->unlinking);
	INIT_XLIST(&uhci->endpoints);

	uhci->root_ctrl.prepare = UHCI_ROOT_PREPARE_CTRL;
	uhci->root_ctrl.post = UHCI_ROOT_POST_CTRL;
	uhci->root_ctrl.free = UHCI_ROOT_FREE;
	uhci->root_ctrl.cancel = UHCI_ROOT_CANCEL_CTRL;
	uhci->root_ctrl.clear_stall = UHCI_ROOT_CLEAR_STALL;
	uhci->root_intr.prepare = UHCI_ROOT_PREPARE_INTR;
	uhci->root_intr.post = UHCI_ROOT_POST_INTR;
	uhci->root_intr.free = UHCI_ROOT_FREE;
	uhci->root_intr.cancel = UHCI_ROOT_CANCEL_INTR;
	uhci->root_intr.clear_stall = UHCI_ROOT_CLEAR_STALL;

	desc.ptr = (unsigned long)UHCI2AREA(uhci);
	desc.len = sizeof(AREA);
	desc.vspace = &KERNEL$VIRTUAL;
	vdma.spl = SPL_X(SPL_ZERO);
	RAISE_SPL(SPL_VSPACE);
	uhci->area_dmaunlock = KERNEL$VIRTUAL.op->vspace_dmalock(&desc, PF_READ | PF_WRITE, &vdma);
	if (__unlikely(vdma.len != desc.len))
		KERNEL$SUICIDE("UHCI: CAN'T DMALOCK AREA");
	uhci->area_dmaaddr = vdma.ptr;

	if (__unlikely(r = UHCI_INIT_QH(uhci, &UHCI2AREA(uhci)->low_ctrl, USB_EP_CTRL, 1, NULL))) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT ALLOCATE MEMORY: %s", dev_name, strerror(-r));
		goto ret2;
	}

	n_ports = UHCI_DETECT_PORTS(uhci);
	if (__unlikely(n_ports <= 0)) {
		KERNEL$SYSLOG(__SYSLOG_HW_BUG, dev_name, "NO ROOT HUB PORTS");
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: NO ROOT HUB PORTS", dev_name);
		r = -ENXIO;
		goto ret3;
	}
	uhci->n_ports = n_ports;
	RAISE_SPL(SPL_USB);
	UHCI_TAKEOVER(id, 0);
	if (__unlikely(r = UHCI_RESET(uhci))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT INITIALIZE CONTROLLER", dev_name);
		goto ret4;
	}
	uhci->irq_ast.fn = UHCI_IRQ;
	if ((r = KERNEL$REQUEST_IRQ(irq, &uhci->irq_ctrl, IRQ_REQUEST_AST_HANDLER | IRQ_REQUEST_SHARED, NULL, &uhci->irq_ast, dev_name)) < 0) {
		KERNEL$SYSLOG(__SYSLOG_SYS_CONFLICT, dev_name, "COULD NOT GET IRQ %d: %s", irq, strerror(-r));
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT GET IRQ", dev_name);
		goto ret4;
	}
	UHCI_CONTINUE_RESET(uhci);
	INIT_TIMER(&uhci->hub_timer);
	uhci->hub_timer.fn = UHCI_HUB_TIMER;
	KERNEL$SET_TIMER(UHCI_HUB_CHECK_TIME, &uhci->hub_timer);
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r = USB$HC_REGISTER(UHCI2USB(uhci), &UHCI_HC_OPERATIONS, uhci->dev_name, errorlevel, plug))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: CAN'T REGISTER USB HOST CONTROLLER: %s", dev_name, strerror(-r));
		goto ret5;
	}
	reg = 1;


	_printf("%s: %s ON PCI: %s\n", dev_name, chip_name, PCI$ID(gstr, id));
	_printf("%s: IO "IO_FORMAT", IRQ %d\n", dev_name, io, irq);

	r = KERNEL$REGISTER_DEVICE(dev_name, "UHCI.SYS", 0, UHCI2USB(uhci), NULL, USB$HC_DCALL, USB$HC_DCALL_TYPE, USB$HC_DCTL, UHCI_UNLOAD, &uhci->lnte, NULL);
	if (r < 0) {
		if (r != -EINTR) _snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "%s: COULD NOT REGISTER DEVICE", dev_name);
		goto ret5;
	}
	uhci->dlrq = KERNEL$TSR_IMAGE();
	strcpy(KERNEL$ERROR_MSG(), dev_name);
	return 0;

	ret5:
	RAISE_SPL(SPL_USB);
	UHCI_WAIT_FOR_LISTS(uhci);
	LOWER_SPL(SPL_ZERO);
	KERNEL$MASK_IRQ(uhci->irq_ctrl);
	RAISE_SPL(SPL_USB);
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	UHCI_WRITE_16(uhci, UHCI_USBSTS, UHCI_READ_16(uhci, UHCI_USBSTS));
	KERNEL$RELEASE_IRQ(uhci->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &uhci->irq_ast);
	LOWER_SPL(SPL_ZERO);
	if (reg) {
		if ((rr = USB$HC_PREPARE_UNREGISTER(UHCI2USB(uhci))))
			KERNEL$SUICIDE("UHCI: FAILED TO UNREGISTER UHCI CONTROLLER: %s", strerror(-rr));
		USB$HC_UNREGISTER(UHCI2USB(uhci));
	}
	RAISE_SPL(SPL_USB);
	KERNEL$DEL_TIMER(&uhci->hub_timer);
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	ret4:
	UHCI_STOP(uhci);
	LOWER_SPL(SPL_ZERO);
	ret3:
	RAISE_SPL(SPL_USB);
	UHCI_FREE_QH(&UHCI2AREA(uhci)->low_ctrl);
	LOWER_SPL(SPL_ZERO);
	ret2:
	uhci->area_dmaunlock(uhci->area_dmaaddr);
	KERNEL$FREE_CONTIG_AREA(UHCI2AREA(uhci), CONTIG_SIZE);
	ret1:
	PCI$FREE_DEVICE(id);
	ret0:
	return r;
}

__COLD_ATTR__ static int UHCI_UNLOAD(void *p, void **release, const char * const argv[])
{
	int r;
	UHCI *uhci = USB2UHCI(p);
	if (argv[0] && argv[1]) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: SYNTAX ERROR", uhci->dev_name);
		return -EBADSYN;
	}
	if ((r = USB$HC_PREPARE_UNREGISTER(UHCI2USB(uhci)))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "UNLOAD %s: UNABLE TO REMOVE HOST CONTROLLER: %s", uhci->dev_name, strerror(-r));
		return r;
	}
	KERNEL$UNREGISTER_DEVICE(uhci->lnte, 0);
	RAISE_SPL(SPL_USB);
	UHCI_WAIT_FOR_LISTS(uhci);
	LOWER_SPL(SPL_ZERO);
	KERNEL$MASK_IRQ(uhci->irq_ctrl);
	RAISE_SPL(SPL_USB);
	UHCI_WRITE_16(uhci, UHCI_USBINTR, 0);
	UHCI_WRITE_16(uhci, UHCI_USBSTS, UHCI_READ_16(uhci, UHCI_USBSTS));
	KERNEL$RELEASE_IRQ(uhci->irq_ctrl, IRQ_REQUEST_MASKED | IRQ_REQUEST_SHARED | IRQ_REQUEST_AST_HANDLER, NULL, &uhci->irq_ast);
	LOWER_SPL(SPL_ZERO);
	USB$HC_UNREGISTER(UHCI2USB(uhci));
	RAISE_SPL(SPL_USB);
	KERNEL$DEL_TIMER(&uhci->hub_timer);
	KERNEL$DEL_TIMER(&uhci->fsbr_timer);
	UHCI_STOP(uhci);
	UHCI_FREE_QH(&UHCI2AREA(uhci)->low_ctrl);
	LOWER_SPL(SPL_ZERO);
	uhci->area_dmaunlock(uhci->area_dmaaddr);
	PCI$FREE_DEVICE(uhci->id);
	*release = uhci->dlrq;
	KERNEL$FREE_CONTIG_AREA(UHCI2AREA(uhci), CONTIG_SIZE);
	return 0;
}

