#include <STDLIB.H>
#include <SPAD/AC.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYSLOG.H>
#include <SPAD/WQ.H>
#include <SPAD/TIMER.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV_KRNL.H>
#include <SETJMP.H>

#include <SPAD/USBHUB.H>

#include "USB.H"

char *USB_DEV_ID(USB_DEV *dev)
{
	static char n[__MAX_STR_LEN];
	_snprintf(n, sizeof n, "%04X:%04X:%04X", (unsigned)dev->dev_desc.idVendor, (unsigned)dev->dev_desc.idProduct, (unsigned)dev->dev_desc.bcdDevice);
	return n;
}

__finline__ unsigned USB_CFG_NUM(USB_DEV_CFG *cfg)
{
	return cfg->cfg_desc.bConfigurationValue;
}

unsigned USB_CFG_ORDER(USB_DEV_CFG *cfg)
{
	return cfg - cfg->dev->confs;
}

char *USB_IF_ID(USB_DEV_INTERFACE *iface)
{
	static char n[__MAX_STR_LEN];
	_snprintf(n, sizeof n, "CONFIGURATION %u, INTERFACE %u, ALT SETTING %u", USB_CFG_NUM(iface->cfg), iface->desc->bInterfaceNumber, iface->desc->bAlternateSetting);
	return n;
}

int USB_GET_DESCRIPTOR(USB_DEV *dev, __u8 rqtype, __u8 dt, __u8 idx, __u16 windex, void *ptr, unsigned len, unsigned min_len, int flags)
{
	int r = USB$SYNC_CTRL(dev->in_endpoints[0], rqtype | USB_REQ_GET_DESCRIPTOR, (dt << 8) + idx, windex, len, ptr, (flags & USB_DESC_QUIET_ERROR ? USBRQ_QUIET_ERROR : 0) | USBRQ_ALLOW_SHORT, USB_CTRL_TIMEOUT, USB_CTRL_RETRIES, NULL, NULL);
	if (__likely(r >= 2)) {
		if (__likely(!(flags & USB_DESC_NO_TYPECHECK)) && __unlikely(((__u8 *)ptr)[1] != dt)) {
			if (!(flags & USB_DESC_QUIET_ERROR) && dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, dev->usb->dev_name, "DEVICE %s RETURNED WRONG DESCRIPTOR TYPE %02X ON DESCRIPTOR REQUEST %02X:%02X", USB_DEV_ID(dev), (unsigned)((__u8 *)ptr)[1], (unsigned)dt, (unsigned)idx);
			return -EPROTO;
		}
	}
	if (__likely(r >= 0)) {
		if (__unlikely(r < min_len)) {
			if (!(flags & USB_DESC_QUIET_ERROR) && dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, dev->usb->dev_name, "DEVICE %s RETURNED TOO SHORT DESCRIPTOR (%d) FOR REQUEST %02X:%02X", USB_DEV_ID(dev), r, (unsigned)dt, (unsigned)idx);
			return -EPROTO;
		}
	} else {
		if (r != -EINTR && r != -ENOLINK && r != -ECONNRESET) {
			if (!(flags & USB_DESC_QUIET_ERROR) && dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, dev->usb->dev_name, "DEVICE %s RETURNED ERROR READING DESCRIPTOR %02X:%02X: %s", USB_DEV_ID(dev), (unsigned)dt, (unsigned)idx, strerror(-r));
		}
	}
	return r;
}

static int USB_GET_DEVICE_DESCRIPTOR(USB_DEV *dev, __u8 dt, __u8 idx, void *ptr, unsigned len, unsigned min_len)
{
	return USB_GET_DESCRIPTOR(dev, USB_CTRL_RQTYPE_DEVICE, dt, idx, 0, ptr, len, min_len, 0);
}

static void FREE_CONFIGURATION(USB_DEV_CFG *cfg)
{
	if (__likely(cfg->valid)) {
		unsigned i;
		if (__likely(cfg->descs)) KERNEL$FREE_CONTIG_AREA(cfg->descs, cfg->descs_allocated_size);
		for (i = 0; i < cfg->n_ifaces; i++) {
			free(cfg->ifaces[i].endpts);
		}
		free(cfg->ifaces);
		cfg->valid = 0;
	}
}

void *USB_FIND_DESCRIPTOR(USB_DEV_CFG *cfg, unsigned *off, __u8 type, unsigned length, int boundary)
{
	int first = 1;
	for (; *off < cfg->descs_size; *off += cfg->descs[*off], first = 0) {
		if (__unlikely(cfg->descs[*off] < 2) || __unlikely(*off + cfg->descs[*off] > cfg->descs_size)) {
			if (cfg->dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, cfg->dev->usb->dev_name, "DEVICE %s HAS INVALID DESCRIPTOR (CONFIGURATION %u, DESCRIPTOR TYPE %u, OFFSET %u)", USB_DEV_ID(cfg->dev), USB_CFG_NUM(cfg), (unsigned)type, *off);
			*off = cfg->descs_size;
			return NULL;
		}
		if (cfg->descs[*off + 1] == type) {
			void *p;
			if (__unlikely(cfg->descs[*off] < length)) {
				if (cfg->dev->usb->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, cfg->dev->usb->dev_name, "DEVICE %s HAS TOO SHORT DESCRIPTOR (CONFIGURATION %u, DESCRIPTOR TYPE %u, OFFSET %u)", USB_DEV_ID(cfg->dev), USB_CFG_NUM(cfg), (unsigned)type, *off);
				continue;
			}
			p = &cfg->descs[*off];
			*off += cfg->descs[*off];
			return p;
		}
		if (!first && cfg->descs[*off + 1] == boundary) return NULL;
	}
	return NULL;
}

static void PARSE_CONFIGURATION(USB_DEV_CFG *cfg)
{
	USB_DESCRIPTOR_INTERFACE *idesc;
	unsigned off = 0;
	int max_ifaces = 0;
	while ((idesc = USB_FIND_DESCRIPTOR(cfg, &off, USB_DT_INTERFACE, USB_DESCRIPTOR_INTERFACE_SIZE, -1))) {
		USB_DEV_INTERFACE *iface, *new_ifaces;
		USB_DEV_ENDPOINT *endpts;
		unsigned n_endpts;
		unsigned i;
		int iface_is_new = 1;
		for (i = 0; i < cfg->n_ifaces; i++) {
			if (cfg->ifaces[i].desc->bInterfaceNumber == idesc->bInterfaceNumber) {
				iface_is_new = 0;
				if (cfg->ifaces[i].desc->bAlternateSetting == idesc->bAlternateSetting) {
					if (cfg->dev->usb->errorlevel >= 1)
						KERNEL$SYSLOG(__SYSLOG_NET_ERROR, cfg->dev->usb->dev_name, "DEVICE %s HAS MULTIPLE DEFINITIONS OF THE SAME INTERFACE (CONFIGURATION %u, INTERFACE %u, ALT SETTING %u)", USB_DEV_ID(cfg->dev), USB_CFG_NUM(cfg), (unsigned)idesc->bInterfaceNumber, (unsigned)idesc->bAlternateSetting);
					continue;
				}
			}
		}
		new_ifaces = __sync_realloc(cfg->ifaces, (cfg->n_ifaces + 1) * sizeof(USB_DEV_INTERFACE));
		if (__unlikely(!new_ifaces)) continue;
		cfg->ifaces = new_ifaces;
		iface = &new_ifaces[cfg->n_ifaces];
		iface->cfg = cfg;
		iface->desc = idesc;
		endpts = __sync_malloc(idesc->bNumEndpoints * sizeof(USB_DEV_ENDPOINT));
		if (__unlikely(!endpts)) continue;
		n_endpts = 0;
		for (i = 0; i < idesc->bNumEndpoints; i++) {
			USB_DESCRIPTOR_ENDPOINT *edesc;
			if (__unlikely(!(edesc = USB_FIND_DESCRIPTOR(cfg, &off, USB_DT_ENDPOINT, USB_DESCRIPTOR_ENDPOINT_SIZE, USB_DT_INTERFACE)))) {
				if (cfg->dev->usb->errorlevel >= 1)
					KERNEL$SYSLOG(__SYSLOG_NET_ERROR, cfg->dev->usb->dev_name, "DEVICE %s HAS NOT ENOUGH ENDPOINT DESCRIPTORS FOR %s", USB_DEV_ID(cfg->dev), USB_IF_ID(iface));
				break;
			}
			endpts[n_endpts].desc = edesc;
			endpts[n_endpts].used = 0;
			n_endpts++;
		}
		iface->endpts = endpts;
		iface->n_endpts = n_endpts;
		iface->attached = 0;
		iface->disabled = 0;
		iface->unplug_code = NULL;
		iface->hub = NULL;
		cfg->n_ifaces++;
		if (iface_is_new) max_ifaces++;
	}
	if (max_ifaces > cfg->dev->max_ifaces) cfg->dev->max_ifaces = max_ifaces;
}

/* Unlock address 0 after some command was successfully received with new
   address. So that even buggy device that successfully completes SET_ADDRESS
   but doesn't use the new address won't mess up the bus */

static void UNLOCK_A0(USB *usb, int *a0locked)
{
	if (*a0locked) {
		MTX_UNLOCK(&usb->addr0lock);
		*a0locked = 0;
	}
}

static int GET_CONFIGURATIONS(USB_DEV *dev)
{
	int r;
	unsigned i;
	USB *usb = dev->usb;
	if (!dev->dev_desc.bDescriptorType) {
		if (__unlikely((r = USB_GET_DEVICE_DESCRIPTOR(dev, USB_DT_DEVICE, 0, &dev->dev_desc, USB_DESCRIPTOR_DEVICE_SIZE, USB_DESCRIPTOR_DEVICE_SIZE)) < 0)) return r;
	}
	if (__unlikely(!dev->dev_desc.bNumConfigurations)) {
		if (usb->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, usb->dev_name, "DEVICE %s HAS NO CONFIGURATIONS", USB_DEV_ID(dev));
		return -EIO;
	}
	dev->n_confs = dev->dev_desc.bNumConfigurations;
	dev->confs = __sync_malloc(dev->n_confs * sizeof(USB_DEV_CFG));
	if (__unlikely(!dev->confs)) {
		dev->n_confs = 0;
		return -ENOMEM;
	}
	memset(dev->confs, 0, dev->dev_desc.bNumConfigurations * sizeof(USB_DEV_CFG));
	for (i = 0; i < dev->dev_desc.bNumConfigurations; i++) {
		unsigned j;
		unsigned l;
		int ll;
		USB_DEV_CFG *cfg = &dev->confs[i];
		cfg->dev = dev;
		if (__unlikely(USB_GET_DEVICE_DESCRIPTOR(dev, USB_DT_CONFIG, i, &cfg->cfg_desc, USB_DESCRIPTOR_CONFIGURATION_SIZE, USB_DESCRIPTOR_CONFIGURATION_SIZE) < 0)) continue;
		l = __16LE2CPU(cfg->cfg_desc.wTotalLength);
		if (__unlikely(l < USB_DESCRIPTOR_CONFIGURATION_SIZE)) {
			if (usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, usb->dev_name, "DEVICE %s HAS TOO SHORT DESCRIPTOR FOR CONFIGURATION %u: %u", USB_DEV_ID(dev), i, l);
			continue;
		}
		for (j = 0; j < i; j++) if (__likely(dev->confs[j].valid) && __unlikely(dev->confs[j].cfg_desc.bConfigurationValue == cfg->cfg_desc.bConfigurationValue)) {
			if (usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, usb->dev_name, "DEVICE %s HAS MULTIPLE ENTRIES FOR CONFIGURATION %u", USB_DEV_ID(dev), USB_CFG_NUM(&dev->confs[j]));
			continue;
		}
		cfg->descs = KERNEL$ALLOC_CONTIG_AREA(l, AREA_DATA | AREA_PHYSCONTIG);
		if (__unlikely(__IS_ERR(cfg->descs))) {
			cfg->descs = NULL;
			continue;
		}
		cfg->descs_allocated_size = l;
		cfg->valid = 1;
		cfg->disabled = 0;
		ll = USB_GET_DEVICE_DESCRIPTOR(dev, USB_DT_CONFIG, i, cfg->descs, l, USB_DESCRIPTOR_CONFIGURATION_SIZE);
		if (__unlikely(ll < 0)) {
			free_cfg:
			FREE_CONFIGURATION(cfg);
			continue;
		}
		cfg->descs_size = ll;
		cfg->ifaces = NULL;
		cfg->n_ifaces = 0;
		/* {
			int i;
			__debug_printf("descs:");
			for (i = 0; i < cfg->descs_size; i++)
				__debug_printf(" %02x", cfg->descs[i]);
			__debug_printf("\n");
		} */
		PARSE_CONFIGURATION(cfg);
		if (__unlikely(!cfg->n_ifaces)) {
			if (usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, usb->dev_name, "DEVICE %s HAS NO INTERFACES FOR CONFIGURATION %u", USB_DEV_ID(dev), i);
			goto free_cfg;
		}
	}
	for (i = 0; i < dev->dev_desc.bNumConfigurations; i++) {
		if (__likely(dev->confs[i].valid)) goto some_valid;
	}
	free(dev->confs);
	dev->confs = NULL;
	dev->n_confs = 0;
	dev->active_cfg = -1;
	return -ENODEV;
	some_valid:
	return 0;
}

int USB_FIND_ADDRESS(USB *usb)
{
	int i;
	for (i = 1; i < USB_MAX_ADDRESSES; i++) if (!usb->devices[i]) return i;
	return -EADDRNOTAVAIL;
}

#define VALID_CTRL_PKTSIZE(p)	(__likely((p) >= 8) && __likely((p) <= 64) && __likely(!((p) & ((p) - 1))))

static __finline__ USB_ENDPOINT *ZERO_ENDPOINT(USB_DEV *dev)
{
	return dev->speed == USB_LOW_SPEED ? dev->usb->zero_ctrl_lowspeed : dev->usb->zero_ctrl_fullspeed;
}

int USB_INIT_DEVICE(USB_DEV *dev, int *a0locked)
{
	int r;
	unsigned pktsize, new_pktsize;
	USB_ENDPOINT *zero_ep;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_USB)))
		KERNEL$SUICIDE("USB_DEVICE_INIT AT SPL %08X", KERNEL$SPL);
	pktsize = dev->speed == USB_LOW_SPEED ? 8 : 64;
	new_pktsize = dev->speed == USB_FULL_SPEED ? 0 : pktsize;
	zero_ep = ZERO_ENDPOINT(dev);
	/*if (__likely((r = USB_GET_DEVICE_DESCRIPTOR(dev, USB_DT_DEVICE, 0, desc, sizeof desc, USB_DESCRIPTOR_DEVICE_OFFSET_BMAXPKTSIZE0 + 1)) >= 0) && __likely(!new_pktsize)) */
	if (__likely(dev->addr != USB_ROOT_HUB_ADDR)) {
		USB_BUG_EARLY_GET_DEVICE_DESCRIPTOR(dev, zero_ep, &new_pktsize);
		if (__unlikely(!VALID_CTRL_PKTSIZE(new_pktsize))) new_pktsize = 0;
		/* maybe reset hub port here, as Linux does ? */
		r = USB$SYNC_CTRL(zero_ep, USB_CTRL_RQTYPE_DEVICE_OUT | USB_REQ_SET_ADDRESS, dev->addr, 0, 0, NULL, 0, USB_CTRL_TIMEOUT, USB_CTRL_RETRIES, NULL, NULL);
		if (__unlikely(r < 0)) return r;
		KERNEL$SLEEP(USB_SET_ADDRESS_WAIT);
		if (__likely(new_pktsize)) pktsize = new_pktsize;
	}

	/* Hmm, we should unlock after we made sure that the device accepted
	   new address. However we can't allocate dynamic memory inside the
	   lock, otherwise VM deadlock happens while resetting mass-storage
	   device. --- So we can't create the endpoint and check it :-/
	   
	   I'm not sure how to solve it now. But the only problem is that faulty
	   device can jam other devices in a short time window. */

	UNLOCK_A0(dev->usb, a0locked);

	dev->in_endpoints[0] = dev->out_endpoints[0] = dev->usb->op->alloc_endpoint(dev->usb, dev->speed, USB_EP_CTRL, dev->addr, 0, pktsize, 0, 1, USB_MAX_PIPE0_SIZE);
	if (__unlikely(__IS_ERR(dev->in_endpoints[0]))) {
		cant_create_pipe:
		r = __PTR_ERR(dev->in_endpoints[0]);
		dev->in_endpoints[0] = NULL;
		dev->out_endpoints[0] = NULL;
		return r;
	}
	if (__unlikely(!new_pktsize)) {
		__u8 desc[USB_DESCRIPTOR_DEVICE_OFFSET_BMAXPKTSIZE0 + 1];
		if (__unlikely((r = USB_GET_DEVICE_DESCRIPTOR(dev, USB_DT_DEVICE, 0, desc, USB_DESCRIPTOR_DEVICE_OFFSET_BMAXPKTSIZE0 + 1, USB_DESCRIPTOR_DEVICE_OFFSET_BMAXPKTSIZE0 + 1)) < 0)) {
			ret_fail:
			if (dev->addr > 0) dev->usb->devices[dev->addr] = NULL;
			dev->usb->op->free_endpoint(dev->in_endpoints[0]);
			dev->in_endpoints[0] = NULL;
			dev->out_endpoints[0] = NULL;
			return r;
		}
		new_pktsize = desc[USB_DESCRIPTOR_DEVICE_OFFSET_BMAXPKTSIZE0];
		if (__unlikely(!VALID_CTRL_PKTSIZE(new_pktsize))) {
			if (dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, dev->usb->dev_name, "DEVICE %s REPORTS BOGUS PACKET SIZE %u", USB_DEV_ID(dev), new_pktsize);
			r = -EPROTO;
			goto ret_fail;
		}
	}
	if (__unlikely(pktsize != new_pktsize)) {
		pktsize = new_pktsize;
		dev->usb->op->free_endpoint(dev->in_endpoints[0]);
		dev->in_endpoints[0] = dev->out_endpoints[0] = dev->usb->op->alloc_endpoint(dev->usb, dev->speed, USB_EP_CTRL, dev->addr, 0, pktsize, 0, 1, USB_MAX_PIPE0_SIZE);
		if (__unlikely(__IS_ERR(dev->in_endpoints[0]))) goto cant_create_pipe;
	}
	dev->pktsize = pktsize;
	if (__unlikely(r = GET_CONFIGURATIONS(dev))) goto ret_fail;
	dev->usb->op->free_endpoint(dev->in_endpoints[0]);
	dev->in_endpoints[0] = dev->out_endpoints[0] = dev->usb->op->alloc_endpoint(dev->usb, dev->speed, USB_EP_CTRL, dev->addr, 0, pktsize, 0, dev->max_ifaces * 2 + 1, USB_MAX_PIPE0_SIZE);
	if (__unlikely(__IS_ERR(dev->in_endpoints[0]))) goto cant_create_pipe;
	return 0;
}

int USB_REINIT_DEVICE(USB_DEV *dev, int *a0locked)
{
	USB_ENDPOINT *zero_ep = ZERO_ENDPOINT(dev);
	USB_DEV_CFG *cfg;
	USB_DEV_INTERFACE *iface, *iface2;
	unsigned i, j;
	int r;
	/* some devices retain the old address after reset, so this command could fail ... test the new address anyway */
	r = USB$SYNC_CTRL(zero_ep, USB_CTRL_RQTYPE_DEVICE_OUT | USB_REQ_SET_ADDRESS, dev->addr, 0, 0, NULL, 0, USB_CTRL_TIMEOUT, USB_CTRL_RETRIES, NULL, NULL);
	KERNEL$SLEEP(USB_SET_ADDRESS_WAIT);
	if (__unlikely(dev->active_cfg == -1))
		KERNEL$SUICIDE("USB_REINIT_DEVICE: HOST %s, DEVICE %s: NO ACTIVE CFG", dev->usb->dev_name, USB_DEV_ID(dev));
	cfg = &dev->confs[dev->active_cfg];
	r = USB$SYNC_CTRL(dev->in_endpoints[0], USB_CTRL_RQTYPE_DEVICE_OUT | USB_REQ_SET_CONFIGURATION, cfg->cfg_desc.bConfigurationValue, 0, 0, NULL, 0, USB_CTRL_TIMEOUT, USB_CTRL_RETRIES, NULL, NULL);
	if (__unlikely(r < 0)) return r;
	UNLOCK_A0(dev->usb, a0locked);
	for (i = 0; i < cfg->n_ifaces; i++) {
		iface = &cfg->ifaces[i];
		if (iface->attached) {
			for (j = 0; j < cfg->n_ifaces; j++) {
				iface2 = &cfg->ifaces[j];
				if (iface->desc->bInterfaceNumber == iface2->desc->bInterfaceNumber && iface->desc->bAlternateSetting != iface2->desc->bAlternateSetting) {
					r = USB$SYNC_CTRL(dev->in_endpoints[0], USB_CTRL_RQTYPE_INTERFACE_OUT | USB_REQ_SET_INTERFACE, iface->desc->bAlternateSetting, iface->desc->bInterfaceNumber, 0, NULL, 0, USB_CTRL_TIMEOUT, USB_CTRL_RETRIES, NULL, NULL);
					if (__unlikely(r < 0)) return r;
					break;
				}
			}
		}
	}
	for (i = 1; i < USB_MAX_ENDPOINTS; i++) {
		USB_ENDPOINT *ep;
		ep = dev->in_endpoints[i];
		if (ep) ep->clear_stall(ep);
		ep = dev->out_endpoints[i];
		if (ep) ep->clear_stall(ep);
	}
	return 0;
}

void USB_FREE_DEVICE(USB_DEV *dev)
{
	unsigned i;
	USBPLUG_UNPLUG(dev);
	if (__unlikely(dev->n_attached)) {
		/* must not free structure, only disallow use */
		for (i = 0; i < USB_MAX_ENDPOINTS; i++) {
			if (dev->in_endpoints[i]) dev->usb->op->disable_endpoint(dev->in_endpoints[i]);
			if (dev->out_endpoints[i]) dev->usb->op->disable_endpoint(dev->out_endpoints[i]);
		}
		dev->hub_port = -1;
		return;
	}
	for (i = 1; i < USB_MAX_ENDPOINTS; i++) {
		if (__unlikely(dev->in_endpoints[i] != NULL))
			KERNEL$SUICIDE("USB_FREE_DEVICE: INPUT ENDPOINT %u LEAKED, HOST %s, DEVICE %s", i, dev->usb->dev_name, USB_DEV_ID(dev));
		if (__unlikely(dev->out_endpoints[i] != NULL))
			KERNEL$SUICIDE("USB_FREE_DEVICE: OUTPUT ENDPOINT %u LEAKED, HOST %s, DEVICE %s", i, dev->usb->dev_name, USB_DEV_ID(dev));
	}
	if (__likely(dev->in_endpoints[0] != NULL)) dev->usb->op->free_endpoint(dev->in_endpoints[0]);
	for (i = 0; i < dev->n_confs; i++) {
		USB_DEV_CFG *cfg = &dev->confs[i];
		FREE_CONFIGURATION(cfg);
	}
	free(dev->confs);
	if (__likely(dev->addr > 0)) dev->usb->devices[dev->addr] = NULL;
	if (__likely(dev != &dev->usb->root_hub)) free(dev);
}

static const USB_ARGS null_args = { NULL, 0, -1, -1, -1 };

int USB$PARSE_PARAMS(const char *opt, const char *optend, const char *str, USB_ARGS **args)
{
	int state = 0;
	static const struct __param_table params[6] = {
		"CTRL", __PARAM_NEWSTRING, 1, __MAX_STR_LEN,
		"ADDRESS", __PARAM_INT, 1, USB_MAX_ADDRESSES,
		"CONFIGURATION", __PARAM_INT, USB_MIN_CONFIGURATION, USB_MAX_CONFIGURATION,
		"INTERFACE", __PARAM_INT, USB_MIN_INTERFACE, USB_MAX_INTERFACE,
		"ALTSETTING", __PARAM_INT, USB_MIN_ALTSETTING, USB_MAX_ALTSETTING,
		NULL, 0, 0, 0,
	};
	void *vars[6];
	if (__unlikely(__strcasexcmp("USB", opt, optend))) return 1;
	if (__unlikely(__likely(!*args))) {
		*args = __sync_malloc(sizeof(USB_ARGS));
		if (__unlikely(!*args)) return errno != EINTR;
		memcpy(*args, &null_args, sizeof(USB_ARGS));
	}
	vars[0] = &(*args)->controller;
	vars[1] = &(*args)->address;
	vars[2] = &(*args)->configuration;
	vars[3] = &(*args)->interface;
	vars[4] = &(*args)->altsetting;
	vars[5] = NULL;
	return __parse_extd_param(&str, &state, params, vars, NULL, NULL, NULL, NULL);
}

void USB$FREE_PARAMS(USB_ARGS *args)
{
	if (__likely(args != NULL)) {
		free(args->controller);
		free(args);
	}
}

static USB_DEV_INTERFACE *ATTACH_DRIVER(char *controller, USB_ARGS *args, int (*test)(USB_DEV_INTERFACE *, long), long data);

USB_DEV_INTERFACE *USB$ATTACH_DRIVER(USB_ARGS *args, int (*test)(USB_DEV_INTERFACE *, long), long data)
{
	int i;
	LOGICAL_LIST_REQUEST lrq;
	if (__likely(args != NULL) && __likely(args->controller != NULL)) return ATTACH_DRIVER(args->controller, args, test, data);
	lrq.prefix = "USB$";
	SYNC_IO(&lrq, KERNEL$LIST_LOGICALS);
	if (__unlikely(lrq.status < 0)) return __ERR_PTR(lrq.status);
	for (i = 0; i < lrq.n_entries; i++) {
		LOGICAL_LIST_ENTRY *le = lrq.entries[i];
		USB_DEV_INTERFACE *iface = ATTACH_DRIVER(le->name, args, test, data);
		if (!__IS_ERR(iface)) {
			KERNEL$FREE_LOGICAL_LIST(&lrq);
			return iface;
		}
	}
	KERNEL$FREE_LOGICAL_LIST(&lrq);
	return __ERR_PTR(-ENODEV);
}

static USB_DEV_INTERFACE *ATTACH_DRIVER(char *controller, USB_ARGS *args, int (*test)(USB_DEV_INTERFACE *, long), long data)
{
	int r;
	USB_ATTACH_CMD attch;
	attch.args = args ? args : &null_args;
	attch.test = test;
	attch.data = data;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), KERNEL$SPL)))
		KERNEL$SUICIDE("USB$ATTACH_DRIVER AT SPL %08X", KERNEL$SPL);
	if (__unlikely((r = KERNEL$DCALL(controller, USB$HC_DCALL_TYPE, USB_CMD_ATTACH, &attch)))) return __ERR_PTR(r);
	return attch.result;
}

const char USB$HC_DCALL_TYPE[] = "USB";

static int ATTACH_DRIVER_TO_USB(USB *volatile usb, USB_ATTACH_CMD *volatile attch);
static int ATTACH_DRIVER_TO_DEVICE(USB_DEV *dev, USB_ATTACH_CMD *attch, jmp_buf jmp);
static int ATTACH_DRIVER_TO_CFG(USB_DEV_CFG *cfg, USB_ATTACH_CMD *attch, jmp_buf jmp);
static int ATTACH_DRIVER_TO_IFACE(USB_DEV_INTERFACE *iface, USB_ATTACH_CMD *attch, int atype, jmp_buf jmp);

int USB$HC_DCALL(void *ptr, const char *dcall_type, int cmd, va_list list)
{
	USB *usb = ptr;
	int r;
	USB_ATTACH_CMD *attch = va_arg(list, USB_ATTACH_CMD *);
	RAISE_SPL(SPL_USB);
	switch (cmd) {
		case USB_CMD_ATTACH:
			r = ATTACH_DRIVER_TO_USB(usb, attch);
			break;
		default:
			r = -EINVAL;
			break;
	}
	LOWER_SPL(SPL_DEV);
	return r;
}

static int ATTACH_DRIVER_TO_USB(USB *volatile usb, USB_ATTACH_CMD *volatile attch)
{
	int r;
	jmp_buf jmp;
	if (__unlikely(usb->shutting_down)) return -ENODEV;
	if (__likely(!(r = setjmp(jmp)))) r = -ENODEV;
	if (__unlikely(attch->args->address == -1)) return ATTACH_DRIVER_TO_DEVICE(&usb->root_hub, attch, jmp);
	if (__unlikely(!attch->args->address)) {
		int i;
		for (i = 1; i < USB_MAX_ADDRESSES; i++) {
			if (!ATTACH_DRIVER_TO_DEVICE(usb->devices[i], attch, jmp)) return 0;
		}
		return r;
	}
	if (__likely(attch->args->address >= 1) && __likely(attch->args->address < USB_MAX_ADDRESSES)) {
		if (!ATTACH_DRIVER_TO_DEVICE(usb->devices[attch->args->address], attch, jmp)) return 0;
		return r;
	}
	return -EINVAL;
}

static int ATTACH_DRIVER_TO_DEVICE(USB_DEV *dev, USB_ATTACH_CMD *attch, jmp_buf jmp)
{
	int i;
	if (__unlikely(!dev) || __unlikely(dev->hub_port == -1) || __unlikely(dev->n_attached == -1)) return -ENODEV;
	for (i = 0; i < dev->n_confs; i++)
		if (__likely(dev->confs[i].valid) && __likely(!dev->confs[i].disabled))
			if (__likely(attch->args->configuration == -1) || dev->confs[i].cfg_desc.bConfigurationValue == attch->args->configuration)
				if (i == dev->active_cfg || __likely(!dev->n_attached))
					if (__likely(!ATTACH_DRIVER_TO_CFG(&dev->confs[i], attch, jmp))) return 0;
	return -ENODEV;
}

static int ATTACH_DRIVER_TO_CFG(USB_DEV_CFG *cfg, USB_ATTACH_CMD *attch, jmp_buf jmp)
{
	unsigned i, j;
	for (i = 0; i < cfg->n_ifaces; i++) {
		if (!cfg->ifaces[i].attached && __likely(!cfg->ifaces[i].disabled) && (attch->args->interface == -1 || attch->args->interface == cfg->ifaces[i].desc->bInterfaceNumber) && (attch->args->altsetting == -1 || attch->args->altsetting == cfg->ifaces[i].desc->bAlternateSetting)) {
			int atype;
			for (j = 0; j < cfg->n_ifaces; j++) if ((atype = cfg->ifaces[j].attached)) {
				if (__unlikely(atype == USB_ATTACH_DEVICE)) return -ENODEV;
				if (__unlikely(atype == USB_ATTACH_CLASS) && cfg->ifaces[i].desc->bInterfaceClass == cfg->ifaces[j].desc->bInterfaceClass) goto skip_this;
				if (__likely(cfg->ifaces[i].desc->bInterfaceNumber == cfg->ifaces[j].desc->bInterfaceNumber)) goto skip_this;
			}
			if ((atype = attch->test(&cfg->ifaces[i], attch->data))) {
				
				if (__unlikely(atype == USB_ATTACH_DEVICE) && cfg->dev->n_attached) goto skip_this;
				if (__unlikely(atype == USB_ATTACH_CLASS)) {
					for (j = 0; j < cfg->n_ifaces; j++) if (cfg->ifaces[j].attached && cfg->ifaces[i].desc->bInterfaceClass == cfg->ifaces[j].desc->bInterfaceClass) goto skip_this;
				}
				if (__likely(!ATTACH_DRIVER_TO_IFACE(&cfg->ifaces[i], attch, atype, jmp))) return 0;
			}
			skip_this:;
		}
	}
	return -ENODEV;
}

static int ATTACH_DRIVER_TO_IFACE(USB_DEV_INTERFACE *iface, USB_ATTACH_CMD *attch, int atype, jmp_buf jmp)
{
	unsigned i;
	int r;
/* need to attach before issuing commands so that device structure remains valid
   in case of unplug */
	iface->cfg->dev->n_attached++;
	iface->attached = atype;
	attch->result = iface;
	if (__unlikely(r = USB_HUB_WAIT_FOR_RESET(iface->cfg->dev))) {
		iface->disabled = 1;
		goto dtch;
	}
	if (USB_CFG_ORDER(iface->cfg) != iface->cfg->dev->active_cfg) {
		iface->cfg->dev->active_cfg = USB_CFG_ORDER(iface->cfg);
		if (__unlikely((r = USB$SYNC_CTRL(iface->cfg->dev->in_endpoints[0], USB_CTRL_RQTYPE_DEVICE_OUT | USB_REQ_SET_CONFIGURATION, iface->cfg->cfg_desc.bConfigurationValue, 0, 0, NULL, 0, USB_CTRL_TIMEOUT, USB_CTRL_RETRIES, iface, NULL)) < 0)) {
			iface->cfg->dev->active_cfg = -1;
			iface->cfg->disabled = 1;
/* while issuing the command, the device might have been unplugged --- if this
   happened, USB$DETACH_DRIVER will free the device */
			dtch:
			USB$DETACH_DRIVER(iface);
			longjmp(jmp, r);
		}
	}
	for (i = 0; i < iface->cfg->n_ifaces; i++) if (__likely(iface->cfg->ifaces[i].desc->bInterfaceNumber == iface->desc->bInterfaceNumber) && __unlikely(iface->cfg->ifaces[i].desc->bAlternateSetting != iface->desc->bAlternateSetting)) {
		if (__unlikely((r = USB$SYNC_CTRL(iface->cfg->dev->in_endpoints[0], USB_CTRL_RQTYPE_INTERFACE_OUT | USB_REQ_SET_INTERFACE, iface->desc->bAlternateSetting, iface->desc->bInterfaceNumber, 0, NULL, 0, USB_CTRL_TIMEOUT, USB_CTRL_RETRIES, iface, NULL)) < 0)) {
			iface->disabled = 1;
			goto dtch;
		}
		break;
	}
	return 0;
}

void USB$DETACH_DRIVER(USB_DEV_INTERFACE *iface)
{
	unsigned i;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_USB), spl)))
		KERNEL$SUICIDE("USB$DETACH_DRIVER AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_USB);
	if (__unlikely(!iface->attached))
		KERNEL$SUICIDE("USB$DETACH_DRIVER: DETACHING NON-ATTACHED INTERFACE, HOST %s, DEVICE %s, %s", iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface));
	for (i = 0; i < iface->n_endpts; i++) if (__likely(iface->endpts[i].used))
		USB$FREE_ENDPOINT(iface, (iface->endpts[i].desc->bEndpointAddress & USB_ENDPOINT_ADDRESS_IN ? iface->cfg->dev->in_endpoints : iface->cfg->dev->out_endpoints)[iface->endpts[i].desc->bEndpointAddress & USB_ENDPOINT_ADDRESS_MASK]);
	iface->attached = 0;
	USBPLUG_CLEAR_IFACE_UNPLUG(iface);
	iface->hub = NULL;
	iface->cfg->dev->n_attached--;
	if (__unlikely(iface->cfg->dev->n_attached < 0)) {
		KERNEL$SUICIDE("USB$DETACH_DRIVER: DEVICE ATTACH COUNT UNDERFLOW: %d", iface->cfg->dev->n_attached);
	}
	if (__likely(!iface->cfg->dev->n_attached)) {
		for (i = 1; i < USB_MAX_ENDPOINTS; i++) {
			if (__unlikely(iface->cfg->dev->in_endpoints[i] != NULL))
				KERNEL$SUICIDE("USB$DETACH_DRIVER: INPUT ENDPOINT %u LEAKED, HOST %s, DEVICE %s, %s", i, iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface));
			if (__unlikely(iface->cfg->dev->out_endpoints[i] != NULL))
				KERNEL$SUICIDE("USB$DETACH_DRIVER: OUTPUT ENDPOINT %u LEAKED, HOST %s, DEVICE %s, %s", i, iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface));
		}
		if (iface->cfg->dev->hub_port == -1) {
			USB_FREE_DEVICE(iface->cfg->dev);
		}
	}
	LOWER_SPLX(spl);
}

USB_ENDPOINT *USB$GET_DEFAULT_ENDPOINT(USB_DEV_INTERFACE *iface)
{
	return iface->cfg->dev->in_endpoints[0];
}

static int VALID_PKTSIZE(int typecode, int speed, unsigned pktsize)
{
	if (USB_IS_CTRL(typecode)) {
		if (speed == USB_LOW_SPEED) return __likely(pktsize == 8);
		if (speed == USB_FULL_SPEED) p8to64: return __likely(pktsize >= 8) && __likely(pktsize <= 64) && __likely(!(pktsize & (pktsize - 1)));
		return __likely(pktsize == 64);
	}
	if (USB_IS_ISO(typecode)) {
		if (speed == USB_FULL_SPEED) return __likely(pktsize <= 1023);
		return __likely(pktsize <= 1024);
	}
	if (USB_IS_INTR(typecode)) {
		if (speed == USB_LOW_SPEED) return __likely(pktsize <= 8);
		if (speed == USB_FULL_SPEED) return __likely(pktsize <= 64);
		return __likely(pktsize == 1024);
	}
	if (speed == USB_FULL_SPEED) goto p8to64;
	return __likely(pktsize == 512);
}

int USB_ENDPOINT_TYPECODE(USB_DESCRIPTOR_ENDPOINT *desc)
{
	unsigned type, in;
	int typecode;
	type = desc->bmAttributes & USB_ENDPOINT_ATTRIBUTE_TRANSFER_MASK;
	in = desc->bEndpointAddress & USB_ENDPOINT_ADDRESS_IN;
	typecode = 
type == USB_ENDPOINT_ATTRIBUTE_TRANSFER_CONTROL ? USB_EP_CTRL :
type == USB_ENDPOINT_ATTRIBUTE_TRANSFER_BULK && in ? USB_EP_BULK_IN :
type == USB_ENDPOINT_ATTRIBUTE_TRANSFER_BULK && !in ? USB_EP_BULK_OUT :
type == USB_ENDPOINT_ATTRIBUTE_TRANSFER_INTERRUPT && in ? USB_EP_INTR_IN :
type == USB_ENDPOINT_ATTRIBUTE_TRANSFER_INTERRUPT && !in ? USB_EP_INTR_OUT :
type == USB_ENDPOINT_ATTRIBUTE_TRANSFER_ISOCHRONOUS && in ? USB_EP_ISO_IN :
type == USB_ENDPOINT_ATTRIBUTE_TRANSFER_ISOCHRONOUS && !in ? USB_EP_ISO_OUT :
-1;
	if (__unlikely(typecode == -1))
		KERNEL$SUICIDE("USB_ENDPOINT_TYPECODE: ERROR IN TABLE, PIPE %02X, ATTR %02X", desc->bEndpointAddress, desc->bmAttributes);
	return typecode;
}

USB_ENDPOINT *USB$GET_ENDPOINT(USB_DEV_INTERFACE *iface, unsigned n, int wanted_type, unsigned max_requests, unsigned max_request_size)
{
	unsigned i_pipe, in;
	int typecode;
	int interval;
	unsigned pktsize;
	USB_DEV_ENDPOINT *endpt;
	USB_ENDPOINT *hc;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_USB), spl)))
		KERNEL$SUICIDE("USB$GET_ENDPOINT AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_USB);
	if (__unlikely(n >= iface->n_endpts)) {
		LOWER_SPLX(spl);
		return __ERR_PTR(-ENOENT);
	}
	endpt = &iface->endpts[n];
	if (__unlikely(endpt->used))
		KERNEL$SUICIDE("USB$GET_ENDPOINT: HOST %s, DEVICE %s, %s: GETTING ENDPOINT %u TWICE", iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), n);
	
	in = endpt->desc->bEndpointAddress & USB_ENDPOINT_ADDRESS_IN;
	i_pipe = endpt->desc->bEndpointAddress & USB_ENDPOINT_ADDRESS_MASK;
	typecode = USB_ENDPOINT_TYPECODE(endpt->desc);

	if (__unlikely(typecode != wanted_type)) {
		LOWER_SPLX(spl);
		return __ERR_PTR(-EPROTOTYPE);
	}

	if (__unlikely(iface->cfg->dev->speed == USB_LOW_SPEED) && (__unlikely(USB_IS_BULK(typecode)) || __unlikely(USB_IS_ISO(typecode)))) {
		if (iface->cfg->dev->usb->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, iface->cfg->dev->usb->dev_name, "LOW-SPEED DEVICE %s, %s, HAS %s ENDPOINT %u", USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), USB$ENDPOINT_TYPE_STRING(typecode), i_pipe);
		LOWER_SPLX(spl);
		return __ERR_PTR(-EPROTO);
	}

	pktsize = endpt->desc->wMaxPacketSize_lo + (endpt->desc->wMaxPacketSize_hi << 8);
	if (__unlikely(!VALID_PKTSIZE(typecode, iface->cfg->dev->speed, pktsize))) {
		if (iface->cfg->dev->usb->errorlevel >= 1)
			KERNEL$SYSLOG(__SYSLOG_NET_ERROR, iface->cfg->dev->usb->dev_name, "%s DEVICE %s, %s, %s ENDPOINT %u HAS INVALID PACKET SIZE %u", USB$SPEED_STRING(iface->cfg->dev->speed), USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), USB$ENDPOINT_TYPE_STRING(typecode), i_pipe, pktsize);
		LOWER_SPLX(spl);
		return __ERR_PTR(-EPROTO);
	}

	interval = endpt->desc->bInterval;
	if (__unlikely(USB_IS_ISO(typecode)) || (USB_IS_INTR(typecode) && __unlikely(iface->cfg->dev->speed == USB_HIGH_SPEED))) {
		interval -= 1;
		if (__likely((unsigned)interval <= 15)) interval = 1 << interval;
		else {
			if (iface->cfg->dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, iface->cfg->dev->usb->dev_name, "DEVICE %s, %s, ENDPOINT %u HAS INVALID INTERVAL %u", USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), i_pipe, endpt->desc->bInterval);
			interval = 0;
		}
	}

	endpt->used = 1;
	hc = iface->cfg->dev->usb->op->alloc_endpoint(iface->cfg->dev->usb, iface->cfg->dev->speed, typecode, iface->cfg->dev->addr, i_pipe, pktsize, interval, max_requests, max_request_size);
	if (__unlikely(__IS_ERR(hc))) {
		endpt->used = 0;
		LOWER_SPLX(spl);
		return hc;
	}

	if (__unlikely(USB_IS_CTRL(typecode))) {
		if (__unlikely(iface->cfg->dev->in_endpoints[i_pipe] != NULL) || __unlikely(iface->cfg->dev->out_endpoints[i_pipe] != NULL)) {
			used:
			if (iface->cfg->dev->usb->errorlevel >= 1)
				KERNEL$SYSLOG(__SYSLOG_NET_ERROR, iface->cfg->dev->usb->dev_name, "DEVICE %s, %s HAS ENDPOINT %u ALREADY USED", USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), i_pipe);
			iface->cfg->dev->usb->op->free_endpoint(hc);
			endpt->used = 0;
			LOWER_SPLX(spl);
			return __ERR_PTR(-EPROTO);
		}
		iface->cfg->dev->in_endpoints[i_pipe] = hc;
		iface->cfg->dev->out_endpoints[i_pipe] = hc;
	} else if (in) {
		if (__unlikely(iface->cfg->dev->in_endpoints[i_pipe] != NULL)) goto used;
		iface->cfg->dev->in_endpoints[i_pipe] = hc;
	} else {
		if (__unlikely(iface->cfg->dev->out_endpoints[i_pipe] != NULL)) goto used;
		iface->cfg->dev->out_endpoints[i_pipe] = hc;
	}
	if (__unlikely(iface->cfg->dev->hub_port == -1)) {
		iface->cfg->dev->usb->op->disable_endpoint(hc);
	}
	if (__likely(!USB_IS_ISO(typecode)))
		USB$SYNC_CTRL(iface->cfg->dev->in_endpoints[0], USB_CTRL_RQTYPE_ENDPOINT_OUT | USB_REQ_CLEAR_FEATURE, USB_ENDPOINT_FEAT_HALT, i_pipe + (USB_IS_IN(typecode) ? USB_ENDPOINT_ADDRESS_IN : 0), 0, NULL, 0, USB_CTRL_TIMEOUT, 1, NULL, NULL);
	LOWER_SPLX(spl);
	return hc;
}

void USB$FREE_ENDPOINT(USB_DEV_INTERFACE *iface, USB_ENDPOINT *endpt)
{
	unsigned i, j;
	int in, out;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_USB), spl)))
		KERNEL$SUICIDE("USB$FREE_ENDPOINT AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_USB);
	if (__unlikely(!endpt))
		KERNEL$SUICIDE("USB$FREE_ENDPOINT: HOST %s, DEVICE %s, %s: NULL POINTER", iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface));
	if (__unlikely(endpt == iface->cfg->dev->in_endpoints[0]))
		KERNEL$SUICIDE("USB$FREE_ENDPOINT: HOST %s, DEVICE %s, %s: TRYING TO FREE DEFAULT CONTROL PIPE", iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface));
	in = out = 0;
	for (i = 1; i < USB_MAX_ENDPOINTS; i++) {
		if (iface->cfg->dev->in_endpoints[i] == endpt) in = 1;
		if (iface->cfg->dev->out_endpoints[i] == endpt) out = 1;
		if (in || out) goto found_i;
	}
	KERNEL$SUICIDE("USB$FREE_ENDPOINT: HOST %s, DEVICE %s, %s: INVALID POINTER TO ENDPOINT", iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface));
	found_i:
	for (j = 0; j < iface->n_endpts; j++) if ((iface->endpts[j].desc->bEndpointAddress & USB_ENDPOINT_ADDRESS_MASK) == i) goto found_j;
	KERNEL$SUICIDE("USB$FREE_ENDPOINT: HOST %s, DEVICE %s, %s: DESCRIPTOR FOR ENDPOINT %u NOT FOUND", iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), i);
	found_j:
	iface->cfg->dev->usb->op->free_endpoint(endpt);
	if (in) iface->cfg->dev->in_endpoints[i] = NULL;
	if (out) iface->cfg->dev->out_endpoints[i] = NULL;
	if (__unlikely(!iface->endpts[j].used))
		KERNEL$SUICIDE("USB$FREE_ENDPOINT: HOST %s, DEVICE %s, %s: ENDPOINT %u NOT MARKED USED", iface->cfg->dev->usb->dev_name, USB_DEV_ID(iface->cfg->dev), USB_IF_ID(iface), j);
	iface->endpts[j].used = 0;
	LOWER_SPLX(spl);
}

int USB_INIT_ROOT_HUB(USB *usb)
{
	int zero = 0;
	USB_ARGS args;
	USB_ATTACH_CMD attch;
	int r;
	usb->root_hub.usb = usb;
	usb->root_hub.active_cfg = -1;
	usb->root_hub.addr = -1;
	usb->root_hub.speed = usb->op->flags & USB_HC_20 ? USB_HIGH_SPEED : USB_FULL_SPEED;
	r = USB_INIT_DEVICE(&usb->root_hub, &zero);
	if (__unlikely(r)) return r;
	args.controller = NULL;
	args.address = -1;
	args.configuration = -1;
	args.interface = -1;
	args.altsetting = -1;
	attch.args = &args;
	attch.test = USB$HUB_TEST;
	attch.data = USB_HUB_PROTOCOL_SINGLE_TT;
	r = ATTACH_DRIVER_TO_USB(usb, &attch);
	if (__unlikely(r)) {
		USB_FREE_DEVICE(&usb->root_hub);
		return r;
	}
	r = USB$HUB_ATTACH(attch.result);
	if (__unlikely(r)) {
		USB_FREE_DEVICE(&usb->root_hub);
		return r;
	}
	return 0;
}

