#ifndef __SIGNAL_H
#define __SIGNAL_H

#include <SYS/TYPES.H>
#include <ARCH/SIGNAL.H>

#define NSIG		32

#define SIGHUP		1
#define SIGINT		2
#define SIGQUIT		3
#define SIGILL		4
#define SIGTRAP		5
#define SIGABRT		6
#define SIGIOT		6
#define SIGBUS		7
#define SIGFPE		8
#define SIGKILL		9
#define SIGUSR1		10
#define SIGSEGV		11
#define SIGUSR2		12
#define SIGPIPE		13
#define SIGALRM		14
#define SIGTERM		15
#define SIGCHLD		17
#define SIGCONT		18
#define SIGSTOP		19
#define SIGTSTP		20
#define SIGTTIN		21
#define SIGTTOU		22
#define SIGURG		23
#define SIGVTALRM	26
#define SIGPROF		27
#define SIGWINCH	28
#define SIGIO		29
#define SIGPOLL		SIGIO
#define SIGMCE		31

extern __const__ char *__const__ sys_siglist[];

#ifndef __SIGNAL_H_TYPEDEFS
#define __SIGNAL_H_TYPEDEFS

typedef unsigned long sigset_t;

typedef long sig_atomic_t;

typedef void (*__sighandler_t)(int);

#define SIG_ERR	((__sighandler_t) -1)
#define SIG_DFL	((__sighandler_t) 0)
#define SIG_IGN	((__sighandler_t) 1)

struct sigaction {
	__sighandler_t sa_handler;
	sigset_t sa_mask;
	int sa_flags;
	void *__pad;
};

#define SA_NOCLDSTOP	0x00000001
#define SA_ONSTACK	0x08000000
#define SA_RESTART	0x10000000
#define SA_INTERRUPT	0x20000000	/* unused */
#define SA_NODEFER	0x40000000
#define SA_RESETHAND	0x80000000

#define SA_NOMASK	SA_NODEFER
#define SA_ONESHOT	SA_RESETHAND

typedef struct sigaltstack {
	void *ss_sp;
	int ss_flags;
	size_t ss_size;
} stack_t;

#define SS_ONSTACK	1
#define SS_DISABLE	2

#endif

#define SIG_BLOCK	0	/* must be 0 --- SETJMP.S */
#define SIG_UNBLOCK	1
#define SIG_SETMASK	2

__BEGIN_DECLS

int sigaction(int __signum, __const__ struct sigaction *__act, struct sigaction *__oldact);
int sigprocmask(int __how, __const__ sigset_t *__set, sigset_t *__oldset);
int sigpending(sigset_t *__set);
int sigsuspend(const sigset_t *__mask);
int sigwait(__const__ sigset_t *__set, int *__sig);
void _thread_sigwait(__const__ sigset_t *__set);
int sigaltstack(__const__ stack_t *ss, stack_t *oss);
__sighandler_t signal(int __signum, __sighandler_t __handler);
int kill(pid_t __pid, int __sig);
int killpg(pid_t __pgrp, int __sig);
int raise(int __sig);
int _thread_raise(int __sig);
int sigblock(int __mask);
int siggetmask(void);
int sigsetmask(int __mask);
int sigpause(int __mask);
int siginterrupt(int __signum, int __flag);

int pthread_kill(pthread_t __thid, int __sig);
int pthread_sigmask(int __how, __const__ sigset_t *__set, sigset_t *__oldset);

#ifdef __NO_INLINES
int sigmask(int __signum);
#else
__static__ __finline__ int sigmask(int __signum)
{
	return (1 << __signum);
}
#endif

#ifdef __NO_INLINES
int sigemptyset(sigset_t *__set);
#else
__static__ __finline__ int sigemptyset(sigset_t *__set)
{
	*__set = 0;
	return 0;
}
#endif

#ifdef __NO_INLINES
int sigfillset(sigset_t *__set);
#else
__static__ __finline__ int sigfillset(sigset_t *__set)
{
	*__set = (sigset_t )-1;
	return 0;
}
#endif

#ifdef __NO_INLINES
int sigaddset(sigset_t *__set, int __signum);
#else
__static__ __finline__ int sigaddset(sigset_t *__set, int __signum)
{
	*__set |= (1 << __signum);
	return 0;
}
#endif

#ifdef __NO_INLINES
int sigdelset(sigset_t *__set, int __signum);
#else
__static__ __finline__ int sigdelset(sigset_t *__set, int __signum)
{
	*__set &= ~(1 << __signum);
	return 0;
}
#endif

#ifdef __NO_INLINES
int sigismember(__const__ sigset_t *__set, int __signum);
#else
__static__ __finline__ int sigismember(__const__ sigset_t *__set, int __signum)
{
	return (*__set >> __signum) & 1;
}
#endif

void psignal(int, const char *);

#define __EXIT_SIGNAL_VAL	(-0x10000)
#define __EXIT_SIGNAL(sig)	(__EXIT_SIGNAL_VAL + (sig))
#define __IS_EXIT_SIGNAL(val)	((val) >= __EXIT_SIGNAL_VAL && (val) < __EXIT_SIGNAL(NSIG))

void KERNEL$INTR_SYSCALL(void);

__END_DECLS

#endif
