#ifndef __SPAD_VM_H
#define __SPAD_VM_H

#include <SYS/TYPES.H>
#include <SPAD/LIST.H>
#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <SPAD/VMDEF.H>
#include <ARCH/VM.H>
#include <ARCH/TIME.H>
#include <SPAD/PROC.H>

__BEGIN_DECLS

typedef struct {
	PROC *proc;
} PROC_TAG;

typedef struct {
	IORQ_HEAD;
	LIST_ENTRY list;
} PREFETCH_TAG;

WQ *KERNEL$ACQUIRE_IO_TAG(PROC_TAG *tag, PROC *proc);
void KERNEL$RELEASE_IO_TAG(PROC_TAG *tag);
void KERNEL$ACQUIRE_PREFETCH_TAG(PREFETCH_TAG *tag, PROC *proc);
static __finline__ void RELEASE_PREFETCH_TAG(PREFETCH_TAG *tag)
{
	DEL_FROM_LIST(&tag->list);
}

void KERNEL$ACQUIRE_WRITEBACK_TAG(PROC *proc);
void KERNEL$RELEASE_WRITEBACK_TAG(PROC *proc);

#define VME_Q_FREEING			255

typedef struct {
	LIST_ENTRY vm_queue;
	LIST_ENTRY wire_list;
	PROC *wired;
	unsigned char queue;
	unsigned char type;
	unsigned short wired_clock;
	u_jiffies_lo_t touchtime;
} VMENTITY;

/* PAGE->flags */
#define PAGE_BUSY		0x00000003
#define PAGE_BUSY_1		0x00000001
#define PAGE_WANTFREE		0x00000004
#define PAGE_WRITEABLE		0x00000008
#define PAGE_ACCOUNTED		0x00000010
#define PAGE_WRITECOUNT		0x00001fe0
#define PAGE_WRITECOUNT_1	0x00000020
#define PAGE_DMALOCKCOUNT	0xffffe000
#define PAGE_DMALOCKCOUNT_1	0x00002000

struct __page {					/* size in ARCH/VMDEF.H */
	LIST_ENTRY hash_entry;
	LIST_ENTRY node_entry;	/* list of pages on fnode */
		/* for pages on freelist, this is lru that points to PAGE, not LIST_HEAD */

	_u_off_t id;		/* positions of these are defined in ARCH/VMDEF.H */
	IO_STUB *release;

	unsigned flags;		/* PAGE_xxx flags */
	VMENTITY e;
	void *fnode;
/* page offsets; must lay on fs block boundary. valid_to may be pointing
   (unaligned) to the end of file on the last page (in this case, remaining
   bytes aren't zeroed).
   The range described by these two entries must be non-empty or (valid_from
   == __PAGE_SIZE && valid_to == 0) for empty range */
	unsigned valid_from;
	unsigned valid_to;
/* don't have to lay on fs block boundary (will be aligned when writing page).
   Must lay within valid_from, valid_to */
	unsigned dirty_from;
	unsigned dirty_to;
	unsigned prefetch_hint;
	XLIST_HEAD mapping;
	WQ wait;
	WQ *(*lockdown)(PAGE *p, int lock);	/* 0 lock, 1 unlock */
	LIST_ENTRY free_entry;
	unsigned io_to;
	__node_id_t node;
#if ___PG_STRUCT_PAGE_PAD > 0
	char pad[___PG_STRUCT_PAGE_PAD];
#endif
};

struct __spage_entry {
	LIST_ENTRY freelist;
	SPAGES *spages;
	unsigned size;	/* 0 - allocated, n - block size. */
		/* Must be set for first and last spage entry in block */
};

struct __spages {
	PAGE *page;
	SPAGE_ENTRY s[2];	/* need to be extended by amount of pages
			in cluster.
			first and last are unused --- they
			just contain size == 0 as a boundary */
};

struct __spage {
	XLIST_HEAD mapping;
	unsigned flags;
	PAGE *page;
	unsigned offset;
	WQ wait;
	unsigned n_pages;
	WQ *(*lockdown)(SPAGE *p, int lock);	/* 0 lock, 1 unlock */
	SPAGE_ENTRY *spage_entry;
};

extern unsigned long KERNEL$STACKPAGE;

PAGE *KERNEL$ALLOC_USER_PAGE(int type);
PAGE *KERNEL$ALLOC_IO_PAGE(int type);
void *KERNEL$ALLOC_KERNEL_PAGE(int type);

void KERNEL$FREE_USER_PAGE(PAGE *pg, int type);
void KERNEL$FREE_KERNEL_PAGE(void *ptr, int type);

typedef struct {
	IORQ_HEAD;
	PAGE *pg;
} PAGE_RELEASE_REQUEST;

extern IO_STUB KERNEL$WAKE_PAGE_RELEASE;

/*
 ->status == -EEXIST -- kernel page
 	     -ENOENT -- page is not in memory
	     -EIO -- caller did not provide release function
	     -ENOMEM -- out of memory
*/

WQ *KERNEL$VM_UNMAP_PAGE(PAGE *p);
WQ *KERNEL$VM_UNSET_WRITEABLE(PAGE *p);
int KERNEL$VM_SCAN_PAGE(PAGE *p);
void KERNEL$VM_PREPARE_PAGE_FOR_MMAP(PAGE *p);

WQ *KERNEL$VM_UNMAP_SPAGE(SPAGE *sp);
int KERNEL$VM_SCAN_SPAGE(SPAGE *sp);

int KERNEL$SET_MEMORY_LIMIT(__u64 memory, __u64 *holes, int n_holes);
__u64 KERNEL$GET_MEMORY_SIZE(int type);
int KERNEL$SET_MAPPED_MEMORY(__p_addr addr, __p_addr len, int pat);
int KERNEL$SET_MEMORY_PAT(__p_addr addr, __p_addr len, int pat);

typedef struct {
/* functions are called at SPL_CACHE and may lower SPL down to SPL_DEV */
	int (*checkmap)(VMENTITY *e);
	void (*write)(VMENTITY *e, PROC *p, int trashing);
	int (*pageout)(VMENTITY *e);
	const char *name;
} VMENTITY_T;

/* pageout returns:
	0 --- entity was freed (with KERNEL$CACHE_REMOVE_VM_ENTITY)
	1 --- entity will be freed soon (after swapout completes)
	2 --- entity can't be freed
	3 --- entity depends on another one, KERNEL$CACHE_TRANSFER_QUEUE_STATE
		was used
	  --- or: retry call very soon (without calling transfer queue state)
*/

int KERNEL$CACHE_REGISTER_VM_TYPE(int *entity, __const__ VMENTITY_T *t);
void KERNEL$CACHE_UNREGISTER_VM_TYPE(int entity);

#define VM_ENTITY_NOSTREAM	1
void KERNEL$CACHE_INSERT_VM_ENTITY(VMENTITY *e, PROC *wire, int flags);
void KERNEL$CACHE_TOUCH_VM_ENTITY(VMENTITY *e, PROC *wire);
void KERNEL$CACHE_REMOVE_VM_ENTITY(VMENTITY *e);
void KERNEL$CACHE_TRANSFER_QUEUE_STATE(VMENTITY *to, VMENTITY *from);

static __finline__ void CACHE_CONSTRUCT_VM_ENTITY(VMENTITY *e)
{
	e->vm_queue.next = NULL;
}

static __finline__ int CACHE_VMENTITY_IS_FREEABLE(VMENTITY *e)
{
	return (!e->queue || e->queue == VME_Q_FREEING) && !e->wired;
}

WQ *KERNEL$MAY_ALLOC(PROC *p, unsigned size);
void KERNEL$NOTIFY_FREE(PROC *p, unsigned size);
void KERNEL$NOTIFY_ALLOC(PROC *p, unsigned size);
/*
 * mode == 0 ==> return NULL if process is below quota
 * mode == 1 ==> return NULL if process is idle
 */
WQ *KERNEL$MAY_DIRTY(PROC *p, unsigned mode);

int KERNEL$QUERY_MEMORY(unsigned long *mem);
unsigned long KERNEL$VM_OOMKILL(void);

void KERNEL$MEMSTAT_DUMP(void);

__END_DECLS

#endif
