/*	$OpenBSD: arc4random.c,v 1.15 2005/11/30 07:51:02 otto Exp $	*/

/*
 * Copyright (c) 1996, David Mazieres <dm@uun.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * Arc4 random number generator for OpenBSD.
 *
 * This code is derived from section 17.1 of Applied Cryptography,
 * second edition, which describes a stream cipher allegedly
 * compatible with RSA Labs "RC4" cipher (the actual description of
 * which is a trade secret).  The same algorithm is used as a stream
 * cipher called "arcfour" in Tatu Ylonen's ssh package.
 *
 * Here the stream cipher has been modified always to include the time
 * when initializing the state.  That makes it impossible to
 * regenerate the same random sequence twice, so this can't be used
 * for encryption, but will generate good random numbers.
 *
 * RC4 is a registered trademark of RSA Laboratories.
 */

#include <STDLIB.H>
#include <UNISTD.H>
#include <FCNTL.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/WQ.H>
#include <SYS/TYPES.H>
#include <SYS/TIME.H>
#include <ARCH/TIME.H>
#include <ERRNO.H>

struct arc4_stream {
	__u8 i;
	__u8 j;
	__u8 s[256];
};

static int stream_initialized = 0;
static int arc4_count = 0;
static int arc4_error = 0;

static struct arc4_stream rs;

__COLD_ATTR__ static void initialize(void)
{
	unsigned i;
	rs.i = 0;
	rs.j = 0;
	for (i = 0; i < 256; i++) rs.s[i] = i;
	stream_initialized = 1;
}

static void arc4_addrandom(__u8 *dat, int datlen)
{
	int n;
	__u8 si;

	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_MALLOC);
	if (__unlikely(!stream_initialized)) initialize();
	rs.i--;
	for (n = 0; n < 256; n++) {
		rs.i = (rs.i + 1);
		si = rs.s[rs.i];
		rs.j = (rs.j + si + dat[n % datlen]);
		rs.s[rs.i] = rs.s[rs.j];
		rs.s[rs.j] = si;
	}
	rs.j = rs.i;
	LOWER_SPLX(spl);
}

static __u32 arc4_getword(void);

u_int32_t arc4random_nonblock(void)
{
	if (arc4_count >= 0) __SUBI(&arc4_count, 1);
	return arc4_getword();
}

u_int32_t arc4random(void)
{
	if (__unlikely(__SUBIS(&arc4_count, 1))) arc4random_stir();
	return arc4_getword();
}

static __finline__ __u8 arc4_getbyte(void)
{
	__u8 si, sj;

	rs.i = (rs.i + 1);
	si = rs.s[rs.i];
	rs.j = (rs.j + si);
	sj = rs.s[rs.j];
	rs.s[rs.i] = sj;
	rs.s[rs.j] = si;
	return rs.s[(si + sj) & 0xff];
}

static __u32 arc4_getword(void)
{
	__u32 val;
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_MALLOC);
	if (__unlikely(!stream_initialized)) initialize();
	val = arc4_getbyte() << 24;
	val |= arc4_getbyte() << 16;
	val |= arc4_getbyte() << 8;
	val |= arc4_getbyte();
	LOWER_SPLX(spl);
	return val;
}

__COLD_ATTR__ int arc4random_stir(void)
{
	int ret;
	union {
		OPENRQ o;
		SIORQ s;
		CLOSERQ c;
		struct timeval tv;
	} u;
	union {
		__u8 s[128];
		struct {
			__u64 tim;
			struct timeval tv;
		} t;
	} data;
	unsigned i;

	u.o.flags = O_RDONLY | _O_NOPOSIX;
	u.o.path = "SYS$RANDOM:/";
	u.o.cwd = NULL;
	SYNC_IO(&u.o, KERNEL$OPEN);
	if (__unlikely(u.o.status < 0)) {
		ret = u.o.status;
		goto stir_empty;
	}
	u.s.h = u.o.status;
	u.s.v.ptr = (unsigned long)&data;
	u.s.v.len = sizeof data;
	u.s.v.vspace = &KERNEL$VIRTUAL;
	u.s.progress = 0;
	read_more:
	SYNC_IO(&u.s, KERNEL$READ);
	if (__unlikely(u.s.status < 0)) {
		ret = u.s.status;
		goto do_close;
	}
	if (__unlikely(u.s.v.len != 0)) goto read_more;
	ret = 0;

	do_close:
	u.c.h = u.s.h;
	SYNC_IO(&u.c, KERNEL$CLOSE);

	stir_empty:
	data.t.tim ^= TIMER_RANDOMNESS();
	gettimeofday(&u.tv, NULL);
	data.t.tv.tv_sec ^= u.tv.tv_sec;
	data.t.tv.tv_usec ^= u.tv.tv_usec;

	arc4_addrandom((__u8 *)&data, sizeof data);
	
	for (i = 0; i < 256; i++) arc4_getword();
	arc4_count = 400000;
	arc4_error = ret;
	return ret;
}

void arc4random_addrandom(unsigned char *dat, int datlen)
{
	if (__unlikely(!datlen)) return;
	arc4_addrandom(dat, datlen);
}

int arc4random_error(void)
{
	return arc4_error;
}

__COLD_ATTR__ void arc4random_invalidate(void)
{
	if (__unlikely(arc4_count > 0)) {
		__barrier();
		arc4_count = 0;
	}
}
