#include <ARCH/BSF.H>
#include <KERNEL/VMDEF.H>
#include <KERNEL/VM_ARCH.H>
#include <SPAD/ALLOC.H>
#include <SPAD/SYNC.H>

#include <KERNEL/VM.H>
#include <SPAD/DEV_KRNL.H>

#if __DEBUG_BOUNCE
#define BOUNCE_DMA_EXCLUDE	0
#else
#define BOUNCE_DMA_EXCLUDE	BOUNCE_DMA64
#endif

#if KERNEL_BOUNCE_COMPILE

#define page_map_count	valid_from
#define page_size_shift	valid_to

struct bounce {
	XLIST_HEAD page_list;
	XLIST_HEAD record_list;
	unsigned long allocated_slots;
	unsigned long reserved_slots;
};

struct bounce_record {
	union {
		LIST_ENTRY list;
		struct {
			__p_addr orig_addr;
			unsigned long len;
		} s;
	} u;
	void *vaddr;
};

#define MIN_BOUNCE_SIZE	(2 << __BSR_CONST(sizeof(struct bounce_record)))

#define N_BOUNCES	(__BSR_CONST(PAGE_CLUSTER_SIZE / MIN_BOUNCE_SIZE) + 1)

struct bounce bounces[N_BOUNCES];

__COLD_ATTR__ static DECL_IOCALL(BOUNCE_PAGE_RELEASE, SPL_TOP, PAGE_RELEASE_REQUEST)
{
	RQ->status = -EEXIST;
	RETURN_AST(RQ);
}

static __finline__ unsigned SIZE2IDX(unsigned long size)
{
	unsigned idx = __BSR((size - 1) | (MIN_BOUNCE_SIZE - 1)) - (__BSR_CONST(MIN_BOUNCE_SIZE) - 1);
#if __DEBUG >= 1
	if (__unlikely(idx >= N_BOUNCES))
		KERNEL$SUICIDE("SIZE2IDX: INVALID SIZE %lX, IDX %X, LIMIT %X", size, idx, N_BOUNCES);
#endif
	return idx;
}

static unsigned long IDX2SIZE(unsigned idx)
{
	return (unsigned long)MIN_BOUNCE_SIZE << idx;
}

__COLD_ATTR__ static struct bounce *FIXUP_REQUESTS_SIZE(unsigned long *requests, unsigned long *size, unsigned *size_shift)
{
	unsigned index;
	if (*size > PAGE_CLUSTER_SIZE) {
		*size += PAGE_CLUSTER_SIZE - 1;
		*requests = *requests * (*size / PAGE_CLUSTER_SIZE);
		*size = PAGE_CLUSTER_SIZE;
	}
	index = SIZE2IDX(*size);
	*size = IDX2SIZE(index);
	*size_shift = index + __BSR_CONST(MIN_BOUNCE_SIZE);
	return &bounces[index];
}

__COLD_ATTR__ static void FREE_BOUNCE_PAGE(PAGE *page)
{
	struct bounce *b;
	struct bounce_record *rec = page->fnode;
	unsigned i;
	for (i = 0; i < PAGE_CLUSTER_SIZE >> page->page_size_shift; i++)
		DEL_FROM_LIST(&rec[i].u.list);
	DEL_FROM_LIST(&page->hash_entry);
	b = &bounces[page->page_size_shift - __BSR_CONST(MIN_BOUNCE_SIZE)];
	b->allocated_slots -= PAGE_CLUSTER_SIZE >> page->page_size_shift;
	KERNEL$FREE_CONTIG_AREA(PAGE_2_VIRT(page), PAGE_CLUSTER_SIZE);
	free(rec);
}

__COLD_ATTR__ static void SLOT_CLEANUP(struct bounce *b)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_VSPACE);
	again:
	if (b->allocated_slots - b->reserved_slots >= PAGE_CLUSTER_SIZE / IDX2SIZE(b - bounces)) {
		PAGE *page;
		XLIST_FOR_EACH(page, &b->page_list, PAGE, hash_entry) {
			if (page->page_map_count)
				continue;
			FREE_BOUNCE_PAGE(page);
			goto again;
		}
	}
	LOWER_SPLX(spl);
}

__COLD_ATTR__ static int ALLOC_RESERVE_BOUNCE(unsigned long requests, unsigned long size)
{
	int spl = KERNEL$SPL;
	unsigned long reserved_so_far = 0;
	PAGE *page = NULL;
	struct bounce_record *rec = NULL;
	unsigned size_shift;
	struct bounce *b = FIXUP_REQUESTS_SIZE(&requests, &size, &size_shift);
	unsigned long diff;

	again:
	RAISE_SPL(SPL_VSPACE);
	diff = b->allocated_slots - b->reserved_slots;
	if (diff) {
		if (diff > requests) diff = requests;
		b->reserved_slots += diff;
		requests -= diff;
		reserved_so_far += diff;
	}
	if (requests && page) {
		unsigned i, o;
		ADD_TO_XLIST(&b->page_list, &page->hash_entry);
		for (i = 0, o = 0; o < PAGE_CLUSTER_SIZE; i++, o += size) {
			ADD_TO_XLIST(&b->record_list, &rec[i].u.list);
			rec[i].vaddr = (char *)PAGE_2_VIRT(page) + o;
		}
		b->allocated_slots += i;
		if (i > requests) i = requests;
		b->reserved_slots += i;
		requests -= i;
		reserved_so_far += i;
		page->release = BOUNCE_PAGE_RELEASE;
		page = NULL;
		rec = NULL;
	}
	LOWER_SPLX(spl);
	if (requests) {
		int r;
		union {
			MALLOC_REQUEST mrq;
		} u;
		void *ptr;
		u.mrq.size = (sizeof(struct bounce_record) * PAGE_CLUSTER_SIZE) >> size_shift;
		SYNC_IO_CANCELABLE(&u.mrq, KERNEL$UNIVERSAL_MALLOC);
		if (u.mrq.status < 0) {
			r = u.mrq.status;
			ret0:
			RAISE_SPL(SPL_VSPACE);
			b->reserved_slots -= reserved_so_far;
			LOWER_SPLX(spl);
			SLOT_CLEANUP(b);
			return r;
		}
		rec = u.mrq.ptr;
		ptr = KERNEL$ALLOC_CONTIG_AREA(PAGE_CLUSTER_SIZE, AREA_DATA | AREA_PHYSCONTIG | AREA_PCIDMA | AREA_ALIGN, (unsigned long)PAGE_CLUSTER_SIZE);
		if (__IS_ERR(ptr)) {
			r = __PTR_ERR(ptr);
			free(rec);
			goto ret0;
		}
		page = VIRT_2_PAGE_ALIGNED(ptr);
		page->fnode = rec;
		page->page_map_count = 0;
		page->page_size_shift = size_shift;
		goto again;
	}
	if (page)
		KERNEL$FREE_CONTIG_AREA(PAGE_2_VIRT(page), PAGE_CLUSTER_SIZE);

	return 0;
}

__COLD_ATTR__ static void FREE_RESERVE_BOUNCE(unsigned long requests, unsigned long size)
{
	int spl = KERNEL$SPL;
	unsigned size_shift;
	struct bounce *b = FIXUP_REQUESTS_SIZE(&requests, &size, &size_shift);

	RAISE_SPL(SPL_VSPACE);
	if (b->reserved_slots < requests)
		KERNEL$SUICIDE("FREE_RESERVE_BOUNCE: RESERVED SLOT COUNT UNDERFLOW: %lX < %lX, SIZE %lX", b->reserved_slots, requests, size);
	b->reserved_slots -= requests;
	LOWER_SPLX(spl);
	SLOT_CLEANUP(b);
}

static void KERNEL_DO_UNBOUNCE1(__u32 paddr);

vspace_dmaunlock_t *KERNEL_DO_BOUNCE(VDESC *desc, int rw, VDMA *vdma)
{
	PAGE *page;
	unsigned idx;
	struct bounce *b;
	struct bounce_record *rec;
	void *ptr;

	SPLVSPACEASSERT("KERNEL_DO_BOUNCE");

	if (__unlikely(((unsigned long)desc->ptr & (PAGE_CLUSTER_SIZE - 1)) + desc->len > PAGE_CLUSTER_SIZE))
		vdma->len = PAGE_CLUSTER_SIZE - ((unsigned long)desc->ptr & (PAGE_CLUSTER_SIZE - 1));
	else
		vdma->len = desc->len;

	idx = SIZE2IDX(vdma->len);
	b = &bounces[idx];
	while (__unlikely(XLIST_EMPTY(&b->record_list))) {
		b++;
		if (__unlikely(b == &bounces[N_BOUNCES]))
			KERNEL$SUICIDE("KERNEL_DO_BOUNCE: OUT OF BOUNCE MEMORY");
	}
	rec = LIST_STRUCT(b->record_list.next, struct bounce_record, u.list);
	DEL_FROM_LIST(&rec->u.list);
	page = VIRT_2_PAGE(rec->vaddr);
	page->page_map_count++;
	LOWER_SPLX(vdma->spl);
	rec->u.s.orig_addr = desc->ptr;
	rec->u.s.len = vdma->len & -(rw & PF_WRITE);
	if (rw & PF_READ) {
		ptr = KERNEL$MAP_PHYSICAL_BANK(desc->ptr);
		memcpy(rec->vaddr, ptr, vdma->len);
		KERNEL$UNMAP_PHYSICAL_BANK(ptr);
	}
	vdma->ptr = KERNEL$VIRT_2_PHYS(rec->vaddr);
	/*__debug_printf("bounce %Lx,%x,%x -> %x,%x\n", desc->ptr, desc->len, rw, ret.ptr, ret.len);*/
	return KERNEL_DO_UNBOUNCE1;
}

static void KERNEL_DO_UNBOUNCE1(__u32 paddr)
{
	KERNEL_DO_UNBOUNCE(paddr);
}

__p_addr KERNEL_DO_UNBOUNCE(__u32 paddr)
{
	PAGE *page;
	struct bounce_record *rec;
	struct bounce *b;
	unsigned offset;
	void *ptr;
	__p_addr orig_addr;
	int spl;

	page = PHYS_2_PAGE(paddr);

	if (__likely(page->release != BOUNCE_PAGE_RELEASE)) {
		return paddr;
	}

	offset = paddr & (PAGE_CLUSTER_SIZE - 1);
#if __DEBUG >= 1
	if (__unlikely(offset & ((1 << page->page_size_shift) - 1)))
		KERNEL$SUICIDE("KERNEL_DO_UNBOUNCE: INVALID POINTER %X, PAGE_SIZE_SHIFT %X", paddr, page->page_size_shift);
#endif
	rec = page->fnode;
	rec += offset >> page->page_size_shift;

	/*__debug_printf("unbounce: %x,%x -> %Lx\n", paddr, rec->u.s.len, rec->u.s.orig_addr);*/

	if (rec->u.s.len) {
		ptr = KERNEL$MAP_PHYSICAL_BANK(rec->u.s.orig_addr);
		memcpy(ptr, rec->vaddr, rec->u.s.len);
		KERNEL$UNMAP_PHYSICAL_BANK(ptr);
	}

	spl = KERNEL$SPL;
	RAISE_SPL(SPL_VSPACE);

#if __DEBUG >= 1
	if (__unlikely(page->page_map_count <= 0))
		KERNEL$SUICIDE("KERNEL_DO_UNBOUNCE: MAP_COUNT UNDERFLOW: %d", page->page_map_count);
#endif
	page->page_map_count--;
	b = &bounces[page->page_size_shift - __BSR_CONST(MIN_BOUNCE_SIZE)];
	orig_addr = rec->u.s.orig_addr;
	ADD_TO_XLIST(&b->record_list, &rec->u.list);
	if (__unlikely(b->allocated_slots - b->reserved_slots >= PAGE_CLUSTER_SIZE >> page->page_size_shift) && !page->page_map_count) {
		FREE_BOUNCE_PAGE(page);
	}

	LOWER_SPLX(spl);

	return orig_addr;
}

#endif

__COLD_ATTR__ void BOUNCE_INIT(void)
{
#if KERNEL_BOUNCE_COMPILE
	unsigned i;
	for (i = 0; i < N_BOUNCES; i++) {
		INIT_XLIST(&bounces[i].page_list);
		INIT_XLIST(&bounces[i].record_list);
		bounces[i].allocated_slots = 0;
		bounces[i].reserved_slots = 0;
	}
#endif
}

static unsigned long rq[BOUNCE_DMA64 + 1] = { 0 };
static unsigned long total_size[BOUNCE_DMA64 + 1] = { 0 };

__COLD_ATTR__ static void UNACCOUNT_BOUNCE(const char *device, unsigned long requests, unsigned long size, int flags)
{
	int spl = KERNEL$SPL;
	RAISE_SPL(SPL_VSPACE);
	if (requests > rq[flags & BOUNCE_DMA64] || size > total_size[flags & BOUNCE_DMA64]) KERNEL$SUICIDE("KERNEL$UNRESERVE_BOUNCE: UNDERRUN, FLAGS %X RESERVED(%lu,%lu), FREEING(%lu, %lu)", flags, rq[flags & BOUNCE_DMA64], total_size[flags & BOUNCE_DMA64], requests, size);
	rq[flags & BOUNCE_DMA64] -= requests;
	total_size[flags & BOUNCE_DMA64] -= size;

	{
		int i;
		int empty = 1;
		for (i = 0; i < sizeof(rq) / sizeof(*rq); i++) {
			if (rq[i] && total_size[i]) {
				empty = 0;
				continue;
			}
			if (!rq[i] && !total_size[i])
				continue;
			KERNEL$SUICIDE("UNACCOUNT_BOUNCE: MISCACCOUNTING AT %X: REQUESTS %lu, TOTAL SIZE %lu", i, rq[i], total_size[i]);
		}
#if KERNEL_BOUNCE_COMPILE
		if (empty) {
			for (i = 0; i < N_BOUNCES; i++) {
				if (bounces[i].allocated_slots || bounces[i].reserved_slots)
					KERNEL$SUICIDE("UNACCOUNT_BOUNCE: BOUNCE BUFFERS LEAKED FOR INDEX %d, ALLOCATED %lu, RESERVED %lu", i, bounces[i].allocated_slots, bounces[i].reserved_slots);
				if (!XLIST_EMPTY(&bounces[i].page_list) || !XLIST_EMPTY(&bounces[i].record_list))
					KERNEL$SUICIDE("UNACCOUNT_BOUNCE: LISTS NOT CLEARED FOR INDEX %d, PAGE LIST %d, RECORD LIST %d", i, XLIST_EMPTY(&bounces[i].page_list), XLIST_EMPTY(&bounces[i].record_list));
			}
			/*__debug_printf("cleaned up\n");*/
		}
#endif
	}

	LOWER_SPLX(spl);
}

__COLD_ATTR__ int KERNEL$RESERVE_BOUNCE(const char *device, unsigned long requests, unsigned long size, int flags)
{
	int r;
	int spl = KERNEL$SPL;

	/*__debug_printf("reserve %ld %ld %X\n", requests, size, flags);*/
	if (!requests || !size)
		return 0;

	RAISE_SPL(SPL_VSPACE);
	rq[flags & BOUNCE_DMA64] += requests;
	total_size[flags & BOUNCE_DMA64] += size;
	LOWER_SPLX(spl);

#if KERNEL_BOUNCE_COMPILE
	if (KERNEL_BOUNCE_ACTIVE && !(flags & BOUNCE_DMA_EXCLUDE)) {
		/*__debug_printf("alloc %ld %ld\n", requests, size);*/
		r = ALLOC_RESERVE_BOUNCE(requests, size);
		if (r)
			goto ret0;
	}
#endif

	return 0;

	ret0:
	UNACCOUNT_BOUNCE(device, requests, size, flags);
	return r;
}

	/* must not block */
__COLD_ATTR__ void KERNEL$UNRESERVE_BOUNCE(const char *device, unsigned long requests, unsigned long size, int flags)
{
	if (!requests || !size)
		return;

#if KERNEL_BOUNCE_COMPILE
	if (KERNEL_BOUNCE_ACTIVE && !(flags & BOUNCE_DMA_EXCLUDE)) {
		FREE_RESERVE_BOUNCE(requests, size);
	}
#endif

	UNACCOUNT_BOUNCE(device, requests, size, flags);
}

__COLD_ATTR__ int KERNEL$RESERVE_BOUNCE_ACTIVE(int flags)
{
	if (!(flags & BOUNCE_DMA_EXCLUDE))
		return KERNEL_BOUNCE_ACTIVE;
	return 0;
}

