#include <STRING.H>
#include <ERRNO.H>
#include <SPAD/LIBC.H>
#include <SPAD/TIMER.H>
#include <KERNEL/PROC.H>
#include <KERNEL/VM.H>
#include <KERNEL/PARAMS.H>
#include <SPAD/QUOTA.H>

#include <SPAD/VM.H>

#define VM_QUEUES			16
#define VM_INIT_QUEUE			0
#define VM_TOUCH_ADVANCE		2
#define VM_MAPPING_ADVANCE		3
#define VM_MAPPING_WIRED_ADVANCE	2
#define VM_STEP				(JIFFIES_PER_SECOND * 60)
#define FREE_LOWER_RATE			(JIFFIES_PER_SECOND * 5)

#define N_TYPES 256

#define CLOCK_SMALL_LIST	1	/* must be 1 */
#define CLOCK_MAPPED_2ND_PASS	2
#define CLOCK_NOT_TOUCHED	4
#define CLOCK_ADD		8

#if __DEBUG >= 4
#define CACHE_CHECK	/* slows down very much */
#endif

static struct {
	VMENTITY_T t;
} ENTITIES[N_TYPES];

unsigned IO_IN_PROGRESS = 0;
unsigned long VM_ENTITIES = 0;
unsigned long VM_ENTITIES_SOFT_LIMIT = -1;
unsigned long VM_ENTITIES_HARD_LIMIT = -1;

static LIST_HEAD VM_QUEUE[VM_QUEUES];
static unsigned long VM_OBJCOUNT[VM_QUEUES];

static LIST_HEAD VM_WIREWAIT[VM_WIRESTEPS];
static unsigned current_wirewait;

unsigned VM_WIRED_STREAM_QUEUE;
unsigned VM_WIRED_STREAM_REACTIVATE;

u_jiffies_lo_t VM_WIREJIFFIES;

static void LOWER_ENTITIES(void);

#ifdef CACHE_CHECK
void CHECK_CACHE_QUEUES(PROC *p);
void CHECK_CACHE_QUEUES(PROC *p)
{
	VMENTITY *e;
	int i;
	unsigned long count;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_CACHE), spl)))
		KERNEL$SUICIDE("CHECK_CACHE_QUEUES AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_CACHE);
	if (p) {
		if (__unlikely((p->vm_wired_small_clock & (CLOCK_SMALL_LIST | CLOCK_MAPPED_2ND_PASS | CLOCK_NOT_TOUCHED)) != (CLOCK_SMALL_LIST | CLOCK_NOT_TOUCHED)))
			KERNEL$SUICIDE("CHECK_CACHE_QUEUES: PROCESS CLOCK %04X", p->vm_wired_small_clock);
		count = 0;
		LIST_FOR_EACH(e, &p->vm_wired_small, VMENTITY, wire_list) {
			if (__unlikely(!(e->wired_clock & CLOCK_SMALL_LIST)))
				KERNEL$SUICIDE("CHECK_CACHE_QUEUES: ENTITY SHOULD BE ON VM_WIRED");
			count++;
		}
		if (__unlikely(count != p->vm_wired_small_n))
			KERNEL$SUICIDE("CHECK_CACHE_QUEUES: VM_WIRED_SMALL_N (%lu) != NUMBER OF ENTRIES ON LIST (%lu)", p->vm_wired_small_n, count);
		LIST_FOR_EACH(e, &p->vm_wired, VMENTITY, wire_list) {
			if (__unlikely(e->wired_clock & CLOCK_SMALL_LIST))
				KERNEL$SUICIDE("CHECK_CACHE_QUEUES: ENTITY SHOULD BE ON VM_WIRED_SMALL");
			count++;
		}
		if (__unlikely(count != p->wireq.q_node_usage))
			KERNEL$SUICIDE("CHECK_CACHE_QUEUES: Q_NODE_USAGE (%lu) != NUMBER OF ENTRIES ON LISTS (%lu)", p->wireq.q_node_usage, count);
	}
	for (i = 0; i < VM_QUEUES; i++) {
		count = 0;
		LIST_FOR_EACH(e, &VM_QUEUE[i], VMENTITY, vm_queue) {
			if (__unlikely(!ENTITIES[e->type].t.pageout)) KERNEL$SUICIDE("CHECK_CACHE_QUEUES: TYPE %d AT VM_QUEUE[%d]", e->type, i);
			if (__unlikely(e->wired != NULL)) KERNEL$SUICIDE("CHECK_CACHE_QUEUES: WIRED ENTITY ON VM_QUEUE[%d]", i);
			count++;
		}
		if (__unlikely(count != VM_OBJCOUNT[i]))
			KERNEL$SUICIDE("CHECK_CACHE_QUEUES: VM_OBJCOUNT[%d] MISCOUNTED: %lu != (COUNTED) %lu", i, VM_OBJCOUNT[i], count);
	}
	count = 0;
	for (i = 0; i < VM_WIRESTEPS; i++) {
		LIST_FOR_EACH(e, &VM_WIREWAIT[i], VMENTITY, vm_queue) {
			if (__unlikely(!ENTITIES[e->type].t.pageout)) KERNEL$SUICIDE("CHECK_CACHE_QUEUES: TYPE %d AT VM_WIREWAIT[%d]", e->type, i);
			if (__unlikely(!e->wired)) KERNEL$SUICIDE("CHECK_CACHE_QUEUES: UNWIRED ENTITY ON VM_WIREWAIT[%d]", i);
			count++;
		}
	}
	if (__unlikely(count != KERNEL$PROC_KERNEL.wireq.q_subtree_usage))
		KERNEL$SUICIDE("CHECK_CACHE_QUEUES: QUOTA LEAK: SUBTREE USAGE %lu != (COUNTED) %lu", KERNEL$PROC_KERNEL.wireq.q_node_usage, count);
	LOWER_SPLX(spl);
}
#else
#define CHECK_CACHE_QUEUES(p) do { } while (0)
#endif

static __finline__ void VM_QUEUE_INSERT(VMENTITY *e, unsigned queue)
{
#if __DEBUG >= 2
	if (__unlikely(e->wired != NULL))
		KERNEL$SUICIDE("VM_QUEUE_INSERT: INSERTING WIRED ENTITY");
	if (__unlikely(queue >= VM_QUEUES))
		KERNEL$SUICIDE("VM_QUEUE_INSERT: INVALID QUEUE %u", queue);
#endif
	e->queue = queue;
	VM_OBJCOUNT[queue]++;
	ADD_TO_LIST_END(&VM_QUEUE[queue], &e->vm_queue);
}

static __finline__ void VM_QUEUE_REMOVE(VMENTITY *e)
{
#if __DEBUG >= 2
	if (__unlikely(e->wired != NULL) || __unlikely(e->queue >= VM_QUEUES))
		KERNEL$SUICIDE("VM_QUEUE_REMOVE: INVALID ENTITY, WIRED %p, QUEUE %u", e->wired, e->queue);
	if (__unlikely(!VM_OBJCOUNT[e->queue]))
		KERNEL$SUICIDE("VM_QUEUE_REMOVE: OBJCOUNT[%u] == 0", e->queue);
#endif
	VM_OBJCOUNT[e->queue]--;
	DEL_FROM_LIST(&e->vm_queue);
}

#if __DEBUG < 2

#define VM_QUEUE_EMPTY(q)	(LIST_EMPTY(&VM_QUEUE[q]))
#define VM_QUEUE_FIRST(q)	(LIST_STRUCT(VM_QUEUE[q].next, VMENTITY, vm_queue))

#else

static int VM_QUEUE_EMPTY(unsigned q)
{
	int m;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_CACHE)))
		KERNEL$SUICIDE("VM_QUEUE_EMPTY AT SPL %08X", KERNEL$SPL);
	m = LIST_EMPTY(&VM_QUEUE[q]);
	if (__unlikely(m != !VM_OBJCOUNT[q])) {
		__barrier();	/* improves register allocation */
		KERNEL$SUICIDE("VM_QUEUE_EMPTY: VM_OBJCOUNT SKEWED ON QUEUE %u: %lu, EMPTY %d", q, VM_OBJCOUNT[q], m);
	}
	return m;
}

static VMENTITY *VM_QUEUE_FIRST(unsigned q)
{
	VMENTITY *e;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_CACHE)))
		KERNEL$SUICIDE("VM_QUEUE_EMPTY AT SPL %08X", KERNEL$SPL);
	e = LIST_STRUCT(VM_QUEUE[q].next, VMENTITY, vm_queue);
	if (__unlikely(!ENTITIES[e->type].t.pageout))
		KERNEL$SUICIDE("VM_QUEUE_FIRST: UNKNOWN ENTITY %d ON QUEUE", e->type);
	if (__unlikely(e->wired != NULL))
		KERNEL$SUICIDE("VM_QUEUE_FIRST: WIRED ENTITY %s ON QUEUE %u", ENTITIES[e->type].t.name, q);
	if (__unlikely(e->queue != q))
		KERNEL$SUICIDE("VM_QUEUE_FIRST: ENTITY %s MISPLACED ON DIFFERENT QUEUE: %u != %u", ENTITIES[e->type].t.name, e->queue, q);
	return e;
}

#endif

__COLD_ATTR__ int KERNEL$CACHE_REGISTER_VM_TYPE(int *entity, const VMENTITY_T *t)
{
	int i;
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_CACHE), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("KERNEL$CACHE_UNREGISTER_VM_TYPE AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_CACHE);
	CHECK_CACHE_QUEUES(NULL);
		/* catch uninitialized variable errors --- do not start from 0 */
	for (i = !!__DEBUG; i < N_TYPES; i++) {
		if (!ENTITIES[i].t.pageout) {
			memcpy(&ENTITIES[i].t, t, sizeof(VMENTITY_T));
			if (__unlikely(!ENTITIES[i].t.pageout)) KERNEL$SUICIDE("KERNEL$CACHE_REGISTER_VM_TYPE: NO PAGEOUT");
			*entity = i;
			LOWER_SPLX(spl);
			return 0;
		}
	}
	LOWER_SPLX(spl);
	return -ENFILE;
}

__COLD_ATTR__ void KERNEL$CACHE_UNREGISTER_VM_TYPE(int entity)
{
	int i;
	int spl;
	VMENTITY *e;
	CHECK_CACHE_QUEUES(NULL);
	if ((unsigned)entity >= N_TYPES || !ENTITIES[entity].t.pageout)
		KERNEL$SUICIDE("KERNEL$CACHE_UNREGISTER_VM_TYPE: ENTITY %d", entity);
	ENTITIES[entity].t.pageout = NULL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_CACHE), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("KERNEL$CACHE_UNREGISTER_VM_TYPE AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_CACHE);
	for (i = 0; i < VM_QUEUES; i++) LIST_FOR_EACH(e, &VM_QUEUE[i], VMENTITY, vm_queue) if (__unlikely(e->type == entity)) KERNEL$SUICIDE("KERNEL$CACHE_UNREGISTER_VM_TYPE: TYPE %s HAS ACTIVE ENTITIES ON VM_QUEUE[%d]", ENTITIES[entity].t.name, i);
	for (i = 0; i < VM_WIRESTEPS; i++) LIST_FOR_EACH(e, &VM_WIREWAIT[i], VMENTITY, vm_queue) if (__unlikely(e->type == entity)) KERNEL$SUICIDE("KERNEL$CACHE_UNREGISTER_VM_TYPE: TYPE %s HAS ACTIVE ENTITIES ON VM_WIREWAIT[%d]", ENTITIES[entity].t.name, i);
	LOWER_SPLX(spl);
}

void CACHE_PROC_CTOR(PROC *proc)
{
	INIT_LIST(&proc->vm_wired);
	INIT_LIST(&proc->vm_wired_small);
	proc->vm_wired_small_n = 0;
	proc->vm_wired_small_clock = CLOCK_SMALL_LIST | CLOCK_NOT_TOUCHED | 0x8000;
	VOID_LIST_ENTRY(&proc->delayed_balance);
}

	/* warning: this does lower SPL */
static void UNWIRE_ONE_ENTITY(VMENTITY *e, PROC *p, int wflag)
{
	p->vm_wired_small_n -= ((unsigned)e->wired_clock / CLOCK_SMALL_LIST) & 1;
	DEL_FROM_LIST(&e->vm_queue);
	DEL_FROM_LIST(&e->wire_list);
	QFREE(&p->wireq, 1, proc_wireq_isroot, proc_wireq_parent, Q_NULL_CALL);
	e->wired = NULL;
	VM_QUEUE_INSERT(e, e->queue);
#if __DEBUG == 1
	e->wire_list.next = NULL;
	e->wire_list.prev = NULL;
#endif
	ENTITIES[e->type].t.write(e, p, __unlikely(!e->queue) ? 1 : wflag);
	RAISE_SPL(SPL_CACHE);
}

static AST delayed_balance_ast = { NULL };
static DECL_XLIST(delayed_balance_list);

static DECL_AST(DELAYED_BALANCE, SPL_DEV, AST)
{
	u_jiffies_lo_t j = KERNEL$GET_JIFFIES_LO();
	RAISE_SPL(SPL_CACHE);
	while (!XLIST_EMPTY(&delayed_balance_list)) {
		PROC *proc = LIST_STRUCT(delayed_balance_list.next, PROC, delayed_balance);
		while (1) {
			int r;
			VMENTITY *e;
			if (__unlikely(KERNEL$GET_JIFFIES_LO() - j > TIMEOUT_JIFFIES(JIFFIES_PER_SECOND / 25))) break;
			TEST_SPL(SPL_DEV, SPL_CACHE);
			if (__unlikely(proc->vm_wired_small_n <= VM_WIRED_STREAM_QUEUE * 3 / 4)) break;
#if __DEBUG >= 1
			if (__unlikely(LIST_EMPTY(&proc->vm_wired_small)))
				KERNEL$SUICIDE("DELAYED_BALANCE: SMALL LIST EMPTY");
#endif
			e = LIST_STRUCT(proc->vm_wired_small.next, VMENTITY, wire_list);
#if __DEBUG >= 1
			if (__unlikely(e->wired != proc))
				KERNEL$SUICIDE("DELAYED_BALANCE: BAD ENTITY ON LIST (%p != %p)", e->wired, proc);
#endif
			r = ENTITIES[e->type].t.checkmap(e);
			RAISE_SPL(SPL_CACHE);
			if (r) {
				if (!(e->wired_clock & CLOCK_MAPPED_2ND_PASS)) {
					e->wired_clock |= CLOCK_MAPPED_2ND_PASS;
					DEL_FROM_LIST(&e->wire_list);
					ADD_TO_LIST_END(&e->wired->vm_wired_small, &e->wire_list);
					continue;
				} else {
					e->wired_clock = 0;
					proc->vm_wired_small_n--;
					DEL_FROM_LIST(&e->wire_list);
					ADD_TO_LIST_END(&e->wired->vm_wired, &e->wire_list);
					continue;
				}
			}
			UNWIRE_ONE_ENTITY(e, proc, 0);
		}
		DEL_FROM_LIST(&proc->delayed_balance);
		VOID_LIST_ENTRY(&proc->delayed_balance);
	}
	delayed_balance_ast.fn = NULL;
	RETURN;
}

static void BALANCE_WIRED_SMALL(PROC *proc)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_CACHE)))
		KERNEL$SUICIDE("BALANCE_WIRED_SMALL AT SPL %08X", KERNEL$SPL);
#endif
	if (__unlikely(!LIST_VOID(&proc->delayed_balance))) return;
	ADD_TO_XLIST(&delayed_balance_list, &proc->delayed_balance);
	if (__unlikely(delayed_balance_ast.fn != NULL)) return;
	delayed_balance_ast.fn = DELAYED_BALANCE;
	CALL_AST(&delayed_balance_ast);
}

/* it is expected later in this file that this function always inserts to vm_wired_small list */
void KERNEL$CACHE_INSERT_VM_ENTITY(VMENTITY *e, PROC *wire, int flags)
{
	QUOTA *pzap;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_CACHE)))
		KERNEL$SUICIDE("KERNEL$CACHE_INSERT_VM_ENTITY AT SPL %08X", KERNEL$SPL);
	if (__unlikely(e->vm_queue.next != NULL))
		KERNEL$SUICIDE("KERNEL$CACHE_INSERT_VM_ENTITY: INSERTING ENTITY TWICE");
		/* +1 to kill warning */
	if (__unlikely(((unsigned)e->type + 1) >= N_TYPES + 1) || __unlikely(!ENTITIES[e->type].t.pageout))
		KERNEL$SUICIDE("KERNEL$CACHE_INSERT_VM_ENTITY: UNKNOWN ENTITY %d", e->type);
	if (__unlikely(!wire))
		KERNEL$SUICIDE("KERNEL$CACHE_INSERT_VM_ENTITY: wire == NULL");
#endif
	CHECK_CACHE_QUEUES(wire);
	ADD_TO_LIST_END(&VM_WIREWAIT[current_wirewait], &e->vm_queue);
	e->queue = VM_INIT_QUEUE;
	e->wired = wire;
	e->touchtime = KERNEL$GET_JIFFIES_LO();
	QALLOC(&wire->wireq, 1, proc_wireq_isroot, proc_wireq_parent, Q_NULL_CALL, pzap, {
		KERNEL$SUICIDE("KERNEL$CACHE_INSERT_VM_ENTITY: QALLOC FAILED");
	});
	VM_ENTITIES++;
	if (flags & VM_ENTITY_NOSTREAM) {
		ADD_TO_LIST_END(&wire->vm_wired, &e->wire_list);
		e->wired_clock = 0;
	} else {
		ADD_TO_LIST_END(&wire->vm_wired_small, &e->wire_list);
		e->wired_clock = wire->vm_wired_small_clock += CLOCK_ADD;
		wire->vm_wired_small_n++;
		if (__unlikely(wire->vm_wired_small_n > VM_WIRED_STREAM_QUEUE)) {
			BALANCE_WIRED_SMALL(wire);
		}
	}
	if (__unlikely(VM_ENTITIES > VM_ENTITIES_SOFT_LIMIT)) {
		RAISE_SPL(SPL_TOP);
		MEMORY_BALANCE();
		LOWER_SPL(SPL_CACHE);
	}
	CHECK_CACHE_QUEUES(wire);
}

void KERNEL$CACHE_TOUCH_VM_ENTITY(VMENTITY *e, PROC *wire)
{
	u_jiffies_lo_t j;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_CACHE)))
		KERNEL$SUICIDE("KERNEL$CACHE_TOUCH_VM_ENTITY AT SPL %08X", KERNEL$SPL);
	if (__unlikely(!wire))
		KERNEL$SUICIDE("KERNEL$CACHE_TOUCH_VM_ENTITY: wire == NULL");
#endif
	CHECK_CACHE_QUEUES(wire);
	if (__unlikely(e->queue == VME_Q_FREEING)) return;
	if (__likely(e->wired != NULL)) {
		if (__likely(e->wired == wire) || e->wired == wire->backptr[e->wired->depth]) {
			if (!(e->wired_clock & CLOCK_SMALL_LIST)) {
				DEL_FROM_LIST(&e->vm_queue);
				goto reactivate;
			}
			if (__unlikely(e->wired_clock & CLOCK_NOT_TOUCHED)) {
				e->wired_clock = e->wired->vm_wired_small_clock & ~CLOCK_NOT_TOUCHED;
				goto ret;
			}
			if ((unsigned short)e->wired->vm_wired_small_clock - e->wired_clock > VM_WIRED_STREAM_REACTIVATE * CLOCK_ADD) {
				e->wired->vm_wired_small_n--;
				DEL_FROM_LIST(&e->wire_list);
				DEL_FROM_LIST(&e->vm_queue);
				goto set_active;
			}
			goto ret;
		}
	} else {
		QUOTA *pzap;
		VM_QUEUE_REMOVE(e);
		QALLOC(&wire->wireq, 1, proc_wireq_isroot, proc_wireq_parent, Q_NULL_CALL, pzap, {
			KERNEL$SUICIDE("KERNEL$CACHE_TOUCH_VM_ENTITY: QALLOC FAILED");
		});
		e->wired = wire;
		set_active:
		ADD_TO_LIST_END(&e->wired->vm_wired, &e->wire_list);
		e->wired_clock = 0;
		reactivate:
		ADD_TO_LIST_END(&VM_WIREWAIT[current_wirewait], &e->vm_queue);
	}
	ret:
	j = KERNEL$GET_JIFFIES_LO();
	if (__unlikely(j - e->touchtime >= VM_STEP)) {
		e->touchtime = j;
		if ((e->queue += VM_TOUCH_ADVANCE) >= VM_QUEUES) e->queue = VM_QUEUES - 1;
		e->wired_clock &= (CLOCK_ADD - 1) & ~CLOCK_NOT_TOUCHED;
	}
}

static void CACHE_REMOVE_VM_ENTITY_FREEING(VMENTITY *e);

void KERNEL$CACHE_REMOVE_VM_ENTITY(VMENTITY *e)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_CACHE)))
		KERNEL$SUICIDE("KERNEL$CACHE_REMOVE_VM_ENTITY AT SPL %08X", KERNEL$SPL);
#endif
	CHECK_CACHE_QUEUES(e->wired);
	VM_ENTITIES--;
	if (__unlikely(e->queue == VME_Q_FREEING)) {
		CACHE_REMOVE_VM_ENTITY_FREEING(e);
		goto ret;
	}
	if (e->wired) {
		e->wired->vm_wired_small_n -= e->wired_clock & CLOCK_SMALL_LIST;
		QFREE(&e->wired->wireq, 1, proc_wireq_isroot, proc_wireq_parent, Q_NULL_CALL);
		DEL_FROM_LIST(&e->wire_list);
		DEL_FROM_LIST(&e->vm_queue);
	} else {
		VM_QUEUE_REMOVE(e);
	}
#if __DEBUG == 1
	e->vm_queue.next = NULL;
#endif
	if (__likely(VM_ENTITIES <= VM_ENTITIES_SOFT_LIMIT)) WQ_WAKE_ALL(&FREE_ENTITIES);
	CHECK_CACHE_QUEUES(e->wired);
	ret:;
}

static void CACHE_REMOVE_VM_ENTITY_FREEING(VMENTITY *e)
{
#if __DEBUG >= 1
	if (__unlikely(!IO_IN_PROGRESS))
		KERNEL$SUICIDE("CACHE_REMOVE_VM_ENTITY_FREEING: IO_IN_PROGRESS UNDERFLOW");
#endif
	IO_IN_PROGRESS--;
#if __DEBUG >= 1
	e->vm_queue.next = NULL;
#endif
	RAISE_SPL(SPL_TOP);
	MEMORY_BALANCE();
	LOWER_SPL(SPL_CACHE);
	CHECK_CACHE_QUEUES(NULL);
}

void ZAP_WIRED_ENTITIES(PROC *p)
{
	VMENTITY *e;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("ZAP_WIRED_ENTITIES AT SPL %08X", KERNEL$SPL);
#endif
	CHECK_CACHE_QUEUES(p);
	RAISE_SPL(SPL_CACHE);
	next_entity:
	TEST_SPL(SPL_DEV, SPL_CACHE);
	if (__unlikely(!LIST_EMPTY(&p->vm_wired_small))) {
		e = LIST_STRUCT(p->vm_wired_small.next, VMENTITY, wire_list);
#if __DEBUG >= 1
		if (__unlikely(!(e->wired_clock & CLOCK_SMALL_LIST)))
			KERNEL$SUICIDE("ZAP_WIRED_ENTITIES: ENTITY SHOULD BE ON VM_WIRED");
#endif
		goto deactivate_entity;
	}
	if (!LIST_EMPTY(&p->vm_wired)) {
		e = LIST_STRUCT(p->vm_wired.next, VMENTITY, wire_list);
#if __DEBUG >= 1
		if (__unlikely(e->wired_clock & CLOCK_SMALL_LIST))
			KERNEL$SUICIDE("ZAP_WIRED_ENTITIES: ENTITY SHOULD BE ON VM_WIRED_SMALL");
#endif
		goto deactivate_entity;
	}
	if (0) deactivate_entity: {
		QUOTA *pzap;
#if __DEBUG >= 1
		if (__unlikely(e->wired != p))
			KERNEL$SUICIDE("ZAP_WIRED_ENTITIES: BAD ENTITY ON LIST (%p != %p)", e->wired, p);
#endif
		QFREE(&p->wireq, 1, proc_wireq_isroot, proc_wireq_parent, Q_NULL_CALL);
		QALLOC(&p->parent->wireq, 1, proc_wireq_isroot, proc_wireq_parent, Q_NULL_CALL, pzap, {
			KERNEL$SUICIDE("ZAP_WIRED_ENTITIES: QALLOC FAILED");
		});
		e->wired = p->parent;
		DEL_FROM_LIST(&e->wire_list);
		if (!(e->wired_clock & CLOCK_SMALL_LIST)) {
			ADD_TO_LIST_END(&p->parent->vm_wired, &e->wire_list);
		} else {
			p->vm_wired_small_n--;
			e->wired_clock = (p->parent->vm_wired_small_clock += CLOCK_ADD) & ~CLOCK_NOT_TOUCHED;
			ADD_TO_LIST_END(&p->parent->vm_wired_small, &e->wire_list);
			p->parent->vm_wired_small_n++;
			if (__unlikely(p->parent->vm_wired_small_n > VM_WIRED_STREAM_QUEUE)) {
				BALANCE_WIRED_SMALL(p->parent);
			}
		}
		goto next_entity;
	}
	DEL_FROM_LIST(&p->delayed_balance);
	VOID_LIST_ENTRY(&p->delayed_balance);
#if __DEBUG >= 1
	if (__unlikely(p->vm_wired_small_n))
		KERNEL$SUICIDE("ZAP_WIRED_ENTITIES: VM_WIRED_SMALL_N LEAKED (%ld)", p->vm_wired_small_n);
	if (__unlikely(p->wireq.q_node_usage))
		KERNEL$SUICIDE("ZAP_WIRED_ENTITIES: WIREQ.Q_NODE_USAGE LEAKED (%ld)", p->wireq.q_node_usage);
#endif
	CHECK_CACHE_QUEUES(p);
	LOWER_SPL(SPL_DEV);
}

DECL_TIMER(wire_timer);
static AST wire_ast;

static void WIRE_TIMER(TIMER *t)
{
	CALL_AST(&wire_ast);
}

static DECL_AST(WIRE_AST, SPL_DEV, AST)
{
	unsigned deact;
	RAISE_SPL(SPL_CACHE);
	CHECK_CACHE_QUEUES(NULL);
	deact = (current_wirewait + 1) & (VM_WIRESTEPS - 1);
	while (1) {
		int r;
		PROC *pw;
		VMENTITY *e;
		TEST_SPL(SPL_DEV, SPL_CACHE);
		if (__unlikely(LIST_EMPTY(&VM_WIREWAIT[deact]))) break;
		e = LIST_STRUCT(VM_WIREWAIT[deact].next, VMENTITY, vm_queue);
#if __DEBUG >= 1
		if (__unlikely(!e->wired))
			KERNEL$SUICIDE("WIRE_TIMER: UNWIRED ENTITY ON VM_WIREWAIT");
#endif
		r = ENTITIES[e->type].t.checkmap(e);
		RAISE_SPL(SPL_CACHE);
		if (r) {
			if (e->wired_clock & CLOCK_SMALL_LIST) {
				if (!(e->wired_clock & CLOCK_MAPPED_2ND_PASS)) {
					e->wired_clock |= CLOCK_MAPPED_2ND_PASS;
					DEL_FROM_LIST(&e->vm_queue);
					ADD_TO_LIST_END(&VM_WIREWAIT[current_wirewait], &e->vm_queue);
					DEL_FROM_LIST(&e->wire_list);
					ADD_TO_LIST_END(&e->wired->vm_wired_small, &e->wire_list);
					continue;
				}
			}
			if (e->queue < VM_QUEUES - (VM_MAPPING_WIRED_ADVANCE - 1)) e->queue += VM_MAPPING_WIRED_ADVANCE - 1;
			KERNEL$CACHE_TOUCH_VM_ENTITY(e, e->wired);
			if (__likely(e != LIST_STRUCT(VM_WIREWAIT[deact].next, VMENTITY, vm_queue))) continue;
		}
		DEL_FROM_LIST(&e->vm_queue);
		DEL_FROM_LIST(&e->wire_list);
		e->wired->vm_wired_small_n -= e->wired_clock & CLOCK_SMALL_LIST;
		QFREE(&e->wired->wireq, 1, proc_wireq_isroot, proc_wireq_parent, Q_NULL_CALL);
		pw = e->wired;
		e->wired = NULL;
		VM_QUEUE_INSERT(e, e->queue);
#if __DEBUG == 1
		e->wire_list.next = NULL;
		e->wire_list.prev = NULL;
#endif
		ENTITIES[e->type].t.write(e, pw, 0);
		RAISE_SPL(SPL_CACHE);
	}
	current_wirewait = deact;
	LOWER_SPL(SPL_DEV);
	LOWER_ENTITIES();
	CHECK_CACHE_QUEUES(NULL);
	KERNEL$SET_TIMER(VM_WIREJIFFIES, &wire_timer);
	RETURN;
}

static void LOWER_ENTITIES(void)
{
	unsigned i, j;
	unsigned long t, f, n;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("LOWER_ENTITIES AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_CACHE);
	n = 0;
	for (i = 0; i < VM_QUEUES; i++) n += VM_OBJCOUNT[i];
	n /= VM_QUEUES;
	f = t = 0;
	for (i = 0, j = 1; i < VM_QUEUES - 1; i++) {
		unsigned newq;
		int r;
		TEST_SPL(SPL_DEV, SPL_CACHE);
		f += VM_OBJCOUNT[i];
		t += n;
		while (f < t) {
			VMENTITY *e;
			if (j <= i) j = i + 1;
			while (__unlikely(VM_QUEUE_EMPTY(j))) {
				j++;
				if (__unlikely(j >= VM_QUEUES))
					goto brk;
			}
			e = VM_QUEUE_FIRST(j);
			r = ENTITIES[e->type].t.checkmap(e);
			RAISE_SPL(SPL_CACHE);
			if (__unlikely(r)) {
				newq = j + VM_MAPPING_ADVANCE;
				if (__unlikely(newq >= VM_QUEUES)) {
					if (__unlikely(j == VM_QUEUES - 1)) newq = VM_QUEUES - 2, f++;
					else newq = VM_QUEUES - 1;
				}
			} else {
				newq = i;
				f++;
			}
			VM_QUEUE_REMOVE(e);
			VM_QUEUE_INSERT(e, newq);
			TEST_SPL(SPL_DEV, SPL_CACHE);
			CHECK_CACHE_QUEUES(NULL);
		}
	}
	brk:
	LOWER_SPL(SPL_DEV);
}

static int UNWIRE_MOST_USED(void)
{
	int done_something = 0;
	QUOTA *qz;
	PROC *pz;
	unsigned long c;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV)))
		KERNEL$SUICIDE("UNWIRE_MOST_USED AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_CACHE);
	QZAP(&KERNEL$PROC_KERNEL.wireq, for_all_wireq_subnodes, for_all_wireq_subnodes_tail, qz, 1);
	pz = LIST_STRUCT(qz, PROC, wireq);
	CHECK_CACHE_QUEUES(pz);
	/*__debug_printf("... most used: %s(%d) - %ld\n", pz->jobname, pz->depth, pz->wireq.q_node_usage);*/
	for (c = (pz->wireq.q_node_usage >> 1) + 1; c > 0; c--) {
		LIST_HEAD *list;
		VMENTITY *e;
		TEST_SPL(SPL_DEV, SPL_CACHE);
		if (__likely(pz->wireq.q_node_usage >> 1 >= pz->vm_wired_small_n)) list = &pz->vm_wired;
		else list = &pz->vm_wired_small;
		if (__unlikely(LIST_EMPTY(list))) break;
		e = LIST_STRUCT(list->next, VMENTITY, wire_list);
#if __DEBUG >= 1
		if (__unlikely(e->wired != pz))
			KERNEL$SUICIDE("UNWIRE_MOST_USED: BAD ENTITY ON LIST (%p != %p)", e->wired, pz);
#endif
		UNWIRE_ONE_ENTITY(e, pz, 1);
		done_something = 1;
	}
	CHECK_CACHE_QUEUES(pz);
	LOWER_SPL(SPL_DEV);
	return done_something;
}

unsigned long CACHE_UPCALL(unsigned long need)
{
	unsigned i;
	again:
	RAISE_SPL(SPL_CACHE);
	i = 0;
	while (need) {
		VMENTITY *e;
		int r;
		unsigned newq;
#if __DEBUG >= 1
		const char *name;
#endif
		CHECK_CACHE_QUEUES(NULL);
		TEST_SPL(SPL_DEV, SPL_CACHE);
		if (__unlikely(VM_QUEUE_EMPTY(i))) {
			i++;
			if (i < VM_QUEUES)
				continue;
			LOWER_SPL(SPL_DEV);
			if (__likely(UNWIRE_MOST_USED()))
				goto again;
			goto lower_entities_ret;
		}
		e = VM_QUEUE_FIRST(i);
#if __DEBUG >= 1
		name = ENTITIES[e->type].t.name;
#endif
		r = ENTITIES[e->type].t.pageout(e);
		RAISE_SPL(SPL_CACHE);
#if __DEBUG >= 1
		if (r && __unlikely(!e->vm_queue.next)) {
			KERNEL$SUICIDE("CACHE_UPCALL: PAGER %s RELEASED OBJECT %p BUT RETURNED NON-ZERO: %d", name, e, r);
		}
#endif
		if (__likely(!r)) {
			need--;
		} else if (__likely(r == 1)) {
			VM_QUEUE_REMOVE(e);
			e->queue = VME_Q_FREEING;
			e->vm_queue.next = (void *)1;	/* needed --- some places check for != NULL */
			e->vm_queue.prev = (void *)1;
			IO_IN_PROGRESS++;
			need--;
		} else if (__likely(r == 2)) {
			/*__debug_printf("%s(2)", ENTITIES[e->type].t.name);*/
			if (i != VM_QUEUES - 1) {
				newq = VM_QUEUES - 1;
			} else {
				newq = VM_QUEUES - 2;
			}
			requeue:
			VM_QUEUE_REMOVE(e);
			VM_QUEUE_INSERT(e, newq);
		} else if (__likely(r == 3)) {
			/*__debug_printf("%s(3)", ENTITIES[e->type].t.name);*/
			if (__unlikely(e->queue == i) && __likely(!e->wired)) {
				if (i) newq = 0;
				else newq = 1;
				goto requeue;
			}
		} else {
			KERNEL$SUICIDE("CACHE_UPCALL: UNKNOWN PAGEOUT RETURN VALUE %d, ENTITY %s", r, ENTITIES[e->type].t.name);
		}
		CHECK_CACHE_QUEUES(NULL);
	}
	LOWER_SPL(SPL_DEV);
	lower_entities_ret:
	LOWER_ENTITIES();
	return need;
}

void KERNEL$CACHE_TRANSFER_QUEUE_STATE(VMENTITY *to, VMENTITY *from)
{
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_CACHE)))
		KERNEL$SUICIDE("KERNEL$CACHE_TRANSFER_QUEUE_STATE AT SPL %08X", KERNEL$SPL);
#endif
	if (__unlikely(from->queue == VME_Q_FREEING)) goto ret;
	KERNEL$CACHE_REMOVE_VM_ENTITY(to);
	if (from->wired) {
		int q;
		KERNEL$CACHE_INSERT_VM_ENTITY(to, from->wired, 0);
		q = from->queue + 1;
		if (__unlikely(q >= VM_QUEUES)) q = VM_QUEUES - 1;
		to->queue = q;
		if (!(from->wired_clock & CLOCK_SMALL_LIST)) {
			DEL_FROM_LIST(&to->wire_list);
			ADD_TO_LIST_END(&from->wired->vm_wired, &to->wire_list);
			to->wired_clock = 0;
			to->wired->vm_wired_small_n--;
		}
		CHECK_CACHE_QUEUES(from->wired);
	} else {
		int q;
		CHECK_CACHE_QUEUES(NULL);
		VM_ENTITIES++;
		to->wired = NULL;
		q = from->queue + 1;
		if (__unlikely(q >= VM_QUEUES)) q = VM_QUEUES - 1;
		VM_QUEUE_INSERT(to, q);
		CHECK_CACHE_QUEUES(NULL);
	}
	ret:;
}

__COLD_ATTR__ const char *CACHE_DUMPQ(void)
{
	static char str[__MAX_STR_LEN];
	char *p = str;
	unsigned i;
	for (i = 0; i < VM_QUEUES; i++) {
		_snprintf(p, str + sizeof(str) - p, "%s%lu", i ? "/" : "", VM_OBJCOUNT[i]);
		p = strchr(p, 0);
	}
	return str;
}

__COLD_ATTR__ void CACHE_INIT(void)
{
	int i;
	current_wirewait = 0;
	for (i = 0; i < VM_QUEUES; i++) INIT_LIST(&VM_QUEUE[i]);
	for (i = 0; i < VM_WIRESTEPS; i++) INIT_LIST(&VM_WIREWAIT[i]);
	memset(VM_OBJCOUNT, 0, sizeof VM_OBJCOUNT);
	memset(ENTITIES, 0, sizeof ENTITIES);
	wire_ast.fn = WIRE_AST;
	wire_timer.fn = WIRE_TIMER;
	KERNEL$SET_TIMER(VM_WIREJIFFIES, &wire_timer);
	CHECK_CACHE_QUEUES(NULL);
}

__COLD_ATTR__ unsigned long CACHE_ACTIVE_ENTITIES(void)
{
	int i;
	long n = VM_ENTITIES;
	for (i = 0; i < VM_QUEUES; i++) n -= VM_OBJCOUNT[i];
	if (__unlikely(n < 0)) n = 0;	/* can happen due to races */
	return n;
}
