#include <SPAD/LINK.H>
#include <SYS/TYPES.H>
#include <STRING.H>
#include <SPAD/LIBC.H>
#include <SPAD/LIST.H>

#include <KERNEL/FDE.H>

/* Subroutines needed for unwinding stack frames for exception handling.  */
/* Copyright (C) 1997, 1998, 1999, 2000, 2001, 2002 Free Software Foundation, Inc.
   Contributed by Jason Merrill <jason@cygnus.com>. */

typedef unsigned long _Unwind_Word;
typedef long _Unwind_Sword;
typedef _Unwind_Word _Unwind_Ptr;

struct dwarf_cie {
	unsigned length;
	int CIE_id;
	unsigned char version;
	unsigned char augmentation[1];
};

struct dwarf_fde {
	unsigned length;
	int CIE_delta;
	char *pc_begin;
	size_t pc_length;
};

struct dwarf_eh_bases {
	void *tbase;
	void *dbase;
	void *func;
};

/* Pointer encodings, from dwarf2.h.  */
#define DW_EH_PE_absptr		0x00
#define DW_EH_PE_omit		0xff

#define DW_EH_PE_uleb128	0x01
#define DW_EH_PE_udata2		0x02
#define DW_EH_PE_udata4		0x03
#define DW_EH_PE_udata8		0x04
#define DW_EH_PE_sleb128	0x09
#define DW_EH_PE_sdata2		0x0A
#define DW_EH_PE_sdata4		0x0B
#define DW_EH_PE_sdata8		0x0C
#define DW_EH_PE_signed		0x08

#define DW_EH_PE_pcrel		0x10
#define DW_EH_PE_textrel	0x20
#define DW_EH_PE_datarel	0x30
#define DW_EH_PE_funcrel	0x40
#define DW_EH_PE_aligned	0x50

#define DW_EH_PE_indirect	0x80

static const unsigned char *
read_uleb128 (const unsigned char *p, _Unwind_Word *val)
{
  unsigned int shift = 0;
  unsigned char byte;
  _Unwind_Word result;

  result = 0;
  do
    {
      byte = *p++;
      result |= (byte & 0x7f) << shift;
      shift += 7;
    }
  while (byte & 0x80);

  *val = result;
  return p;
}

static const unsigned char *
read_sleb128 (const unsigned char *p, _Unwind_Sword *val)
{
  unsigned int shift = 0;
  unsigned char byte;
  _Unwind_Word result;

  result = 0;
  do
    {
      byte = *p++;
      result |= (byte & 0x7f) << shift;
      shift += 7;
    }
  while (byte & 0x80);

  /* Sign-extend a negative value.  */
  if (shift < 8 * sizeof(result) && (byte & 0x40) != 0)
    result |= -(1L << shift);

  *val = (_Unwind_Sword) result;
  return p;
}

static const unsigned char *
read_encoded_value_with_base (unsigned char encoding, _Unwind_Ptr base,
			      const unsigned char *p, _Unwind_Ptr *val)
{
  union unaligned
    {
      void *ptr;
      unsigned u2 __attribute__ ((mode (HI)));
      unsigned u4 __attribute__ ((mode (SI)));
      unsigned u8 __attribute__ ((mode (DI)));
      signed s2 __attribute__ ((mode (HI)));
      signed s4 __attribute__ ((mode (SI)));
      signed s8 __attribute__ ((mode (DI)));
    } __attribute__((__packed__));

  union unaligned *u = (union unaligned *) p;
  _Unwind_Ptr result;

  if (encoding == DW_EH_PE_aligned)
    {
      _Unwind_Ptr a = (_Unwind_Ptr)p;
      a = (a + sizeof (void *) - 1) & - sizeof(void *);
      result = *(_Unwind_Ptr *) a;
      p = (const unsigned char *)(a + sizeof (void *));
    }
  else
    {
      switch (encoding & 0x0f)
	{
	case DW_EH_PE_absptr:
	  result = (_Unwind_Ptr) u->ptr;
	  p += sizeof (void *);
	  break;

	case DW_EH_PE_uleb128:
	  {
	    unsigned int shift = 0;
	    unsigned char byte;

	    result = 0;
	    do
	      {
		byte = *p++;
		result |= (_Unwind_Ptr)(byte & 0x7f) << shift;
		shift += 7;
	      }
	    while (byte & 0x80);
	  }
	  break;

	case DW_EH_PE_sleb128:
	  {
	    unsigned int shift = 0;
	    unsigned char byte;

	    result = 0;
	    do
	      {
		byte = *p++;
		result |= (_Unwind_Ptr)(byte & 0x7f) << shift;
		shift += 7;
	      }
	    while (byte & 0x80);

	    if (shift < 8 * sizeof(result) && (byte & 0x40) != 0)
	      result |= -(1L << shift);
	  }
	  break;

	case DW_EH_PE_udata2:
	  result = u->u2;
	  p += 2;
	  break;
	case DW_EH_PE_udata4:
	  result = u->u4;
	  p += 4;
	  break;
	case DW_EH_PE_udata8:
	  result = u->u8;
	  p += 8;
	  break;

	case DW_EH_PE_sdata2:
	  result = u->s2;
	  p += 2;
	  break;
	case DW_EH_PE_sdata4:
	  result = u->s4;
	  p += 4;
	  break;
	case DW_EH_PE_sdata8:
	  result = u->s8;
	  p += 8;
	  break;

	default:
	  KERNEL$SUICIDE("read_encoded_value_with_base: invalid encoding");
	}

      if (result != 0)
	{
	  result += ((encoding & 0x70) == DW_EH_PE_pcrel
		     ? (_Unwind_Ptr)u : base);
	  if (encoding & DW_EH_PE_indirect)
	    result = *(_Unwind_Ptr *)result;
	}
    }

  *val = result;
  return p;
}

static unsigned int
size_of_encoded_value (unsigned char encoding)
{
  if (encoding == DW_EH_PE_omit)
    return 0;

  switch (encoding & 0x07)
    {
    case DW_EH_PE_absptr:
      return sizeof (void *);
    case DW_EH_PE_udata2:
      return 2;
    case DW_EH_PE_udata4:
      return 4;
    case DW_EH_PE_udata8:
      return 8;
    }
  KERNEL$SUICIDE("size_of_encoded_value: invalid encoding");
}

static int
get_cie_encoding (struct dwarf_cie *cie)
{
  const unsigned char *aug, *p;

  aug = cie->augmentation;
  if (aug[0] != 'z')
    return DW_EH_PE_absptr;

  p = aug + strlen ((char *)aug) + 1;	/* Skip the augmentation string.  */
  p = read_uleb128 (p, (void *)&KERNEL$LIST_END); /* Skip code alignment.  */
  p = read_sleb128 (p, (void *)&KERNEL$LIST_END); /* Skip data alignment.  */
  p++;					/* Skip return address column.  */

  aug++;				/* Skip 'z' */
  p = read_uleb128 (p, (void *)&KERNEL$LIST_END); /* Skip augmentation length.  */
  while (1)
    {
      /* This is what we're looking for.  */
      if (*aug == 'R')
	return *p;
      /* Personality encoding and pointer.  */
      else if (*aug == 'P')
	{
	  /* ??? Avoid dereferencing indirect pointers, since we're
	     faking the base address.  Gotta keep DW_EH_PE_aligned
	     intact, however.  */
	  p = read_encoded_value_with_base (*p & 0x7F, 0, p + 1, (void *)&KERNEL$LIST_END);
	}
      /* LSDA encoding.  */
      else if (*aug == 'L')
	p++;
      /* Otherwise end of string, or unknown augmentation.  */
      else
	return DW_EH_PE_absptr;
      aug++;
    }
}

static _Unwind_Ptr
base_from_object (unsigned char encoding, __f_off tbase, __f_off dbase)
{
  if (encoding == DW_EH_PE_omit)
    return 0;

  switch (encoding & 0x70)
    {
    case DW_EH_PE_absptr:
    case DW_EH_PE_pcrel:
    case DW_EH_PE_aligned:
      return 0;

    case DW_EH_PE_textrel:
      return (_Unwind_Ptr) tbase;
    case DW_EH_PE_datarel:
      return (_Unwind_Ptr) dbase;
    }
  KERNEL$SUICIDE("base_from_object: invalid encoding");
}

struct dwarf_fde *FIND_FDE(void *pc, struct dwarf_eh_bases *bases, __f_off start, __f_off end, __f_off tbase, __f_off dbase)
{
#define fde ((struct dwarf_fde *)start)
	struct dwarf_cie *cie, *last_cie = NULL;
	_Unwind_Ptr base = 0, func;
	int encoding = DW_EH_PE_absptr;
	bases->tbase = (void *)tbase;
	bases->dbase = (void *)dbase;
		/*__debug_printf("pc: %x, tbase %x, dbase %x\n", pc, tbase, dbase);*/
	while (start + sizeof(struct dwarf_fde) <= end) {
		if (__unlikely(!fde->CIE_delta)) goto skip;
		cie = (struct dwarf_cie *)((char *)&fde->CIE_delta - fde->CIE_delta);
		if (__unlikely(cie != last_cie)) {
			last_cie = cie;
			encoding = get_cie_encoding(cie);
			base = base_from_object (encoding, tbase, dbase);
			/*__debug_printf("encoding: %x\n", encoding);*/
		}
		if (__likely(encoding == DW_EH_PE_absptr)) {
			/*__debug_printf("%x - %x\n", fde->pc_begin, fde->pc_length);*/
			if (__likely(fde->pc_begin > (char *)pc)) goto skip;
			if (__likely(fde->pc_begin + fde->pc_length > (char *)pc)) {
				goto found;
			}
		} else {
			_Unwind_Ptr pc_begin, pc_range, mask;
			const unsigned char *p;
			p = read_encoded_value_with_base(encoding, base, (const unsigned char *)&fde->pc_begin, &pc_begin);
			read_encoded_value_with_base(encoding & 0x0F, 0, p, &pc_range);
			mask = size_of_encoded_value(encoding);
			if (mask < sizeof (void *)) mask = (1L << (mask << 3)) - 1;
			else mask = -1;
			/*__debug_printf("%x - %x\n", pc_begin, pc_range);*/
			if (__unlikely((pc_begin & mask) == 0)) {
				/*__debug_printf("skp\n");*/
				goto skip;
			}
			if (__unlikely((_Unwind_Ptr)pc - pc_begin < pc_range))
				goto found;
		}
		skip:
		start = (__f_off)&fde->CIE_delta + fde->length;
	}
			/*__debug_printf("not found\n");*/
	return NULL;

	found:
			/*__debug_printf("found\n");*/
	read_encoded_value_with_base(encoding, base_from_object(encoding, tbase, dbase), (unsigned char *)&fde->pc_begin, &func);
	bases->func = (void *)func;
	return fde;
#undef fde
}
