#include <STRING.H>
#include <STDLIB.H>
#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <ARCH/IO.H>
#include <KERNEL/ASM.H>

#include <SPAD/IRQ.H>
#include <KERNEL/IRQ.H>

struct irq_controller IRQC = { 0 };

#define MAX_SHARED_IRQS	15

struct irq_handler {
	int flags;
	IRQ_STUB *call;
	void *data;
	const char *desc;
};

struct irq {
	IRQ_HANDLE *handle;
	AST multi_irq_ast;
	AST *multi_irq_asts[MAX_SHARED_IRQS + 1];
	struct irq_handler handlers[MAX_SHARED_IRQS + 1];
};

__COLD_ATTR__ void KERNEL$DI_HW(void)
{
	IRQC.di_hw();
}

__COLD_ATTR__ void KERNEL$EI_HW(void)
{
	IRQC.ei_hw();
}

static struct irq **irqs;

__COLD_ATTR__ static DECL_RT_IRQ_HANDLER(UNHANDLED_IRQ)
{
	int irq = (int)DATA;
	KERNEL$SUICIDE("UNHANDLED_IRQ: IRQ %d", irq);
	IRQ_RETURN;
}

static DECL_RT_IRQ_HANDLER(DISPATCH_SHIRQ)
{
	struct irq *irq = DATA;
	struct irq_handler *ih = &irq->handlers[0];
	AST **astp = &irq->multi_irq_asts[0];
	do {
		if (__unlikely(!(ih->flags & IRQ_REQUEST_AST_HANDLER))) {
			AST *ast;
			__u32 dummy1, dummy2;
			__asm__ volatile ("CALL *%3":"=a"(ast),"=d"(dummy1),"=c"(dummy2):"r"(ih->call),"a"(ih->data):"cc","memory");
			if (ast) *astp++ = ast;
		} else {
			KERNEL_MASK_IRQ_DI(irq->handle);
			*astp++ = ih->data;
		}
	} while ((++ih)->flags);
	if (__likely(astp != &irq->multi_irq_asts[0])) {
		KERNEL_MASK_IRQ_DI(irq->handle);
		*astp = NULL;
		IRQ_POST_AST(&irq->multi_irq_ast);
	}
	IRQ_RETURN;
}

static DECL_IRQ_AST(MULTI_IRQ_AST, SPL_ALMOST_TOP, AST)
{
	struct irq *irq = GET_STRUCT(RQ, struct irq, multi_irq_ast);
	AST **ast;
	AST *ret;
	for (ast = &irq->multi_irq_asts[1]; *ast; ast++)
		CALL_AST(*ast);
	ret = irq->multi_irq_asts[0];
	KERNEL$UNMASK_IRQ(irq->handle);
	RETURN_AST(ret);
}

__COLD_ATTR__ static int N_HANDLERS(struct irq *irq)
{
	int n;
	for (n = 0; irq->handlers[n].flags; n++)
		if (__unlikely((unsigned)n >= MAX_SHARED_IRQS))
			KERNEL$SUICIDE("N_HANDLERS: HANDLER LIST NOT TERMINATED");
	return n;
}

__COLD_ATTR__ int KERNEL$REQUEST_IRQ(int n, IRQ_HANDLE **handle, int flags, IRQ_STUB *call, void *data, const char *desc)
{
	struct irq *irq;
	struct irq_handler *h;
	int n_handlers;
	int spl;
	int enable;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), spl = KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$REQUEST_IRQ AT SPL %08X", KERNEL$SPL);
	if (__unlikely(!((!(flags & IRQ_REQUEST_EXCLUSIVE) ^ !(flags & IRQ_REQUEST_SHARED)) &
			 (!(flags & IRQ_REQUEST_RT_HANDLER) ^ !(flags & IRQ_REQUEST_AST_HANDLER)))))
		inval:KERNEL$SUICIDE("KERNEL$REQUEST_IRQ: INVALID PARAMETERS: IRQ %d, FLAGS %X, CALL %p, DATA %p, DEVICE %s", n, flags, call, data, desc);
	if (__likely(flags & IRQ_REQUEST_AST_HANDLER) && __unlikely(call))
		goto inval;
	if (__unlikely((unsigned)n >= IRQC.n_irqs) || __unlikely(!irqs[n]))
		return -ERANGE;
	irq = irqs[n];
	RAISE_SPL(SPL_VSPACE);
	if (irq->handlers[0].flags & IRQ_REQUEST_EXCLUSIVE ||
	    (irq->handlers[0].flags && flags & IRQ_REQUEST_EXCLUSIVE)) {
		LOWER_SPLX(spl);
		return -EBUSY;
	}
	n_handlers = N_HANDLERS(irq);
	if (__unlikely(n_handlers >= MAX_SHARED_IRQS)) {
		LOWER_SPLX(spl);
		return -EOVERFLOW;
	}
	h = &irq->handlers[n_handlers];
	KERNEL$DI();
	h->flags = flags;
	h->call = call;
	h->data = data;
	h[1].flags = 0;
	enable = 0;
	if (!n_handlers) {
		if (!(flags & IRQ_REQUEST_AST_HANDLER)) IRQC.irq_request_rt(n, call, data);
		else IRQC.irq_request_ast(n, data);
		if (!(flags & IRQ_REQUEST_MASKED)) enable = 1;
		IRQC.irq_startup(n);
	} else {
		IRQC.irq_request_rt(n, &DISPATCH_SHIRQ, (void *)irq);
		if (flags & IRQ_REQUEST_MASKED) KERNEL$MASK_IRQ(irq->handle);
	}
	*handle = irq->handle;	/* this must be set before we actually enable the interrupt */
	KERNEL$EI();
	if (enable) KERNEL$UNMASK_IRQ(irq->handle);
	LOWER_SPLX(spl);
	return 0;
}

__COLD_ATTR__ void KERNEL$RELEASE_IRQ(IRQ_HANDLE *ctrl, int flags, IRQ_STUB *call, void *data)
{
	struct irq *irq;
	struct irq_handler *h;
	int n_handlers;
	int spl;
	int i, n, eoi;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_VSPACE), spl = KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$RELEASE_IRQ AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_VSPACE);
	for (n = 0; n < IRQC.n_irqs; n++) if (__likely(irqs[n] != NULL)) {
		irq = irqs[n];
		if (__likely(irq->handle != ctrl))
			continue;
		n_handlers = N_HANDLERS(irq);
		for (i = 0; i < n_handlers; i++) {
			h = &irq->handlers[i];
			if (__likely((h->flags & (IRQ_REQUEST_RT_HANDLER | IRQ_REQUEST_AST_HANDLER)) == (flags & (IRQ_REQUEST_RT_HANDLER | IRQ_REQUEST_AST_HANDLER))) && __likely(h->call == call) && __likely(h->data == data))
				goto found;
		}
	}
	KERNEL$SUICIDE("KERNEL$RELEASE_IRQ: RELEASING IRQ THAT WAS NOT REGISTERED, FLAGS %X, CALL %p, DATA %p", flags, call, data);
	found:
	KERNEL$DI();
	eoi = 0;
	if (__likely(n_handlers == 1)) {
		IRQC.irq_shutdown(n, flags & IRQ_REQUEST_MASKED);
		IRQC.irq_request_rt(n, &UNHANDLED_IRQ, (void *)n);
	} else {
		if (__unlikely(flags & IRQ_REQUEST_MASKED)) eoi = 1;
	}
		/* move one more --- the terminating 0 entry */
	memmove(h, h + 1, sizeof(struct irq_handler) * (n_handlers - i));
	KERNEL$EI();
	if (eoi)
		KERNEL$UNMASK_IRQ(irq->handle);
	LOWER_SPLX(spl);
}

__COLD_ATTR__ void IRQ_INIT(void)
{
	int n;
	irqs = malloc(IRQC.n_irqs * sizeof(struct irq *));
	if (!irqs) {
		__critical_printf("CAN'T ALLOCATE MEMORY FOR IRQ POINTERS\n");
		HALT_KERNEL();
	}
	for (n = 0; n < IRQC.n_irqs; n++) {
		IRQ_HANDLE *h = IRQC.irq_get_handle(n);
		if (!h) {
			irqs[n] = NULL;
		} else {
			struct irq *irq = malloc(sizeof(struct irq));
			if (!irq) {
				__critical_printf("CAN'T ALLOCATE MEMORY FOR IRQ\n");
				HALT_KERNEL();
			}
			irqs[n] = irq;
			irq->handle = h;
			irq->multi_irq_ast.fn = MULTI_IRQ_AST;
			irq->handlers[0].flags = 0;
			IRQC.irq_request_rt(n, &UNHANDLED_IRQ, (void *)n);
		}
	}
	KERNEL$EI();
}

