#include <SPAD/VM.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/LIST.H>
#include <STRING.H>
#include <SPAD/DIR_KRNL.H>
#include <KERNEL/ASM.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/PROC.H>
#include <TIME.H>
#include <STDLIB.H>
#include <SPAD/HASH.H>
#include <SYS/STAT.H>
#include <SPAD/CAPA.H>

#include <KERNEL/FSDLDR.H>
#include <KERNEL/MFS.H>

#define MAX_FILENAME	47
#define MAX_SIZE	(0x80000000 - PAGE_CLUSTER_SIZE)
#define PAGE_HASH	1024
#define FILE_HASH	64

typedef struct {
	LIST_ENTRY hash;
	unsigned size;
	XLIST_HEAD handles;
	char filename[MAX_FILENAME + 1];
} MFSFILE;

static XLIST_HEAD page_hash[PAGE_HASH];
static XLIST_HEAD file_hash[FILE_HASH];

#define HASH_PAGE(file, offset)	(((unsigned long)(file) / 64 * 7 + (unsigned long)(file) / (64 * PAGE_HASH) * 9 + ((offset) >> (PG_SIZE_BITS + PG_CLUSTER_BITS))) & (PAGE_HASH - 1))

static PAGE *find_page(MFSFILE *f, unsigned off)
{
	XLIST_HEAD *l = &page_hash[HASH_PAGE(f, off)];
	PAGE *p;
	off &= ~(PAGE_CLUSTER_SIZE - 1);
	XLIST_FOR_EACH(p, l, PAGE, hash_entry) {
		if (__likely(p->fnode == f) && __likely(p->id == off)) {
			if (__unlikely(l->next != &p->hash_entry)) {
				DEL_FROM_LIST(&p->hash_entry);
				ADD_TO_XLIST(l, &p->hash_entry);
			}
			return p;
		}
	}
	KERNEL$SUICIDE("find_page: FILE %s, SIZE %u, PAGE %u", f->filename, f->size, off);
}

static void test_page_not_exists(MFSFILE *f, unsigned off)
{
#if __DEBUG >= 2
	XLIST_HEAD *l = &page_hash[HASH_PAGE(f, off)];
	PAGE *p;
	off &= ~(PAGE_CLUSTER_SIZE - 1);
	XLIST_FOR_EACH_UNLIKELY(p, l, PAGE, hash_entry) {
		if (__unlikely(p->fnode == f) && __unlikely(p->id == off)) {
			KERNEL$SUICIDE("test_page_not_exists: PAGE EXISTS: FILE %s, SIZE %u, PAGE %u", f->filename, f->size, off);
		}
	}
#endif
}

static const HANDLE_OPERATIONS mfs_dir_operations;
static const HANDLE_OPERATIONS mfs_file_operations;

static IO_STUB FILE_PAGE_RELEASE;

static __finline__ void SETUP_MFS_PAGE(PAGE *p)
{
	KERNEL$VM_PREPARE_PAGE_FOR_MMAP(p);
	p->release = FILE_PAGE_RELEASE;
}

static DECL_IOCALL(FILE_PAGE_RELEASE, SPL_FS, PAGE_RELEASE_REQUEST)
{
	PAGE *p1, *p2;
	void *v1, *v2;
	WQ *wq;
	p1 = RQ->pg;
	if (__unlikely(p1->release != FILE_PAGE_RELEASE)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_PAGE_RELEASE);
	if (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(p1)) != NULL)) {
		WQ_WAIT(wq, RQ, KERNEL$WAKE_PAGE_RELEASE);
		RETURN;
	}
	p2 = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED);
	if (__unlikely(!p2)) {
		if (__unlikely(KERNEL$OOM(VM_TYPE_WIRED_UNMAPPED))) {
			RQ->status = -ENOMEM;
			RETURN_AST(RQ);
		}
		WQ_WAIT(&KERNEL$FREEMEM_WAIT, RQ, KERNEL$WAKE_PAGE_RELEASE);
		RETURN;
	}
	LOWER_SPL(SPL_FS);
	SETUP_MFS_PAGE(p2);
	p2->fnode = p1->fnode;
	p2->id = p1->id;
	ADD_TO_LIST_BEFORE(&p1->hash_entry, &p2->hash_entry);
	DEL_FROM_LIST(&p1->hash_entry);
	v1 = KERNEL$MAP_PHYSICAL_PAGE(p1);
	v2 = KERNEL$MAP_PHYSICAL_PAGE(p2);
	memcpy(v2, v1, PAGE_CLUSTER_SIZE);
	KERNEL$UNMAP_PHYSICAL_BANK(v1);
	KERNEL$UNMAP_PHYSICAL_BANK(v2);
	KERNEL$FREE_USER_PAGE(p1, VM_TYPE_WIRED_UNMAPPED);
	RQ->status = 0;
	RETURN_AST(RQ);
}

static void mfs_kill_handles(MFSFILE *f)
{
	while (!XLIST_EMPTY(&f->handles)) {
		HANDLE *h = LIST_STRUCT(f->handles.next, HANDLE, fnode_entry);
		KERNEL$DETACH_HANDLE(h);
	}
}

static WQ *mfs_extend_file(MFSFILE *f, unsigned newsize, int one_more)
{
	PAGE *p;
	char *pp;
	if (__unlikely(newsize + one_more > MAX_SIZE)) return __ERR_PTR(-EFBIG);
	if (f->size > newsize) return NULL;

	while ((f->size + PAGE_CLUSTER_SIZE - 1) >> (PG_SIZE_BITS + PG_CLUSTER_BITS) < ((newsize + PAGE_CLUSTER_SIZE - 1 + one_more) >> (PG_SIZE_BITS + PG_CLUSTER_BITS))) {
		p = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED);
		if (!p) {
			if (__unlikely(KERNEL$OOM(VM_TYPE_WIRED_UNMAPPED))) {
				return __ERR_PTR(-ENOMEM);
			}
			return &KERNEL$FREEMEM_WAIT;
		}
		SETUP_MFS_PAGE(p);
		p->fnode = f;
		p->id = (f->size + PAGE_CLUSTER_SIZE - 1) & ~(PAGE_CLUSTER_SIZE - 1);
		test_page_not_exists(f, p->id);
		ADD_TO_XLIST(&page_hash[HASH_PAGE(f, (unsigned long)p->id)], &p->hash_entry);
		pp = KERNEL$MAP_PHYSICAL_PAGE(p);
		memset(pp, 0, PAGE_CLUSTER_SIZE);
		KERNEL$UNMAP_PHYSICAL_BANK(pp);
		f->size = (unsigned long)p->id + PAGE_CLUSTER_SIZE;
	}
	f->size = newsize;
	return NULL;
}

static WQ *mfs_truncate_file(MFSFILE *f, unsigned newsize)
{
	unsigned offs;
	if (__unlikely(newsize >= f->size)) return NULL;
	for (offs = (newsize + PAGE_CLUSTER_SIZE - 1) & ~(PAGE_CLUSTER_SIZE - 1); offs < f->size; offs += PAGE_CLUSTER_SIZE) {
		PAGE *p = find_page(f, offs);
		WQ *wq;
		if (__unlikely((wq = KERNEL$VM_UNMAP_PAGE(p)) != NULL)) {
			return wq;
		}
		LOWER_SPL(SPL_FS);
	}
	for (offs = (newsize + PAGE_CLUSTER_SIZE - 1) & ~(PAGE_CLUSTER_SIZE - 1); offs < f->size; offs += PAGE_CLUSTER_SIZE) {
		PAGE *p = find_page(f, offs);
		DEL_FROM_LIST(&p->hash_entry);
		KERNEL$FREE_USER_PAGE(p, VM_TYPE_WIRED_UNMAPPED);
	}
	if (__unlikely((f->size = newsize) & (PAGE_CLUSTER_SIZE - 1))) {
		char *pp = KERNEL$MAP_PHYSICAL_PAGE(find_page(f, newsize - 1));
		memset(pp + (newsize & (PAGE_CLUSTER_SIZE - 1)), 0, PAGE_CLUSTER_SIZE - (newsize & (PAGE_CLUSTER_SIZE - 1)));
		KERNEL$UNMAP_PHYSICAL_BANK(pp);
	}
	return NULL;
}

static void *mfs_lookup(HANDLE *h, char *str, int open_flags)
{
	unsigned hash;
	XLIST_HEAD *l;
	char *e;
	MFSFILE *f;
	if (__unlikely(str[0] == '^')) return __ERR_PTR(-EBADMOD);
	if (__unlikely(!str[0])) return __ERR_PTR(-ENOENT);
	e = str;
	hash = 0;
	quickcasehash(e, *e, hash);
	l = &file_hash[hash & (FILE_HASH - 1)];
	XLIST_FOR_EACH(f, l, MFSFILE, hash) if (__likely(!__strcasexcmp(f->filename, str, e))) {
		if (__unlikely(l->next != &f->hash)) {
			DEL_FROM_LIST(&f->hash);
			ADD_TO_XLIST(l, &f->hash);
		}
		h->fnode = f;
		h->op = &mfs_file_operations;
		return NULL;
	}
	return __ERR_PTR(-ENOENT);
}

static void *mfs_create(HANDLE *h, char *str, int open_flags)
{
	unsigned hash;
	XLIST_HEAD *l;
	char *e;
	MFSFILE *f;
	if (str[0] == '^') return __ERR_PTR(-EINVAL);
	if (!str[0]) return __ERR_PTR(-EINVAL);
	if (strlen(str) > MAX_FILENAME) return __ERR_PTR(-ENAMETOOLONG);
	e = str;
	hash = 0;
	quickcasehash(e, *e, hash);
	l = &file_hash[hash & (FILE_HASH - 1)];
	XLIST_FOR_EACH_UNLIKELY(f, l, MFSFILE, hash) if (__unlikely(!__strcasexcmp(f->filename, str, e)))
		return __ERR_PTR(-EEXIST);
	if (__unlikely(open_flags & _O_MKDIR)) return __ERR_PTR(-ENOOP);

	if (__unlikely(!(f = malloc(sizeof(MFSFILE))))) {
		if (__unlikely(KERNEL$OOM(VM_TYPE_WIRED_MAPPED)))
			return __ERR_PTR(-ENOMEM);
		return &KERNEL$FREEMEM_WAIT;
	}
	ADD_TO_XLIST(l, &f->hash);
	f->size = 0;
	INIT_XLIST(&f->handles);
	strcpy(f->filename, str);
	__upcase(f->filename);
	h->fnode = f;
	h->op = &mfs_file_operations;
	return NULL;
}

static void *mfs_delete(HANDLE *h, IORQ *rq, int open_flags, HANDLE *hp)
{
	WQ *wq;
	MFSFILE *f = h->fnode;
	if (__unlikely(f == hp->fnode)) return __ERR_PTR(-EBUSY);
	if (__unlikely(open_flags & (_O_MKDIR | _O_DELETE_PROC))) return __ERR_PTR(-ENOTDIR);
	mfs_kill_handles(f);
	if (__unlikely((wq = mfs_truncate_file(f, 0)) != NULL)) {
		WQ_WAIT_F(wq, rq);
		LOWER_SPL(SPL_FS);
		return (void *)3;
	}
#if __DEBUG >= 3
	{
		unsigned u;
		PAGE *p;
		for (u = 0; u < PAGE_HASH; u++) XLIST_FOR_EACH_UNLIKELY(p, &page_hash[u], PAGE, hash_entry) if (__unlikely(p->fnode == f)) KERNEL$SUICIDE("mfs_delete: PAGE %lu LEAKED WHEN DELETING FILE %s", (unsigned long)p->id, f->filename);
	}
#endif
	DEL_FROM_LIST(&f->hash);
	free(f);
	return NULL;
}

static void *mfs_instantiate(HANDLE *h, IORQ *rq, int open_flags)
{
	MFSFILE *f = h->fnode;
	if (__unlikely(open_flags & O_TRUNC)) {
		WQ *wq;
		if (__unlikely(!(open_flags & _O_KRNL_WRITE))) return __ERR_PTR(-EPERM);
		if (__unlikely((wq = mfs_truncate_file(f, 0)) != NULL)) return wq;
	}
	ADD_TO_XLIST(&f->handles, &h->fnode_entry);
	return NULL;
}

static void mfs_detach(HANDLE *h)
{
	DEL_FROM_LIST(&h->fnode_entry);
	h->fnode = NULL;
}

static void mfs_init_root(HANDLE *h, void *fs_)
{
	h->fnode = NULL;
	h->op = &mfs_dir_operations;
}

static DECL_IOCALL(mfs_isdir, SPL_FS, IORQ)
{
	RQ->status = -EISDIR;
	RETURN_AST(RQ);
}

#define sio
#include "MFSIO.I"

#undef sio
#include "MFSIO.I"

static __finline__ void VFS_FILLSTAT(struct stat *stat, MFSFILE *f)
{
	if (f) {
		/*stat->st_ino = (unsigned long)f;*/
		stat->st_mode = 0644 | S_IFREG;
		stat->st_nlink = 1;
		stat->st_size = f->size;
		stat->st_blksize = PAGE_CLUSTER_SIZE;
		stat->st_blocks = (unsigned)(f->size + PAGE_CLUSTER_SIZE - 1) / PAGE_CLUSTER_SIZE * (PAGE_CLUSTER_SIZE / 512);
		stat->st_atime = stat->st_mtime = stat->st_ctime = time(NULL);
	} else {
		/*stat->st_ino = 2;*/
		stat->st_mode = 0755 | S_IFDIR;
		stat->st_nlink = 1;
		stat->st_blksize = PAGE_CLUSTER_SIZE;
		stat->st_atime = stat->st_mtime = stat->st_ctime = time(NULL);
	}
}

#define VFS_FN_SPL	SPL_FS

#include <SPAD/VFS_FN.H>

static void DO_STATFS(IOCTLRQ *RQ)
{
	static struct statfs stat;
	unsigned h;
	MFSFILE *f;
	PAGE *p;
	unsigned long files = 0, pages = 0;
	unsigned long mem[KQM_N_ENTRIES];
	__u64 lowmem;
	int r;
	memset(&stat, 0, sizeof stat);
	if (__unlikely(KERNEL$QUERY_MEMORY(mem))) memset(mem, 0, sizeof mem);
	lowmem = KERNEL$GET_MEMORY_SIZE(VM_TYPE_WIRED_MAPPED);
	for (h = 0; h < FILE_HASH; h++) XLIST_FOR_EACH_UNLIKELY(f, &file_hash[h], MFSFILE, hash) files++;
	for (h = 0; h < PAGE_HASH; h++) XLIST_FOR_EACH_UNLIKELY(p, &page_hash[h], PAGE, hash_entry) pages++;
	stat.f_bsize = PAGE_CLUSTER_SIZE;
	stat.f_iosize = PAGE_CLUSTER_SIZE;
	stat.f_bfree = mem[VM_TYPE_CACHED_MAPPED] + mem[VM_TYPE_CACHED_UNMAPPED] + mem[VM_TYPE_USER_MAPPED] + mem[VM_TYPE_USER_UNMAPPED] + mem[KQM_KERNEL_FREE] + mem[KQM_IO_FREE] + mem[KQM_HIGH_FREE];
	stat.f_blocks = stat.f_bfree + pages;
	stat.f_bavail = stat.f_bfree;
	stat.f_files = lowmem / sizeof(MFSFILE);
	if (files < stat.f_files) stat.f_ffree = stat.f_files - files;
	else stat.f_ffree = 0;
	stat.f_fsid.val[0] = (unsigned long)page_hash;
	stat.f_fsid.val[1] = (unsigned long)file_hash;
	stat.f_namelen = MAX_FILENAME;
	stat.f_flags = MNT_NOATIME;
	strcpy(stat.f_fstypename, "MFS");
	strcpy(stat.f_fsmntonname, "MFS:/");
	strcpy(stat.f_fsmntfromname, "MEMORY");
	memcpy(&stat.f_type, stat.f_fstypename, sizeof stat.f_type);
	if (__unlikely((r = KERNEL$PUT_IOCTL_STRUCT(RQ, &stat, sizeof stat)) == 1)) {
		DO_PAGEIN_NORET(RQ, &RQ->v, PF_WRITE);
		return;
	}
	RQ->status = r;
	CALL_AST(RQ);
	return;
}

static DECL_IOCALL(mfs_dir_ioctl, SPL_FS, IOCTLRQ)
{
	if (__unlikely(RQ->handle->op != &mfs_dir_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_FS));
	switch (RQ->ioctl) {
		case IOCTL_READDIR: {
			int r;
			unsigned h;
			MFSFILE *ptr, *f, *mf;
			struct readdir_buffer *buf;
			struct dirent *de;
			vspace_unmap_t *unmap;
			void *map;
			RAISE_SPL(SPL_VSPACE);
			map = RQ->v.vspace->op->vspace_map(&RQ->v, PF_RW, SPL_X(SPL_FS), &unmap);
			if (__unlikely(!map)) DO_PAGEIN(RQ, &RQ->v, PF_RW);
			if (__unlikely(__IS_ERR(map))) {
				r = __PTR_ERR(map);
				goto ret_0;
			}
			if (__unlikely((int)(unsigned long)map & (__MALLOC_ALIGN - 1)) || __unlikely(RQ->v.len > PAGE_CLUSTER_SIZE)) {
				r = -EINVAL;
				goto ret;
			} else if (__unlikely(RQ->v.len < DIRENT_START_FROM_COOKIE_SIZE(sizeof(int)))) {
				small:
				r = -EMSGSIZE;
				goto ret;
			}
			buf = map;
			buf->dirent_start = DIRENT_START_FROM_COOKIE_SIZE(sizeof(int));
			de = DIR_START(buf);
			if ((char *)de + DIRENT_LENGTH(MAX_FILENAME) > (char *)buf + RQ->v.len) goto small;
			if (buf->cookie_size != 2 * sizeof(unsigned long)) buf->cookie_size = 2 * sizeof(unsigned long), h = 0, ptr = NULL;
			else h = ((unsigned long *)buf->cookie)[0], ptr = (void *)((unsigned long *)buf->cookie)[1];
			if (__unlikely(h >= FILE_HASH)) {
				r = -EINVAL;
				goto ret;
			}
			next_entry:
			mf = NULL;
			XLIST_FOR_EACH(f, &file_hash[h], MFSFILE, hash) {
				if (f > ptr && (!mf || f < mf)) mf = f;
			}
			if (mf) {
				de->d_reclen = DIRENT_LENGTH(de->d_namlen = strlen(mf->filename));
				de->d_type = DT_REG;
				strcpy(de->d_name, mf->filename);
				de = DIR_NEXT(de);
				((unsigned long *)buf->cookie)[0] = h;
				((unsigned long *)buf->cookie)[1] = (unsigned long)mf;
				buf->end = 0;
			} else if (h < FILE_HASH - 1) {
				/*((unsigned long *)buf->cookie)[0] = h + 1;
				((unsigned long *)buf->cookie)[1] = NULL;
				buf->end = 0;*/
				h++;
				ptr = NULL;
				goto next_entry;
			} else {
				buf->end = 1;
			}
			buf->total_size = (char *)de - (char *)buf;
			r = 0;

			ret:
			unmap(map);
			ret_0:
			RQ->status = r;
			RETURN_AST(RQ);
		}

		case IOCTL_STAT: {
			int r;
			r = VFS_STAT(RQ, NULL);
			if (__likely(r <= 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
		}

		case IOCTL_STATFS: {
			DO_STATFS(RQ);
			RETURN;
		}

		case IOCTL_FCNTL_GETFL:
		case IOCTL_FCNTL_SETFL: {
			if (__likely(!VFS_GETSETFL(RQ))) RETURN_AST(RQ);
			RETURN;
		}

		case IOCTL_FSYNC: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}

		case IOCTL_UTIMES: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}

		case IOCTL_GET_CAPABILITIES: {
			RQ->status = CAP_ALL;
			RETURN_AST(RQ);
		}

		default: {
			RQ->status = -EISDIR;
			RETURN_AST(RQ);
		}
	}
}

static DECL_IOCALL(mfs_file_ioctl, SPL_FS, IOCTLRQ)
{
	if (__unlikely(RQ->handle->op != &mfs_file_operations)) RETURN_IORQ_LSTAT(RQ, KERNEL$WAKE_IOCTL);
	RQ->tmp1 = (unsigned long)KERNEL$WAKE_IOCTL;
	TEST_LOCKUP_ENTRY(RQ, RETURN);
	SWITCH_PROC_ACCOUNT(RQ->handle->name_addrspace, SPL_X(SPL_FS));
	switch (RQ->ioctl) {
		case IOCTL_STAT: {
			int r;
			r = VFS_STAT(RQ, RQ->handle->fnode);
			if (__likely(r <= 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			DO_PAGEIN(RQ, &RQ->v, PF_WRITE);
		}

		case IOCTL_STATFS: {
			DO_STATFS(RQ);
			RETURN;
		}

		case IOCTL_LSEEK: {
			int r;
			r = VFS_LSEEK(RQ, ((MFSFILE *)RQ->handle->fnode)->size);
			if (__likely(r <= 0)) {
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (r == 2) RETURN;
			DO_PAGEIN(RQ, &RQ->v, PF_RW);
		}

		case IOCTL_TRUNCATE: {
			int r;
			_u_off_t size;
			WQ *wq;
			MFSFILE *f = RQ->handle->fnode;
			if (__unlikely(r = KERNEL$GET_IOCTL_STRUCT(RQ, &size, sizeof size))) {
				if (r == 1) DO_PAGEIN(RQ, &RQ->v, PF_READ);
				RQ->status = r;
				RETURN_AST(RQ);
			}
			if (__unlikely(size != (unsigned)size)) {
				RQ->status = -EFBIG;
				RETURN_AST(RQ);
			}
			wq = size <= f->size ? mfs_truncate_file(f, size) : mfs_extend_file(f, size, 0);
			if (__unlikely(wq != NULL)) {
				if (__unlikely(__IS_ERR(wq))) {
					RQ->status = __PTR_ERR(wq);
					RETURN_AST(RQ);
				}
				WQ_WAIT_F(wq, RQ);
				LOWER_SPL(SPL_FS);
				RETURN;
			}
			RQ->status = 0;
			RETURN_AST(RQ);
		}

		case IOCTL_FCNTL_GETFL:
		case IOCTL_FCNTL_SETFL: {
			if (__likely(!VFS_GETSETFL(RQ))) RETURN_AST(RQ);
			RETURN;
		}

		case IOCTL_FSYNC: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}

		case IOCTL_UTIMES: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}

		case IOCTL_CAN_MMAP: {
			RQ->status = 0;
			RETURN_AST(RQ);
		}

		case IOCTL_GET_CAPABILITIES: {
			RQ->status = CAP_ALL;
			RETURN_AST(RQ);
		}

		default: {
			RQ->status = -ENOOP;
			RETURN_AST(RQ);
		}
	}
}

static PAGE *MFS_GET_PAGE(HANDLE *h, __v_off idx, int wr)
{
	MFSFILE *f = h->fnode;
	if (__unlikely(wr & ~PF_RW)) return __ERR_PTR(-EINVAL);
	if (__unlikely((idx & ~(__v_off)(PAGE_CLUSTER_SIZE - 1)) >= f->size)) return __ERR_PTR(-ERANGE);
	return find_page(f, idx);
}

static const HANDLE_OPERATIONS mfs_dir_operations = {
	SPL_X(SPL_FS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	KERNEL$NO_VSPACE_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,
	mfs_lookup,
	mfs_create,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	mfs_isdir,
	mfs_isdir,
	mfs_isdir,
	mfs_isdir,
	mfs_dir_ioctl,
	KERNEL$NO_OPERATION,
	KERNEL$NO_OPERATION,
};

static const HANDLE_OPERATIONS mfs_file_operations = {
	SPL_X(SPL_FS),
	KERNEL$NO_VSPACE_GET,
	KERNEL$NO_VSPACE_PUT,
	KERNEL$NO_VSPACE_MAP,
	KERNEL$NO_VSPACE_DMALOCK,
	KERNEL$NO_VSPACE_DMA64LOCK,
	KERNEL$NO_VSPACE_PHYSLOCK,
	KERNEL$NO_VSPACE_GET_PAGEIN_RQ,
	MFS_GET_PAGE,
	KERNEL$NO_VSPACE_SWAP_OP,
	NULL,
	NULL,
	NULL,
	mfs_delete,
	NULL,
	NULL,
	mfs_instantiate,
	NULL,
	mfs_detach,
	NULL,
	mfs_read,
	mfs_write,
	mfs_aread,
	mfs_awrite,
	mfs_file_ioctl,
	KERNEL$BIO_EMU,
	KERNEL$NO_OPERATION,
};

static void MFS_LOAD_FILENAMES(void)
{
	long cookie = 0;
	do {
		char filename[256];
		char err[__MAX_STR_LEN];
		long len;
		int j;
		MFSFILE *f;
		char *fn;
		unsigned hash;
		RESET_HOLD_JIFFIES();
		if (__unlikely(FSD_DIR(filename, &cookie, err))) {
			__critical_printf("UNABLE TO GET LIST OF FILES: %s\n", err);
			HALT_KERNEL();
		}
		if (__unlikely(strlen(filename) >= 256))
			KERNEL$SUICIDE("MFS_LOAD_FILENAMES: LEN %ld", (long)strlen(filename));
		if (__unlikely(strlen(filename) > MAX_FILENAME)) {
			__critical_printf("IGNORING FILE %s: NAME LEN %d > %d\n", filename, (int)strlen(filename), MAX_FILENAME);
			continue;
		}
		/* if (__unlikely(!_strcasecmp(filename, "KERNEL.SYS"))) continue; ... needed for smp */
		len = FSD_OPEN(filename, err);
		if (__unlikely(len < 0)) {
			__critical_printf("ERROR LOADING %s: OPEN: %s\n", filename, err);
			HALT_KERNEL();
		}
		if (__unlikely(len >= MAX_SIZE)) {
			__critical_printf("ERROR LOADING %s: SIZE %ld >= %d\n", filename, len, MAX_SIZE);
			HALT_KERNEL();
		}
		if (__unlikely(!(f = malloc(sizeof(MFSFILE))))) {
			__critical_printf("ERROR LOADING %s: OUT OF MEMORY FOR FILE DESCRIPTOR\n", filename);
			HALT_KERNEL();
		}
		fn = filename;
		hash = 0;
		quickcasehash(fn, *fn, hash);
		hash &= FILE_HASH - 1;
		ADD_TO_XLIST(&file_hash[hash], &f->hash);
		f->size = 0;
		INIT_XLIST(&f->handles);
		strcpy(f->filename, filename);
		__upcase(f->filename);
		for (j = 0; j < len; j += PAGE_CLUSTER_SIZE) {
			PAGE *p;
			void *v;
			unsigned lx;
			RESET_HOLD_JIFFIES();
			p = KERNEL$ALLOC_USER_PAGE(VM_TYPE_WIRED_UNMAPPED);
			if (__unlikely(!p)) {
				__critical_printf("ERROR LOADING %s: OUT OF MEMORY\n", filename);
				HALT_KERNEL();
			}
			SETUP_MFS_PAGE(p);
			p->fnode = f;
			p->id = j;
			test_page_not_exists(f, p->id);
			ADD_TO_XLIST(&page_hash[HASH_PAGE(f, j)], &p->hash_entry);
			v = KERNEL$MAP_PHYSICAL_PAGE(p);
			memset(v, 0, PAGE_CLUSTER_SIZE);
			lx = j + PAGE_CLUSTER_SIZE < len ? PAGE_CLUSTER_SIZE : len - j;
			if (__unlikely(FSD_READ(v, lx, err))) {
				__critical_printf("ERROR LOADING %s: READ: %s\n", filename, err);
				HALT_KERNEL();
			}
			f->size = j + lx;
			KERNEL$UNMAP_PHYSICAL_BANK(v);
		}
		if (__unlikely(FSD_CLOSE(err))) {
			__critical_printf("ERROR LOADING %s: CLOSE: %s\n", filename, err);
			HALT_KERNEL();
		}
	} while (cookie);
	VM_BOOT_GET_MORE_MEM();
}

static void *mfs_lnte;

void MFS_INIT(void)
{
	int i;
	int r;
	for (i = 0; i < PAGE_HASH; i++) INIT_XLIST(&page_hash[i]);
	for (i = 0; i < FILE_HASH; i++) INIT_XLIST(&file_hash[i]);
	MFS_LOAD_FILENAMES();
	r = KERNEL$REGISTER_DEVICE(MFS_DEVICE_NAME, "KERNEL.SYS", 0, NULL, mfs_init_root, NULL, NULL, NULL, NULL, &mfs_lnte, ":DMAN_SKIP", ":DMAN_SKIP_LINKS", NULL);
	if (r < 0) {
		__critical_printf(MFS_DEVICE_NAME ": COULD NOT REGISTER DEVICE: %s\n", strerror(-r));
		HALT_KERNEL();
	}
	r = KERNEL$CREATE_LOGICAL("LIB__MFS", "MFS:/", 0);
	if (r < 0) {
		__critical_printf("LIB__MFS: COULD NOT CREATE LINK TO MFS: %s\n", strerror(-r));
		HALT_KERNEL();
	}
}
