#ifndef __KERNEL_PROC_H
#define __KERNEL_PROC_H

/*
quota:	subproc
	handles
	iorqs
	subphysiorqs
	subvmspace

pri:	sched_pri
	vm_unwiretime
*/

#include <SPAD/PROC.H>
#include <KERNEL/KSPAWN.H>
#include <SYS/TYPES.H>
#include <SPAD/LIST.H>
#include <SPAD/QUOTA.H>
#include <SPAD/DEV_KRNL.H>
#include <SPAD/SCHED.H>
#include <SPAD/TIMER.H>
#include <KERNEL/DEV.H>
#include <SPAD/VM.H>
#include <SPAD/LIBPROC.H>

void ARCH_PROC_CTOR(PROC *proc);
void ARCH_PROC_INIT(PROC *proc, PROC *previos);
void ARCH_PROC_DESTROY(PROC *proc);
void ARCH_PROC_FREE(PROC *proc);

#define DEFAULT_HANDLE_HASH_SIZE	64
#define RESERVED_HANDLES		16
#define RESERVED_UIORQS			8

#define RESERVED_HANDLES_MIN(p)		8
#define RESERVED_UIORQS_MIN(p)		4

#define RESERVED_HANDLES_MAX(p)		16
#define RESERVED_UIORQS_MAX(p)		8

#define FREE_HANDLES_IN_PASS		4
#define FREE_UIORQS_IN_PASS		3

#define ALLOC_FORGET_TIME		(5 * JIFFIES_PER_SECOND)

#define MAX_IO_SCHED			64	/* must be <= 2^16 */

typedef struct {
	PROC *proc;
	IORQ *iorq;	/* points to process addrspace */
	LIST_ENTRY list;
	union {
		IORQ iorq;
		SIORQ siorq;
		AIORQ aiorq;
		IOCTLRQ ioctlrq;
		OPENRQ openrq;
		CLOSERQ closerq;
	} u;
	HANDLE *vspace;
	int sys;
	IORQ pfw;
} UIORQ;

#define LN_CACHE_SIZE		4
#ifdef __32_BIT
#define LN_CACHE_NAMELEN	23
#else
#define LN_CACHE_NAMELEN	19
#endif

typedef struct {
	char name[LN_CACHE_NAMELEN];
	unsigned char touch;
	int h;
	PROC *proc;
} LN_CACHE_ENTRY;

struct __proc {
#include <KERNEL/PROC_HEA.I>
	struct __handle_vsp vspace;

	PROC *parent;
	XLIST_HEAD children;
	LIST_ENTRY child_entry;
	XLIST_HEAD *handles;
	int handle_hash_mask;

	LNT *parent_lnt;
	int ln_mode;
	FFILE **parent_forktable;
	int parent_forktable_n;
	KSPAWNRQ *parent_rq;

	int flags;

	TIMER timer;
	IORQ timer_iorq;
	int timer_iorq_posted;

	QUOTA_R hq;
	QUOTA_R uq;
	WQ free_resources;

	XLIST_HEAD uiorqs;
	int n_reserved_uiorqs;
	XLIST_HEAD reserved_uiorq_list;
	UIORQ reserved_uiorqs[RESERVED_UIORQS];

	int n_reserved_handles;
	XLIST_HEAD reserved_handle_list;
	XLIST_HEAD default_handles[DEFAULT_HANDLE_HASH_SIZE];
	HANDLE reserved_handles[RESERVED_HANDLES];

	LN_CACHE_ENTRY ln_cache[LN_CACHE_SIZE];

	int on_reclaim_list;
	LIST_ENTRY reclaim_list;

	QUOTA_R ioq;
	XLIST_HEAD prefetch;
	WQ ioq_wait;
	int writeback;

	QUOTA pgtblq;
	QUOTA vmq;

	IORQ fault_iorq;
	void *fault_address;
	int fault_access;
	PROC *fault_proc;

	unsigned long alloc_rate;
	u_jiffies_lo_t alloc_rate_time;
	unsigned long vm_wired_small_n;
	unsigned short vm_wired_small_clock;
	LIST_HEAD vm_wired_small;
	LIST_HEAD vm_wired;
	QUOTA wireq;
	LIST_ENTRY delayed_balance;

	SCHED sch __ALIGN_ATTR__(__CPU_CACHELINE_ALIGN > sizeof(unsigned long) * 16 ? sizeof(unsigned long) * 16 : __CPU_CACHELINE_ALIGN);

	VMENTITY vme;

	unsigned long created_alloc_rate;
	u_jiffies_lo_t created_alloc_rate_time;

	QUOTA procq;
	WQ shutdown;

	PROC *fork_child;
	PROC *fork_parent;

	unsigned depth;
	PROC *backptr[MAX_PROC_DEPTH];

	int xcpt_type;
	void *xcpt_address;
	void *xcpt_ip;
	long xcpt_error;

	sched_unsigned iosch_mininvpri;
	sched_unsigned iosch_maxinvpri;
	sched_unsigned ios_unblock_time;
	sched_unsigned ios_accumulated_time;
	IO_SCHED ios[MAX_IO_SCHED];

	char jobname[9];

	long error;
	char error_msg[__MAX_STR_LEN];
#include <KERNEL/PROC_END.I>
};

#define PR_RUNDOWN_1		0x01
#define PR_RUNDOWN_2		0x02
#define PR_RUNDOWN_3		0x03
#define PR_RUNDOWN_4		0x04
#define PR_RUNDOWN_5		0x05
#define PR_RUNDOWN_6		0x06
#define PR_RUNDOWN_FINISH	0x07
#define PR_RUNDOWN		0x0f

#define PR_HARDFAULT		0x10
#define PR_SOFTFAULT		0x20
#define PR_WQ_WAIT		0x30
#define PR_IO_WAIT		0x40
#define PR_SHUTDOWN		0x50
#define PR_NEW			0x60
#define PR_BLOCKREASON		0x70

#define PR_XCPT_OTHERPROC	0x80

#include <KERNEL/PROCARCH.H>

extern PROC *PROC_CURRENT;
extern int PROC_CURRENT_LOCK;
extern WQ PROC_CURRENT_LOCK_WAIT;
extern PROC *PROC_RUN;
extern PROC *PROC_FPU;
extern unsigned KERNEL_HOLD_JIFFIES;

static __finline__ void RESET_HOLD_JIFFIES(void)
{
	KERNEL_HOLD_JIFFIES = 0;
}

extern int PROC_INITIALIZED;

void SET_PROC_CURRENT(PROC *p);
void KERNEL_PROC_INIT(void);
int FOR_ALL_PROCS(PROC *from, int (*fn)(PROC *p));
void DELAYED_SHUTDOWN(void);

HANDLE *PROC_ALLOC_HANDLE(PROC *p, int n, WQ **wq);
void PROC_FREE_HANDLE(HANDLE *h);
UIORQ *PROC_ALLOC_UIORQ(PROC *p, WQ **wq);
void PROC_FREE_UIORQ(UIORQ *u);

int PROC_VM_FAULT(PROC *p, void *addr, int acc);
int PROC_OTHER_VM_FAULT(PROC *p, PROC *pp, void *addr, int acc);
void PROC_WAIT(PROC *p, WQ *wq);
int SYSCALL_RETURN(PROC *p, unsigned long ret);

void __NORET_ATTR__ VM_FAULT_EXCEPTION(void *addr, int wr, void *ip);
void __NORET_ATTR__ USER_EXCEPTION(unsigned long except, void *addr, void *ip);
void DO_END_SYSCALL(void);
void USER_SYSCALL(unsigned long arg1, unsigned long arg2, unsigned long arg3);

#define procq2proc(q)	LIST_STRUCT(q, PROC, procq)
#define hq2proc(q)	LIST_STRUCT(q, PROC, hq)
#define uq2proc(q)	LIST_STRUCT(q, PROC, uq)
#define vmq2proc(q)	LIST_STRUCT(q, PROC, vmq)
#define pgtblq2proc(q)	LIST_STRUCT(q, PROC, pgtblq)
#define ioq2proc(q)	LIST_STRUCT(q, PROC, ioq)
#define wireq2proc(q)	LIST_STRUCT(q, PROC, wireq)
#define sch2proc(q)	LIST_STRUCT(q, PROC, sch)
#define iosch2proc(q)	(LIST_STRUCT(q, IO_SCHED, sch)->p)

#define proc_procq_parent(q)	(&(procq2proc(q)->parent)->procq)
#define proc_procq_isroot(q)	(!(procq2proc(q)->parent))
#define proc_hq_parent(q)	(&(hq2proc(q)->parent)->hq)
#define proc_hq_isroot(q)	(!(hq2proc(q)->parent))
#define proc_uq_parent(q)	(&(uq2proc(q)->parent)->uq)
#define proc_uq_isroot(q)	(!(uq2proc(q)->parent))
#define proc_vmq_parent(q)	(&(vmq2proc(q)->parent)->vmq)
#define proc_vmq_isroot(q)	(!(vmq2proc(q)->parent))
#define proc_pgtblq_parent(q)	(&(pgtblq2proc(q)->parent)->pgtblq)
#define proc_pgtblq_isroot(q)	(!(pgtblq2proc(q)->parent))
#define proc_ioq_parent(q)	(&(ioq2proc(q)->parent)->ioq)
#define proc_ioq_isroot(q)	(!(ioq2proc(q)->parent))
#define proc_wireq_parent(q)	(&(wireq2proc(q)->parent)->wireq)
#define proc_wireq_isroot(q)	(!(wireq2proc(q)->parent))
#define proc_sch_parent(q)	(&(sch2proc(q)->parent)->sch)
#define proc_sch_isroot(q)	(!(sch2proc(q)->parent))
#define proc_iosch_parent(q)	(&((IO_SCHED *)((char *)iosch2proc(q)->parent + ((char *)(q) - (char *)LIST_STRUCT(q, IO_SCHED, sch)->p)))->sch)
#define proc_iosch_isroot(q)	(!(iosch2proc(q)->parent))

#define for_all_procq_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, procq)->children.next, PROC, child_entry)->procq; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->procq; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, procq)->child_entry.next, PROC, child_entry)->procq)
#define for_all_procq_subnodes_tail(v, p)

#define for_all_hq_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, hq)->children.next, PROC, child_entry)->hq; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->hq; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, hq)->child_entry.next, PROC, child_entry)->hq)
#define for_all_hq_subnodes_tail(v, p)

#define for_all_uq_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, uq)->children.next, PROC, child_entry)->uq; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->uq; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, uq)->child_entry.next, PROC, child_entry)->uq)
#define for_all_uq_subnodes_tail(v, p)

#define for_all_vmq_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, vmq)->children.next, PROC, child_entry)->vmq; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->vmq; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, vmq)->child_entry.next, PROC, child_entry)->vmq)
#define for_all_vmq_subnodes_tail(v, p)

#define for_all_pgtblq_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, pgtblq)->children.next, PROC, child_entry)->pgtblq; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->pgtblq; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, pgtblq)->child_entry.next, PROC, child_entry)->pgtblq)
#define for_all_pgtblq_subnodes_tail(v, p)

#define for_all_ioq_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, ioq)->children.next, PROC, child_entry)->ioq; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->ioq; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, ioq)->child_entry.next, PROC, child_entry)->ioq)
#define for_all_ioq_subnodes_tail(v, p)

#define for_all_wireq_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, wireq)->children.next, PROC, child_entry)->wireq; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->wireq; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, wireq)->child_entry.next, PROC, child_entry)->wireq)
#define for_all_wireq_subnodes_tail(v, p)

#define for_all_sch_subnodes(v, p) for (v = &LIST_STRUCT(LIST_STRUCT(p, PROC, sch)->children.next, PROC, child_entry)->sch; v != &LIST_STRUCT(&KERNEL$LIST_END, PROC, child_entry)->sch; v = &LIST_STRUCT(LIST_STRUCT(v, PROC, sch)->child_entry.next, PROC, child_entry)->sch)
#define for_all_sch_subnodes_tail(v, p)

typedef struct {
	char *name;
	int type;
	long min, max;
	int offset;
} PROC_OPTION;

#define OPT_LONG		1
#define OPT_LONG_DISADV		2
#define OPT_INV_PRI		3

#define procoffset(field)	__offsetof(PROC, field)

extern const PROC_OPTION PROC_OPTIONS[];

void ZAP_WIRED_ENTITIES(PROC *p);
void CACHE_INIT(void);
unsigned long CACHE_ACTIVE_ENTITIES(void);
void CACHE_PROC_CTOR(PROC *proc);

static __finline__ void UNIFY_ALLOC_RATE(PROC *p)
{
	u_jiffies_lo_t forget_cycles = (KERNEL$GET_JIFFIES_LO() - p->alloc_rate_time) & -(1 << __BSR_CONST(ALLOC_FORGET_TIME));
	p->alloc_rate_time += forget_cycles;
	forget_cycles >>= __BSR_CONST(ALLOC_FORGET_TIME);
	if (__unlikely(forget_cycles >= sizeof(unsigned long) * 8 - __BSR_CONST(ALLOC_FORGET_TIME))) forget_cycles = sizeof(unsigned long) * 8 - __BSR_CONST(ALLOC_FORGET_TIME) - 1;
	p->alloc_rate >>= forget_cycles;
}

#define FIND_DIFFERING_PROC(a_, b_, diff_, found)			\
do {									\
	unsigned depth_ = 0;						\
	while (1) {							\
		if (__unlikely(depth_ > (a_)->depth) || __unlikely((a_)->backptr[depth_] != (b_)->backptr[depth_])) {					\
			(diff_) = (b_)->backptr[depth_];		\
			{						\
				found;					\
			}						\
			__unreachable_code();				\
		}							\
		depth_++;						\
		if (__unlikely(depth_ > (b_)->depth)) break;		\
	}								\
} while (0)

#endif
