/*
 * A fast, small, non-recursive O(nlog n) sort for the Linux kernel
 *
 * Jan 23 2005  Matt Mackall <mpm@selenic.com>
 *
 * Modified by Mikulas Patocka
 */

#include <SYS/TYPES.H>
#include <STDLIB.H>

static void generic_swap(void *a, void *b, size_t size)
{
	char t;

	do {
		t = *(char *)a;
		*(char *)a = *(char *)b;
		a = (char *)a + 1;
		*(char *)b = t;
		b = (char *)b + 1;
	} while (--size);
}

static void long_multiple_swap(void *a, void *b, size_t size)
{
	unsigned long t;

	size /= sizeof(unsigned long);

	do {
		t = *(unsigned long *)a;
		*(unsigned long *)a = *(unsigned long *)b;
		a = (unsigned long *)a + 1;
		*(unsigned long *)b = t;
		b = (unsigned long *)b + 1;
	} while (--size);
}

static void long_swap(void *a, void *b, size_t size)
{
	unsigned long t = *(unsigned long *)a;
	*(unsigned long *)a = *(unsigned long *)b;
	*(unsigned long *)b = t;
}

/**
 * sort - sort an array of elements
 * @base: pointer to data to sort
 * @num: number of elements
 * @size: size of each element
 * @cmp_func: pointer to comparison function
 * @swap_func: pointer to swap function or NULL
 *
 * This function does a heapsort on the given array. You may provide a
 * swap_func function optimized to your element type.
 *
 * Sorting time is O(n log n) both on average and worst-case. While
 * qsort is about 20% faster on average, it suffers from exploitable
 * O(n*n) worst-case behavior and extra memory requirements that make
 * it less suitable for kernel use.
 */

static void sort(void *base_, size_t num, size_t size,
	  int (*cmp_func)(const void *, const void *),
	  void (*swap_func)(void *, void *, size_t size))
{
	char *base = base_;
	size_t i, n, c, r;

	/* pre-scale counters for performance */
	i = (num / 2) * size;
	if (__unlikely(!i))
		return;
	n = num * size;

	/* heapify */
	while (i) {
		i -= size;
		for (r = i; r * 2 + size < n; r = c) {
			c = r * 2 + size;
			if (c < n - size &&
					cmp_func(base + c, base + c + size) < 0)
				c += size;
			if (cmp_func(base + r, base + c) >= 0)
				break;
			swap_func(base + r, base + c, size);
		}
	}

	/* sort */
	for (i = n - size; i > 0; i -= size) {
		swap_func(base, base + i, size);
		for (r = 0; r * 2 + size < i; r = c) {
			c = r * 2 + size;
			if (c < i - size &&
					cmp_func(base + c, base + c + size) < 0)
				c += size;
			if (cmp_func(base + r, base + c) >= 0)
				break;
			swap_func(base + r, base + c, size);
		}
	}
}

void qsort(void *base, size_t num, size_t size, int (*cmp_func)(const void *, const void *))
{
	void (*swap_func)(void *, void *, size_t size);

	swap_func = generic_swap;
	if (__likely(!((unsigned long)base & (sizeof(unsigned long) - 1)))) {
		if (__likely(size == sizeof(unsigned long))) swap_func = long_swap;
		else if (__likely(!(size & (sizeof(unsigned long) - 1)))) swap_func = long_multiple_swap;
	}

	sort(base, num, size, cmp_func, swap_func);
}
