#include <SPAD/AC.H>
#include <SPAD/READDIR.H>
#include <SPAD/DIR_KRNL.H>
#include <SPAD/ALLOC.H>
#include <SPAD/THREAD.H>
#include <MALLOC.H>
#include <SPAD/VM.H>
#include <ERRNO.H>
#include <STRING.H>
#include <SPAD/LIBC.H>
#include <STDLIB.H>
#include <PATHS.H>
#include <VALUES.H>
#include <SPAD/DEV_KRNL.H>
#include <KERNEL/DEV.H>
#include <KERNEL/VMDEF.H>

#define ENTRIES_DIRECT_ALLOCATED	256
#define ENTRIES_CONTIG	(PAGE_CLUSTER_SIZE / sizeof(struct dirent *))

#define SMALL_DIRSIZE	1024

AST_STUB READDIR_GOT_DATA;
AST_STUB READDIR_CLUSTER_ALLOCATED;
static int CMP_DIRPTR(const void *d1_, const void *d2_);

typedef struct __n_readdir_rq N_READDIR_RQ;

typedef struct {
	IORQ_HEAD;
	int h;
	FFILE *f;
	unsigned long n_entries;
	struct dirent **entries;
	unsigned long n_entries_allocated;
	struct readdir_buffer *readdir_pages;
	struct readdir_buffer *readdir_last_page;
	READDIR_RQ *rq;
	IOCTLRQ irq;
	union {
		THREAD_RQ trq;
		OPENRQ orq;
		MALLOC_REQUEST mrq;
	} u;
	LOGICAL_LIST_REQUEST lrq;
	size_t mo_idx;
	int lerr;
	N_READDIR_RQ *list;
	char *ne;
} I_READDIR_RQ;

struct __n_readdir_rq {
	READDIR_RQ rrq;
	N_READDIR_RQ *next;
	I_READDIR_RQ *ir;
	char path[1];
};

static AST_STUB RD_OPEN, RD_DONE, RD_MULTIOPEN, RD_M_ALLOC, RD_M_RD, RD_M_CRQ;
static IO_STUB I_READDIR;
static long READDIR_EXTEND_ENTRIES(void *r_);
static long READDIR_M_ALLOC_ENTRIES(void *r_);

DECL_IOCALL(KERNEL$READDIR, SPL_DEV, READDIR_RQ)
{
	I_READDIR_RQ *ir;
	FFILE *f;
	unsigned s;
	RQ->n_entries = 0;
	RQ->entries = NULL;
	RQ->n_entries_allocated = 0;
	RQ->readdir_pages = NULL;
	if (__unlikely(!(ir = calloc(sizeof(I_READDIR_RQ), 1)))) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$READDIR, sizeof(I_READDIR_RQ));
		RETURN;
	}
	ir->rq = RQ;
	ir->u.orq.fn = RD_OPEN;
	ir->u.orq.path = RQ->path;
	ir->u.orq.flags = O_RDONLY | RQ->flags;
	ir->u.orq.cwd = RQ->cwd;
	f = open_file_name(&ir->u.orq, &s);
	if (__unlikely(__IS_ERR(f))) {
		free(ir);
		if (f == __ERR_PTR(-ENOMEM)) {
			KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$READDIR, s);
			RETURN;
		}
		RQ->status = __PTR_ERR(f);
		RETURN_AST(RQ);
	}
	if (__unlikely(s)) {
		ir->f = f;
		ir->ne = strrchr(f->path, ':');
		if (__unlikely(!ir->ne) || __unlikely(ir->ne == f->path))
			KERNEL$SUICIDE("KERNEL$READDIR: PATH DOESN'T CONTAIN COLON: \"%s\" -> \"%s\"", RQ->path, f->path);
		ir->lrq.prefix = malloc(ir->ne - f->path);
		if (__unlikely(!ir->lrq.prefix)) {
			free(ir->f);
			free(ir);
			KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$READDIR, ir->ne - f->path);
			RETURN;
		}
		ir->ne--;
		*(char *)mempcpy(ir->lrq.prefix, f->path, ir->ne - f->path) = 0;
		ir->lrq.fn = RD_MULTIOPEN;
		RETURN_IORQ_CANCELABLE(&ir->lrq, KERNEL$LIST_LOGICALS, RQ);
	}
	free(f);
	RETURN_IORQ_CANCELABLE(&ir->u.orq, KERNEL$OPEN, RQ);
}

static DECL_AST(RD_OPEN, SPL_DEV, OPENRQ)
{
	I_READDIR_RQ *ir = GET_STRUCT(RQ, I_READDIR_RQ, u.orq);
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), ir->rq);
	if (__unlikely(ir->u.orq.status < 0)) {
		READDIR_RQ *rq = ir->rq;
		rq->status = ir->u.orq.status;
		free(ir);
		RETURN_AST(rq);
	}
	ir->fn = RD_DONE;
	ir->h = ir->u.orq.status;
	RETURN_IORQ_CANCELABLE(ir, I_READDIR, ir->rq);
}

static DECL_AST(RD_DONE, SPL_DEV, I_READDIR_RQ)
{
	READDIR_RQ *rq = RQ->rq;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), rq);
	rq->n_entries = RQ->n_entries;
	rq->entries = RQ->entries;
	rq->n_entries_allocated = RQ->n_entries_allocated;
	rq->readdir_pages = RQ->readdir_pages;
	if (__unlikely((rq->status = RQ->status) < 0)) KERNEL$FREE_READDIR(rq);
	KERNEL$FAST_CLOSE(RQ->h);
	free(RQ);
	RETURN_AST(rq);
}

static int CMP_DIRPTR(const void *d1_, const void *d2_)
{
	const struct dirent *d1 = *(struct dirent **)d1_;
	const struct dirent *d2 = *(struct dirent **)d2_;
	return strcmp(d1->d_name, d2->d_name);
}

static __finline__ void SORTENT(struct dirent **e, unsigned long *n)
{
	unsigned long i, j, sub;
	if (__unlikely(!*n)) return;
	qsort(e, *n, sizeof(struct dirent *), CMP_DIRPTR);
	for (i = *n - 1; i > 0; i--) {
		if (__unlikely(!strcmp(e[i - 1]->d_name, e[i]->d_name))) goto shift;
	}
	shift_done:
	return;

	shift:
	sub = 0;
	j = 1;
	for (i = 1; i < *n; i++) {
		if (__likely(strcmp(e[j - 1]->d_name, e[i]->d_name))) {
			e[j++] = e[i];
		} else sub++;
	}
	*n -= sub;
	goto shift_done;
}

static DECL_IOCALL(I_READDIR, SPL_DEV, I_READDIR_RQ)
{
	struct readdir_buffer *buf;
	if (!RQ->entries) {
		RQ->entries = malloc(ENTRIES_DIRECT_ALLOCATED * sizeof(struct dirent *));
		if (__unlikely(!RQ->entries)) {
			KERNEL$MEMWAIT((IORQ *)RQ, I_READDIR, ENTRIES_DIRECT_ALLOCATED * sizeof(struct dirent *));
			RETURN;
		}
		RQ->n_entries_allocated = ENTRIES_DIRECT_ALLOCATED;
		RQ->n_entries = 0;

		if (__unlikely(!(buf = malloc(SMALL_DIRSIZE)))) {
			KERNEL$MEMWAIT((IORQ *)RQ, I_READDIR, SMALL_DIRSIZE);
			RETURN;
		}
		buf->page = 0;
		RQ->irq.v.len = SMALL_DIRSIZE;
	} else {
		if (__unlikely(!(buf = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED)))) {
			KERNEL$MEMWAIT((IORQ *)RQ, I_READDIR, PAGE_CLUSTER_SIZE);
			RETURN;
		}
		buf->page = 1;
		RQ->irq.v.len = PAGE_CLUSTER_SIZE;
	}
	buf->next = NULL;
	buf->dirent_start = 0;
	if (!RQ->readdir_pages) {
		RQ->readdir_pages = buf;
		buf->cookie_size = 0;
	} else {
		struct readdir_buffer *last = RQ->readdir_last_page;
		last->next = buf;
		buf->cookie_size = last->cookie_size;
		memcpy(buf->cookie, last->cookie, buf->cookie_size & COOKIE_SIZE_MASK);
	}
	RQ->readdir_last_page = buf;
	RQ->irq.fn = READDIR_GOT_DATA;
	RQ->irq.h = RQ->h;
	RQ->irq.ioctl = IOCTL_READDIR;
	RQ->irq.param = 0;
	RQ->irq.v.ptr = (unsigned long)buf;
	RQ->irq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->irq, KERNEL$IOCTL, RQ);
}

DECL_AST(READDIR_GOT_DATA, SPL_DEV, IOCTLRQ)
{
	int i;
	struct dirent *d, *e;
	struct readdir_buffer *b;
	I_READDIR_RQ *r = GET_STRUCT(RQ, I_READDIR_RQ, irq);
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), r);
	if (__unlikely(RQ->status < 0)) {
		if (RQ->status == -EMSGSIZE) {
			if (__likely(r->readdir_last_page->total_size > r->readdir_last_page->dirent_start)) goto noerr;
			if (r->readdir_pages != r->readdir_last_page || r->readdir_last_page->page)
				KERNEL$SUICIDE("READDIR_GOT_DATA: UNEXPECTED EMSGSIZE");
			free(r->readdir_last_page);
			r->readdir_pages = NULL;
			RETURN_IORQ_LSTAT(r, I_READDIR);
		}
		if (RQ->status == -ENOOP) RQ->status = -ENOTDIR;
		r->status = RQ->status;
		RETURN_AST(r);
	}
	noerr:
	b = r->readdir_last_page;
	d = DIR_START(b);
	e = (struct dirent *)((char *)b + b->total_size);
	i = 0;
	while (d < e) {
		d = DIR_NEXT(d);
		i++;
	}
	if (__unlikely(r->n_entries + i > r->n_entries_allocated)) goto extend;
	d = DIR_START(b);
	while (d < e) {
		r->entries[r->n_entries++] = d;
		d = DIR_NEXT(d);
		i++;
	}
	if (__likely(b->end)) {
		SORTENT(r->entries, &r->n_entries);
		r->status = 0;
		RETURN_AST(r);
	}
	RETURN_IORQ_LSTAT(r, I_READDIR);

	extend:
	r->u.trq.fn = READDIR_CLUSTER_ALLOCATED;
	r->u.trq.thread_main = READDIR_EXTEND_ENTRIES;
	r->u.trq.p = r;
	r->u.trq.error = NULL;
	r->u.trq.cwd = NULL;
	r->u.trq.std_in = -1;
	r->u.trq.std_out = -1;
	r->u.trq.std_err = -1;
	r->u.trq.dlrq = NULL;
	r->u.trq.thread = NULL;
	r->u.trq.spawned = 0;
	RETURN_IORQ_CANCELABLE(&r->u.trq, KERNEL$THREAD, r);
}

static long READDIR_EXTEND_ENTRIES(void *r_)
{
	I_READDIR_RQ *r = r_;
	unsigned long new_n_entries_allocated;
	struct dirent **new_entries;
	new_n_entries_allocated = r->n_entries_allocated * 2;
	if (__unlikely(new_n_entries_allocated > MAXULONG / sizeof(struct dirent *))) {
		return -EOVERFLOW;
	}
	new_entries = KERNEL$ALLOC_CONTIG_AREA(new_n_entries_allocated * sizeof(struct dirent *), AREA_DATA);
	if (__unlikely(__IS_ERR(new_entries))) {
		return __PTR_ERR(new_entries);
	}
	memcpy(new_entries, r->entries, r->n_entries * sizeof(struct dirent *));
	if (r->n_entries_allocated <= ENTRIES_DIRECT_ALLOCATED) free(r->entries);
	else KERNEL$FREE_CONTIG_AREA(r->entries, r->n_entries_allocated * sizeof(struct dirent *));
	r->entries = new_entries;
	r->n_entries_allocated = new_n_entries_allocated;
	return 0;
}

DECL_AST(READDIR_CLUSTER_ALLOCATED, SPL_DEV, THREAD_RQ)
{
	I_READDIR_RQ *r = GET_STRUCT(RQ, I_READDIR_RQ, u.trq);
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), r);
	if (__unlikely(RQ->status < 0)) {
		r->status = RQ->status;
		RETURN_AST(r);
	}
	RETURN_AST(&r->irq);
}

void KERNEL$FREE_READDIR(READDIR_RQ *r)
{
	struct readdir_buffer *b, *c;
	for (b = r->readdir_pages; b; b = c) {
		c = b->next;
		if (b->page) KERNEL$FREE_KERNEL_PAGE(b, VM_TYPE_WIRED_MAPPED);
		else free(b);
	}
	r->readdir_pages = NULL;
	if (r->entries) {
		if (r->n_entries_allocated <= ENTRIES_DIRECT_ALLOCATED) free(r->entries);
		else KERNEL$FREE_CONTIG_AREA(r->entries, r->n_entries_allocated * sizeof(struct dirent *));
	}
	r->entries = NULL;
	r->n_entries = 0;
}

static void free_ir(I_READDIR_RQ *ir)
{
	while (ir->list) {
		N_READDIR_RQ *nrq = ir->list;
		ir->list = nrq->next;
		KERNEL$FREE_READDIR(&nrq->rrq);
		free(nrq);
	}
	KERNEL$FREE_LOGICAL_LIST(&ir->lrq);
	free(ir->lrq.prefix);
	free(ir->f);
	free(ir);
}

static DECL_AST(RD_MULTIOPEN, SPL_DEV, LOGICAL_LIST_REQUEST)
{
	I_READDIR_RQ *ir = GET_STRUCT(RQ, I_READDIR_RQ, lrq);
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), ir->rq);
	if (__unlikely(ir->lrq.status < 0)) {
		READDIR_RQ *rq = ir->rq;
		rq->status = ir->lrq.status;
		free(ir->lrq.prefix);
		free(ir->f);
		free(ir);
		RETURN_AST(rq);
	}
	ir->list = NULL;
	ir->lerr = -ENOLNM;
	if (__unlikely(!ir->lrq.n_entries)) {
		READDIR_RQ *rq = ir->rq;
		rq->status = -ENOLNM;
		free_ir(ir);
		RETURN_AST(rq);
	}
	ir->mo_idx = 0;
	ir->u.mrq.fn = RD_M_ALLOC;
	ir->u.mrq.size = sizeof(N_READDIR_RQ) + strlen(ir->lrq.entries[0]->name) + strlen(ir->ne + 1);
	RETURN_IORQ_CANCELABLE(&ir->u.mrq, KERNEL$UNIVERSAL_MALLOC, ir->rq);
}

static DECL_AST(RD_M_ALLOC, SPL_DEV, MALLOC_REQUEST)
{
	I_READDIR_RQ *ir = GET_STRUCT(RQ, I_READDIR_RQ, u.mrq);
	N_READDIR_RQ *nrq;
	const char *lnm;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), ir->rq);
	if (__unlikely(ir->u.mrq.status < 0)) {
		READDIR_RQ *rq = ir->rq;
		rq->status = ir->lrq.status;
		free_ir(ir);
		RETURN_AST(rq);
	}
	nrq = ir->u.mrq.ptr;
	lnm = ir->lrq.entries[ir->mo_idx]->name;
	strcpy(stpcpy(nrq->path, lnm), ir->ne + 1);
	nrq->rrq.fn = RD_M_RD;
	nrq->rrq.flags = ir->rq->flags;
	nrq->rrq.path = nrq->path;
	nrq->rrq.cwd = NULL;
	nrq->ir = ir;
	/*
	 * This is a dirty hack. If someone executes readdir .:/ and he is
	 * in background, we do not want him to block on reading STDIN:/
	 */
	if (__unlikely(!strcmp(lnm, _LNM_STDIN)) ||
	    __unlikely(!strcmp(lnm, _LNM_STDOUT)) ||
	    __unlikely(!strcmp(lnm, _LNM_STDERR))) {
		nrq->rrq.status = 0;
		nrq->rrq.entries = NULL;
		nrq->rrq.n_entries = 0;
		nrq->rrq.readdir_pages = NULL;
		RETURN_AST(&nrq->rrq);
	}
	RETURN_IORQ_CANCELABLE(&nrq->rrq, KERNEL$READDIR, ir->rq);
}

static DECL_AST(RD_M_RD, SPL_DEV, READDIR_RQ)
{
	N_READDIR_RQ *nrq = GET_STRUCT(RQ, N_READDIR_RQ, rrq);
	I_READDIR_RQ *ir = nrq->ir;
	unsigned long ne;
	READDIR_RQ *rq;
	int e;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), ir->rq);
	if (__unlikely(nrq->rrq.status < 0)) {
		KERNEL$FREE_READDIR(&nrq->rrq);
		if (nrq->rrq.status == -EINTR) {
			e = -EINTR;
			goto ret_e;
		}
		if (ir->lerr < 0) {
			if (MULTIOPEN_ERROR_PRI(ir->lerr) < MULTIOPEN_ERROR_PRI(nrq->rrq.status)) ir->lerr = nrq->rrq.status;
		}
	} else {
		ir->lerr = 1;
		nrq->next = ir->list;
		ir->list = nrq;
	}
	if (__likely(++ir->mo_idx < ir->lrq.n_entries)) {
		ir->u.mrq.fn = RD_M_ALLOC;
		ir->u.mrq.size = sizeof(N_READDIR_RQ) + strlen(ir->lrq.entries[ir->mo_idx]->name) + strlen(ir->ne + 1);
		RETURN_IORQ_CANCELABLE(&ir->u.mrq, KERNEL$UNIVERSAL_MALLOC, ir->rq);
	}
	if (__unlikely(!ir->list)) {
		e = ir->lerr;
		goto ret_e;
	}
	ne = 0;
	for (nrq = ir->list; nrq; nrq = nrq->next) {
		ne += nrq->rrq.n_entries;
		if (__unlikely(ne > MAXULONG / sizeof(struct dirent *))) {
			e = -EOVERFLOW;
			goto ret_e;
		}
	}
	if (__unlikely(!ne)) {
		e = 0;
		goto ret_e;
	}
	ir->rq->n_entries = ne;
	ir->u.trq.fn = RD_M_CRQ;
	ir->u.trq.thread_main = READDIR_M_ALLOC_ENTRIES;
	ir->u.trq.p = ir;
	ir->u.trq.error = NULL;
	ir->u.trq.cwd = NULL;
	ir->u.trq.std_in = -1;
	ir->u.trq.std_out = -1;
	ir->u.trq.std_err = -1;
	ir->u.trq.dlrq = NULL;
	ir->u.trq.thread = NULL;
	ir->u.trq.spawned = 0;
	RETURN_IORQ_CANCELABLE(&ir->u.trq, KERNEL$THREAD, ir->rq);


ret_e:
	rq = ir->rq;
	rq->status = e;
	free_ir(ir);
	RETURN_AST(rq);
}

static long READDIR_M_ALLOC_ENTRIES(void *r_)
{
	I_READDIR_RQ *ir = r_;
	struct dirent **new_entries = KERNEL$ALLOC_CONTIG_AREA(ir->rq->n_entries * sizeof(struct dirent *), AREA_DATA);
	if (__unlikely(__IS_ERR(new_entries))) {
		return __PTR_ERR(new_entries);
	}
	ir->rq->entries = new_entries;
	return 0;
}

static DECL_AST(RD_M_CRQ, SPL_DEV, THREAD_RQ)
{
	I_READDIR_RQ *ir = GET_STRUCT(RQ, I_READDIR_RQ, u.trq);
	READDIR_RQ *rq = ir->rq;
	N_READDIR_RQ *nrq, *nnrq;
	unsigned long ne;
	struct readdir_buffer **b;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), rq);
	if (__unlikely(ir->u.trq.status < 0)) {
		rq->n_entries = 0;
		rq->status = ir->u.trq.status;
		free_ir(ir);
		RETURN_AST(rq);
	}
	ne = 0;
	b = &rq->readdir_pages;
	for (nrq = ir->list; nrq; ) {
		memcpy(&rq->entries[ne], nrq->rrq.entries, nrq->rrq.n_entries * sizeof(struct dirent *));
		ne += nrq->rrq.n_entries;
		if (nrq->rrq.readdir_pages) {
			*b = nrq->rrq.readdir_pages;
			while (*b) b = &(*b)->next;
		}
		nnrq = nrq->next;
		free(nrq);
		nrq = nnrq;
	}
	rq->n_entries = ne;
	SORTENT(rq->entries, &rq->n_entries);
	ir->list = NULL;
	rq->status = 0;
	free_ir(ir);
	RETURN_AST(rq);
}



