#include <SPAD/LIBC.H>
#include <SPAD/AC.H>
#include <SPAD/WQ.H>
#include <SPAD/TIMER.H>
#include <ERRNO.H>
#include <SPAD/ALLOC.H>
#include <SPAD/DEV.H>
#include <SPAD/LIST.H>
#include <ARCH/SETUP.H>
#include <SPAD/CD.H>
#include <SPAD/DEV_KRNL.H>
#include <KERNEL/VMDEF.H>

#include <KERNEL/THREAD.H>
#include <SPAD/THREAD.H>
#include <SPAD/SYNC.H>
#include <KERNEL/SELECT.H>
#include <SETJMP.H>
#include <SCHED.H>

#define THREAD_ENTRIES_1	\
	LIST_ENTRY run_list;

#define THREAD_ENTRIES_2		\
	int sleep;			\
	THREAD_RQ *rq;			\
	struct select_context *sctx;

#define S_RUN		0
#define S_ASLEEP	1
#define S_SLEEP		2

DECL_LIST(THREAD_READY);

#include <KERNEL/THREADMD.I>

static AST_STUB UNBLOCK;

static IO_STUB THREAD_WAKE_UP;
static void TIME_WAKE_UP(TIMER *tm);
static void TIME_CANCELABLE_WAKE_UP(TIMER *tm);

static IORQ SCHED_AST;
static int SCHED_AST_PENDING;

void *KERNEL$ALLOC_THREAD_SYNC(void)
{
	THREAD *t;
	KERNEL$THREAD_MAY_BLOCK();
	while (__unlikely(!(t = ALLOC_THREAD_STRUCT()))) {
		int r;
		if (__unlikely(r = KERNEL$MEMWAIT_SYNC(PAGE_CLUSTER_SIZE))) return __ERR_PTR(r);
	}
	return t;
}

void KERNEL$FREE_THREAD(void *t)
{
	FREE_THREAD_STRUCT(t);
}

DECL_IOCALL(KERNEL$THREAD, SPL_THREAD, THREAD_RQ)
{
	THREAD *t;
#if __DEBUG >= 1
	if (__unlikely(RQ->spawned)) KERNEL$SUICIDE("KERNEL$THREAD: SPAWNED FIELD NOT ZEROED (%d)", RQ->spawned);
#endif
	t = RQ->thread;
	if (__likely(!t) && __unlikely(!(t = ALLOC_THREAD_STRUCT()))) {
		KERNEL$MEMWAIT((IORQ *)RQ, KERNEL$THREAD, PAGE_CLUSTER_SIZE);
		RETURN;
	}
	INIT_THREAD_STRUCT(t);
	ADD_TO_LIST_END(&THREAD_READY, &t->run_list);
	t->sleep = S_RUN;
	t->rq = RQ;
	t->sctx = NULL;
	RQ->spawned = 1;
	if (!SCHED_AST_PENDING) {
		SCHED_AST_PENDING = 1;
		RETURN_AST(&SCHED_AST);
	}
	RETURN;
}

static void START_THREAD(void)
{
	long r;
	THREAD *t;
	LOWER_SPL(SPL_ZERO);
	t = CURRENT_THREAD();
	r = t->rq->thread_main(t->rq->p);
	if (__unlikely((r & RQS_PROC_MASK) == RQS_PROC)) r = 0;
	t->rq->status = r;
	/* CALL_AST(t->rq); done in END_THREAD */
	LOWER_SPL(SPL_ZERO);
	RAISE_SPL(SPL_TIMER);
	FREE_THREAD_SELECT_STORAGE(&t->sctx);
	RAISE_SPL(SPL_THREAD);
	END_THREAD(t, t->rq, t->rq->thread);
}

int KERNEL$SET_THREAD_STACK_SIZE(size_t size)
{
	if (__unlikely(size > PAGE_CLUSTER_SIZE)) return -EINVAL;
	return 0;
}

size_t KERNEL$GET_THREAD_STACK_SIZE(void)
{
	return PAGE_CLUSTER_SIZE;
}

IORQ *KERNEL$PREPARE_BLOCK(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$PREPARE_BLOCK: CALLED FROM MAIN THREAD");
#endif
	if (__unlikely(t->sleep == S_ASLEEP)) KERNEL$SUICIDE("KERNEL$PREPARE_BLOCK: THREAD ALREADY PREPARED FOR BLOCKING");
	t->sleep = S_ASLEEP;
	return (IORQ *)t->rq;
}

void KERNEL$BLOCK(IORQ *rq)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$BLOCK: CALLED FROM MAIN THREAD");
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$BLOCK CALLED AT SPL %08X", KERNEL$SPL);
#endif
	if (__unlikely(t->sleep != S_RUN)) {
		int s = KERNEL$SPL;
		RAISE_SPL(SPL_THREAD);
		if (__likely(t->sleep != S_RUN)) {
			t->sleep = S_SLEEP;
			BLOCK_THREAD(t);
		}
		LOWER_SPLX(s);
	}
}

void KERNEL$SYNC_IO_CANCELABLE(IORQ *rq, IO_STUB *call)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE: CALLED FROM MAIN THREAD");
	if (__unlikely(t != RQ_THREAD(rq))) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE: REQUEST IS NOT ON CALLER'S STACK");
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SYNC_IO_CANCELABLE CALLED AT SPL %08X", KERNEL$SPL);
#endif
	t->sleep = S_ASLEEP;
	rq->fn = UNBLOCK;
	CALL_IORQ_CANCELABLE_EXPR(rq, call, t->rq);
	if (__unlikely(t->sleep != S_RUN)) {
		int s = KERNEL$SPL;
		RAISE_SPL(SPL_THREAD);
		if (__likely(t->sleep != S_RUN)) {
			t->sleep = S_SLEEP;
			BLOCK_THREAD(t);
		}
		LOWER_SPLX(s);
	}
	KERNEL$THREAD_MAY_BLOCK();
}

void KERNEL$SYNC_IO_SIGINTERRUPT(IORQ *rq, IO_STUB *call)
{
	return KERNEL$SYNC_IO(rq, call);
}

void KERNEL$SYNC_IO(IORQ *rq, IO_STUB *call)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$SYNC_IO: CALLED FROM MAIN THREAD");
	if (__unlikely(t != RQ_THREAD(rq))) KERNEL$SUICIDE("KERNEL$SYNC_IO: REQUEST IS NOT ON CALLER'S STACK");
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SYNC_IO CALLED AT SPL %08X", KERNEL$SPL);
#endif
	t->sleep = S_ASLEEP;
	rq->fn = UNBLOCK;
	CALL_IORQ_EXPR(rq, call);
	if (__unlikely(t->sleep != S_RUN)) {
		int s = KERNEL$SPL;
		RAISE_SPL(SPL_THREAD);
		if (__likely(t->sleep != S_RUN)) {
			t->sleep = S_SLEEP;
			BLOCK_THREAD(t);
		}
		LOWER_SPLX(s);
	}
	KERNEL$THREAD_MAY_BLOCK();
}

static DECL_AST(UNBLOCK, SPL_THREAD, IORQ)
{
	THREAD *t = RQ_THREAD(RQ);
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_THREAD), t->rq);
	if (__unlikely(t->sleep == S_SLEEP)) {
		ADD_TO_LIST_END(&THREAD_READY, &t->run_list);
		if (!SCHED_AST_PENDING) {
			t->sleep = S_RUN;
			SCHED_AST_PENDING = 1;
			RETURN_AST(&SCHED_AST);
		}
	}
#if __DEBUG >= 1
	else if (__unlikely(t->sleep != S_ASLEEP)) KERNEL$SUICIDE("UNBLOCK: THREAD IS RUNNING, STATE %d", t->sleep);
#endif
	t->sleep = S_RUN;
	RETURN;
}

AST_STUB *KERNEL$UNBLOCK(void)
{
	return UNBLOCK;
}

void KERNEL$THREAD_MAY_BLOCK(void)
{
#if __DEBUG >= 1
	if (__unlikely(CURRENT_THREAD() == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$THREAD_MAY_BLOCK: CALLED FROM MAIN THREAD");
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$THREAD_MAY_BLOCK CALLED AT SPL %08X", KERNEL$SPL);
#endif
	TEST_SPLX(SPL_X(SPL_ZERO), KERNEL$SPL);
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS))
		/*WQ_WAIT_SYNC(&KERNEL$LOCKUP_EVENTS);*/
		KERNEL$THREAD_YIELD();
}

int sched_yield(void)
{
	KERNEL$THREAD_YIELD();
	return 0;
}

void KERNEL$THREAD_YIELD(void)
{
	int spl;
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(CURRENT_THREAD() == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$THREAD_YIELD: CALLED FROM MAIN THREAD");
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$THREAD_YIELD CALLED AT SPL %08X", KERNEL$SPL);
#endif
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	ADD_TO_LIST_END(&THREAD_READY, &t->run_list);
	if (!SCHED_AST_PENDING) {
		SCHED_AST_PENDING = 1;
		CALL_AST(&SCHED_AST);
	}
	BLOCK_THREAD(t);
	LOWER_SPLX(spl);
}

void KERNEL$WQ_WAIT_SYNC(WQ *wq __WQ_LINE_ARG_DECL)
{
	IORQ io;
	int spl;
	THREAD *t;
#if __DEBUG >= 1
	if (__unlikely(CURRENT_THREAD() == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$WQ_WAIT_SYNC: CALLED FROM MAIN THREAD");
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$WQ_WAIT_SYNC CALLED AT SPL %08X", KERNEL$SPL);
#endif
	io.status = RQS_PROCESSING;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	io.tmp1 = (unsigned long)THREAD_WAKE_UP;
	KERNEL$WQ_WAIT(wq, &io __WQ_LINE_ARG_PASS);
	t = CURRENT_THREAD();
	t->sleep = S_SLEEP;
	BLOCK_THREAD(t);
	LOWER_SPLX(spl);
}

static DECL_IOCALL(THREAD_WAKE_UP, SPL_THREAD, IORQ)
{
	THREAD *t = RQ_THREAD(RQ);
#if __DEBUG >= 1
	if (__unlikely(t->sleep != S_SLEEP)) KERNEL$SUICIDE("THREAD_WAKE_UP: THREAD IS RUNNING, STATE %d", t->sleep);
#endif
	t->sleep = S_RUN;
	ADD_TO_LIST_END(&THREAD_READY, &t->run_list);
	if (!SCHED_AST_PENDING) {
		SCHED_AST_PENDING = 1;
		RETURN_AST(&SCHED_AST);
	}
	RETURN;
}

#if __DEBUG_WQ
/* this is racy, but it's only debug information */
static int wq_wait_sync_cancelable_lineno;
#endif

static DECL_IOCALL(WAIT_CANCELABLE, SPL_THREAD, IORQ)
{
	RQ->tmp1 = (unsigned long)KERNEL$SUCCESS;
#if __DEBUG_WQ
	KERNEL$WQ_WAIT((WQ *)RQ->tmp3, RQ, wq_wait_sync_cancelable_lineno);
#else
	KERNEL$WQ_WAIT((WQ *)RQ->tmp3, RQ);
#endif
	RETURN;
}

long KERNEL$WQ_WAIT_SYNC_CANCELABLE(WQ *wq __WQ_LINE_ARG_DECL)
{
	IORQ io;
#if __DEBUG_WQ
	wq_wait_sync_cancelable_lineno = line;
#endif
	io.tmp3 = (unsigned long)wq;
	KERNEL$SYNC_IO_CANCELABLE(&io, WAIT_CANCELABLE);
	return io.status;
}

long KERNEL$WQ_WAIT_SYNC_SIGINTERRUPT(WQ *wq __WQ_LINE_ARG_DECL)
{
	KERNEL$WQ_WAIT_SYNC(wq __WQ_LINE_ARG_PASS);
	return 0;
}

void KERNEL$SLEEP(jiffies_t j)
{
	TIMER tm;
	THREAD *t;
	int spl;
#if __DEBUG >= 1
	if (__unlikely(CURRENT_THREAD() == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$SLEEP: CALLED FROM MAIN THREAD");
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SLEEP CALLED AT SPL %08X", KERNEL$SPL);
#endif
	INIT_TIMER(&tm);
	tm.fn = TIME_WAKE_UP;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	KERNEL$SET_TIMER(j, &tm);
	t = CURRENT_THREAD();
	t->sleep = S_SLEEP;
	BLOCK_THREAD(t);
	LOWER_SPLX(spl);
}

static void TIME_WAKE_UP(TIMER *tm)
{
	THREAD *t = RQ_THREAD((IORQ *)tm);
	LOWER_SPL(SPL_THREAD);
#if __DEBUG >= 1
	if (__unlikely(t->sleep != S_SLEEP)) KERNEL$SUICIDE("TIME_WAKE_UP: THREAD IS RUNNING, STATE %d", t->sleep);
#endif
	t->sleep = S_RUN;
	ADD_TO_LIST_END(&THREAD_READY, &t->run_list);
	if (!SCHED_AST_PENDING) {
		SCHED_AST_PENDING = 1;
		CALL_AST(&SCHED_AST);
	}
}

typedef struct {
	TIMER t;
	WQ wq;
} SC;

int KERNEL$SLEEP_CANCELABLE(jiffies_t j)
{
	SC sc;
	int spl, r;
#if __DEBUG >= 1
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), KERNEL$SPL))) KERNEL$SUICIDE("KERNEL$SLEEP_CANCELABLE CALLED AT SPL %08X", KERNEL$SPL);
#endif
	INIT_TIMER(&sc.t);
	sc.t.fn = TIME_CANCELABLE_WAKE_UP;
	WQ_INIT(&sc.wq, "KERNEL$SLEEP_CANCELABLE");
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_THREAD);
	KERNEL$SET_TIMER(j, &sc.t);
	r = WQ_WAIT_SYNC_CANCELABLE(&sc.wq);
	if (__unlikely(sc.t.fn != NULL)) KERNEL$DEL_TIMER(&sc.t);
	LOWER_SPLX(spl);
	return r;
}

static void TIME_CANCELABLE_WAKE_UP(TIMER *t)
{
	SC *sc = LIST_STRUCT(t, SC, t);
	LOWER_SPL(SPL_THREAD);
	sc->t.fn = NULL;
	WQ_WAKE_ALL(&sc->wq);
}

void KERNEL$THREAD_WAIT_FOR_LONGJMP(jmp_buf j)
{
}

char *KERNEL$ERROR_MSG(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$ERROR_MSG: CALLED FROM MAIN THREAD");
	if (__unlikely(!t->rq->error)) KERNEL$SUICIDE("KERNEL$ERROR_MSG: NO ERROR PLACE");
#endif
	return t->rq->error;
}

void *KERNEL$TSR_IMAGE(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$TSR_IMAGE: CALLED FROM MAIN THREAD");
#endif
	t->rq->tsr = 1;
	return KERNEL$DLRQ();
}

CWD *KERNEL$CWD(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$CWD: CALLED FROM MAIN THREAD");
#endif
	return t->rq->cwd;
}

void SET_THREAD_CWD(CWD *cwd)
{
	THREAD *t;
	int spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_THREAD), spl)))
		KERNEL$SUICIDE("SET_THREAD_CWD AT SPL %08X", spl);
	RAISE_SPL(SPL_THREAD);
	t = CURRENT_THREAD();
	if (__likely(!t->rq->cwd)) t->rq->cwd = cwd;
	else KERNEL$FREE_CWD(cwd);
	LOWER_SPLX(spl);
}

int KERNEL$STDIN(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$STDIN: CALLED FROM MAIN THREAD");
#endif
	return t->rq->std_in;
}

int KERNEL$STDOUT(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$STDOUT: CALLED FROM MAIN THREAD");
#endif
	return t->rq->std_out;
}

int KERNEL$STDERR(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$STDERR: CALLED FROM MAIN THREAD");
#endif
	return t->rq->std_err;
}

void *KERNEL$DLRQ(void)
{
	THREAD *t = CURRENT_THREAD();
#if __DEBUG >= 1
	if (__unlikely(t == &MAIN_THREAD)) KERNEL$SUICIDE("KERNEL$DLRQ: CALLED FROM MAIN THREAD");
#endif
	return t->rq->dlrq;
}

THREAD_RQ *KERNEL$THREAD_RQ(void)
{
	THREAD *t = CURRENT_THREAD();
	return t->rq;
}

int KERNEL$IS_INIT_THREAD(void)
{
	return 0;
}

struct select_context **GET_THREAD_SELECT_STORAGE(void)
{
	THREAD *t = CURRENT_THREAD();
	return &t->sctx;
}

static DECL_AST(SCHED_AST_FN, SPL_ZERO, AST)
{
	if (__unlikely(CURRENT_THREAD() != &MAIN_THREAD))
		KERNEL$SUICIDE("SCHED_AST_FN: NOT MAIN THREAD");
	retry:
	RAISE_SPL(SPL_THREAD);
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
		SCHED_AST.status = RQS_PROCESSING;
		WQ_WAIT(&KERNEL$LOCKUP_EVENTS, &SCHED_AST, KERNEL$SUCCESS);
		RETURN;
	}
	if (!LIST_EMPTY(&THREAD_READY)) {
		THREAD *t = LIST_STRUCT(THREAD_READY.next, THREAD, run_list);
		DEL_FROM_LIST(&t->run_list);
		RUN_THREAD(t);
		LOWER_SPL(SPL_ZERO);
		goto retry;
	}
	SCHED_AST_PENDING = 0;
	RETURN;
}

__COLD_ATTR__ void THREAD_INIT(void)
{
	SCHED_AST.fn = &SCHED_AST_FN;
	SCHED_AST_PENDING = 0;
	MD_THREAD_INIT();
	if (CURRENT_THREAD() != &MAIN_THREAD) KERNEL$SUICIDE("THREAD_INIT: CURRENT THREAD(%p) != MAIN THREAD(%p)", CURRENT_THREAD(), &MAIN_THREAD);
}

