#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/IOCTL.H>
#include <SPAD/CD.H>
#include <SYS/PARAM.H>
#include <TIME.H>
#include <LIB/KERNEL/UIO.H>
#include <LIB/KERNEL/USIGNAL.H>
#include <KERNEL/DEV.H>
#include <KERNEL/THREAD.H>
#include <KERNEL/VMDEF.H>
#include <SIGNAL.H>
#include <STDLIB.H>
#include <LIMITS.H>
#include <SPAD/THREAD.H>
#include <SPAD/HASH.H>
#include <SPAD/CAPA.H>

#include <FCNTL.H>
#include <UNISTD.H>
#include <SYS/STAT.H>
#include <SYS/STATFS.H>
#include <SYS/STATVFS.H>
#include <SYS/TIME.H>
#include <UTIME.H>
#include <SYS/UIO.H>
#include <STDIO.H>

#define make_cwd(cwd, cwde, dir)			\
do {							\
	cwde.name = (char *)(dir);			\
	cwde.len = strlen(dir);				\
	cwde.lnlen = strchr(dir, ':') - (dir);		\
	INIT_XLIST(&cwd.cwds);				\
	ADD_TO_XLIST(&cwd.cwds, &cwde.list);		\
} while (0)

int getdtablesize(void)
{
	if (__unlikely(!FEATURE_TEST(FEATURE_USERSPACE))) return (PG_SIZE * PG_BANK) / sizeof(HANDLE *);
	return OPEN_MAX;
}

static void open_mods(int h, int oflags)
{
	CHHRQ ch;
	ch.h = h;
	if (__unlikely(oflags & _O_CLOSE)) return;
	if (__unlikely(oflags & O_NONBLOCK)) {
		ch.option = "NONBLOCK";
		ch.value = "";
		SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	}
	if (__likely(oflags & O_DIRECT)) {
		ch.option = "DIRECT";
		ch.value = "";
		SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	}
}

static int open_special(CWD *cwd, const char *path, int oflags)
{
	char *p;
	OPENRQ o;
	p = (char *)path;
	if (__likely((oflags & (O_NONBLOCK | O_CREAT | _O_MKDIR | _O_DELETE)) == O_NONBLOCK)) {
/* optimization ... O_NONBLOCK is quite common and so don't do additional
   syscalls in KERNEL$CHANGE_HANDLE */
		char *q;
		p = alloca(strlen(path) + /*strlen("/^NONBLOCK")*/ 10 + 1);
		q = stpcpy(p, path);
		if (__likely(*p) && __unlikely(q[-1] == '/')) q--;
		stpcpy(q, "/^NONBLOCK");
	}
	o.flags = oflags;
	o.path = p;
	o.cwd = cwd;
	SYNC_IO(&o, KERNEL$OPEN);
	if (__unlikely(o.status < 0)) {
		if (__unlikely(o.path == path)) goto really_err;
		o.path = path;
		SYNC_IO(&o, KERNEL$OPEN);
		if (__unlikely(o.status < 0)) {
			really_err:
			errno = -o.status;
			return -1;
		}
	}
	if (__unlikely(oflags & O_DIRECT) || (__unlikely(o.path == path) && __unlikely(oflags & O_NONBLOCK))) {
		open_mods(o.status, oflags);
	}
	return o.status;
}

#include "UNX1OPEN.I"
#define AT
#include "UNX1OPEN.I"

int creat(const char *path, mode_t mode)
{
	return open(path, O_WRONLY | O_CREAT | O_TRUNC, mode);
}

int close(int h)
{
	CLOSERQ c;
	c.h = h;
	/* cancel is specified in posix ... however it is not good idea,
	   because it can cause handle leak. Test cancel at least when
	   close finished */
	/*__cancel_in(SYNC_IO_CANCELABLE(&c, KERNEL$CLOSE));*/
	SYNC_IO_SIGINTERRUPT(&c, KERNEL$CLOSE);
	if (__unlikely(c.h != -1)) SYNC_IO(&c, KERNEL$CLOSE);
	__cancel_test();
	if (__unlikely(c.status < 0)) {
		errno = -c.status;
		return -1;
	}
	return 0;
}

int retry_eintr(int h)
{
	struct stat st;
	/* returning EINTR on non-blocking sockets will
	   cause that Xserver randomly disconnects clients */
	int fl = fcntl(h, F_GETFL);
	if (__unlikely(fl & O_NONBLOCK)) return 1;
	if (__unlikely(fstat(h, &st))) return 0;
	return __unlikely(S_ISREG(st.st_mode)) || __unlikely(S_ISDIR(st.st_mode)) || __unlikely(S_ISBLK(st.st_mode));
}

static int ioerror(int h, long status, int wr)
{
	/*{
		int x;
		static char buf[256];
		extern const char *__progname;
		_snprintf(buf, sizeof buf, "ioerror (%s): %d, %ld, %d\n", __progname, h, status, wr);
		x = open("HOME:/IOERROR.LOG", O_WRONLY | O_CREAT | O_APPEND);
		if (x != -1) {
			write(x, buf, strlen(buf));
			close(x);
		}
	}*/
	if (__unlikely(status == -ENOLNM && (unsigned)h <= 2)) {
		raise_sighup();
		if (!wr) return 0;
		status = -EIO;
	}
	if (__unlikely(status == -EPIPE) && __likely(wr)) _thread_raise(SIGPIPE);
	errno = -status;
	return -1;
}


ssize_t read(int h, void *buf, size_t size)
{
	SIORQ s;
	s.h = h;
	s.v.ptr = (unsigned long)buf;
	s.v.len = size;
	s.v.vspace = &KERNEL$VIRTUAL;
	s.progress = 0;
	/*__debug_printf("read(%d, %s, %p, %lx)\n", h, KERNEL$HANDLE_PATH(h), buf, (unsigned long)size);*/
	retry:
	__cancel_in(SYNC_IO_SIGINTERRUPT(&s, KERNEL$READ));
	if (__likely(s.status >= 0)) return s.status;
	if (__unlikely(s.status == -EINTR) && __unlikely(retry_eintr(s.h))) {
		if (__likely(s.v.len != 0)) goto retry;
		return s.progress;
	}
	if (__unlikely(s.progress != 0)) return s.progress;
	/*__debug_printf("read(%d, %s)->%ld\n", h, KERNEL$HANDLE_PATH(h), -s.status);*/
	return ioerror(s.h, s.status, 0);
}

ssize_t write(int h, const void *buf, size_t size)
{
	SIORQ s;
	s.h = h;
	s.v.ptr = (unsigned long)buf;
	s.v.len = size;
	s.v.vspace = &KERNEL$VIRTUAL;
	s.progress = 0;
	retry:
	__cancel_in(SYNC_IO_SIGINTERRUPT(&s, KERNEL$WRITE));
	if (__likely(s.status >= 0)) return s.status;
	if (__unlikely(s.status == -EINTR) && __unlikely(retry_eintr(s.h))) {
		if (__likely(s.v.len != 0)) goto retry;
		return s.progress;
	}
	if (__unlikely(s.progress != 0)) return s.progress;
	return ioerror(s.h, s.status, 1);
}

ssize_t pread(int h, void *buf, size_t size, off_t offset)
{
	int f;
	AIORQ a;
	a.h = h;
	a.v.ptr = (unsigned long)buf;
	f = KERNEL$HANDLE_FLAGS(h);
	if (__unlikely(f & _O_SOCKET)) {
		if (__unlikely(f == -1)) return -1;
		errno = ESPIPE;
		return -1;
	}
	a.v.len = size;
	if (__unlikely(!size)) return 0;
	a.v.vspace = &KERNEL$VIRTUAL;
	a.pos = offset;
	a.progress = 0;
	retry:
	__cancel_in(SYNC_IO_SIGINTERRUPT(&a, KERNEL$AREAD));
	if (__likely(a.status >= 0)) return a.status;
	if (__unlikely(a.status == -EINTR) && __unlikely(retry_eintr(a.h))) {
		if (__likely(a.v.len != 0)) goto retry;
		return a.progress;
	}
	if (__unlikely(a.progress != 0)) return a.progress;
	return ioerror(a.h, a.status, 0);
}

ssize_t pwrite(int h, const void *buf, size_t size, off_t offset)
{
	int f;
	AIORQ a;
	a.h = h;
	a.v.ptr = (unsigned long)buf;
	f = KERNEL$HANDLE_FLAGS(h);
	if (__unlikely(f & _O_SOCKET)) {
		if (__unlikely(f == -1)) return -1;
		errno = ESPIPE;
		return -1;
	}
	a.v.len = size;
	if (__unlikely(!size)) return 0;
	a.v.vspace = &KERNEL$VIRTUAL;
	a.pos = offset;
	a.progress = 0;
	retry:
	__cancel_in(SYNC_IO_SIGINTERRUPT(&a, KERNEL$AWRITE));
	if (__likely(a.status >= 0)) return a.status;
	if (__unlikely(a.status == -EINTR) && __unlikely(retry_eintr(a.h))) {
		if (__likely(a.v.len != 0)) goto retry;
		return a.progress;
	}
	if (__unlikely(a.progress != 0)) return a.progress;
	return ioerror(a.h, a.status, 1);
}

static int should_fcntl_nonblock(int h, int rwi)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_RWV_NONBLOCK;
	io.param = rwi;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	return io.status;
}

/* the _O_SOCKET test is useless but it saves one syscall */

#define rvwv(name, rw, rwi)						\
ssize_t name(int h, const struct iovec *vec, int count)		\
{									\
	ssize_t total_bytes = 0;					\
	int ofl = -2;							\
	while (count) {							\
		ssize_t s;						\
		if (__unlikely((ssize_t)vec->iov_len < 0)) goto einval;	\
		s = rw(h, vec->iov_base, vec->iov_len);			\
		if (__unlikely(s != vec->iov_len)) {			\
			if (__likely(s >= 0) || __likely(!total_bytes)) total_bytes += s;\
			goto ret;					\
		}							\
		total_bytes += s;					\
		if (__unlikely(total_bytes < 0)) {			\
			einval:						\
			errno = EINVAL;					\
			total_bytes = -1;				\
			goto ret;					\
		}							\
		vec++;							\
		count--;						\
		if (__unlikely(ofl == -2) && __likely(total_bytes != 0) && __likely(count)) {\
			int errno_save = errno;				\
			int f = KERNEL$HANDLE_FLAGS(h);			\
			ofl = -1;					\
			if (__likely(f & _O_SOCKET) || should_fcntl_nonblock(h, rwi)) {\
				ofl = fcntl(h, F_GETFL);		\
				if (__unlikely(ofl == -1)) continue;	\
				if (__likely(!(ofl & O_NONBLOCK))) {	\
					fcntl(h, F_SETFL, ofl | O_NONBLOCK);\
				} else {				\
					ofl = -1;			\
				}					\
			}						\
			errno = errno_save;				\
		}							\
	}								\
	ret:								\
	if (ofl >= 0) {							\
		int errno_save = errno;					\
		fcntl(h, F_SETFL, ofl);					\
		errno = errno_save;					\
	}								\
	return total_bytes;						\
}

rvwv(readv, read, PARAM_RWV_READ)
rvwv(writev, write, PARAM_RWV_WRITE)

off_t slow_lseek(int h, off_t offset, int whence)
{
	IOCTLRQ i;
	off_t oo[2];
	off_t *o;
	o = (off_t *)(((unsigned long)&oo + sizeof(off_t) - 1) & ~(unsigned long)(sizeof(off_t) - 1));
	*o = offset;
	i.h = h;
	i.ioctl = IOCTL_LSEEK;
	i.param = whence;
	i.v.ptr = (unsigned long)o;
	i.v.len = sizeof(off_t);
	i.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		if (i.status == -ENOOP) errno = ESPIPE;
		else errno = -i.status;
		return -1;
	}
	return *o;
}

int ftruncate(int h, off_t offset)
{
	IOCTLRQ i;
	off_t oo[2];
	off_t *o = (off_t *)(((unsigned long)&oo + sizeof(off_t) - 1) & ~(unsigned long)(sizeof(off_t) - 1));
	*o = offset;
	i.h = h;
	i.ioctl = IOCTL_TRUNCATE;
	i.param = 0;
	i.v.ptr = (unsigned long)o;
	i.v.len = sizeof(off_t);
	i.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		errno = -i.status;
		return -1;
	}
	return 0;
}

int truncate(const char *filename, off_t offset)
{
	int r;
	int h = open(filename, O_WRONLY | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = ftruncate(h, offset);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int mkdir(const char *filename, mode_t mode)
{
	return open(filename, O_CREAT | O_EXCL | _O_MKDIR | _O_CLOSE | _O_NOPOSIX, mode);
}

int mkdirat(int dirfd, const char *filename, mode_t mode)
{
	return openat(dirfd, filename, O_CREAT | O_EXCL | _O_MKDIR | _O_CLOSE | _O_NOPOSIX, mode);
}

int rmdir(const char *filename)
{
	return open(filename, _O_DELETE | _O_MKDIR | _O_CLOSE | _O_NOPOSIX);
}

int unlink(const char *filename)
{
	return open(filename, _O_DELETE | _O_CLOSE | _O_NOPOSIX);
}

int unlinkat(int dirfd, const char *filename, int flags)
{
	return openat(dirfd, filename, _O_DELETE | _O_CLOSE | _O_NOPOSIX | (flags & _O_MKDIR));
}

int linkat(int olddirfd, const char *oldpath, int newdirfd, const char *newpath, int flags)
{
	int save_errno = errno;
	int f, t;
	int r, w, p;
	char *buffer = __sync_malloc(PAGE_CLUSTER_SIZE);
	if (__unlikely(!buffer)) return -1;
	if (__unlikely((f = openat(olddirfd, oldpath, O_RDONLY | _O_NOPOSIX)) < 0)) {
		free(buffer);
		return -1;
	}
	if (__unlikely((t = openat(newdirfd, newpath, O_CREAT | O_WRONLY | O_TRUNC | O_EXCL | _O_NOPOSIX, 0644)) < 0)) {
		close(f);
		free(buffer);
		return -1;
	}
	rr:
	r = read(f, buffer, PAGE_CLUSTER_SIZE);
	if (__unlikely(r <= 0)) {
		struct stat st;
		int rr;
		if (!r && __likely(!fstat(f, &st))) {
			struct timeval tv[2];
			tv[0].tv_sec = st.st_atime;
			tv[0].tv_usec = 0;
			tv[1].tv_sec = st.st_mtime;
			tv[1].tv_usec = 0;
			futimes(t, tv);
		}
		close_error:
		rr = close(t);
		if (__likely(!r)) r = rr;
		close(f);
		if (__unlikely(r < 0)) unlinkat(newdirfd, newpath, 0);
		free(buffer);
		if (__likely(!r)) errno = save_errno;
		return r;
	}
	p = 0;
	ww:
	w = write(t, buffer + p, r - p);
	if (__unlikely(w <= 0)) {
		if (__unlikely(!w)) errno = EEOF;
		r = -1;
		goto close_error;
	}
	p += w;
	if (__unlikely(p < r)) goto ww;
	goto rr;
}

int link(const char *oldpath, const char *newpath)
{
	return linkat(AT_FDCWD, oldpath, AT_FDCWD, newpath, 0);
}

int _is_absolute(const char *path)
{
	const char *p = path;
	if (__unlikely(*p == '/')) return _ABS_CURROOT;
	while (1) {
		char c = *p++;
		if (__unlikely(c == ':')) {
			if (__likely(*p == '/')) return _ABS_TOTAL;
			else return _ABS_LOGREL;
		}
		if (__unlikely(c == '/')) return _ABS_NO;
		if (__unlikely(!c)) return _ABS_NO;
	}
}

int _is_local(const char *path)
{
	int r;
	const char *colon = strchr(path, ':');
	char *cpy;
	if (__unlikely(!colon)) return 1;
	cpy = __sync_malloc(colon - path + 3);
	if (__unlikely(!cpy)) return -1;
	memcpy(cpy, path, colon - path + 1);
	cpy[colon - path + 1] = '/';
	cpy[colon - path + 2] = 0;
	r = open(cpy, _O_NOACCESS | _O_NOPOSIX | _O_CLOSE);
	free(cpy);
	return !r;
}

char *_join_paths(const char *dir, const char *path)
{
	FFILE *f;
	OPENRQ rq;
	CWD cwd;
	CWDE cwde;
	unsigned s;
	if (__unlikely(!dir) || __unlikely(_is_absolute(dir) != _ABS_TOTAL)) {
		if (_is_absolute(path) == _ABS_TOTAL) {
			size_t sl = strlen(path) + 1;
			char *s = __sync_malloc(sl);
			if (__likely(s != NULL)) memcpy(s, path, sl);
			return s;
		}
		errno = ENOCWD;
		return NULL;
	}
	rq.path = path;
	rq.cwd = &cwd;
	make_cwd(cwd, cwde, dir);
	retry:
	f = open_file_name(&rq, &s);
	if (__unlikely(f == __ERR_PTR(-ENOMEM))) {
		if (__unlikely(KERNEL$MEMWAIT_SYNC(s))) {
			return NULL;
		}
		goto retry;
	}
	if (__unlikely(__IS_ERR(f))) {
		errno = -__PTR_ERR(f);
		return NULL;
	}
	memmove(f, f->path, strlen(f->path) + 1);
	return (char *)f;
}

int symlink(const char *oldpath, const char *newpath)
{
	return symlinkat(oldpath, AT_FDCWD, newpath);
}

int symlinkat(const char *oldpath, int newdirfd, const char *newpath)
{
	int h, r;
	char *p, *sl, *c, *fp;
	h = openat(newdirfd, newpath, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	p = KERNEL$HANDLE_PATH(h);
	sl = strrchr(p, '/');
	c = strchr(p, ':');
	if (__likely(c + 1 != sl)) {
		*sl = 0;	/* don't try this out of kernel */
	}
	fp = _join_paths(p, oldpath);
	KERNEL$FAST_CLOSE(h);
	if (__unlikely(!fp)) {
		return -1;
	}
	r = linkat(AT_FDCWD, fp, newdirfd, newpath, 0);
	free(fp);
	return r;
}

int mknod(const char *filename, mode_t mode, dev_t dev)
{
	return mknodat(AT_FDCWD, filename, mode, dev);
}

int mknodat(int dirfd, const char *filename, mode_t mode, dev_t dev)
{
	errno = EPERM;
	return -1;
}

int mkfifo(const char *pathname, mode_t mode)
{
	return mkfifoat(AT_FDCWD, pathname, mode);
}

int mkfifoat(int dirfd, const char *pathname, mode_t mode)
{
	errno = EPERM;
	return -1;
}

int readlink(const char *filename, char *buf, size_t bufsiz)
{
	return readlinkat(AT_FDCWD, filename, buf, bufsiz);
}

int readlinkat(int dirfd, const char *filename, char *buf, size_t bufsiz)
{
	errno = EINVAL;
	return -1;
}

int eaccess(const char *filename, int mode)
{
	return faccessat(AT_FDCWD, filename, mode, AT_EACCESS);
}

int access(const char *filename, int mode)
{
	return faccessat(AT_FDCWD, filename, mode, 0);
}

static const int acc[8] = {
_O_NOACCESS | _O_NOPOSIX | _O_CLOSE,
O_RDONLY | _O_NOPOSIX,
O_WRONLY | _O_NOPOSIX | _O_CLOSE,
O_RDWR | _O_NOPOSIX,
O_RDONLY | _O_NOPOSIX | _O_CLOSE,
O_RDONLY | _O_NOPOSIX,
O_RDWR | _O_NOPOSIX | _O_CLOSE,
O_RDWR | _O_NOPOSIX,
};

static const int accdir[8] = {
_O_NOACCESS | _O_NOPOSIX,
_O_NOACCESS | _O_NOPOSIX,
_O_NOACCESS | _O_NOPOSIX,
_O_NOACCESS | _O_NOPOSIX,
O_RDONLY | _O_NOPOSIX,
O_RDONLY | _O_NOPOSIX,
O_RDONLY | _O_NOPOSIX,
O_RDONLY | _O_NOPOSIX,
};

static int access_exec(int h, int mode, int type);
static int access_dir(int h, int mode);

int faccessat(int dirfd, const char *filename, int mode, int flags)
{
	int h, e;
	if (__unlikely(mode & ~(R_OK | W_OK | X_OK))) {
		errno = EINVAL;
		return -1;
	}
	e = errno;
	if (__unlikely((h = openat(dirfd, filename, acc[mode])) < 0)) {
		if (errno == EISDIR || errno == EACCES) {
			if ((h = openat(dirfd, filename, accdir[mode])) >= 0) {
				errno = e;
				return access_exec(h, mode, 2);
			}
		}
		if (mode == X_OK && errno == ENOENT) {
			h = openat(dirfd, filename, acc[mode] | _O_NOOPEN | _O_NOMULTIOPEN);
			if (h >= 0) {
				errno = e;
				return access_exec(h, mode, 1);
			}
			errno = ENOENT;
		}
		return -1;
	}
	if (__likely(!(mode & X_OK))) return 0;
	return access_exec(h, mode, 0);
}

static int access_exec(int h, int mode, int type)
{
	struct stat st;
	char *p, *q, *pp;
	int e, ee;
	if (__unlikely(type == 1)) goto test_new;
	if (__unlikely(fstat(h, &st))) goto close_ret_1;
	if (S_ISDIR(st.st_mode)) {
		return access_dir(h, mode);
	}
	if (type == 2) goto close_ret_1;
	if (__unlikely(S_ISFIFO(st.st_mode)) || __unlikely(S_ISCHR(st.st_mode)) || __unlikely(S_ISSOCK(st.st_mode))) {
		goto eaccess_close;
	}
	test_new:
	p = KERNEL$HANDLE_PATH(h);
	e = errno;
	if (is_cmd(p) || is_exec(p)) {
		ok:
		close(h);
		errno = e;
		return 0;
	}

	/* this should be consistent with execve */
	q = strrchr(p, '.');
	if (q) if (__unlikely(!_strcasecmp(q, ".EXE")) || __unlikely(!_strcasecmp(q, ".CMD"))) goto fail;
	pp = __sync_malloc(strlen(p) + 5);
	if (__unlikely(!pp)) goto fail;
	stpcpy(stpcpy(pp, p), ".EXE");
	ee = errno;
	if (is_exec(pp)) {
		free_ok:
		free(pp);
		goto ok;
	}
	stpcpy(stpcpy(pp, p), ".CMD");
	if (is_cmd(pp)) goto free_ok;
	free(pp);
	errno = ee;

	fail:
	if (errno == ENOEXEC) eaccess_close: errno = EACCES;
	close_ret_1:
	close(h);
	return -1;
}

static int access_dir(int h, int mode)
{
	IOCTLRQ io;
	io.h = h;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	io.ioctl = IOCTL_GET_CAPABILITIES;
	io.param = 0;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		if (io.status == -ENOOP) errno = EINVAL;
		else errno = -io.status;
		return -1;
	}
	if (mode & R_OK && __unlikely(!(io.status & CAP_SCANDIR))) goto fail;
	if (mode & W_OK && __unlikely(!(io.status & CAP_CREATE))) goto fail;
	if (mode & X_OK && __unlikely(!(io.status & CAP_LOOKUP))) goto fail;
	close(h);
	return 0;
	fail:
	close(h);
	errno = EACCES;
	return -1;
}

static int _emptystat(IOCTLRQ *i)
{
#define buf	((struct stat *)(unsigned long)i->v.ptr)
	int flg;
	char *p;
#if 0
	ino_t hash;
#endif
	memset(buf, 0, sizeof(struct stat));
	flg = KERNEL$HANDLE_FLAGS(i->h);
	if (__unlikely(flg == -1)) {
		errno = EBADF;
		return -1;
	}
	if (__unlikely(i->status != -ENOOP) && !(flg & (_O_PIPE | _O_SOCKET))) {
		errno = -i->status;
		return -1;
	}
	p = KERNEL$HANDLE_PATH(i->h);
	if (__unlikely(!p)) {
		errno = EBADF;
		return -1;
	}
#if 0
	hash = 0;
	quickcasehash(p, *p != ':', hash);
	buf->st_dev = hash;
	quickcasehash(p, *p, hash);
	buf->st_ino = buf->st_rdev = hash;
#endif
	if (__unlikely(flg & _O_SOCKET)) {
		buf->st_mode = S_IFSOCK | 0600;
		buf->st_blksize = 512;
	} else if (__unlikely(flg & _O_PIPE)) {
		buf->st_mode = S_IFIFO | 0600;
		buf->st_blksize = 1 << __BSR(PIPE_SIZE);
	} else {
		buf->st_mode = S_IFCHR | 0644;
		buf->st_blksize = PAGE_CLUSTER_SIZE;
	}
	buf->st_nlink = 1;
	buf->st_uid = 1;
	buf->st_gid = 1;
	buf->st_atime = buf->st_mtime = buf->st_ctime = time(NULL);
	return 0;
#undef buf
}

int fstat(int h, struct stat *buf)
{
	IOCTLRQ i;
	i.h = h;
	i.ioctl = IOCTL_STAT;
	i.param = 0;
	i.v.ptr = (unsigned long)buf;
	i.v.len = sizeof(struct stat);
	i.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		return _emptystat(&i);
	}
	return 0;
}

int stat(const char *filename, struct stat *buf)
{
	int r;
	int h = open(filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fstat(h, buf);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int lstat(const char *filename, struct stat *buf)
{
	return stat(filename, buf);
}

int fstatat(int dirfd, const char *filename, struct stat *buf, int flags)
{
	int r;
	int h = openat(dirfd, filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fstat(h, buf);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int fstatfs(int h, struct statfs *buf)
{
	IOCTLRQ i;
	i.h = h;
	i.ioctl = IOCTL_STATFS;
	i.param = 0;
	i.v.ptr = (unsigned long)buf;
	i.v.len = sizeof(struct statfs);
	i.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		errno = -i.status;
		return -1;
	}
	return 0;
}

int statfs(const char *filename, struct statfs *buf)
{
	const char *p, *pp;
	int r;
	int spl;
	int h = open(filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fstatfs(h, buf);
	if (__unlikely(r)) goto cl_ret_err;
	spl = KERNEL$SPL;
	if (__likely(SPLX_BELOW(spl, SPL_X(SPL_ZERO)))) RAISE_SPL(SPL_BOTTOM);
	if (_is_absolute(filename) & _ABS_LOGREL) {
		p = filename;
	} else {
		p = KERNEL$GET_CWD(KERNEL$CWD());
		if (__unlikely(!p)) goto no_cwd;
	}
	pp = strchr(p, ':');
	if (__likely(pp != p) && __unlikely(pp[-1] == '.')) buf->f_flags |= MNT_UNION;
	no_cwd:
	LOWER_SPLX(spl);
	KERNEL$FAST_CLOSE(h);
	return 0;
	cl_ret_err:
	KERNEL$FAST_CLOSE(h);
	return -1;
}

static void statfs2statvfs(struct statvfs *vfs, struct statfs *fs)
{
	memset(vfs, 0, sizeof(struct statvfs));
	if (fs->f_flags & MNT_RDONLY) vfs->f_flag |= ST_RDONLY;
	vfs->f_frsize = fs->f_bsize;
	vfs->f_bsize = fs->f_iosize;
	vfs->f_blocks = fs->f_blocks;
	vfs->f_bfree = fs->f_bfree;
	vfs->f_bavail = fs->f_bavail;
	vfs->f_files = fs->f_files;
	vfs->f_ffree = fs->f_ffree;
	vfs->f_favail = fs->f_favail;
	vfs->f_fsid = fs->f_fsid.val[0];
	vfs->f_namemax = fs->f_namelen;
	strlcpy(vfs->f_basetype, fs->f_fstypename, sizeof vfs->f_basetype);
}

int fstatvfs(int h, struct statvfs *vfs)
{
	int r;
	struct statfs fs;
	if (__unlikely(r = fstatfs(h, &fs))) return r;
	statfs2statvfs(vfs, &fs);
	return 0;
}

int statvfs(const char *filename, struct statvfs *vfs)
{
	int r;
	struct statfs fs;
	if (__unlikely(r = statfs(filename, &fs))) return r;
	statfs2statvfs(vfs, &fs);
	return 0;
}

int CHECK_HANDLE(int h)
{
	IOCTLRQ io;
	/*if (__unlikely(!KERNEL$HANDLE_PATH(h))) {
		errno = EBADF;
		return -1;
	}*/
	io.h = h;
	io.ioctl = IOCTL_NOP;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__likely(!io.status) || __likely(io.status == -ENOOP))
		return 0;
	errno = -io.status;
	return -1;
}

int fchmod(int h, mode_t mode)
{
	return CHECK_HANDLE(h);
}

int chmod(const char *filename, mode_t mode)
{
	int r;
	int h = open(filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fchmod(h, mode);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int fchmodat(int dirfd, const char *filename, mode_t mode, int flags)
{
	int r;
	int h = openat(dirfd, filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fchmod(h, mode);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int fchown(int h, uid_t owner, gid_t group)
{
	return CHECK_HANDLE(h);
}

int chown(const char *filename, uid_t owner, gid_t group)
{
	int r;
	int h = open(filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fchown(h, owner, group);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int lchown(const char *filename, uid_t owner, gid_t group)
{
	return chown(filename, owner, group);
}

int fchownat(int dirfd, const char *filename, uid_t owner, gid_t group, int flags)
{
	int r;
	int h = openat(dirfd, filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fchown(h, owner, group);
	KERNEL$FAST_CLOSE(h);
	return r;
}

static mode_t sys_umask;

mode_t umask(mode_t mask)
{
	mode_t p = sys_umask;
	sys_umask = mask;
	return p;
}

int futimes(int h, const struct timeval *times)
{
	IOCTLRQ i;
	struct timeval tv[2];
	if (__unlikely(!times)) {
		memset(tv, 0, sizeof tv);
		tv[0].tv_sec = time(&tv[1].tv_sec);
		times = tv;
	}
	i.h = h;
	i.ioctl = IOCTL_UTIMES;
	i.param = 0;
	i.v.ptr = (unsigned long)times;
	i.v.len = sizeof(struct timeval) * 2;
	i.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		errno = -i.status;
		return -1;
	}
	return 0;
}

int utimes(const char *filename, const struct timeval *times)
{
	int r;
	int h = open(filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = futimes(h, times);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int lutimes(const char *filename, const struct timeval *times)
{
	return utimes(filename, times);
}

int futimesat(int dirfd, const char *filename, const struct timeval *times)
{
	int r;
	int h = openat(dirfd, filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = futimes(h, times);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int utime(const char *filename, const struct utimbuf *timep)
{
	struct timeval tv[2];
	memset(tv, 0, sizeof tv);
	if (__likely(timep != NULL)) {
		tv[0].tv_sec = timep->actime;
		tv[1].tv_sec = timep->modtime;
	} else {
		tv[0].tv_sec = time(&tv[1].tv_sec);
	}
	return utimes(filename, tv);
}

int fsync(int fd)
{
	IOCTLRQ i;
	i.h = fd;
	i.ioctl = IOCTL_FSYNC;
	i.param = 0;
	i.v.ptr = 0;
	i.v.len = 0;
	i.v.vspace = &KERNEL$VIRTUAL;
	retry:
	__cancel_test();
	SYNC_IO_SIGINTERRUPT(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		if (i.status == -ENOOP) return 0;
		if (__unlikely(i.status == -EINTR) && __unlikely(retry_eintr(i.h))) goto retry;
		errno = -i.status;
		return -1;
	}
	return 0;
}

int fdatasync(int fd)
{
	IOCTLRQ i;
	i.h = fd;
	i.ioctl = IOCTL_FSYNC;
	i.param = PARAM_FSYNC_ONLY_DATA;
	i.v.ptr = 0;
	i.v.len = 0;
	i.v.vspace = &KERNEL$VIRTUAL;
	retry:
	SYNC_IO_SIGINTERRUPT(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		if (i.status == -ENOOP) return 0;
		if (__unlikely(i.status == -EINTR) && __unlikely(retry_eintr(i.h))) goto retry;
		errno = -i.status;
		return -1;
	}
	return 0;
}

int chdir(const char *path)
{
	CWD *c;
	CDRQ cr;
	c = KERNEL$CWD();
	cr.cwd = c;
	cr.path = path;
	SYNC_IO(&cr, KERNEL$CD);
	if (__unlikely(cr.status < 0)) {
		errno = -cr.status;
		return -1;
	}
	if (__unlikely(!c)) SET_THREAD_CWD(cr.cwd);
	return 0;
}

int fchdir(int fd)
{
	char *c = KERNEL$HANDLE_PATH(fd);
	if (__unlikely(!c)) {
		errno = EBADF;
		return -1;
	}
	return chdir(c);
}

static char *put_string(char *buf, size_t size, const char *string, int auto_retry)
{
	size_t l = strlen(string) + 1;
	if (!size) {
		if (__unlikely(buf != NULL)) {
			errno = EINVAL;
			return NULL;
		}
		size = l;
	} else {
		if (__unlikely(size < l)) {
			errno = ERANGE;
			return NULL;
		}
	}
	if (!buf) {
		alloc_again:
		buf = malloc(size);
		if (__unlikely(!buf)) {
			buf = (char *)!KERNEL$MEMWAIT_SYNC(size);
			if (auto_retry & (unsigned long)buf) goto alloc_again;
			return buf;
		}
	}
	memcpy(buf, string, l);
	return buf;
}

char *get_current_dir_name(void)
{
	return getcwd(NULL, 0);
}

char *getwd(char *buf)
{
	return getcwd(buf, PATH_MAX);
}

char *getcwd(char *buf, size_t size)
{
	char *c, *ret;
	int spl;
	if (__likely(SPLX_BELOW(spl = KERNEL$SPL, SPL_X(SPL_BOTTOM)))) RAISE_SPL(SPL_BOTTOM);
	retry:
	c = KERNEL$GET_CWD(KERNEL$CWD());
	if (__unlikely(!c)) {
		errno = ENOCWD;
		ret = NULL;
		goto ret;
	}
	ret = put_string(buf, size, c, 0);
	if (__unlikely(ret == (void *)1)) goto retry;
	if (__likely(ret != NULL)) __upcase(ret);
	ret:
	LOWER_SPLX(spl);
	return ret;
}

char *getcwdat(int dirfd, char *buf, size_t size)
{
	char *c, *ret;
	if (__unlikely(dirfd == AT_FDCWD)) return getcwd(buf, size);
	c = KERNEL$HANDLE_PATH(dirfd);
	if (__unlikely(!c)) {
		errno = EBADF;
		return NULL;
	}
	ret = put_string(buf, size, c, 1);
	if (__likely(ret != NULL)) __upcase(ret);
	return ret;
}

char *_fullpath(char *buf, const char *path, size_t size)
{
	return _fullpathat(buf, AT_FDCWD, path, size);
}

char *_fullpathat(char *buf, int dirfd, const char *path, size_t size)
{
	char *str;
	int h = openat(dirfd, path, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return NULL;
	str = put_string(buf, size, KERNEL$HANDLE_PATH(h), 1);
	KERNEL$FAST_CLOSE(h);
	return str;
}

long fpathconf(int fd, int pc)
{
	struct statfs statfs;
	IOCTLRQ io;
	switch (pc) {
		case _PC_LINK_MAX: return 1;
		case _PC_MAX_CANON:
		case _PC_MAX_INPUT:
			io.h = fd;
			io.ioctl = IOCTL_FPATHCONF;
			io.param = pc;
			io.v.ptr = 0;
			io.v.len = 0;
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
			if (__unlikely(io.status < 0)) {
				errno = -io.status;
				return -1;
			}
			return io.status;
		case _PC_NAME_MAX:
			if (fstatfs(fd, &statfs)) return -1;
			return statfs.f_namelen;
		case _PC_PATH_MAX: return PATH_MAX;
		case _PC_PIPE_BUF: return PIPE_BUF;
		case _PC_CHOWN_RESTRICTED: return 1;
		case _PC_NO_TRUNC: return 0;
		case _PC_VDISABLE: return 1;
		default:
			errno = EINVAL;
			return -1;
	}
}

long pathconf(const char *filename, int pc)
{
	long r;
	int h = open(filename, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = fpathconf(h, pc);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int rename(const char *oldpath, const char *newpath)
{
	int r;
	int h = open(oldpath, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = open(newpath, _O_NOACCESS | _O_NOPOSIX | _O_RENAME | _O_CLOSE, h);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int renameat(int olddirfd, const char *oldpath, int newdirfd, const char *newpath)
{
	int r;
	int h = openat(olddirfd, oldpath, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
	if (__unlikely(h < 0)) return h;
	r = openat(newdirfd, newpath, _O_NOACCESS | _O_NOPOSIX | _O_RENAME | _O_CLOSE, h);
	KERNEL$FAST_CLOSE(h);
	return r;
}

int sync(void)
{
	int i;
	LOGICAL_LIST_REQUEST llr;
	llr.prefix = "";
	SYNC_IO(&llr, KERNEL$LIST_LOGICALS);
	if (__unlikely(llr.status < 0)) {
		errno = -llr.status;
		return -1;
	}
	for (i = 0; i < llr.n_entries; i++) {
		IOCTLRQ io;
		char *name = __sync_malloc(strlen(llr.entries[i]->name) + 3);
		int h;
		if (__unlikely(!name)) {
			KERNEL$FREE_LOGICAL_LIST(&llr);
			return -1;
		}
		strcpy(stpcpy(name, llr.entries[i]->name), ":/");
		h = open(name, _O_NOACCESS | _O_NOPOSIX | _O_NOOPEN);
		free(name);
		if (__unlikely(h < 0)) continue;
		io.h = h;
		io.ioctl = IOCTL_FSYNC;
		io.param = PARAM_FSYNC_ASYNC | PARAM_FSYNC_TREE;
		io.v.ptr = 0;
		io.v.len = 0;
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&io, KERNEL$IOCTL);
		KERNEL$FAST_CLOSE(h);
	}
	KERNEL$FREE_LOGICAL_LIST(&llr);
	return 0;
}

int pipe(int *fd)
{
	FPIPE *p;
	int spl;
	int h0, h1;
	int r;
	if (__unlikely((h0 = open("SYS$PIPE:/R", O_RDONLY | _O_PIPE)) < 0)) {
		ret_err_enosys:
		if (errno == ENOLNM) errno = ENOSYS;
		return -1;
	}
	if (__unlikely((h1 = open("SYS$PIPE:/W", O_WRONLY | _O_PIPE)) < 0)) {
		close(h0);
		goto ret_err_enosys;
	}
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), KERNEL$SPL)))
		KERNEL$SUICIDE("pipe AT SPL %08X", KERNEL$SPL);
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_DEV);
	while (__unlikely(!(p = KERNEL$ALLOC_KERNEL_PAGE(VM_TYPE_WIRED_MAPPED)))) {
		if (__unlikely(r = KERNEL$MEMWAIT_SYNC(PAGE_CLUSTER_SIZE))) {
			close(h0);
			close(h1);
			LOWER_SPLX(spl);
			errno = -r;
			return -1;
		}
	}
	p->l_refcount = 2;
	p->readers = 1;
	p->writers = 1;
	p->start = 0;
	p->end = 0;
	p->read_flags = 0;
	p->write_flags = 0;
	set_pipe_buffer(h0, p);
	set_pipe_buffer(h1, p);
	fd[0] = h0;
	fd[1] = h1;
	LOWER_SPLX(spl);
	return 0;
}

