#include <SPAD/READDIR.H>
#include <DIRENT.H>
#include <SPAD/SYNC.H>
#include <SPAD/CD.H>
#include <STDLIB.H>
#include <UNISTD.H>

struct __dir {
	READDIR_RQ r;
	unsigned long pos;
	int fd;
	char *name;
	MTX mtx;
};

DIR *opendir(const char *name)
{
	READDIR_RQ r;
	DIR *d = __sync_malloc(sizeof(DIR));
	int spl;
	if (__unlikely(!d)) return NULL;
	spl = KERNEL$SPL;
	if (__likely(SPLX_BELOW(spl, SPL_X(SPL_BOTTOM)))) RAISE_SPL(SPL_BOTTOM);
	d->name = _join_paths(KERNEL$GET_CWD(KERNEL$CWD()), name);
	if (__unlikely(!d->name)) {
		LOWER_SPLX(spl);
		goto free_d_ret;
	}
	LOWER_SPLX(spl);
	d->fd = -1;
	r.cwd = NULL;
	r.flags = 0;
	r.path = d->name;
	SYNC_IO(&r, KERNEL$READDIR);
	if (__unlikely(r.status != 0)) {
		errno = -r.status;
		free(d->name);
		free_d_ret:
		free(d);
		return NULL;
	}
	memcpy(&d->r, &r, sizeof(READDIR_RQ));
	MTX_INIT(&d->mtx, "KERNEL$DIR");
	d->pos = 0;
	return d;
}

DIR *fdopendir(int fd)
{
	DIR *d;
	const char *name = KERNEL$HANDLE_PATH(fd);
	if (__unlikely(!name)) {
		errno = EBADF;
		return NULL;
	}
	d = opendir(name);
	if (__unlikely(!d)) return NULL;
	d->fd = fd;
	return d;
}

struct dirent *readdir(DIR *d)
{
	if (__unlikely(d->pos >= d->r.n_entries)) return NULL;
	return d->r.entries[d->pos++];
}

int readdir_r(DIR *d, struct dirent *entry, struct dirent **result)
{
	struct dirent *de;
	MTX_LOCK_SYNC(&d->mtx);
	de = readdir(d);
	if (__unlikely(!de)) {
		*result = NULL;
	} else {
		memcpy(*result = entry, de, de->d_reclen);
	}
	MTX_UNLOCK(&d->mtx);
	return 0;
}

void rewinddir(DIR *d)
{
	READDIR_RQ r;
	r.cwd = NULL;
	r.flags = 0;
	r.path = d->name;
	SYNC_IO(&r, KERNEL$READDIR);
	MTX_LOCK_SYNC(&d->mtx);
	KERNEL$FREE_READDIR(&d->r);
	if (__likely(!r.status)) {
		memcpy(&d->r, &r, sizeof(READDIR_RQ));
	}
	MTX_UNLOCK(&d->mtx);
	d->pos = 0;
}

int closedir(DIR *d)
{
	if (__unlikely(d->fd != -1)) KERNEL$FAST_CLOSE(d->fd);
	KERNEL$FREE_READDIR(&d->r);
	free(d->name);
	free(d);
	return 0;
}

off_t telldir(DIR *d)
{
	unsigned long pos;
	MTX_LOCK_SYNC(&d->mtx);
	pos = d->pos;
	MTX_UNLOCK(&d->mtx);
	return pos;
}

void seekdir(DIR *d, off_t offset)
{
	MTX_LOCK_SYNC(&d->mtx);
	d->pos = offset;
	MTX_UNLOCK(&d->mtx);
}

int dirfd(const DIR *d_)
{
	DIR *d = (DIR *)d_;	/* lame but makes prototype ok */
	MTX_LOCK_SYNC(&d->mtx);
	if (d->fd < 0) {
		d->fd = open(d->name, O_RDONLY | _O_NOOPEN);
	}
	MTX_UNLOCK(&d->mtx);
	return d->fd;
}
