#include <ERRNO.H>
#include <STDARG.H>
#include <STRING.H>
#include <TERMIOS.H>
#include <SPAD/IOCTL.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SIGNAL.H>
#include <LIB/KERNEL/USIGNAL.H>
#include <SYS/STAT.H>
#include <UNISTD.H>
#include <VALUES.H>
#include <SYS/TYPES.H>
#include <SPAD/TTY.H>
#include <PATHS.H>
#include <SPAD/READDIR.H>
#include <DIRENT.H>
#include <SPAD/DL.H>

#include <SYS/IOCTL.H>
#include <SYS/KD.H>
#include <KERNEL/DEV.H>

#define IOC_USE_TTY
#include <SPAD/IOC.H>

IOC_GEN_GET_MODE
IOC_GEN_GET_MODE_DATA
IOC_GEN_SET_MODE_INT

static int kd_get_mode(int h);

unsigned char intr_key = 3;
unsigned char quit_key = 28;
/*unsigned char stop_key = 26;*/

static unsigned char set_intr_key = 3;
static unsigned char set_quit_key = 28;
static unsigned char set_stop_key = 26;

static int get_tty(int h, struct termios *t)
{
	int r;
	if (__unlikely((r = get_mode(h, IOCTL_TTY_GET_TERM_FLAGS, 0, 1)) < 0)) return r;
	memset(t, 0, sizeof(struct termios));
	t->c_iflag = IGNBRK | (r & TTYF_RAW ? 0 : ICRNL);
	t->c_oflag = r & TTYF_O_NL_CRNL ? OPOST | ONLCR : 0;
	t->c_cflag = CREAD | CS8;
	t->c_lflag = (r & TTYF_RAW ? 0 : ICANON) | (r & TTYF_NOECHO ? 0 : ECHOCTL | ECHO | ECHOK | ECHOE | ECHOKE) | ISIG;
	t->c_cc[VEOF] = 4;
	t->c_cc[VERASE] = 127;
	t->c_cc[VKILL] = 21;
	t->c_cc[VERASE2] = 8;
	t->c_cc[VLNEXT] = 22;
	t->c_cc[VMIN] = 1;
	t->c_cc[VINTR] = set_intr_key;
	t->c_cc[VQUIT] = set_quit_key;
	t->c_cc[VSUSP] = set_stop_key;
	if (__unlikely(set_intr_key != intr_key) || __unlikely(set_quit_key != quit_key)) t->c_lflag &= ~ISIG;
	t->c_ispeed = B38400;
	t->c_ospeed = B38400;
	r = get_mode(h, IOCTL_TTY_GET_SPEED, PARAM_ISPEED, 0);
	if (__likely(r >= 0)) t->c_ispeed = r;
	r = get_mode(h, IOCTL_TTY_GET_SPEED, PARAM_OSPEED, 0);
	if (__likely(r >= 0)) t->c_ospeed = r;
	return 0;
}

static int set_tty(int h, struct termios *t)
{
	int r;
	unsigned qk;
	int kd = kd_get_mode(h);
	int ignore_signal = kd >= 0 && kd != K_XLATE && kd != K_UNICODE;
	set_intr_key = t->c_cc[VINTR];
	qk = !ignore_signal && t->c_lflag & ISIG ? set_intr_key : 0;
	if (__unlikely(intr_key != qk)) {
		INSTALL_INTR_KEY(0, intr_key, 0);
		INSTALL_INTR_KEY(0, intr_key = qk, SIGINT);
	}
	set_quit_key = t->c_cc[VQUIT];
	qk = !ignore_signal && t->c_lflag & ISIG ? set_quit_key : 0;
	if (__unlikely(quit_key != qk)) {
		INSTALL_INTR_KEY(0, quit_key, 0);
		INSTALL_INTR_KEY(0, quit_key = qk, SIGQUIT);
	}
	set_stop_key = t->c_cc[VSUSP];
/*
	qk = !ignore_signal && t->c_lflag & ISIG ? set_stop_key : 0;
	if (__unlikely(stop_key != qk)) {
		INSTALL_INTR_KEY(0, stop_key, 0);
		INSTALL_INTR_KEY(0, stop_key = qk, SIGTSTP);
	}
*/
	if (__unlikely(r = set_mode_int(h, "ECHO", ((unsigned)t->c_lflag / ECHO) & 1, 1))) return r;
	if (__unlikely(r = set_mode_int(h, "ICANON", ((unsigned)t->c_lflag / ICANON) & 1, 1))) return r;
	if (__unlikely(r = set_mode_int(h, "ONLCR", (t->c_oflag & (OPOST | ONLCR)) == (OPOST | ONLCR), 1))) return r;
	read(h, NULL, 0);
	return 0;
}

static int set_tty_x(int h, struct termios *t)
{
	int er, r;
	r = set_tty(h, t);
	if (__unlikely((unsigned)h >= 3) || __unlikely(r)) return r;
	er = errno;
	if (h != 0) set_tty(0, t);
	if (h != 1) set_tty(1, t);
	if (h != 2) set_tty(2, t);
	if (!r) errno = er;
	return 0;
}

static __finline__ int drain_tty(int h)
{
	return get_mode(h, IOCTL_FSYNC, 0, 1);
}

static __finline__ int flush_tty(int h, int type)
{
	return get_mode(h, IOCTL_TTY_FLUSH, type, 1);
}

static int tty_get_winsize(int h, struct winsize *w)
{
	int r;
	struct tty_window_size ws;
	if (__unlikely((r = get_mode_data(h, IOCTL_TTY_GET_WINSIZE, 0, &ws, sizeof ws, 1)) < 0)) return r;
	w->ws_col = ws.ws_col;
	w->ws_row = ws.ws_row;
	w->ws_xpixel = ws.ws_xpixel;
	w->ws_ypixel = ws.ws_ypixel;
	return 0;
}

static int tty_set_winsize(int h, struct winsize *w)
{
	char valstr[48];
	CHHRQ ch;
	if (__unlikely(_snprintf(valstr, sizeof(valstr), "%d,%d,%d,%d", w->ws_col, w->ws_row, w->ws_xpixel, w->ws_ypixel) < 0)) {
		errno = ERANGE;
		return -1;
	}
	ch.h = h;
	ch.option = "WINSIZE";
	ch.value = valstr;
	SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	if (__unlikely(ch.status < 0)) {
		if (ch.status == -EBADMOD || ch.status == -ENOENT || ch.status == -ENOOP) ch.status = -ENOTTY;
		if (ch.status == -ENOLNM && (unsigned)ch.h <= 2) {
			raise_sighup();
			ch.status = -EIO;
		}
		errno = -ch.status;
		return -1;
	}
	return 0;
}

static int nread(int h, int *n)
{
	IOCTLRQ io;
	io.h = h;
	io.ioctl = IOCTL_NREAD;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status == -ENOOP)) {
		struct stat st;
		off_t s;
		if (fstat(h, &st)) return -1;
		s = lseek(h, 0, SEEK_CUR);
		if (__unlikely(s == (off_t)-1)) return -1;
		if (__unlikely(s > st.st_size)) *n = 0;
		else if (__unlikely(st.st_size - s > MAXINT)) *n = MAXINT;
		else *n = st.st_size - s;
		return 0;
	}
	if (__unlikely(io.status < 0)) {
		errno = -io.status;
		return -1;
	}
	if (__unlikely(io.status >= MAXINT)) *n = MAXINT;
	else *n = io.status;
	return 0;
}

static int kd_get_mode(int h)
{
	return get_mode(h, IOCTL_TTY_GETKBDMODE, 0, 1);
}

static int kd_set_mode(int h, int mode)
{
	struct termios t;
	int r = set_mode_int(h, "KBDRAW", mode, 1);
	if (__unlikely(r)) return r;
	r = get_tty(h, &t);
	if (__unlikely(r)) {
		read_ret:
		read(h, NULL, 0);
		return r;
	}
	r = set_tty(h, &t);
	if (__unlikely(r)) goto read_ret;
	return 0;
}

static int kd_get_leds(int h)
{
	return get_mode(h, IOCTL_TTY_GETLED, 0, 1);
}

static int kd_set_leds(int h, int leds)
{
	int r = set_mode_int(h, "KBDLED", leds, 1);
	if (__unlikely(r)) return r;
	read(h, NULL, 0);
	return 0;
}


struct extender_library {
	struct extender_library *next;
	DLRQ dl;
	translate_unx_fn *translate_unx;
};

static struct extender_library *extender_libraries = NULL;

static int scan_extenders(int (*try_this)(DLRQ *dl, void *data), void *data);
static int ioc_extender(DLRQ *dl, void *data);

struct ioc_pack {
	int h;
	unsigned long io;
	va_list va;
	int retval;
};

struct ioc_hash_entry {
	struct ioc_hash_entry *next;
	unsigned long io;
	ioc_fn *fn;
};

#define IOC_HASH_SIZE	256
#define IOC_HASH(io)	((unsigned)((io) + (((io) >> 8) * 35)) % IOC_HASH_SIZE)

static struct ioc_hash_entry *ioc_hash[IOC_HASH_SIZE] = { NULL };

int ioctl(int h, unsigned long io, ...)
{
	struct ioc_pack pack;
	switch (io) {
		case TIOCGETA: {
			struct termios *t;
			va_start(pack.va, io);
			t = va_arg(pack.va, struct termios *);
			va_end(pack.va);
			return get_tty(h, t);
		}
		case TIOCSETA: {
			struct termios *t;
			va_start(pack.va, io);
			t = va_arg(pack.va, struct termios *);
			va_end(pack.va);
			return set_tty_x(h, t);
		}
		case TIOCSETAW: {
			struct termios *t;
			drain_tty(h);
			va_start(pack.va, io);
			t = va_arg(pack.va, struct termios *);
			va_end(pack.va);
			return set_tty_x(h, t);
		}
		case TIOCSETAF: {
			struct termios *t;
			drain_tty(h);
			flush_tty(h, FREAD);
			va_start(pack.va, io);
			t = va_arg(pack.va, struct termios *);
			va_end(pack.va);
			return set_tty_x(h, t);
		}
		case TIOCSPGRP: {
			return CHECK_HANDLE(h);
		}
		case TIOCGPGRP: {
			pid_t *i;
			if (__unlikely(CHECK_HANDLE(h)))
				return -1;
			va_start(pack.va, io);
			i = va_arg(pack.va, pid_t *);
			va_end(pack.va);
			*i = getpgrp();
			return 0;
		}
		case TIOCSBRK: {
			return CHECK_HANDLE(h);
		}
		case TIOCDRAIN: {
			return drain_tty(h);
		}
		case TIOCFLUSH: {
			int c;
			va_start(pack.va, io);
			c = va_arg(pack.va, int);
			va_end(pack.va);
			return flush_tty(h, c);
		}
		case TIOCSTOP: {
			return CHECK_HANDLE(h);
		}
		case TIOCSTART: {
			return CHECK_HANDLE(h);
		}
		case TIOCGWINSZ: {
			struct winsize *w;
			va_start(pack.va, io);
			w = va_arg(pack.va, struct winsize *);
			va_end(pack.va);
			return tty_get_winsize(h, w);
		}
		case TIOCSWINSZ: {
			struct winsize *w;
			va_start(pack.va, io);
			w = va_arg(pack.va, struct winsize *);
			va_end(pack.va);
			return tty_set_winsize(h, w);
		}
		case FIOCLEX: {
			return fcntl(h, F_SETFD, FD_CLOEXEC);
		}
		case FIONCLEX: {
			return fcntl(h, F_SETFD, 0);
		}
		case FIONBIO: {
			int *c;
			int fl;
			va_start(pack.va, io);
			c = va_arg(pack.va, int *);
			va_end(pack.va);
			if (__unlikely((fl = fcntl(h, F_GETFL)) == -1)) return -1;
			return fcntl(h, F_SETFL, (fl & ~O_NONBLOCK) | (*c ? O_NONBLOCK : 0));
		}
		case FIONREAD: {
			int *n;
			va_start(pack.va, io);
			n = va_arg(pack.va, int *);
			va_end(pack.va);
			return nread(h, n);
		}
		case KDGKBMODE: {
			int c = kd_get_mode(h);
			if (__unlikely(c < 0)) return c;
			va_start(pack.va, io);
			*va_arg(pack.va, int *) = c;
			va_end(pack.va);
			return 0;
		}
		case KDSKBMODE: {
			int c;
			va_start(pack.va, io);
			c = va_arg(pack.va, int);
			va_end(pack.va);
			return kd_set_mode(h, c);
		}
		case KDGETLED: {
			int c = kd_get_leds(h);
			if (__unlikely(c < 0)) return c;
			va_start(pack.va, io);
			*va_arg(pack.va, char *) = c;
			va_end(pack.va);
			return 0;
		}
		case KDSETLED: {
			int c;
			va_start(pack.va, io);
			c = va_arg(pack.va, int);
			va_end(pack.va);
			return kd_set_leds(h, c);
		}
		default: {
			int r;
			struct ioc_hash_entry *e;
			va_start(pack.va, io);
			for (e = ioc_hash[IOC_HASH(io)]; e; e = e->next)
				if (__likely(e->io == io)) {
					r = e->fn(h, io, pack.va);
					va_end(pack.va);
					return r;
				}
			pack.h = h;
			pack.io = io;
			r = scan_extenders(ioc_extender, &pack);
			va_end(pack.va);
			if (__unlikely(r)) {
				/*if (r > 0) __debug_printf("UNKNOWN IOCTL %x\n", io);*/
				errno = r >= 0 ? ENOTTY : -r;
				return -1;
			}
			return pack.retval;
		}
	}
}

static int ioc_extender(DLRQ *dl, void *data)
{
	struct ioc_pack *pack = data;
	unsigned long ioc_table_ptr;
	struct ioc_entry *ioc_table;
	if (__unlikely(KERNEL$DL_GET_SYMBOL(dl, "IOC$TABLE", &ioc_table_ptr)))
		return 1;
	for (ioc_table = *(struct ioc_entry **)ioc_table_ptr; ioc_table->fn; ioc_table++) {
		if (__unlikely(pack->io >= ioc_table->ioctl) &&
		    __likely(pack->io < ioc_table->ioctl + ioc_table->n)) {
			pack->retval = ioc_table->fn(pack->h, pack->io, pack->va);
			return 0;
		}
	}
	return 1;
}


static const struct {
	const char *unx, *spd;
} unx_trans[] = {
	"/dev/null", _PATH_DEVNULL,
	"/dev/zero", _PATH_DEVZERO,
	"/dev/stdin", _LNM_STDIN":/",
	"/dev/stdout", _LNM_STDOUT":/",
	"/dev/stderr", _LNM_STDERR":/",
	"/dev/random", "SYS$RANDOM:/STRONG",
	"/dev/urandom", "SYS$RANDOM:/",
	NULL, NULL,
};

static int translate_unx_extender(DLRQ *dl, void *data);

const char *open_translate_unx(const char *path, const char **to_free)
{
	int i;
	struct extender_library *l;
	const char *str;
	if (__unlikely(*to_free != NULL))
		KERNEL$SUICIDE("open_translate_unx: to_free UNINITIALIZED");

	for (i = 0; unx_trans[i].unx; i++)
		if (!strcmp(path, unx_trans[i].unx))
			return unx_trans[i].spd;

	for (l = extender_libraries; l; l = l->next)
		if (__likely(l->translate_unx != NULL))
			if (__likely((str = l->translate_unx(path, to_free)) != NULL))
				return str;

	*to_free = path;
	if (__unlikely(scan_extenders(translate_unx_extender, to_free))) {
		*to_free = NULL;
		return path;
	}

	return *to_free;
}

static int translate_unx_extender(DLRQ *dl, void *data)
{
	const char **path = data;
	const char *to_free = NULL;
	const char *ret;
	unsigned long translate_unx;
	if (__unlikely(KERNEL$DL_GET_SYMBOL(dl, "IOC$TRANSLATE_UNX", &translate_unx)))
		return 1;

	ret = ((translate_unx_fn *)translate_unx)(*path, &to_free);
	if (ret) {
		if (__unlikely(to_free != NULL)) {
			if (__unlikely(to_free != ret)) {
				memmove((char *)to_free, ret, strlen(ret) + 1);
			}
		} else {
			int saved_errno = errno;
			to_free = __sync_malloc(strlen(ret) + 1);
			errno = saved_errno;
			if (__unlikely(!to_free))
				return 1;
			strcpy((char *)to_free, ret);
		}
		*path = to_free;
		return 0;
	}

	return 1;
}


static int instantiate_extender(DLRQ *dl);

static int scan_extenders(int (*try_this)(DLRQ *dl, void *data), void *data)
{
	unsigned long e;
	READDIR_RQ rd;
	DLRQ dl;
	rd.cwd = NULL;
	rd.flags = 0;
	rd.path = "LIB.:/";
	SYNC_IO(&rd, KERNEL$READDIR);
	if (__unlikely(rd.status != 0))
		return 1;
	for (e = 0; e < rd.n_entries; e++) {
		struct dirent *d = rd.entries[e];
		if (__likely(d->d_namlen >= 4) && __unlikely(!_strcasecmp(d->d_name + d->d_namlen - 4, ".IOC"))) {
			struct extender_library *l;
			for (l = extender_libraries; l; l = l->next)
				if (!_strcasecmp(l->dl.filename, d->d_name))
					goto skip_this;
			dl.type = 0;
			strlcpy(dl.filename, d->d_name, sizeof dl.filename);
			strcpy(dl.modname, "IOC");
			dl.error[0] = 0;
			SYNC_IO(&dl, KERNEL$DL_LOAD_MODULE);
			if (__unlikely(dl.status != 0))
				goto skip_this;
			if (try_this(&dl, data)) {
				SYNC_IO(&dl, KERNEL$DL_UNLOAD_MODULE);
				goto skip_this;
			}
			if (__unlikely(instantiate_extender(&dl)))
				SYNC_IO(&dl, KERNEL$DL_UNLOAD_MODULE);
			KERNEL$FREE_READDIR(&rd);
			return 0;
		}
		skip_this:;
	}

	KERNEL$FREE_READDIR(&rd);
	return 1;
}

static int instantiate_extender(DLRQ *dl)
{
	unsigned long ioc_sym;
	struct extender_library *l;
	struct ioc_hash_entry *ioc_entries = NULL;
	MALLOC_REQUEST mrq;
	static MTX_DECL(extender_mtx, "KERNEL$EXTENDER_MTX");

	if (__unlikely(!FEATURE_TEST(FEATURE_USERSPACE)))
		goto ret;

	MTX_LOCK_SYNC(&extender_mtx);

	for (l = extender_libraries; l; l = l->next)
		if (!_strcasecmp(l->dl.filename, dl->filename))
			goto unlock_ret;

	if (__likely(!KERNEL$DL_GET_SYMBOL(dl, "IOC$TABLE", &ioc_sym))) {
		struct ioc_entry *ioc_table;
		for (ioc_table = *(struct ioc_entry **)ioc_sym; ioc_table->fn; ioc_table++) {
			unsigned long io;
			for (io = ioc_table->ioctl; io < ioc_table->ioctl + ioc_table->n; io++) {
				struct ioc_hash_entry *e;

				for (e = ioc_hash[IOC_HASH(io)]; e; e = e->next)
					if (__unlikely(e->io == io))
						goto free_ioc_entries_ret;

				mrq.size = sizeof(struct ioc_hash_entry);
				SYNC_IO(&mrq, KERNEL$UNIVERSAL_MALLOC);
				if (__unlikely(mrq.status != 0))
					goto free_ioc_entries_ret;
				e = mrq.ptr;
				e->io = io;
				e->fn = ioc_table->fn;
				e->next = ioc_entries;
				ioc_entries = e;
			}
		}
	}

	mrq.size = sizeof(struct extender_library);
	SYNC_IO(&mrq, KERNEL$UNIVERSAL_MALLOC);
	if (__unlikely(mrq.status != 0))
		goto free_ioc_entries_ret;
	l = mrq.ptr;

	memcpy(&l->dl, dl, sizeof(DLRQ));
	l->translate_unx = NULL;

	if (__likely(!KERNEL$DL_GET_SYMBOL(dl, "IOC$TRANSLATE_UNX", &ioc_sym))) {
		l->translate_unx = (translate_unx_fn *)ioc_sym;
	}

	while (ioc_entries) {
		struct ioc_hash_entry *next = ioc_entries->next;
		unsigned hash = IOC_HASH(ioc_entries->io);
		ioc_entries->next = ioc_hash[hash];
		__barrier();
		ioc_hash[hash] = ioc_entries;
		ioc_entries = next;
	}

	l->next = extender_libraries;
	__barrier();
	extender_libraries = l;

	MTX_UNLOCK(&extender_mtx);
	return 0;

	free_ioc_entries_ret:
	while (ioc_entries) {
		struct ioc_hash_entry *next = ioc_entries->next;
		free(ioc_entries);
		ioc_entries = next;
	}
	unlock_ret:
	MTX_UNLOCK(&extender_mtx);
	ret:
	return 1;
}

