#include <STDLIB.H>
#include <UNISTD.H>
#include <SIGNAL.H>
#include <SYS/TYPES.H>
#include <SYS/TIME.H>
#include <TIME.H>
#include <ERRNO.H>

#include <ARCH/BSF.H>
#include <ARCH/BT.H>
#include <SPAD/DEV.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <SPAD/ALLOC.H>
#include <SPAD/WQ.H>
#include <SPAD/TIMER.H>
#include <LIB/KERNEL/USIGNAL.H>

#include <SYS/POLL.H>
#include <KERNEL/SELECT.H>

struct _select_rq_unpaded {
	IOCTLRQ io;
	struct select_context *ctx;
};

#define _SELECT_RQ_PAD	(sizeof(struct _select_rq_unpaded) <= 64 ? 64 : sizeof(struct _select_rq_unpaded) <= 128 ? 128 : 256)

union select_rq {
	struct _select_rq_unpaded s;
	char pad[_SELECT_RQ_PAD];
};

struct select_context {
	LIST_ENTRY list;
	unsigned n;
	unsigned outstanding;
	int flags;
	struct fd_set *read;
	struct fd_set *write;
	union select_rq *rrq;
	union select_rq *wrq;
	WQ wait;
	struct fd_set *rwait;
	struct fd_set *wwait;
	struct fd_set *rr;
	struct fd_set *rw;
	unsigned nwait;
	unsigned activated;
	TIMER timer;
	WQ cleanup;
	WQ *wq_on_cancel;
};

#define CTX_DETACHED	1

static DECL_XLIST(select_list);

#define event(EVENT, xwait, rx, xrq)					\
static DECL_AST(EVENT, SPL_TIMER, IOCTLRQ)				\
{									\
	union select_rq *rq = LIST_STRUCT(RQ, union select_rq, s.io);	\
	struct select_context *ctx = rq->s.ctx;				\
	ctx->outstanding--;						\
	if (__unlikely(ctx->flags & CTX_DETACHED)) {			\
		if (!ctx->outstanding) {				\
			DEL_FROM_LIST(&ctx->list);			\
			free(ctx);				\
		}							\
		RETURN;							\
	}								\
	if (__likely(rq->s.io.h < ctx->nwait)) {			\
		if (__likely(ctx->xwait != NULL) && FD_ISSET(rq->s.io.h, ctx->xwait)) {									\
			if (__likely(rq->s.io.status != -EINTR)) {	\
				FD_SET(rq->s.io.h, ctx->rx);		\
				ctx->activated++;			\
				WQ_WAKE_ONE(&ctx->wait);		\
			} else {					\
				ctx->outstanding++;			\
				RETURN_IORQ(&rq->s.io, KERNEL$IOCTL);	\
			}						\
		}							\
	}								\
	FD_CLR(rq->s.io.h, ctx->xrq);					\
	WQ_WAKE_ALL(&ctx->cleanup);					\
	RETURN;								\
}

event(READ_EVENT_HAPPENED, rwait, rr, read)
event(WRITE_EVENT_HAPPENED, wwait, rw, write)

static void SELECT_TIMEOUT(TIMER *t)
{
	struct select_context *ctx;
	LOWER_SPL(SPL_TIMER);
	ctx = LIST_STRUCT(t, struct select_context, timer);
	ctx->timer.fn = NULL;
	WQ_WAKE_ONE(&ctx->wait);
}

void FREE_THREAD_SELECT_STORAGE(struct select_context **ctxp)
{
	struct select_context *ctx;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_TIMER)))
		KERNEL$SUICIDE("FREE_THREAD_SELECT_STORAGE AT SPL %08X", KERNEL$SPL);
	ctx = *ctxp;
	if (!ctx) return;
	if (ctx->timer.fn) KERNEL$DEL_TIMER(&ctx->timer);
	WQ_WAKE_ALL(&ctx->wait);
	WQ_WAKE_ALL(&ctx->cleanup);
	if (ctx->outstanding) {
		ctx->flags |= CTX_DETACHED;
		*ctxp = NULL;
		return;
	} else {
		DEL_FROM_LIST(&ctx->list);
		*ctxp = NULL;
		free(ctx);
		return;
	}
}

static int extend_context(struct select_context **ctxp, unsigned n)
{
	struct select_context *ctx;
	unsigned i;
	n = 2 << __BSR(n - 1);
	alloc_again:
	FREE_THREAD_SELECT_STORAGE(ctxp);
	if (__unlikely(n > (((size_t)-1 - sizeof(struct select_context)) / (sizeof(union select_rq) * 2 + 1)))) {
		return -ENOMEM;
	}
	if (__unlikely(!(ctx = calloc(1, sizeof(struct select_context) + (size_t)n / (NBBY / 2) + (size_t)n * sizeof(union select_rq) * 2)))) {
		int r;
		if (__unlikely(r = KERNEL$MEMWAIT_SYNC(sizeof(struct select_context) + (size_t)n / (NBBY / 2) + (size_t)n * sizeof(union select_rq) * 2))) return r;
		goto alloc_again;
	}
	ctx->n = n;
	ctx->read = (struct fd_set *)(ctx + 1);
	ctx->write = (struct fd_set *)((char *)ctx->read + n / NBBY);
	ctx->rrq = (union select_rq *)((char *)ctx->write + n / NBBY);
	ctx->wrq = ctx->rrq + n;
	for (i = 0; i < n; i++) {
		ctx->rrq[i].s.io.fn = READ_EVENT_HAPPENED;
		ctx->rrq[i].s.io.h = i;
		ctx->rrq[i].s.io.ioctl = IOCTL_SELECT_READ;
		ctx->rrq[i].s.io.param = 1;
		ctx->rrq[i].s.io.v.vspace = &KERNEL$VIRTUAL;
		ctx->rrq[i].s.ctx = ctx;
		ctx->wrq[i].s.io.fn = WRITE_EVENT_HAPPENED;
		ctx->wrq[i].s.io.h = i;
		ctx->wrq[i].s.io.ioctl = IOCTL_SELECT_WRITE;
		ctx->wrq[i].s.io.param = 1;
		ctx->wrq[i].s.io.v.vspace = &KERNEL$VIRTUAL;
		ctx->wrq[i].s.ctx = ctx;
	}
	WQ_INIT(&ctx->wait, "KERNEL$UNX6_CTX_WAIT");
	WQ_INIT(&ctx->cleanup, "KERNEL$UNX6_CTX_CLEANUP");
	INIT_TIMER(&ctx->timer);
	ADD_TO_XLIST(&select_list, &ctx->list);
	*ctxp = ctx;
	return 0;
}

#include <KERNEL/SYSCALL.H>

int select(int n, fd_set *read_fds, fd_set *write_fds, fd_set *except_fds, struct timeval *timeout)
{
	struct fd_set *rr, *rw;
	struct select_context **ctxp;
	struct select_context *ctx;
	unsigned us, i, nn;
	int r;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO)))
		KERNEL$SUICIDE("select AT SPL %08X", KERNEL$SPL);
	if (timeout && __unlikely(__tv_invalid(timeout))) {
			/* mplayer uses 1000000 */
		goto einval;
	}
	if (__unlikely(n <= 0)) {
		if (__likely(!n)) {
			if (__unlikely(!timeout)) goto no_timeout;
			us = timeout->tv_usec;
			timeout->tv_usec = (unsigned)timeout->tv_usec * 1000;
			r = nanosleep((struct timespec *)timeout, NULL);
			timeout->tv_usec = us;
			return r;
		}
		einval:
		errno = EINVAL;
		return -1;
	}
	no_timeout:
	nn = (n + NFDBITS - 1) & ~(NFDBITS - 1);
	RAISE_SPL(SPL_TIMER);
	ctxp = GET_THREAD_SELECT_STORAGE();
	if (__unlikely(!*ctxp) || __unlikely(nn > (*ctxp)->n)) {
		if (__unlikely(r = extend_context(ctxp, nn))) {
			errno = -r;
			LOWER_SPL(SPL_ZERO);
			return -1;
		}
	}
	ctx = *ctxp;
	if (__unlikely((ctx->nwait | (unsigned long)ctx->wq_on_cancel)))
		KERNEL$SUICIDE("select CALLED RECURSIVELY");

	rr = alloca(nn / (NBBY / 2));
	memset(rr, 0, nn / (NBBY / 2));
	rw = (struct fd_set *)((char *)rr + nn / NBBY);
	ctx->rwait = read_fds;
	ctx->wwait = write_fds;
	ctx->rr = rr;
	ctx->rw = rw;
	ctx->nwait = nn;
	ctx->activated = 0;
	for (i = 0; i < nn / NFDBITS; i++) {
		fd_mask fdm;
		if (read_fds && __unlikely((fdm = read_fds->fds_bits[i] & ~ctx->read->fds_bits[i]) != 0)) {
			ctx->read->fds_bits[i] |= fdm;
			do {
				union select_rq *rq;
				unsigned j = __BSCAN(fdm);
				fdm = __BVR(fdm, j);
				rq = &ctx->rrq[i * NFDBITS + j];
				CALL_IORQ(&rq->s.io, KERNEL$IOCTL);
				ctx->outstanding++;
			} while (fdm);
		}
		if (write_fds && __unlikely((fdm = write_fds->fds_bits[i] & ~ctx->write->fds_bits[i]) != 0)) {
			ctx->write->fds_bits[i] |= fdm;
			do {
				union select_rq *rq;
				unsigned j = __BSCAN(fdm);
				fdm = __BVR(fdm, j);
				rq = &ctx->wrq[i * NFDBITS + j];
				CALL_IORQ(&rq->s.io, KERNEL$IOCTL);
				ctx->outstanding++;
			} while (fdm);
		}
	}
	LOWER_SPL(SPL_BOTTOM);	/* allow callbacks but do not allow signal! */
	if (ctx->activated) {
		RAISE_SPL(SPL_TIMER);
		a1:
		if (read_fds) memcpy(read_fds, rr, nn / NBBY);
		if (write_fds) memcpy(write_fds, rw, nn / NBBY);
		if (__unlikely(except_fds != NULL)) memset(except_fds, 0, nn / NBBY);
		ctx->nwait = 0;
		r = ctx->activated;
		LOWER_SPL(SPL_ZERO);
		return r;
	}
	RAISE_SPL(SPL_TIMER);
	if (__unlikely(ctx->activated)) goto a1;
	if (__unlikely(ctx->timer.fn != NULL)) KERNEL$DEL_TIMER(&ctx->timer), ctx->timer.fn = NULL;
	if (timeout) {
		u_jiffies_t j;
		if (!timeout->tv_sec && !timeout->tv_usec) goto a1;
		ctx->timer.fn = SELECT_TIMEOUT;
		TV_2_JIFFIES(timeout, j);
		KERNEL$SET_TIMER(j, &ctx->timer);
	}
	r = WQ_WAIT_SYNC_SIGINTERRUPT(&ctx->wait);
		/* signal might have deallocated ctx */
	if (__unlikely(*ctxp != ctx) || __unlikely(!ctx->nwait)) {
		LOWER_SPL(SPL_ZERO);
		errno = EINTR;
		return -1;
	}
	if (ctx->timer.fn) KERNEL$DEL_TIMER(&ctx->timer), ctx->timer.fn = NULL;
	if (__unlikely(r)) {
		errno = -r;
		ctx->nwait = 0;
		LOWER_SPL(SPL_ZERO);
		return -1;
	}
	if (__unlikely(!ctx->activated) && __unlikely(!timeout))
		KERNEL$SUICIDE("select: SOMEONE UNKNOWN WOKEN US UP");
	goto a1;
}

void SIGNAL_CANCEL_SELECT(void)
{
	struct select_context *ctx = *GET_THREAD_SELECT_STORAGE();
	if (ctx) {
		if (__unlikely(ctx->wq_on_cancel != NULL)) {
			WQ_WAKE_ALL_PL(ctx->wq_on_cancel);
			ctx->wq_on_cancel = NULL;
		}
		ctx->nwait = 0;
		WQ_WAKE_ALL(&ctx->wait);
	}
}

int CLOSE_SELECT_HANDLE(int h, IORQ *rq)
{
	struct select_context *ctx;
	IORQ *rrq;
#if __DEBUG >= 1
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_DEV))) KERNEL$SUICIDE("CLOSE_SELECT_HANDLE AT SPL %08X", KERNEL$SPL);
#endif
	RAISE_SPL(SPL_TIMER);
	rrq = rq;
	XLIST_FOR_EACH_UNLIKELY(ctx, &select_list, struct select_context, list) if (__likely(!(ctx->flags & CTX_DETACHED))) {
		if (__unlikely((unsigned)h >= ctx->n)) continue;
		if (__unlikely(FD_ISSET(h, ctx->read))) {
			KERNEL$CIO((IORQ *)(void *)&ctx->rrq[h].s.io);
			if (__likely(rq != NULL)) {
				WQ_WAIT_F(&ctx->cleanup, rq);
				rq = NULL;
			}
		}
		if (__unlikely(FD_ISSET(h, ctx->write))) {
			KERNEL$CIO((IORQ *)(void *)&ctx->wrq[h].s.io);
			if (__likely(rq != NULL)) {
				WQ_WAIT_F(&ctx->cleanup, rq);
				rq = NULL;
			}
		}
	}
	LOWER_SPL(SPL_DEV);
 	if (__unlikely(h <= 2)) {
		if (__unlikely(rrq != NULL)) return CANCEL_INTR_KEYS(h, rq);
		else REFRESH_INTR_KEYS(h);
	}
	return !rq;
}

void SELECT_FORK(void)
{
	RAISE_SPL(SPL_TIMER);
	while (!XLIST_EMPTY(&select_list)) {
		struct select_context *ctx = LIST_STRUCT(select_list.next, struct select_context, list);
		ctx->outstanding = 0;
		FREE_THREAD_SELECT_STORAGE(&ctx);
	}
	*GET_THREAD_SELECT_STORAGE() = NULL;
	LOWER_SPL(SPL_DEV);
}

static void poll_test_extra(int flg, struct pollfd *fdp);
static int poll_1(struct pollfd *fd, int timeout);

int poll(struct pollfd *fds, unsigned nfds, int timeout)
{
	unsigned npollfd;
	unsigned i, e;
	struct fd_set *read_fds;
	struct fd_set *write_fds;
	fd_mask *fd_ptr;
	struct pollfd **fdptr;
	struct timeval tv;
	int r;
	unsigned retn;
	if (__unlikely(KERNEL$SPL != SPL_X(SPL_ZERO)))
		KERNEL$SUICIDE("poll AT SPL %08X", KERNEL$SPL);
	if (nfds == 1 && __likely(fds[0].fd >= 0) && __likely(fds[0].events & (POLLIN | POLLOUT))) return poll_1(fds, timeout);
	npollfd = 0;
	for (i = 0; i < nfds; i++) {
		if (__unlikely(fds[i].fd >= (int)npollfd)) npollfd = (fds[i].fd + 1 + NFDBITS - 1) & ~(NFDBITS - 1);
	}
	read_fds = alloca(npollfd / (NBBY / 2) + npollfd * sizeof(struct pollfd *));
	memset(read_fds, 0, npollfd / (NBBY / 2) + npollfd * sizeof(struct pollfd *));
	write_fds = (struct fd_set *)((char *)read_fds + npollfd / NBBY);
	fdptr = (struct pollfd **)((char *)write_fds + npollfd / NBBY);
	retn = 0;
	RAISE_SPL(SPL_BOTTOM);	/* prevent setting errno */
	for (i = 0; i < nfds; i++) {
		unsigned fd;
		unsigned ev;
		char *p;
		fds[i].revents = 0;
		if (__unlikely((int)(fd = fds[i].fd) < 0)) continue;
		p = KERNEL$HANDLE_PATH(fd);
		if (__unlikely(!p)) {
			fds[i].revents = POLLNVAL;
			retn++;
			timeout = 0;
			continue;
		}
		if (__unlikely(fdptr[fd] != NULL)) {
			LOWER_SPL(SPL_ZERO);
			errno = EINVAL;
			return -1;
		}
		fdptr[fd] = &fds[i];
		ev = fds[i].events;
		if (ev & POLLIN) {
			FD_SET(fd, read_fds);
		}
		if (ev & POLLOUT) {
			FD_SET(fd, write_fds);
		}
	}
	LOWER_SPL(SPL_ZERO);
	if (timeout > 0) {
		tv.tv_sec = (unsigned)timeout / 1000;
		tv.tv_usec = ((unsigned)timeout % 1000) * 1000;
	} else {
		tv.tv_sec = 0;
		tv.tv_usec = 0;
	}
	r = select(npollfd, read_fds, write_fds, NULL, __likely(timeout < 0) ? NULL : &tv);
	if (__unlikely(r < 0)) return r;
	if (!r) goto ret_retn;
	RAISE_SPL(SPL_BOTTOM);	/* prevent setting errno */
	fd_ptr = &read_fds->fds_bits[0];
	for (e = POLLIN; e != 2 * POLLOUT - POLLIN; e += POLLOUT - POLLIN) {
		for (i = 0; i < npollfd / NFDBITS; i++) {
			fd_mask fdm = *fd_ptr++;
			while (__unlikely(fdm != 0)) {
				int h;
				int flg;
				unsigned j = __BSCAN(fdm);
				fdm = __BVR(fdm, j);
				h = i * NFDBITS + j;
				retn += !fdptr[h]->revents;
				fdptr[h]->revents |= e;
				flg = KERNEL$HANDLE_FLAGS(h);
				if (__unlikely(flg & (_O_SOCKET | _O_PIPE))) poll_test_extra(flg, fdptr[h]);
			}
		}
	}
	LOWER_SPL(SPL_ZERO);
	ret_retn:
	return retn;
}

static void poll_test_extra(int flg, struct pollfd *fdp)
{
	int ev;
	IOCTLRQ io;
	if (__unlikely(flg == -1)) {
		fdp->revents = POLLNVAL;
		return;
	}
	io.h = fdp->fd;
	io.ioctl = IOCTL_POLL;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	ev = io.status;
	if (__unlikely(ev == -ENOOP)) return; /* driver doesn't support POLLERR | POLLHUP */
	if (__unlikely(ev < 0)) ev = POLLERR;
	ev &= fdp->events | (POLLERR | POLLHUP | POLLNVAL);
	if (__likely(ev != 0)) fdp->revents = ev;
}

struct poll_1 {
	IOCTLRQ rio;
	IOCTLRQ wio;
	WQ wq;
};

static void POLL_1_TIMEOUT(TIMER *t)
{
	struct select_context *ctx;
	LOWER_SPL(SPL_TIMER);
	ctx = LIST_STRUCT(t, struct select_context, timer);
	ctx->timer.fn = NULL;
	if (ctx->wq_on_cancel) WQ_WAKE_ONE(ctx->wq_on_cancel);
}

static DECL_AST(POLL_1_FINISH, SPL_TIMER, IOCTLRQ)
{
	struct poll_1 *p;
	RQ->fn = NULL;
	if (__likely(RQ->ioctl == IOCTL_SELECT_READ)) p = GET_STRUCT(RQ, struct poll_1, rio);
	else p = GET_STRUCT(RQ, struct poll_1, wio);
	WQ_WAKE_ONE(&p->wq);
	RETURN;
}

static int poll_1(struct pollfd *fd, int timeout)
{
	struct select_context **ctxp;
	struct select_context *ctx;
	struct poll_1 p;
	unsigned ev;
	int prs, pws;
	int r;
	int flg;
	if (!timeout) {
		flg = KERNEL$HANDLE_FLAGS(fd->fd);
		if (__likely(flg & (_O_SOCKET | _O_PIPE))) {
				/* optimization --- do not need to send rqs */
			fd->revents = 0;
			poll_test_extra(flg, fd);
			goto ret_r;
		}
	}
	p.rio.fn = NULL;
	p.rio.status = -EINTR;
	p.wio.fn = NULL;
	p.wio.status = -EINTR;
	WQ_INIT(&p.wq, "KERNEL$POLL_1");
	RAISE_SPL(SPL_TIMER);
	ctxp = GET_THREAD_SELECT_STORAGE();
	if (__unlikely(!*ctxp)) {
		if (__unlikely(r = extend_context(ctxp, NFDBITS))) {
			errno = -r;
			LOWER_SPL(SPL_ZERO);
			return -1;
		}
	}
	ctx = *ctxp;
	if (__unlikely((ctx->nwait | (unsigned long)ctx->wq_on_cancel)))
		KERNEL$SUICIDE("poll_1 CALLED RECURSIVELY");
	ctx->wq_on_cancel = &p.wq;
	if (fd->events & POLLIN) {
		p.rio.fn = POLL_1_FINISH;
		p.rio.h = fd->fd;
		p.rio.ioctl = IOCTL_SELECT_READ;
		p.rio.param = !!timeout;
		p.rio.v.ptr = 0;
		p.rio.v.len = 0;
		p.rio.v.vspace = &KERNEL$VIRTUAL;
		CALL_IORQ(&p.rio, KERNEL$IOCTL);
	}
	if (fd->events & POLLOUT) {
		p.wio.fn = POLL_1_FINISH;
		p.wio.h = fd->fd;
		p.wio.ioctl = IOCTL_SELECT_WRITE;
		p.wio.param = !!timeout;
		p.wio.v.ptr = 0;
		p.wio.v.len = 0;
		p.wio.v.vspace = &KERNEL$VIRTUAL;
		CALL_IORQ(&p.wio, KERNEL$IOCTL);
	}
	if (__unlikely(ctx->timer.fn != NULL)) KERNEL$DEL_TIMER(&ctx->timer), ctx->timer.fn = NULL;
	r = 0;
	if (__likely(timeout)) {
		if (timeout > 0) {
			u_jiffies_t j;
			ctx->timer.fn = POLL_1_TIMEOUT;
			MSEC_2_JIFFIES(timeout, j);
			KERNEL$SET_TIMER(j, &ctx->timer);
		}
		r = WQ_WAIT_SYNC_SIGINTERRUPT(&p.wq);
	}
	TEST_SPL(SPL_ZERO, SPL_TIMER);
	while (__unlikely(((unsigned long)p.rio.fn | (unsigned long)p.wio.fn) != 0)) {
		KERNEL$CIO((IORQ *)(void *)&p.rio);
		KERNEL$CIO((IORQ *)(void *)&p.wio);
		WQ_WAIT_SYNC(&p.wq);
	}
	if (__unlikely(*ctxp != ctx) || __unlikely(!ctx->wq_on_cancel)) {
		LOWER_SPL(SPL_ZERO);
		errno = EINTR;
		return -1;
	}
	if (ctx->timer.fn) KERNEL$DEL_TIMER(&ctx->timer), ctx->timer.fn = NULL;
	ctx->wq_on_cancel = NULL;
	LOWER_SPL(SPL_ZERO);
	if (__unlikely(r)) {
		errno = -r;
		return -1;
	}
	prs = p.rio.status;
	pws = p.wio.status;
	if (__unlikely(prs == -EBADF) || __unlikely(pws == -EBADF)) {
		ev = POLLNVAL;
		goto ret;
	}
	ev = 0;
	if (prs != -EINTR && prs != -EWOULDBLOCK) ev = POLLIN;
	if (pws != -EINTR && pws != -EWOULDBLOCK) ev |= POLLOUT;
	ret:
	fd->revents = ev;
	flg = KERNEL$HANDLE_FLAGS(fd->fd);
	if (__unlikely(flg & (_O_SOCKET | _O_PIPE))) poll_test_extra(flg, fd);
	ret_r:
	r = !!fd->revents;
	return r;
}


