#include <KERNEL/VMPROC.H>
#include <SPAD/DEV_KRNL.H>
#include <KERNEL/UACCESS.H>
#include <KERNEL/PROC.H>
#include <KERNEL/VM_ARCH.H>
#include <KERNEL/VM.H>
#include <STRING.H>

unsigned long PROC_VSPACE_GET(VDESC *desc, const VBUF *buf);
unsigned long PROC_VSPACE_PUT(VDESC *desc, const VBUF *buf);
unsigned long PROC_VSPACE_GET_SLOW(VDESC *desc, const VBUF *buf);
unsigned long PROC_VSPACE_PUT_SLOW(VDESC *desc, const VBUF *buf);
static void *PROC_VSPACE_MAP(VDESC *desc, int rw, int spl, vspace_unmap_t **unmap);
static vspace_dmaunlock_t *PROC_VSPACE_DMALOCK(VDESC *desc, int rw, VDMA *dma);
static vspace_dma64unlock_t *PROC_VSPACE_DMA64LOCK(VDESC *desc, int rw, VDMA64 *dma);
static vspace_physunlock_t *PROC_VSPACE_PHYSLOCK(VDESC *desc, int rw, VPHYS *dma);
static IORQ *PROC_VSPACE_GET_PAGEIN_RQ(VDESC *desc, IORQ *rq, int wr);

const HANDLE_OPERATIONS PROC_VSPACE_OPERATIONS = {
	SPL_X(SPL_DEV),
	PROC_VSPACE_GET,
	PROC_VSPACE_PUT,
	PROC_VSPACE_MAP,
	PROC_VSPACE_DMALOCK,
	PROC_VSPACE_DMA64LOCK,
	PROC_VSPACE_PHYSLOCK,
	PROC_VSPACE_GET_PAGEIN_RQ,
};

static void PROC_VSPACE_UNMAP(void *ptr)
{
	UNLOCKDOWN_PROC();
}

static void PROC_VSPACE_UNMAP_PAGE(void *ptr)
{
	int spl;
	PAGE *p;
	p = PHYS_2_PAGE(KERNEL$UNMAP_PHYSICAL_BANK_ADDR(ptr));
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_VSPACE);
	p->lockdown(p, 1);
	LOWER_SPLX(spl);
}

static void PROC_VSPACE_UNMAP_SPAGE(void *ptr)
{
	int spl;
	__p_addr phys;
	unsigned pos;
	SPAGE *s;
	phys = KERNEL$UNMAP_PHYSICAL_BANK_ADDR(ptr);
	pos = ((unsigned)phys & (PAGE_CLUSTER_SIZE - 1)) >> PG_SIZE_BITS;
	s = (SPAGE *)((SPAGES *)PHYS_2_PAGE(phys)->fnode)->s[pos + 1].freelist.next;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_VSPACE);
	s->lockdown(s, 1);
	LOWER_SPLX(spl);
}

static void *PROC_VSPACE_MAP(VDESC *desc, int rw, int spl, vspace_unmap_t **unmap)
{
	PROC *p;
	unsigned long addr;
	SPLVSPACEASSERT("PROC_VSPACE_MAP");
	if (__unlikely(!desc->len)) {
		LOWER_SPLX(spl);
		return *unmap = KERNEL$NULL_VSPACE_UNMAP;
	}
	p = GET_STRUCT(desc->vspace, PROC, vspace);
	addr = (unsigned long)(void *)(unsigned long)desc->ptr;
	if (__unlikely(desc->ptr != (__v_off)addr) || __unlikely(desc->len > PAGE_CLUSTER_SIZE) || __unlikely(((unsigned)addr & (PAGE_CLUSTER_SIZE - 1)) + desc->len > PAGE_CLUSTER_SIZE)) {
		ret_einval:
		LOWER_SPLX(spl);
		return __ERR_PTR(-EINVAL);
	}
	RAISE_SPL(SPL_TOP);
	if (__unlikely(PROC_CURRENT != p)) {
		if (__unlikely(PROC_CURRENT_LOCK)) {
			PAGE *pg;
			LOWER_SPL(SPL_VSPACE);
			pg = VM_ARCH_GET_PAGE(p, (unsigned long)addr, rw & PF_WRITE);
			addr &= PAGE_CLUSTER_SIZE - 1;
			if (__unlikely(!pg)) {
				ret_null:
				LOWER_SPLX(spl);
				return NULL;
			}
			if (__unlikely(!IS_PAGE_POINTER(pg))) {
#define sp	((SPAGE *)pg)
				*unmap = PROC_VSPACE_UNMAP_SPAGE;
				if (__unlikely(addr + desc->len > sp->n_pages << PG_SIZE_BITS)) goto ret_einval;
				if (__unlikely(sp->lockdown(sp, 0) != NULL)) goto ret_null;
				LOWER_SPLX(spl);
				return KERNEL$MAP_PHYSICAL_BANK(PAGE_2_PHYS(sp->page) | (sp->offset + addr));
#undef sp
			}
			*unmap = PROC_VSPACE_UNMAP_PAGE;
			if (__unlikely(pg->lockdown(pg, 0) != NULL)) goto ret_null;
			LOWER_SPLX(spl);
			return KERNEL$MAP_PHYSICAL_BANK(PAGE_2_PHYS(pg) | addr);
		}
		SET_PROC_CURRENT(p);
	}
	LOCKDOWN_PROC();
	LOWER_SPL(SPL_VSPACE);
	*unmap = PROC_VSPACE_UNMAP;
	if (__unlikely(utest_access(p, (void *)addr, rw & PF_WRITE))) {
		PROC_VSPACE_UNMAP(NULL);
		goto ret_null;
	}
	LOWER_SPLX(spl);
	return (void *)addr;
}

#define FN		static vspace_dmaunlock_t *PROC_VSPACE_DMALOCK(VDESC *desc, int rw, VDMA *dma)
#define FN_NAME		"PROC_VSPACE_DMALOCK"
#define UNL		PROC_VSPACE_DMAUNLOCK
#define UNL_SPAGE	PROC_VSPACE_DMAUNLOCK_SPAGE
#define UNL_ZERO	PROC_VSPACE_DMAUNLOCK_ZERO
#define UNL_TYPE	__u32
#define RET_ZERO	{ LOWER_SPLX(dma->spl); dma->ptr = dma->len = 0; return NULL; }
#define RET_MAP(a,l,rw,unl) { VDESC vd; vd.ptr = (a); vd.len = (l); vd.vspace = &KERNEL$PHYSICAL; KERNEL$PHYSICAL.op->vspace_dmalock(&vd, rw, dma); return unl;}
#define DO_UNMAP(x)	PHYS_2_PAGE(KERNEL$PHYSICAL_UNMAP_DMA(x))

#include "VSPROCM.I"

#define FN		static vspace_dma64unlock_t *PROC_VSPACE_DMA64LOCK(VDESC *desc, int rw, VDMA64 *dma)
#define FN_NAME		"PROC_VSPACE_DMA64LOCK"
#define UNL		PROC_VSPACE_DMA64UNLOCK
#define UNL_SPAGE	PROC_VSPACE_DMA64UNLOCK_SPAGE
#define UNL_ZERO	PROC_VSPACE_DMA64UNLOCK_ZERO
#define UNL_TYPE	__u64
#define RET_ZERO	{ LOWER_SPLX(dma->spl); dma->ptr = dma->len = 0; return NULL; }
#define RET_MAP(a,l,rw,unl) { VDESC vd; vd.ptr = (a); vd.len = (l); vd.vspace = &KERNEL$PHYSICAL; KERNEL$PHYSICAL.op->vspace_dma64lock(&vd, rw, dma); return unl; }
#define DO_UNMAP(x)	PHYS_2_PAGE(KERNEL$PHYSICAL_UNMAP_DMA64(x))

#include "VSPROCM.I"

#define FN		static vspace_physunlock_t *PROC_VSPACE_PHYSLOCK(VDESC *desc, int rw, VPHYS *dma)
#define FN_NAME		"PROC_VSPACE_PHYSUNLOCK"
#define UNL		PROC_VSPACE_PHYSUNLOCK
#define UNL_SPAGE	PROC_VSPACE_PHYSUNLOCK_SPAGE
#define UNL_TYPE	__p_addr
#define RET_ZERO	{ LOWER_SPLX(dma->spl); dma->ptr = dma->len = 0; return NULL; }
#define RET_MAP(a,l,rw,unl) { LOWER_SPLX(dma->spl); dma->ptr = (a); dma->len = l; return unl; }

#define DO_UNMAP(x)	PHYS_2_PAGE(x)

#include "VSPROCM.I"

unsigned long PROC_VSPACE_GET_SLOW(VDESC *desc, const VBUF *buf)
{
	PROC *p;
	PAGE *pg;
	unsigned off;
	unsigned long l;
	char *v;
	p = GET_STRUCT(desc->vspace, PROC, vspace);
	RAISE_SPL(SPL_VSPACE);
	pg = VM_ARCH_GET_PAGE(p, (unsigned long)desc->ptr, 0);
	if (__unlikely(!pg)) {
		ret0:
		LOWER_SPLX(buf->spl);
		return 0;
	}
	if (__unlikely(!IS_PAGE_POINTER(pg))) {
#define sp	((SPAGE *)pg)
		if (__unlikely(sp->lockdown(sp, 0) != NULL)) goto ret0;
		LOWER_SPLX(buf->spl);
		off = (unsigned)desc->ptr & (PAGE_CLUSTER_SIZE - 1);
		l = desc->len;
		if (l > buf->len) l = buf->len;
		if (__unlikely(off >= sp->n_pages << PG_SIZE_BITS)) {
			if (__unlikely(l > PAGE_CLUSTER_SIZE - off)) l = PAGE_CLUSTER_SIZE - off;
			memset(buf->ptr, 0, l);
			goto unlock_sp;
		}
		if (__unlikely(l > (sp->n_pages << PG_SIZE_BITS) - off))
			l = (sp->n_pages << PG_SIZE_BITS) - off;
		v = KERNEL$MAP_PHYSICAL_PAGE(sp->page);
		memcpy(buf->ptr, v + sp->offset + off, l);
		KERNEL$UNMAP_PHYSICAL_BANK(v);
		unlock_sp:
		RAISE_SPL(SPL_VSPACE);
		sp->lockdown(sp, 1);
		goto ret_l_spl;
#undef sp
	}
	if (__unlikely(pg->lockdown(pg, 0) != NULL)) goto ret0;
	LOWER_SPLX(buf->spl);
	off = (unsigned)desc->ptr & (PAGE_CLUSTER_SIZE - 1);
	l = desc->len;
	if (l > buf->len) l = buf->len;
	if (__unlikely(l > PAGE_CLUSTER_SIZE - off)) l = PAGE_CLUSTER_SIZE - off;
	v = KERNEL$MAP_PHYSICAL_PAGE(pg);
	memcpy(buf->ptr, v + off, l);
	KERNEL$UNMAP_PHYSICAL_BANK(v);
	RAISE_SPL(SPL_VSPACE);
	pg->lockdown(pg, 1);
	ret_l_spl:
	LOWER_SPLX(buf->spl);
	desc->ptr += l;
	desc->len -= l;
	return l;
}

unsigned long PROC_VSPACE_PUT_SLOW(VDESC *desc, const VBUF *buf)
{
	PROC *p;
	PAGE *pg;
	unsigned off;
	unsigned long l;
	char *v;
	p = GET_STRUCT(desc->vspace, PROC, vspace);
	RAISE_SPL(SPL_VSPACE);
	pg = VM_ARCH_GET_PAGE(p, (unsigned long)desc->ptr, PF_WRITE);
	/* spage can't be here ... they can't have write access */
	if (__unlikely(!pg) || __unlikely(pg->lockdown(pg, 0) != NULL)) {
		LOWER_SPLX(buf->spl);
		return 0;
	}
	LOWER_SPLX(buf->spl);
	off = (unsigned)desc->ptr & (PAGE_CLUSTER_SIZE - 1);
	l = desc->len;
	if (l > buf->len) l = buf->len;
	if (__unlikely(l > PAGE_CLUSTER_SIZE - off)) l = PAGE_CLUSTER_SIZE - off;
	v = KERNEL$MAP_PHYSICAL_PAGE(pg);
	memcpy(v + off, buf->ptr, l);
	KERNEL$UNMAP_PHYSICAL_BANK(v);
	RAISE_SPL(SPL_VSPACE);
	pg->lockdown(pg, 1);
	LOWER_SPLX(buf->spl);
	desc->ptr += l;
	desc->len -= l;
	return l;
}

static IORQ *pagein_rq = NULL;
static VDESC pagein_vdesc;
static int pagein_wr;
static IORQ pagein_io;
static WQ_DECL(pagein_wait, "KERNEL$PAGEIN_WAIT");
static IO_STUB PROC_PAGEIN_RQ_CONT;

static IORQ *PROC_VSPACE_GET_PAGEIN_RQ(VDESC *desc, IORQ *rq, int wr)
{
	SPLVSPACEASSERT("PROC_VSPACE_GET_PAGEIN_RQ");
	if (__unlikely(pagein_rq != NULL)) {
		WQ_WAIT_F(&pagein_wait, rq);
		return NULL;
	}
	pagein_rq = rq;
	memcpy(&pagein_vdesc, desc, sizeof(VDESC));
	pagein_wr = wr;
	pagein_io.tmp1 = (unsigned long)PROC_PAGEIN_RQ_CONT;
	return &pagein_io;
}

static DECL_IOCALL(PROC_PAGEIN_RQ_CONT, SPL_DEV, IORQ)
{
#define p GET_STRUCT(pagein_vdesc.vspace, PROC, vspace)
	IORQ *caller;
	PAGE *pg;
	WQ *wq;
	unsigned long addr;
	unsigned x;
	if (__unlikely(KERNEL$LOCKUP_LEVEL >= LOCKUP_LEVEL_ALL_IORQS)) {
		/* WQ_WAIT_F(&KERNEL$LOCKUP_EVENTS, pagein_rq);
		goto free_pagein; --- this would livelock */
		WQ_WAIT_F(&KERNEL$LOCKUP_EVENTS, RQ);
		RETURN;
	}
	SWITCH_PROC_ACCOUNT(p, SPL_X(SPL_DEV));
	addr = (unsigned long)pagein_vdesc.ptr;
	if (__unlikely(pagein_vdesc.ptr != (__v_off)addr)) {
		pagein_rq->status = -ERANGE;
		caller = pagein_rq;
		RAISE_SPL(SPL_VSPACE);
		pagein_rq = NULL;
		WQ_WAKE_ALL(&pagein_wait);
		RETURN_AST(caller);
	}
	nextpage:
	t1:
	/*__debug_printf("testacc(%x,%x)\n", addr, pagein_vdesc.len);*/
	RAISE_SPL(SPL_VSPACE);
	pg = VM_ARCH_GET_PAGE(p, (unsigned long)addr, pagein_wr & PF_WRITE);
	if (__unlikely(!pg)) {
		LOWER_SPL(SPL_DEV);
		if (__likely(!VM_FAULT(p, (void *)addr, pagein_wr & PF_WRITE, pagein_rq))) goto t1;
		/* free_pagein: */
		RAISE_SPL(SPL_VSPACE);
		free_pagein_rq_splvspace:
		pagein_rq = NULL;
		WQ_WAKE_ALL(&pagein_wait);
		RETURN;
	}
	if (__unlikely(!IS_PAGE_POINTER(pg))) {
#define sp	((SPAGE *)pg)
		if (__unlikely((wq = sp->lockdown(sp, 0)) != NULL)) {
			wait_on_wq_splvspace:
			WQ_WAIT_F(wq, pagein_rq);
			goto free_pagein_rq_splvspace;
		}
		sp->lockdown(sp, 1);
#undef sp
	} else {
		if (__unlikely((wq = pg->lockdown(pg, 0)) != NULL)) {
			goto wait_on_wq_splvspace;
		}
		pg->lockdown(pg, 1);
	}
	LOWER_SPL(SPL_DEV);
	x = PAGE_CLUSTER_SIZE - (addr & (PAGE_CLUSTER_SIZE - 1));
	if (x < pagein_vdesc.len && __likely(KERNEL$LOCKUP_LEVEL < LOCKUP_LEVEL_ONE_PASS)) {
		addr += x;
		pagein_vdesc.len -= x;
		SWITCH_PROC_ACCOUNT(p, SPL_X(SPL_DEV));
		goto nextpage;
	}
	/*{
		char *sym1, *sym2;
		unsigned long off1, off2;
		sym1 = KERNEL$DL_GET_SYMBOL_NAME((void *)pagein_rq->tmp1, &off1, 0);
		sym2 = KERNEL$DL_GET_SYMBOL_NAME((void *)pagein_rq->fn, &off2, 0);
		__debug_printf("success.............. %Lx,%x (%d) (iorq %s+%ld, ast %s+%ld)\n", pagein_vdesc.ptr, addr - (unsigned)pagein_vdesc.ptr + pagein_vdesc.len, pagein_wr & PF_WRITE, sym1, off1, sym2, off2);
	}*/

	caller = pagein_rq;
	RAISE_SPL(SPL_VSPACE);
	pagein_rq = NULL;
	WQ_WAKE_ALL(&pagein_wait);
	RETURN_IORQ_LSTAT(caller, (IO_STUB *)caller->tmp1);
#undef p
}
