#include <UNISTD.H>
#include <FCNTL.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SYS/TYPES.H>
#include <SPAD/LIST.H>
#include <SPAD/WQ.H>
#include <SPAD/LIBC.H>
#include <SPAD/THREAD.H>
#include <SPAD/DL.H>
#include <TIME.H>
#include <VALUES.H>

#include "CHRCACHE.H"

#define MAX_AGE		30
#define REALLOC_STEP	32

static MTX_DECL(mtx, "CHRCACHE$MTX");
static THREAD_RQ *mtx_owner = NULL;	/* need recursive mutex */

#define LOCK							\
do {								\
	THREAD_RQ *trq = KERNEL$THREAD_RQ();			\
	if (__likely(trq != mtx_owner)) {			\
		MTX_LOCK_SYNC(&mtx);				\
		mtx_owner = trq;				\
		locked = 1;					\
	} else locked = 0;					\
} while (0)

#define UNLOCK							\
do {								\
	if (__likely(locked)) {					\
		mtx_owner = NULL;				\
		MTX_UNLOCK(&mtx);				\
	}							\
} while (0)

static DECL_XLIST(chrcache);

static void PRUNE_CACHE(int all)
{
	char done_something;
	CHRCACHE *chr;
	time_t t = time(NULL);
	again:
	done_something = 0;
	XLIST_FOR_EACH(chr, &chrcache, CHRCACHE, list) {
		if (chr->leave) {
			done_something |= 1;
		}
		if (chr->refcount) {
			continue;
		}
		if (__unlikely(t - chr->last_use > MAX_AGE) || __unlikely(all)) {
			CHRCACHE *prev = LIST_STRUCT(chr->list.prev, CHRCACHE, list);
			DEL_FROM_LIST(&chr->list);
			chr->destructor(chr);
			free(chr);
			chr = prev;
			done_something = 2;
		}
	}
	if (done_something >= 2) goto again;
	if (__unlikely(all) && __unlikely(!done_something) && __unlikely(!XLIST_EMPTY(&chrcache))) {
		chr = LIST_STRUCT(chrcache.next, CHRCACHE, list);
		KERNEL$SUICIDE("PRUNE_CACHE: CACHE ENTITY OF TYPE %d REFERENCED WHEN UNLOADING LIBRARY", chr->type);
	}
}

static void NULL_DESTRUCT(CHRCACHE *chr)
{
}

CHRCACHE *CHRCACHE_OPEN(const char *name, int type, const char *default_ext, int size_of, int (*init)(CHRCACHE *chr, unsigned long h))
{
	CHRCACHE *chr;
	int h;
	char *j;
	char locked;
	LOCK;
	XLIST_FOR_EACH(chr, &chrcache, CHRCACHE, list) {
		if (!_strcasecmp(chr->name, name) && __likely(type == chr->type)) {
			chr->refcount++;
			goto ret_chr;
		}
	}
	PRUNE_CACHE(0);
	if (!(type & TYPEFLAG_FILE)) {
		h = -1;
		goto skip_open;
	}
	j = _join_paths("ETC.:/CHARSETS/", name);
	if (__unlikely(!j)) {
		ret_err:
		UNLOCK;
		return NULL;
	}
	if (__likely(strrchr(j, '.') < strrchr(j, '/')) && default_ext) {
		char *k;
		k = __sync_malloc(strlen(j) + strlen(default_ext) + 2);
		if (__unlikely(!k)) {
			free(j);
			goto ret_err;
		}
		stpcpy(stpcpy(stpcpy(k, j), "."), default_ext);
		free(j);
		j = k;
	}
	h = open(j, O_RDONLY | _O_NOPOSIX);
	free(j);
	if (__unlikely(h < 0)) goto ret_err;
	skip_open:
	chr = __sync_malloc(size_of + strlen(name) + 1);
	if (__unlikely(!chr)) {
		close_ret_err:
		if (h >= 0) close(h);
		goto ret_err;
	}
	chr->name = strcpy((char *)chr + size_of, name);
	chr->type = type;
	chr->leave = 0;
	chr->refcount = 1;
	chr->last_use = 0;
	chr->destructor = NULL_DESTRUCT;
	if (__unlikely(init(chr, h >= 0 ? h : (unsigned long)name))) {
		chr->destructor(chr);
		free(chr);
		goto close_ret_err;
	}
	if (h >= 0) close(h);
	ADD_TO_XLIST(&chrcache, &chr->list);
	ret_chr:
	UNLOCK;
	return chr;
}

void CHRCACHE_CLOSE(CHRCACHE *chr, int keep, int type_check)
{
	char locked;
	if (__unlikely(chr->type != type_check)) KERNEL$SUICIDE("CHRCACHE_CLOSE: CLOSING CHRCACHE ENTRY WITH WRONG TYPE: %d != %d", chr->type, type_check);
	LOCK;
	if (__unlikely(chr->refcount <= 0)) KERNEL$SUICIDE("CHRCACHE_CLOSE: CLOSING INVALID CHRCACHE ENTRY, REFCOUNT %d, TYPE %d", chr->refcount, chr->type);
	if (__likely(keep)) --chr->refcount, time(&chr->last_use);
	else if (__likely(!--chr->refcount)) PRUNE_CACHE(0);
	UNLOCK;
}

int READ_BY_LINES(CHRCACHE *chr, int h, int (*process_line)(CHRCACHE *chr, char *line, size_t len))
{
	char *buf = NULL, *nl;
	size_t len = 0, as, pos, ll;
	ssize_t r;
	read_more:
	as = __alloc_size(buf);
	if (__unlikely(as == len)) {
		buf = __sync_reallocf(buf, len + REALLOC_STEP);
		if (__unlikely(!buf)) return -1;
		as = __alloc_size(buf);
		if (__unlikely(as > MAXINT)) {
			errno = EFBIG;
			free_ret_err:
			free(buf);
			return -1;
		}
	}
	if (__unlikely((r = read(h, buf + len, as - len)) <= 0)) {
		if (__unlikely(r < 0)) return -1;
		if (__unlikely(len)) {
			errno = EFTYPE;
			goto free_ret_err;
		}
		return 0;
	}
	len += r;
	pos = 0;
	next_line:
	if (!(nl = memchr(buf + pos, '\n', len - pos))) {
		if (pos) memmove(buf, buf + pos, len -= pos);
		goto read_more;
	}
	ll = nl - (buf + pos);
	if (__likely(ll != 0) && nl[-1] == '\r') ll--;
	if (__unlikely(process_line(chr, buf + pos, ll))) goto free_ret_err;
	pos = (nl - buf) + 1;
	goto next_line;
}

DECL_IOCALL(DLL$UNLOAD, SPL_DEV, DLINITRQ)
{
	PRUNE_CACHE(1);
	RQ->status = 0;
	RETURN_AST(RQ);
}
