#include <STDLIB.H>
#include <UNISTD.H>
#include <STRING.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/IOCTL.H>
#include <SPAD/TTY.H>
#include <ARCH/TIME.H>
#include <SYS/TIME.H>

#include <GPM.H>

#define DOUBLE_CLICK_TIME	(JIFFIES_PER_SECOND / 2)

static Gpm_Connect *conn = NULL;

int gpm_flag = 0;
int gpm_fd = -1;
int gpm_consolefd = -1;

static struct tty_window_size ws;

int gpm_mx, gpm_my;

static void get_terminal_size(void)
{
	IOCTLRQ io;
	ws.ws_col = 80;
	ws.ws_row = 25;
	ws.ws_xpixel = ws.ws_col * 9;
	ws.ws_ypixel = ws.ws_row * 16;
	io.h = 0;
	io.ioctl = IOCTL_TTY_GET_WINSIZE;
	io.param = 0;
	io.v.ptr = (unsigned long)&ws;
	io.v.len = sizeof ws;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(!ws.ws_col)) ws.ws_col = 80;
	if (__unlikely(!ws.ws_row)) ws.ws_row = 25;
	ws.ws_xpixel /= ws.ws_col;
	ws.ws_ypixel /= ws.ws_row;
	if (__unlikely(!ws.ws_xpixel)) ws.ws_xpixel = 8;
	if (__unlikely(!ws.ws_ypixel)) ws.ws_ypixel = ws.ws_xpixel * ws.ws_col * 3 / 4 / (__likely(ws.ws_row) ? ws.ws_row : 1);
	gpm_mx = ws.ws_col - gpm_zerobased;
	gpm_my = ws.ws_row - gpm_zerobased;
}

static int open_handle(Gpm_Connect *conn)
{
	CHHRQ ch;
	char *c, *d;
	int h;
	int check;
	gpm_consolefd = 0;
	check = 0;
	ch.h = 0;
	ch.option = "MOUSE";
	if (conn) {
		if (__likely(conn->eventMask != 0)) ch.value = "CHAR", check = 1;
		else ch.value = "COPYPASTE";
	} else {
		ch.value = "COPYPASTE";
	}
	if (check) {
		struct mouse_info mi;
		IOCTLRQ io;
		io.h = 0;
		io.ioctl = IOCTL_TTY_MOUSE_INFO;
		io.param = 0;
		io.v.ptr = (unsigned long)&mi;
		io.v.len = sizeof(struct mouse_info);
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&io, KERNEL$IOCTL);
		if (io.status < 0) return -1;
	}
	SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
	if (__unlikely(ch.status < 0)) return -1;
	c = KERNEL$HANDLE_PATH(0);
	if (__unlikely(!c)) return -1;
	d = alloca(strlen(c) + 14);
	strcpy(stpcpy(d, c), "/^MOUSESELECT");
	h = open(d, O_RDONLY | _O_NOPOSIX);
	if (__unlikely(h == -1)) return -1;
	if (gpm_fd == -1) gpm_fd = h;
	else dup2(h, gpm_fd), close(h);
	read(0, NULL, 0);
	get_terminal_size();
	return 0;
}

int Gpm_Open(Gpm_Connect *c, int vc)
{
	Gpm_Connect *cc;
	c->pid = getpid();
	c->vc = vc;
	if (__unlikely(!(cc = malloc(sizeof(Gpm_Connect))))) return -1;
	memcpy(cc, c, sizeof(Gpm_Connect));
	if (open_handle(cc)) {
		free(cc);
		return -1;
	}
	cc->next = conn;
	conn = cc;
	gpm_flag++;
	return gpm_fd;
}

int Gpm_Close(void)
{
	Gpm_Connect *c;
	if (__unlikely(!conn)) return 0;
	c = conn;
	conn = c->next;
	free(c);
	open_handle(conn);
	gpm_flag--;
	return __unlikely(conn != 0) ? -1 : 0;
}

static int lastx = 0, lasty = 0;
static struct mouse_state last_mouse_state = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
static int clicks = 0;
static u_jiffies_t last_click = 0;

int gpm_zerobased;

int Gpm_GetEvent(Gpm_Event *event)
{
	IOCTLRQ io;
	struct mouse_state m;
	if (__unlikely(!gpm_flag)) return 0;
	io.h = gpm_fd;
	io.ioctl = IOCTL_TTY_MOUSE_EVENT;
	io.param = 0;
	io.v.ptr = (unsigned long)&m;
	io.v.len = sizeof(struct mouse_state);
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_SIGINTERRUPT(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) return -1;
	/*if (!memcmp(&last_mouse_state, &m, sizeof(struct mouse_state))) return -1;*/
	memset(event, 0, sizeof(Gpm_Event));
	if (m.buttons & 1) event->buttons |= GPM_B_LEFT;
	if (m.buttons & 2) event->buttons |= GPM_B_RIGHT;
	if (m.buttons & 4) event->buttons |= GPM_B_MIDDLE;
	if (m.buttons & 8) event->buttons |= GPM_B_FOURTH;
	if (m.buttons & 16) event->buttons |= GPM_B_FIFTH;
	if (m.buttons & 32) event->buttons |= GPM_B_SIXTH;
	if (!event->buttons && last_mouse_state.buttons & 63) {
		if (last_mouse_state.buttons & 1) event->buttons |= GPM_B_LEFT;
		if (last_mouse_state.buttons & 2) event->buttons |= GPM_B_RIGHT;
		if (last_mouse_state.buttons & 4) event->buttons |= GPM_B_MIDDLE;
		if (last_mouse_state.buttons & 8) event->buttons |= GPM_B_FOURTH;
		if (last_mouse_state.buttons & 16) event->buttons |= GPM_B_FIFTH;
		if (last_mouse_state.buttons & 32) event->buttons |= GPM_B_SIXTH;
	}
	event->wdx = m.wx;
	event->wdy = -m.wy;
	event->modifiers = 0;
	event->vc = conn->vc;

	get_terminal_size();
	event->x = m.x / (int)ws.ws_xpixel;
	if (__unlikely(event->x >= (int)ws.ws_col)) event->x = ws.ws_col - 1;
	if (__unlikely(event->x < 0)) event->x = 0;
	event->y = m.y / (int)ws.ws_ypixel;
	if (__unlikely(event->y >= (int)ws.ws_row)) event->y = ws.ws_row - 1;
	if (__unlikely(event->y < 0)) event->y = 0;

	if (m.y <= 0) event->margin |= GPM_TOP;
	else if (m.y >= (int)ws.ws_row * (int)ws.ws_ypixel - 1) event->margin |= GPM_BOT;
	else if (m.x <= 0) event->margin |= GPM_LFT;
	else if (m.x >= (int)ws.ws_col * (int)ws.ws_xpixel - 1) event->margin |= GPM_RGT;

	if (__unlikely(conn->eventMask & conn->defaultMask & GPM_SMOOTH)) {
		event->dx = m.rx - last_mouse_state.rx;
		event->dy = m.ry - last_mouse_state.ry;
	} else {
		event->dx = lastx - event->x;
		event->dy = lasty - event->y;
	}
	lastx = event->x;
	lasty = event->y;

	if (m.buttons & ~last_mouse_state.buttons) event->type = GPM_DOWN;
	else if (~m.buttons & last_mouse_state.buttons) event->type = GPM_UP;
	else if (m.buttons) event->type = GPM_DRAG;
	else event->type = GPM_MOVE;

	if (__likely(event->type == GPM_MOVE)) event->clicks = 0;
	else {
		if (__likely(event->type == GPM_DOWN)) {
			u_jiffies_t j = KERNEL$GET_JIFFIES();
			if (__likely(j - last_click > DOUBLE_CLICK_TIME)) clicks = 0;
			else {
				clicks++;
				if (__unlikely(clicks == 3)) clicks = 0;
			}
			last_click = j;
		}
		event->clicks = clicks;
		event->type |= GPM_SINGLE << clicks;
	}

	memcpy(&last_mouse_state, &m, sizeof(struct mouse_state));
	/*if (!(event->type & conn->eventMask)) return -1;*/
	event->x += 1 - gpm_zerobased;
	event->y += 1 - gpm_zerobased;
	return 1;
}

int Gpm_GetSnapshot(Gpm_Event *event)
{
	IOCTLRQ io;
	struct mouse_info mi;
	Gpm_Event dummy;
	if (!event) event = &dummy;
	if (__unlikely(gpm_fd < 0)) return -1;
	io.h = gpm_fd;
	io.ioctl = IOCTL_TTY_MOUSE_INFO;
	io.param = 0;
	io.v.ptr = (unsigned long)&mi;
	io.v.len = sizeof(struct mouse_info);
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) return -1;
	memset(event, 0, sizeof(Gpm_Event));
	Gpm_GetEvent(event);
	event->dx = ws.ws_col;
	event->dy = ws.ws_row;
	return mi.buttons;
}

int gpm_morekeys = 0;
int gpm_hflag = 0;
int gpm_visiblepointer = 0;
Gpm_Handler *gpm_handler = NULL;
void *gpm_data = NULL;
struct timeval gpm_timeout = { 10, 0 };

int Gpm_Getc(FILE *f)
{
	static Gpm_Event ev = { 0 };
	fd_set fds;
	int fd, result, max;
	setvbuf(f, NULL, _IONBF, 0);
	if (__unlikely(!gpm_flag)) return getc(f);
	if (gpm_morekeys) return (*gpm_handler)(&ev, gpm_data);
	if (__unlikely(gpm_fd == -1)) return getc(f);
	fd = fileno(f);
	gpm_hflag = 0;
	max = (gpm_fd > fd) ? gpm_fd : fd;
	while (1) {
		if (gpm_visiblepointer) GPM_DRAWPOINTER(&ev);
		FD_ZERO(&fds);
		FD_SET(fd, &fds);
		FD_SET(gpm_fd, &fds);
		if (__unlikely(select(max + 1, &fds, NULL, NULL, NULL) == -1)) return fgetc(f);
		if (FD_ISSET(fd, &fds)) return fgetc(f);
		if (Gpm_GetEvent(&ev) && gpm_handler
		    && (result = (*gpm_handler)(&ev, gpm_data))) {
			gpm_hflag = 1;
			return result;
		}
	}
}

int Gpm_Wgetch(void *win)
{
	return Gpm_Getc(stdin);
}

int Gpm_DrawPointer(int x, int y, int fd)
{
	return 0;
}

int Gpm_CharsQueued ()
{
	return 0;
}

int Gpm_Repeat(int msec)
{
	int fd;
	fd_set fds;
	struct timeval tv;
	if (__unlikely(msec < 0)) {
		errno = EINVAL;
		return -1;
	}
	tv.tv_sec = msec / 1000;
	tv.tv_usec = msec % 1000 * 1000;
	fd = gpm_flag ? gpm_fd : 0;
	FD_ZERO(&fds);
	FD_SET(fd, &fds);
	return (select(fd + 1, &fds, NULL, NULL, &tv) == 0);
}

int Gpm_FitValuesM(int *x, int *y, int margin)
{
	if (margin == -1) {
		if (*x < !gpm_zerobased) *x = !gpm_zerobased;
		if (*x > gpm_mx) *x = gpm_mx;
		if (*y < !gpm_zerobased) *y = !gpm_zerobased;
		if (*y > gpm_my) *y = gpm_my;
		return 0;
	}
#if 0
	/* not needed because Spad gpm never returns values out of window */
	switch (margin) {
		case GPM_TOP: (*y)++; break;
		case GPM_BOT: (*y)--; break;
		case GPM_RGT: (*x)--; break;
		case GPM_LFT: (*x)++; break;
	}
#endif
	return 0;
}

