#include <ARCH/BITOPS.H>
#include <SPAD/DEV_KRNL.H>
#include <ERRNO.H>
#include <LIB/KERNEL/UASM.H>
#include <KERNEL/SYSCALL.H>
#include <LIB/KERNEL/ULIBC.H>
#include <LIB/KERNEL/IPC.H>
#include <LIB/KERNEL/UTHREAD.H>
#include <SPAD/BIO.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <STRING.H>
#include <STDLIB.H>
#include <PATHS.H>
#include <LIMITS.H>
#include <SYS/STAT.H>
#include <KERNEL/SELECT.H>
#include <SPAD/PKT.H>
#include <SPAD/TIMER.H>

#include <LIB/KERNEL/UIO.H>
#include <FCNTL.H>
#include <UNISTD.H>
#include <KERNEL/DEV.H>

#define MAX_REPOST_TIME		2

int KERNEL$__NEEDCANCEL = 0;
void (*KERNEL$__CANCEL_START_FN)(void) = NULL;
void (*KERNEL$__CANCEL_END_FN)(void) = NULL;
void (*KERNEL$__CANCEL_TEST_FN)(void) = NULL;

int __keep_open_files = 0;

void UNBLOCK_LNT(void)
{
	SYSCALL1(SYS_LN_UNBLOCK);
}

HANDLE *KERNEL$GET_HANDLE(int h)
{
	return (HANDLE *)h;
}

__finline__ char *KERNEL$HANDLE_PATH(int h)
{
	FFILE *f;
	if (__unlikely((unsigned)h >= udata.n_handles) || __unlikely(!(f = HANDLE_PTR(h)))) {
		if (__likely(h == BLOCK_HANDLE)) return "SYS$BLOCK:/";
		if (KERNEL$SPL == SPL_X(SPL_ZERO)) errno = EBADF;
		return NULL;
	}
	return f->path;
}

int KERNEL$HANDLE_FLAGS(int h)
{
	FFILE *f;
	if (__unlikely((unsigned)h >= udata.n_handles) || __unlikely(!(f = HANDLE_PTR(h)))) {
		if (__likely(h == BLOCK_HANDLE)) return 0;
		if (KERNEL$SPL == SPL_X(SPL_ZERO)) errno = EBADF;
		return -1;
	}
	return f->flags;
}

static WQ_DECL(repost_queue, "KERNEL$REPOST_QUEUE");

static void repost_timer_fn(TIMER *);

static DECL_TIMER(repost_timer);
static char repost_timer_active = 0;

static void init_repost_queue(void);

__COLD_ATTR__ static void init_repost_queue_inc(void)
{
	unsigned tm = time(NULL);
	unsigned l = tm - udata.last_repost_time;
	if (__unlikely(l < MAX_REPOST_TIME)) {
		if (__likely(udata.repost_time < MAX_REPOST_TIME * JIFFIES_PER_SECOND)) udata.repost_time++;
	} else {
		l -= MAX_REPOST_TIME;
		if (__unlikely(l >= udata.repost_time)) udata.repost_time = 0;
		else udata.repost_time -= l;
		udata.last_repost_time += l;
	}
	init_repost_queue();
}

__COLD_ATTR__ static void init_repost_queue(void)
{
	if (!repost_timer_active) {
		repost_timer_active = 1;
		repost_timer.fn = repost_timer_fn;
		KERNEL$SET_TIMER(udata.repost_time, &repost_timer);
	}
}

__COLD_ATTR__ static void repost_timer_fn(TIMER *t)
{
	udata.uiorq_overflow = 0;
	udata.last_repost_time = time(NULL);
	repost_timer_active = 0;
	WQ_WAKE_ONE(&repost_queue);
	if (!WQ_EMPTY(&repost_queue)) init_repost_queue();
}

static int test_open_file_delete(const char *path)
{
	int i;
	for (i = 0; i < udata.n_handles; i++) {
		FFILE *f;
		if (__unlikely((f = HANDLE_PTR(i)) != NULL)) {
			if (__unlikely(!_strcasecmp(f->path, path))) {
				f->flags |= _O_PENDING_DELETE;
				return 1;
			}
		}
	}
	return 0;
}

static int extend_handles(void)
{
	FFILE **nh, **oh;
	if (__unlikely((unsigned)(udata.n_handles << 1) > MAX_HANDLE + 1)) {
		return -EMFILE;
	}
	if (__unlikely(!(nh = malloc((udata.n_handles << 1) * sizeof(HANDLE *))))) return -ENOMEM;
	memcpy(nh, udata.handles, udata.n_handles * sizeof(HANDLE *));
	memset(nh + udata.n_handles, 0, udata.n_handles * sizeof(HANDLE *));
	oh = udata.handles;
	__barrier();
	udata.handles = nh;
	__barrier();
	udata.n_handles <<= 1;
	__barrier();
	free(oh);
	return 0;
}

static int alloc_fd(unsigned flags)
{
	int i, r;
	if (__unlikely(flags & _O_NOPOSIX)) goto rescan;
	
	for (i = 0; i < POSIX_RESERVED_HANDLES; i++) {
		if (__unlikely(!udata.handles[i])) return i;
	}

	rescan:
	if (__unlikely(udata.last_handle < POSIX_RESERVED_HANDLES)) udata.last_handle = POSIX_RESERVED_HANDLES;
	for (i = udata.last_handle; i < udata.n_handles; i++) {
		if (__likely(!udata.handles[i])) {
			if (__unlikely(i == SWAPPER_HANDLE) || __unlikely(i == BLOCK_HANDLE)) continue;
			udata.last_handle = i + 1;
			return i;
		}
	}
	for (i = POSIX_RESERVED_HANDLES; i < udata.last_handle && i < udata.n_handles; i++) {
		if (__unlikely(!udata.handles[i])) {
			if (__unlikely(i == SWAPPER_HANDLE) || __unlikely(i == BLOCK_HANDLE)) continue;
			udata.last_handle = i + 1;
			return i;
		}
	}

	i = udata.n_handles;
	if (__unlikely(r = extend_handles())) return r;
	if (__unlikely(i == SWAPPER_HANDLE) || __unlikely(i == BLOCK_HANDLE)) goto rescan;
	return i;
}

static AST_STUB OPEN_AST;

DECL_IOCALL(KERNEL$OPEN, SPL_DEV, OPENRQ)
{
	int i;
	FFILE *f;
	unsigned s;
	if (__unlikely(RQ->flags & O_ASYNC)) {
		RQ->status = -ENOSYS;
		RETURN_AST(RQ);
	}
	f = open_file_name(RQ, &s);
	if (__unlikely(__IS_ERR(f))) {
		RQ->status = __PTR_ERR(f);
		RETURN_AST(RQ);
	}
	if (__unlikely(s) && __likely(!(RQ->flags & _O_NOMULTIOPEN))) {
		MULTIOPEN(RQ, f);
		RETURN;
	}
	if (__unlikely(RQ->flags & _O_DELETE) && __unlikely(__keep_open_files) && test_open_file_delete(f->path)) {
		free(f);
		RQ->status = 0;
		RETURN_AST(RQ);
	}
	f->pos = 0;
	f->flags = RQ->flags + 1;
	/*	if (__unlikely((*(__u32 *)f->path & __32BE2CPU(0xdfdfdf00)) == __32BE2CPU(0x53544400)) && (!_memcasecmp(f->path, _LNM_STDIN":/", 7) || !_memcasecmp(f->path, _LNM_STDOUT":/", 8) || !_memcasecmp(f->path, _LNM_STDERR":/", 8))) f->flags |= _O_NOFASTSEEK;	*/
	f->l_refcount = 1;
	f->aux = 0;
	INIT_XLIST(&f->blobs);
	/*__debug_printf("open: %s, %p, %d\n", RQ->path, f, f->flags);*/

	i = alloc_fd(RQ->flags);
	/*__debug_printf("allocated: %s->%d\n", RQ->path, i);*/
	if (__unlikely(i < 0)) {
		RQ->status = i;
		UNREF_FILE(f);
		RETURN_AST(RQ);
	}
	__barrier();
	udata.handles[i] = f;
	RQ->tmp3 = KERNEL_AST_TMP3(KERNEL_CALL_OPEN);
	RQ->real_ast = RQ->fn;
	RQ->handle = i;
	RQ->fn = OPEN_AST;
	if (__unlikely(RQ->flags & _O_NOOPEN)) {
		if (__unlikely(RQ->flags & _O_CLOSE)) RQ->status = 0;
		else RQ->status = i;
		RETURN_AST(RQ);
	}
	RAISE_SPL(SPL_TOP);
	IO_ENABLE_KERNEL_CANCEL(RQ, RETURN);
	SYSCALL3(SYS_OPEN | (__unlikely(RQ->flags & _O_NOOPEN_CALL) ? 0 : ((RQ->flags + 1) & SYS_EXTENDED_PARAM)), (unsigned long)RQ, i);
	RETURN;
}

static __finline__ void FREE_HANDLE(unsigned h)
{
	FFILE *f = HANDLE_PTR(h);
	udata.handles[h] = NULL;
	__barrier();
	UNREF_FILE(f);
}

static DECL_AST(OPEN_AST, SPL_DEV, OPENRQ)
{
	RQ->fn = RQ->real_ast;
	if (__unlikely(RQ->flags & _O_CLOSE) || __unlikely(RQ->status < 0)) {
		FREE_HANDLE(RQ->handle);
		SYSCALL2(SYS_FAST_CLOSE, RQ->handle);
		if (__unlikely((RQ->status & RQS_PROC_MASK) == RQS_PROC)) {
			WQ_WAIT(&repost_queue, RQ, KERNEL$OPEN);
			RAISE_SPL(SPL_TOP);
			init_repost_queue_inc();
			RETURN;
		}
		if (__unlikely(RQ->status == -ESRCH)) RQ->status = 0;
	}
	RETURN_AST(RQ);
}

static AST_STUB CLOSE_AST;
static AST_STUB FAKE_CLOSE_AST;

DECL_IOCALL(KERNEL$CLOSE, SPL_DEV, CLOSERQ)
{
	FFILE *f;
	RQ->tmp1 = (unsigned long)KERNEL$CLOSE;
	if (__unlikely(CLOSE_SELECT_HANDLE(RQ->h, (IORQ *)RQ))) RETURN;
	RQ->real_ast = RQ->fn;
	RQ->tmp3 = KERNEL_AST_TMP3(KERNEL_CALL_CLOSE);
	RQ->fn = CLOSE_AST;
	if (__likely((unsigned)RQ->h < udata.n_handles) && __likely((f = HANDLE_PTR(RQ->h)) != NULL)) {
		if (__unlikely(f->flags & _O_NOOPEN)) {
			SYSCALL2(SYS_FAST_CLOSE, RQ->h);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
	}
	RAISE_SPL(SPL_TOP);
	IO_ENABLE_KERNEL_CANCEL(RQ, RETURN);
	if (__likely(SYSCALL3(SYS_CLOSE, (unsigned long)RQ, RQ->h) != 0)) {
		RQ->status = 0;
		RETURN_AST(RQ);
	}
	RETURN;
}

static DECL_AST(CLOSE_AST, SPL_DEV, CLOSERQ)
{
	RQ->fn = RQ->real_ast;
	if (__unlikely((RQ->status & RQS_PROC_MASK) == RQS_PROC)) {
		WQ_WAIT(&repost_queue, RQ, KERNEL$CLOSE);
		RAISE_SPL(SPL_TOP);
		init_repost_queue_inc();
		RETURN;
	}
	if (__unlikely((unsigned)RQ->h >= udata.n_handles) || __unlikely(!udata.handles[RQ->h])) RQ->status = -EBADF;
	else FREE_HANDLE(RQ->h);
	RQ->h = -1;
	RETURN_AST(RQ);
}

static DECL_IOCALL(FAKE_CLOSE, SPL_DEV, CLOSERQ)
{
	FFILE *f;
	RQ->real_ast = RQ->fn;
	RQ->tmp3 = KERNEL_AST_TMP3(KERNEL_CALL_FAKE_CLOSE);
	RQ->fn = FAKE_CLOSE_AST;
	if (__likely((unsigned)RQ->h < udata.n_handles) && __likely((f = HANDLE_PTR(RQ->h)) != NULL)) {
		if (__unlikely(f->flags & _O_NOOPEN)) {
			SYSCALL2(SYS_FAST_CLOSE, RQ->h);
			RQ->status = 0;
			RETURN_AST(RQ);
		}
	}
	RAISE_SPL(SPL_TOP);
	IO_ENABLE_KERNEL_CANCEL(RQ, RETURN);
	if (__likely(SYSCALL3(SYS_CLOSE, (unsigned long)RQ, RQ->h) != 0)) {
		RQ->status = 0;
		RETURN_AST(RQ);
	}
	RETURN;
}

DECL_AST(FAKE_CLOSE_AST, SPL_DEV, CLOSERQ)
{
	RQ->fn = RQ->real_ast;
	if (__unlikely((RQ->status & RQS_PROC_MASK) == RQS_PROC)) {
		WQ_WAIT(&repost_queue, RQ, FAKE_CLOSE);
		RAISE_SPL(SPL_TOP);
		init_repost_queue_inc();
		RETURN;
	}
	if (__unlikely((unsigned)RQ->h >= udata.n_handles) || __unlikely(!udata.handles[RQ->h])) RQ->status = -EBADF;
	RETURN_AST(RQ);
}

void KERNEL$FAST_CLOSE(int h)
{
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("KERNEL$FAST_CLOSE AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_DEV);
	if (__unlikely((unsigned)h >= udata.n_handles || __unlikely(!udata.handles[h]))) goto ret;
		/*KERNEL$SUICIDE("KERNEL$FAST_CLOSE(%d): BAD HANDLE", h);*/
	FREE_HANDLE(h);
	ret:
	SYSCALL2(SYS_FAST_CLOSE, h);
	LOWER_SPLX(spl);
}

static __finline__ void fake_close(int h)
{
	CLOSERQ c;
	c.h = h;
	SYNC_IO(&c, FAKE_CLOSE);
}

void fake_close_all(void)
{
	int i;
	for (i = 0; i < udata.n_handles; i++) if (__unlikely(udata.handles[i] != NULL)) {
		fake_close(i);
	}
	for (i = 0; i < udata.n_handles; i++) if (__unlikely(udata.handles[i] != NULL)) {
		__ORI(&HANDLE_PTR(i)->flags, _O_NOOPEN);
	}
}

void set_pipe_buffer(int h, FPIPE *buf)
{
	HANDLE_PTR(h)->pos = (off_t)(unsigned long)buf;
}

static AST_STUB CHANGE_HANDLE_DONE;

DECL_IOCALL(KERNEL$CHANGE_HANDLE, SPL_DEV, CHHRQ)
{
	int r;
	unsigned long l;
	FFILE *f, *nf;
	if (__unlikely((unsigned)RQ->h >= udata.n_handles) || __unlikely(!(f = HANDLE_PTR(RQ->h)))) {
		RQ->status = -EBADF;
		RETURN_AST(RQ);
	}
	if (__unlikely(f->flags & _O_PIPE)) {
		RQ->status = -EINVAL;
		RETURN_AST(RQ);
	}
	l = sizeof(FFILE) + strlen(f->path) + 1 + (RQ->value ? strlen(RQ->option) + strlen(RQ->value) + 3 : 0);
	if (__unlikely(!(nf = malloc(l)))) {
		RQ->status = -ENOMEM;
		RETURN_AST(RQ);
	}
	r = substitute_option(f->path, nf->path, RQ->option, RQ->value);
	if (__unlikely(r <= 0)) {
		free_ret_r:
		RQ->status = r;
		free(nf);
		RETURN_AST(RQ);
	}
	nf->pos = f->pos;
	nf->flags = f->flags;
	nf->l_refcount = 0;
	nf->aux = f->aux;
	INIT_XLIST(&nf->blobs);
	r = alloc_fd(_O_NOPOSIX);
	if (__unlikely(r < 0)) goto free_ret_r;
	__barrier();
	udata.handles[r] = (void *)((unsigned long)nf | FD_CLOEXEC);
	RQ->ioctlrq.fn = &CHANGE_HANDLE_DONE;
	RQ->ioctlrq.h = r;
	RQ->ioctlrq.ioctl = IOCTL_NOP;
	RQ->ioctlrq.param = 0;
	RQ->ioctlrq.v.ptr = 0;
	RQ->ioctlrq.v.len = 0;
	RQ->ioctlrq.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ_CANCELABLE(&RQ->ioctlrq, KERNEL$IOCTL, RQ);
}

static DECL_AST(CHANGE_HANDLE_DONE, SPL_DEV, IOCTLRQ)
{
	CHHRQ *rq = GET_STRUCT(RQ, CHHRQ, ioctlrq);
	int i;
	FFILE *f, *nf;
	IO_DISABLE_CHAIN_CANCEL(SPL_X(SPL_DEV), rq);
	nf = HANDLE_PTR(rq->ioctlrq.h);
	if (__unlikely(rq->ioctlrq.status < 0) && __unlikely(rq->ioctlrq.status != -ENOOP)) {
		udata.handles[rq->ioctlrq.h] = NULL;
		SYSCALL2(SYS_FAST_CLOSE, rq->ioctlrq.h);
		free(nf);
		rq->status = rq->ioctlrq.status;
		RETURN_AST(rq);
	}
	f = HANDLE_PTR(rq->h);
	nf->pos = f->pos;
	nf->flags = f->flags;
	nf->aux = f->aux;
	MOVE_XLIST_HEAD(&nf->blobs, &f->blobs);
	__barrier();
	INIT_XLIST(&f->blobs);
	for (i = 0; i < udata.n_handles; i++) {
		if (__unlikely(HANDLE_PTR(i) == f)) {
			nf->l_refcount++;
			udata.handles[i] = (void *)((unsigned long)nf | ((unsigned long)udata.handles[i] & 1));
			SYSCALL2(SYS_FAST_CLOSE, i);
			CLOSE_SELECT_HANDLE(i, NULL);
			if (__likely(UNREF_FILE(f))) break;
		}
	}
	if (__unlikely(!nf->l_refcount))
		KERNEL$SUICIDE("CHANGE_HANDLE_DONE: HANDLE %d(%s) DISAPPEARED WHILE CHANGING", rq->h, nf->path);
	udata.handles[rq->ioctlrq.h] = NULL;
	SYSCALL2(SYS_FAST_CLOSE, rq->ioctlrq.h);
	rq->status = 0;
	RETURN_AST(rq);
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$SET_HANDLE_BLOB, SPL_DEV, HBLOBRQ)
{
	FFILE *f;
	FBLOB_REF *ref;
	void *to_free_1, *to_free_2;
	unsigned i;
	if (__unlikely((unsigned)RQ->h >= udata.n_handles) || __unlikely(!(f = HANDLE_PTR(RQ->h)))) {
		RQ->status = -EBADF;
		RETURN_AST(RQ);
	}
	ref = malloc(sizeof(FBLOB_REF));
	if (__unlikely(!ref)) {
		RQ->status = -ENOMEM;
		RETURN_AST(RQ);
	}
	SET_FILE_BLOB(f, ref, RQ->blob, &to_free_1, &to_free_2);
	for (i = 0; i < udata.n_handles; i++) {
		if (__unlikely(HANDLE_PTR(i) == f)) {
			SYSCALL2(SYS_FAST_CLOSE, i);
			CLOSE_SELECT_HANDLE(i, NULL);
		}
	}
	free(to_free_1);
	free(to_free_2);
	RQ->status = 0;
	RETURN_AST(RQ);
}

void USR_SWAP_ALL_HANDLES(void)
{
	unsigned i;
	for (i = 0; i < udata.n_handles; i++) if (HANDLE_PTR(i)) SYSCALL2(SYS_FAST_CLOSE, i);
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_READ, SPL_TOP, SIORQ)
{
	RQ->fn = (AST_STUB *)RQ->tmp2;
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, KERNEL$READ);
	WQ_WAIT(&repost_queue, RQ, KERNEL$READ);
	init_repost_queue_inc();
	RETURN;
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_WRITE, SPL_TOP, SIORQ)
{
	RQ->fn = (AST_STUB *)RQ->tmp2;
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, KERNEL$WRITE);
	WQ_WAIT(&repost_queue, RQ, KERNEL$WRITE);
	init_repost_queue_inc();
	RETURN;
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_AREAD, SPL_TOP, AIORQ)
{
	RQ->fn = (AST_STUB *)RQ->tmp2;
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, KERNEL$AREAD);
	WQ_WAIT(&repost_queue, RQ, KERNEL$AREAD);
	init_repost_queue_inc();
	RETURN;
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_AWRITE, SPL_TOP, AIORQ)
{
	RQ->fn = (AST_STUB *)RQ->tmp2;
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, KERNEL$AWRITE);
	WQ_WAIT(&repost_queue, RQ, KERNEL$AWRITE);
	init_repost_queue_inc();
	RETURN;
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_IOCTL, SPL_TOP, IOCTLRQ)
{
	RQ->fn = (AST_STUB *)RQ->tmp2;
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, KERNEL$IOCTL);
	WQ_WAIT(&repost_queue, RQ, KERNEL$IOCTL);
	init_repost_queue_inc();
	RETURN;
}

static AST_STUB KERNEL_REPOST_OPEN_SPL_DEV;

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_OPEN, SPL_TOP, OPENRQ)
{
	RQ->fn = KERNEL_REPOST_OPEN_SPL_DEV;
	RETURN_AST(RQ);
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_OPEN_SPL_DEV, SPL_DEV, OPENRQ)
{
	RQ->fn = RQ->real_ast;
	FREE_HANDLE(RQ->handle);
	SYSCALL2(SYS_FAST_CLOSE, RQ->handle);
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, KERNEL$OPEN);
	RAISE_SPL(SPL_TOP);
	WQ_WAIT(&repost_queue, RQ, KERNEL$OPEN);
	init_repost_queue_inc();
	RETURN;
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_CLOSE, SPL_TOP, CLOSERQ)
{
	RQ->fn = RQ->real_ast;
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, KERNEL$CLOSE);
	WQ_WAIT(&repost_queue, RQ, KERNEL$CLOSE);
	init_repost_queue_inc();
	RETURN;
}

__COLD_ATTR__ static DECL_AST(KERNEL_REPOST_FAKE_CLOSE, SPL_TOP, CLOSERQ)
{
	RQ->fn = RQ->real_ast;
	if (__likely(!udata.uiorq_overflow)) RETURN_IORQ_LSTAT(RQ, FAKE_CLOSE);
	WQ_WAIT(&repost_queue, RQ, FAKE_CLOSE);
	init_repost_queue_inc();
	RETURN;
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$WAKE_READ, SPL_TOP, SIORQ)
{
	RETURN_IORQ_LSTAT(RQ, RQ->handle->op->read);
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$WAKE_WRITE, SPL_TOP, SIORQ)
{
	RETURN_IORQ_LSTAT(RQ, RQ->handle->op->write);
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$WAKE_AREAD, SPL_TOP, AIORQ)
{
	RETURN_IORQ_LSTAT(RQ, RQ->handle->op->aread);
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$WAKE_AWRITE, SPL_TOP, AIORQ)
{
	RETURN_IORQ_LSTAT(RQ, RQ->handle->op->awrite);
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$WAKE_IOCTL, SPL_TOP, IOCTLRQ)
{
	RETURN_IORQ_LSTAT(RQ, RQ->handle->op->ioctl);
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$WAKE_BIO, SPL_TOP, BIORQ)
{
	RETURN_IORQ_LSTAT(RQ, RQ->handle->op->bio);
}

__COLD_ATTR__ DECL_IOCALL(KERNEL$WAKE_PKTIO, SPL_TOP, PACKET)
{
	RETURN_IORQ_LSTAT(RQ, RQ->handle->op->pktio);
}

static void call_dup(int h)
{
	IOCTLRQ io;
	if (__unlikely(HANDLE_PTR(h)->flags & _O_NOOPEN)) return;
	io.h = h;
	io.ioctl = IOCTL_DUP;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
}

static int dup_fd(int src, int dest, int calldup)
{
	int r;
	FFILE *f;
	while (__unlikely((unsigned)dest >= udata.n_handles)) {
		if (__unlikely(dest > MAX_HANDLE)) {
			return -EMFILE;
		}
		if (__unlikely(r = extend_handles())) return r;
	}
	(udata.handles[dest] = f = HANDLE_PTR(src))->l_refcount++;
	if (__unlikely(f->flags & (_O_PIPE | _O_SOCKET)) && __likely(calldup)) {
		call_dup(src);
	}
	return dest;
}

int fcntl(int h, int op, ...)
{
	/*FFILE *f;*/
	int i;
	int r;
	va_list args;
	int spl;
	IOCTLRQ io;
	CHHRQ c;
	__cancel_test();
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL))) {
		if (__unlikely(op != F_SETFD) && __unlikely(op != F_GETFD))
			KERNEL$SUICIDE("fcntl AT SPL %08X", KERNEL$SPL);
	} else RAISE_SPL(SPL_DEV);
	if (__unlikely((unsigned)h >= udata.n_handles) || __unlikely(!udata.handles[h])) {
		errno = EBADF;
		r = -1;
		goto ret;
	}
	switch (op) {
		case F_DUPFD:
			va_start(args, op);
			i = va_arg(args, int);
			va_end(args);
			while ((__likely(i < udata.n_handles) && udata.handles[i]) || __unlikely(i == SWAPPER_HANDLE) || __unlikely(i == BLOCK_HANDLE)) i++;
			r = dup_fd(h, i, 1);
			if (__unlikely(r < 0)) errno = -r, r = -1;
			break;
		case F_GETFD:
			r = (int)(unsigned long)udata.handles[h] & FD_CLOEXEC;
			break;
		case F_SETFD:
			va_start(args, op);
			i = va_arg(args, int);
			va_end(args);
			if (__unlikely(i & ~FD_CLOEXEC)) goto einval;
			udata.handles[h] = (FFILE *)(((unsigned long)udata.handles[h] & ~(unsigned long)FD_CLOEXEC) | (i & FD_CLOEXEC));
			r = 0;
			break;
		case F_GETFL:
			r = (HANDLE_PTR(h)->flags - 1) & __O_FCNTL_GETFL_MASK;
			io.h = h;
			io.ioctl = IOCTL_FCNTL_GETFL;
			io.param = r;
			io.v.ptr = 0;
			io.v.len = 0;
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
			if (__unlikely(io.status >= 0)) r = io.status;
			break;
		case F_SETFL:
			va_start(args, op);
			i = va_arg(args, int);
			va_end(args);
			if (__unlikely(i & O_ASYNC)) {
				errno = ENOSYS;
				r = -1;
				break;
			}
			io.h = h;
			io.ioctl = IOCTL_FCNTL_SETFL;
			io.param = i;
			io.v.ptr = 0;
			io.v.len = 0;
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
			HANDLE_PTR(h)->flags = (HANDLE_PTR(h)->flags & ~__O_FCNTL_SETFL_MASK) | (i & __O_FCNTL_SETFL_MASK);
			c.h = h;
			c.option = "NONBLOCK";
			c.value = i & O_NONBLOCK ? "" : NULL;
			SYNC_IO(&c, KERNEL$CHANGE_HANDLE);
			c.option = "DIRECT";
			c.value = i & O_DIRECT ? "" : NULL;
			SYNC_IO(&c, KERNEL$CHANGE_HANDLE);
			r = 0;
			break;
		default:
			einval:
			errno = EINVAL;
			r = -1;
			break;
	}
	ret:
	if (__likely(SPLX_BELOW(spl, SPL_X(SPL_DEV)))) LOWER_SPLX(spl);
	return r;
}

int dup(int h)
{
	return fcntl(h, F_DUPFD, 0);
}

int dup2(int h, int to)
{
	int r;
	int spl;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL)))
		KERNEL$SUICIDE("dup2 AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_DEV);
	again:
	if (__unlikely((unsigned)h >= udata.n_handles) || __unlikely(!udata.handles[h])) {
		errno = EBADF;
		LOWER_SPLX(spl);
		return -1;
	}
	if (__unlikely(h == to)) {
		LOWER_SPLX(spl);
		return 0;
	}
	if (__likely(to < udata.n_handles) && __unlikely(udata.handles[to] != NULL)) {
		close(to);
		goto again;
	}
	r = dup_fd(h, to, 1);
	if (__unlikely(r < 0)) errno = -r, r = -1;
	LOWER_SPLX(spl);
	return r;
}

void _close_unlocked(int h)
{
	FREE_HANDLE(h);
	SYSCALL2(SYS_FAST_CLOSE, h);
}

int _dup_noposix_unlocked(int h)
{
	int i;
	i = alloc_fd(_O_NOPOSIX);
	if (__unlikely(i < 0)) return i;
	return dup_fd(h, i, 0);
}

int _dup_noposix(int h, IORQ *rq, IO_STUB *fn)
{
	int spl, r;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL))) KERNEL$SUICIDE("_dup_noposix AT SPL %08X", spl);
	RAISE_SPL(SPL_DEV);
	if (__unlikely((unsigned)h >= udata.n_handles) || __unlikely(!udata.handles[h])) {
		r = -EBADF;
		goto rr;
	}
	r = _dup_noposix_unlocked(h);
	rr:
	LOWER_SPLX(spl);
	if (__unlikely(r < 0)) {
		rq->status = r;
		CALL_AST(rq);
		return -1;
	}
	return r;
}

static int add_lnm(LOGICAL_LIST_REQUEST *r, char *name, int flags, size_t prefix_len)
{
	LOGICAL_LIST_ENTRY *le, **entries;
	if (_memcasecmp(name, r->prefix, prefix_len)) return 0;
	if (__unlikely(!(entries = realloc(r->entries, (r->n_entries + 1) * sizeof(char *))))) {
		e0:
		r->status = -ENOMEM;
		KERNEL$FREE_LOGICAL_LIST(r);
		return -1;
	}
	r->entries = entries;
	if (__unlikely(!(le = malloc(sizeof(LOGICAL_LIST_ENTRY) + strlen(name)))))
		goto e0;
	r->entries[r->n_entries++] = le;
	le->driver_name = NULL;
	le->flags = flags;
	strcpy(le->name, name);
	ADD_TO_XLIST(&r->freelist, &le->list);
	return 0;
}

#define LIST_LOGICALS_BUFFER	1024

DECL_IOCALL(KERNEL$LIST_LOGICALS, SPL_DEV, LOGICAL_LIST_REQUEST)
{
	long r;
	int i;
	LNTE *ln;
	DECL_LN_LIST_STAT(LIST_LOGICALS_BUFFER) ln_list_stat;
	size_t prefix_len = strlen(RQ->prefix);
	RQ->entries = NULL;
	RQ->n_entries = 0;
	INIT_XLIST(&RQ->freelist);
	for (i = 0; i < LNT_HASH_SIZE; i++) XLIST_FOR_EACH_UNLIKELY(ln, &LN->hash[i], LNTE, hash) {
		if (__unlikely(add_lnm(RQ, ln->name, ln->handle, prefix_len))) RETURN_AST(RQ);
	}
	ln_list_stat.depth = 0;
	ln_list_stat.hash = 0;
	ln_list_stat.ptr = NULL;
	next_entry:
	if (__likely((r = SYSCALL3(SYS_LN_LIST, (unsigned long)&ln_list_stat, LIST_LOGICALS_BUFFER)) >= 0)) {
		char *end = ln_list_stat.result + r;
		char *ptr = ln_list_stat.result;
		ln_list_stat.depth = ln_list_stat.depth_res;
		ln_list_stat.hash = ln_list_stat.hash_res;
		ln_list_stat.ptr = ln_list_stat.ptr_res;
		while (ptr != end) {
#if __DEBUG >= 1
			if (__unlikely(ptr > end))
				KERNEL$SUICIDE("KERNEL$LIST_LOGICALS: RESULT MISMATCH: %p > %p", ptr, end);
#endif
			if (__unlikely(add_lnm(RQ, ptr, LNTE_PARENT, prefix_len))) RETURN_AST(RQ);
			ptr += strlen(ptr) + 1;
		}
		if (__unlikely(ln_list_stat.depth >= 0)) goto next_entry;
	} else if (__unlikely(r < 0)) {
		RQ->status = r;
		KERNEL$FREE_LOGICAL_LIST(RQ);
		RETURN_AST(RQ);
	}
	SORT_LOGICAL_LIST(RQ);
	RQ->status = 0;
	RETURN_AST(RQ);
}

int UIO_MAKE_FORKTABLE(FFILE ***forktable, int *forktable_n)
{
	FFILE **ft;
	int n;
	int i;
/*__debug_printf("UIO_MAKE_FORK_TABLE(%d,%s)\n", SYSCALL1(SYS_DEPTH), __progname);*/
	for (n = udata.n_handles - 1; n >= 0; n--) if (__unlikely(udata.handles[n] != NULL)) break;
	n++;
	*forktable_n = n;
	ft = malloc(n * sizeof(FFILE *));
	*forktable = ft;
	if (__unlikely(!ft)) return -1;
	for (i = 0; i < n; i++) {
		FFILE *f = ft[i] = HANDLE_PTR(i);
		if (__likely(f != NULL)) {
			f->l_refcount++;
		}
	}
	for (i = 0; i < n; i++) {
		FFILE *f = ft[i];
		if (f && __unlikely(f->flags & (_O_PIPE | _O_SOCKET))) call_dup(i);
	}
	return 0;
}

void UIO_FREE_FORKTABLE(FFILE **forktable, int forktable_n)
{
	int i;
/*__debug_printf("UIO_FREE_FORK_TABLE(%d,%s)\n", SYSCALL1(SYS_DEPTH), __progname);*/
	for (i = 0; i < forktable_n; i++) {
		FFILE *f = forktable[i];
		if (__likely(f != NULL)) UNREF_FILE(f);
	}
	free(forktable);
}

void UIO_FORK(void)
{
	int i;
/*__debug_printf("UIO_FORK(%d,%s)\n", SYSCALL1(SYS_DEPTH), __progname);*/

	while (__unlikely(WQ_GET_ONE(&repost_queue) != NULL)) ;
	repost_timer_active = 0;
	INIT_TIMER(&repost_timer);

	LOWER_SPL(SPL_MALLOC);
	for (i = 0; i < udata.n_handles; i++) {
		FFILE *f = HANDLE_PTR(i);
		if (__unlikely(f != NULL)) {
			FREE_PIPE(f);
			*(int *)(void *)&f->pos = i;
			__barrier();
			f->flags = (f->flags & ~_O_PENDING_DELETE) | _O_KRNL_FORKED;
		}
	}
	SHM_FORK();
}

off_t lseek(int h, off_t offset, int whence)
{
	int spl;
	FFILE *f;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_DEV), spl = KERNEL$SPL))) goto slow;
	RAISE_SPL(SPL_DEV);
	if (__unlikely((unsigned)h >= udata.n_handles)) goto lower_slow;
	f = HANDLE_PTR(h);
	if (__unlikely(!f)) goto lower_slow;
	if (__unlikely(f->flags & (_O_NOFASTSEEK | _O_PIPE | _O_SOCKET | _O_KRNL_FORKED))) {
		if (__unlikely(f->flags & (_O_PIPE | _O_SOCKET))) {
			LOWER_SPLX(spl);
			errno = ESPIPE;
			return -1;
		}
		goto lower_slow;
	}
	if (__unlikely(whence != SEEK_SET)) {
		off_t cp;
		if (__unlikely(whence != SEEK_CUR)) goto lower_slow;
		if (__unlikely(!TEST_AND_SET_OFFSET_AVAILABLE)) goto lower_slow;
		again:
		cp = f->pos;	/* SMPFIX: use atomic read,
					use atomic write in all drivers */
		offset += cp;
		if (__unlikely(TEST_AND_SET_OFFSET(&f->pos, cp, offset) != 0)) goto again;
	} else {
		SET_OFFSET(&f->pos, offset);
	}
	f->pos = offset;
	LOWER_SPLX(spl);
	return offset;
	lower_slow:
	LOWER_SPLX(spl);
	slow:
	return slow_lseek(h, offset, whence);
}

void UIO_CLEANUP(int status)
{
	int i;
	if (__unlikely(udata.exiting)) return;
	THREAD_FORK();		/* prevent other threads from running */
	udata.exiting = 1;
	KERNEL$BPL = SPL_X(SPL_ZERO);
	if (SPLX_BELOW(SPL_X(SPL_BOTTOM), KERNEL$SPL)) LOWER_SPL(SPL_BOTTOM);
	udata.sigblock = -1;
	SHM_CLEANUP();
	if (__unlikely(SUBPROC_CTL != NULL)) LIBPROC_CLEANUP(__IS_EXIT_SIGNAL(status));
	fake_close_all();
	for (i = 0; i < udata.n_handles; i++) if (__unlikely(udata.handles[i] != NULL)) {
		if (__unlikely(HANDLE_PTR(i)->flags & _O_PENDING_DELETE))
		 	close(i);
	}
	if (__likely(!status)) SOCKET_CLEANUP();
	while (__unlikely(!XLIST_EMPTY(&PENDING_DELETE_LIST))) KERNEL$SLEEP(1);
	ULIBC_WAIT_FOR_CP();
}

#define MAXLEN	8
static char *const inithandles[3] = { _LNM_STDIN":/", _LNM_STDOUT":/", _LNM_STDERR":/" };

void UIO_INIT(void)
{
	int i;
	for (i = 0; i < 3; i++) {
		FFILE *f = malloc(sizeof(FFILE) + MAXLEN + 1);
		if (__unlikely(!f)) continue;
		f->pos = 0;
		f->flags = _O_KRNL_READ | _O_KRNL_WRITE;
		f->l_refcount = 1;
		f->aux = 0;
		INIT_XLIST(&f->blobs);
		strcpy(f->path, inithandles[i]);
#if __DEBUG >= 1
		if (__unlikely(udata.handles[i] != NULL)) KERNEL$SUICIDE("UIO_INIT: HANDLE %d ALREADY OPEN", i);
#endif
		udata.handles[i] = f;
	}
}

void UIO_DUPSTD(void)
{
	int i;
	for (i = 0; i < 3; i++) {
		struct stat st;
		FFILE *f = HANDLE_PTR(i);
		if (__likely(f != NULL) && __likely(!fstat(i, &st))) {
			if (__unlikely(S_ISFIFO(st.st_mode))) {
				f->flags |= _O_PIPE;
				goto d;
			} else if (__unlikely(S_ISSOCK(st.st_mode))) {
				f->flags |= _O_SOCKET;
				d:
				call_dup(i);
			}
		}
	}
}

void UIO_RST_INIT(void)
{
	udata.repost_ast_fn[KERNEL_CALL_READ] = KERNEL_REPOST_READ;
	udata.repost_ast_fn[KERNEL_CALL_WRITE] = KERNEL_REPOST_WRITE;
	udata.repost_ast_fn[KERNEL_CALL_AREAD] = KERNEL_REPOST_AREAD;
	udata.repost_ast_fn[KERNEL_CALL_AWRITE] = KERNEL_REPOST_AWRITE;
	udata.repost_ast_fn[KERNEL_CALL_IOCTL] = KERNEL_REPOST_IOCTL;
	udata.repost_ast_fn[KERNEL_CALL_OPEN] = KERNEL_REPOST_OPEN;
	udata.repost_ast_fn[KERNEL_CALL_CLOSE] = KERNEL_REPOST_CLOSE;
	udata.repost_ast_fn[KERNEL_CALL_FAKE_CLOSE] = KERNEL_REPOST_FAKE_CLOSE;
	WQ_INIT(&KERNEL$LOGICAL_WAIT, "KERENL$LOGICAL_WAIT");
}

