#include <SPAD/LIBC.H>
#include <STRING.H>
#include <STDLIB.H>
#include <SIGNAL.H>
#include <KERNEL/SYSCALL.H>
#include <UNISTD.H>
#include <SPAD/TIMER.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>
#include <LIB/KERNEL/UIO.H>
#include <NETINET/DHCP.H>

#include <SYS/UTSNAME.H>
#include <LIB/KERNEL/ULIBC.H>

#define _smsg	"PROCESS COMMITTED SUICIDE: "

int (* volatile KERNEL$SUICIDE_DUMP)(void) = NULL;
void * volatile KERNEL$SUICIDE_DUMP_PARAM = NULL;

void KERNEL$SUICIDE(const char *fmt, ...)
{
	char exitmsg[__MAX_STR_LEN];
	va_list args;
	va_start(args, fmt);
	strcpy(exitmsg, _smsg);
	_vsnprintf(exitmsg + strlen(_smsg), __MAX_STR_LEN - strlen(_smsg), fmt, args);
	va_end(args);
	__critical_printf("%s\n", exitmsg);
	KERNEL$STACK_DUMP();
	_exit_msg(__EXIT_SIGNAL(SIGSEGV), exitmsg);
}

void _exit_msg(int status, const char *msg)
{
	UIO_CLEANUP(status);
	SYSCALL3(SYS_EXIT, status, (unsigned long)msg);
	while (1) SYSCALL3(SYS_EXIT, __EXIT_SIGNAL(SIGABRT), (unsigned long)"TRIED TO RESUME AFTER EXIT");
}

void _background(char *msg)
{
	SYSCALL3(SYS_EXIT, -EBACKGROUND, (unsigned long)msg);
}

void _stop(char *msg)
{
	SYSCALL3(SYS_EXIT, -ESTOPPED, (unsigned long)msg);
}

#define CP_BUFFER_SIZE	4096

static char cp_buffer[CP_BUFFER_SIZE];
static int cp_buffer_size = 0;

static SIORQ cp_sio = { NULL };

static void flush_cp_buffer(void);

static DECL_AST(flush_cp_ast, SPL_TOP, SIORQ)
{
	cp_sio.fn = NULL;
	if (__unlikely(cp_sio.status <= 0) || __likely(cp_sio.status == cp_buffer_size)) cp_buffer_size = 0;
	else {
		memmove(cp_buffer, cp_buffer + cp_sio.status, cp_buffer_size - cp_sio.status);
		cp_buffer_size -= cp_sio.status;
		flush_cp_buffer();
	}
	RETURN;
}

static void flush_cp_buffer(void)
{
	if (__unlikely(cp_sio.fn != NULL) || __unlikely(!cp_buffer_size)) return;
	cp_sio.fn = flush_cp_ast;
	cp_sio.h = 2;
	cp_sio.v.ptr = (unsigned long)cp_buffer;
	cp_sio.v.len = cp_buffer_size;
	cp_sio.v.vspace = &KERNEL$VIRTUAL;
	cp_sio.progress = 0;
	CALL_IORQ(&cp_sio, KERNEL$WRITE);
}

static void cp_call(void *x, const char *data, int len)
{
	again:
	if (__unlikely(cp_buffer_size == CP_BUFFER_SIZE)) {
		/*
		if (len > CP_BUFFER_SIZE) data += len - CP_BUFFER_SIZE, len = CP_BUFFER_SIZE;
		memmove(cp_buffer, cp_buffer + len, (CP_BUFFER_SIZE - len));
		memcpy(cp_buffer + (CP_BUFFER_SIZE - len), data, len);
		*/
	} else if (cp_buffer_size + len > CP_BUFFER_SIZE) {
		int l = CP_BUFFER_SIZE - cp_buffer_size;
		cp_call(NULL, data, l);
		data += l;
		len -= l;
		goto again;
	} else {
		memcpy(cp_buffer + cp_buffer_size, data, len);
		cp_buffer_size += len;
	}
	if (__unlikely(cp_buffer_size >= CP_BUFFER_SIZE / 2)) flush_cp_buffer();
}

int __critical_printf(const char *fmt, ...)
{
	int spl;
	int i;
	va_list args;
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	va_start(args, fmt);
	i = __vxprintf(cp_call, NULL, fmt, args);
	va_end(args);
	flush_cp_buffer();
	LOWER_SPLX(spl);
	return i;
}

int __critical_vprintf(const char *fmt, va_list args)
{
	return __vxprintf(cp_call, NULL, fmt, args);
}

void ULIBC_WAIT_FOR_CP(void)
{
	if (__unlikely(cp_sio.fn != NULL)) KERNEL$SLEEP(1);
}

void ULIBC_FORK(void)
{
	cp_buffer_size = 0;
	cp_sio.fn = NULL;
}

static int syscalls_handle = -1;

static int open_syscalls(void)
{
	if (__likely(syscalls_handle != -1)) return 0;
	syscalls_handle = open("SYS$SYSCALLS:/", O_RDONLY | _O_NOPOSIX);
	if (__unlikely(syscalls_handle == -1)) {
		if (errno == ENOLNM) errno = ENOSYS;
		return -1;
	}
	fcntl(syscalls_handle, F_SETFD, FD_CLOEXEC);
	return 0;
}

int uname(struct utsname *buf)
{
	char *e;
	int save;
	IOCTLRQ i;
	if (__unlikely(open_syscalls())) return -1;
	i.h = syscalls_handle;
	i.ioctl = IOCTL_SYSCALLS_UNAME;
	i.param = 0;
	i.v.ptr = (unsigned long)buf;
	i.v.len = sizeof(struct utsname);
	i.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&i, KERNEL$IOCTL);
	if (__unlikely(i.status < 0)) {
		errno = -i.status;
		return -1;
	}
	save = errno;
	if ((e = getenv("HOSTNAME"))) {
		strlcpy(buf->nodename, e, sizeof buf->nodename);
	} else if (!buf->nodename[0]) {
		int n;
		n = dhcp_option(PF_INET, BOOTP_HOST_NAME, buf->nodename, sizeof buf->nodename - 1);
		if (n >= 0) buf->nodename[n] = 0;
	}
	if (buf->nodename[0]) {
		if (__unlikely((e = getenv("DOMAINNAME")) != NULL)) {
			strlcpy(buf->domainname, e, sizeof buf->domainname);
		} else if (__likely(!buf->domainname[0])) {
			int n;
			n = dhcp_option(PF_INET, BOOTP_DOMAIN_NAME, buf->domainname, sizeof buf->domainname - 1);
			if (n >= 0) buf->domainname[n] = 0;
		}
	} else {
		strlcpy(buf->nodename, "SPAD", sizeof buf->nodename);
	}
	errno = save;
	return 0;
}

int KERNEL$QUERY_MEMORY(unsigned long *mem)
{
	IOCTLRQ i;
	if (__unlikely(open_syscalls())) return -errno;
	i.h = syscalls_handle;
	i.ioctl = IOCTL_SYSCALLS_QUERY_MEMORY;
	i.param = 0;
	i.v.ptr = (unsigned long)mem;
	i.v.len = KQM_N_ENTRIES * sizeof(unsigned long);
	i.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&i, KERNEL$IOCTL);
	return i.status;
}

