#include <ARCH/BARRIER.H>
#include <SYS/TYPES.H>
#include <LIB/KERNEL/UASM.H>
#include <KERNEL/SYSCALL.H>
#include <STRING.H>
#include <SIGNAL.H>
#include <LIB/KERNEL/USIGNAL.H>

#include <KERNEL/TIME.H>
#include <SPAD/TIMER.H>
#include <KERNEL/TIMER.H>
#include <SYS/TIME.H>
#include <SPAD/SCHED.H>

void KERNEL$SET_TIMER(jiffies_t time, TIMER *t)
{
	TIMER *l, *lt;
	int spl;
#if __DEBUG >= 1
	if (__unlikely(t->list.next != __TIMER_MAGIC)) KERNEL$SUICIDE("KERNEL$SET_TIMER: ADDING ACTIVE TIMER %p:%"__64_format"X (%p)", t, (__u64)time, t->list.next);
	if (__unlikely(!t->fn)) KERNEL$SUICIDE("KERNEL$SET_TIMER: ADDING TIMER %p:%"__64_format"X WITH NULL FUNCTION", t, (__u64)time);
#endif
	if (__unlikely(time < 0)) time = 0;

	__TIMER_SET_JIFFIES(t, time + KERNEL$GET_JIFFIES());
	/*__debug_printf("set: %LX\n", time + KERNEL$GET_JIFFIES());*/
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TOP);
	lt = GET_STRUCT(&udata.timer_list, TIMER, list);
	XLIST_FOR_EACH(l, &udata.timer_list, TIMER, list) if ((jiffies_t)(__TIMER_GET_JIFFIES(l) - __TIMER_GET_JIFFIES(t)) > 0) {
		ADD_TO_LIST_BEFORE(&l->list, &t->list);
		goto ok;
	} else {
		lt = l;
	}
	ADD_TO_LIST_AFTER(&lt->list, &t->list);
	ok:
	if (__likely(GET_STRUCT(udata.timer_list.next, TIMER, list) == t)) {
		/*__debug_printf("set k: %LX\n", time);*/
		SYSCALL3(SYS_SET_TIMER, time, time >> 31 >> 1);
	}
	LOWER_SPLX(spl);
}

void KERNEL$DEL_TIMER(TIMER *t)
{
	int spl = KERNEL$SPL;
#if __DEBUG >= 1
	if (__unlikely(t->list.next == __TIMER_MAGIC)) KERNEL$SUICIDE("KERNEL$DEL_TIMER: DELETING INACTIVE TIMER %p", t);
#endif
	RAISE_SPL(SPL_TOP);
	DEL_FROM_LIST(&t->list);
#if __DEBUG >= 1
	t->list.next = __TIMER_MAGIC;
#endif
	LOWER_SPLX(spl);
}

static DECL_AST(USER_TIMER, SPL_TIMER, IORQ)
{
	TIMER *t;
	jiffies_t j;
	/*__debug_printf("TIMER: %d\n", SYSCALL1(SYS_DEPTH));*/
	udata.timer_available = 1;
	__barrier();
	WQ_WAKE_ALL(&KERNEL$LOGICAL_WAIT);
	PROCESS_PENDING_SIGNALS();
	j = KERNEL$GET_JIFFIES();
	RAISE_SPL(SPL_TOP);
	while ((t = LIST_STRUCT(udata.timer_list.next, TIMER, list)) != LIST_STRUCT(&KERNEL$LIST_END, TIMER, list)) {
		jiffies_t jj;
		if ((jj = __TIMER_GET_JIFFIES(t) - j) > 0) {
			SYSCALL3(SYS_SET_TIMER, jj, jj >> 31 >> 1);
			RETURN;
		}
		/*__debug_printf("TIMER: call: %d\n", SYSCALL1(SYS_DEPTH));*/
		DEL_FROM_LIST(&t->list);
#if __DEBUG >= 1
		t->list.next = __TIMER_MAGIC;
#endif
		t->fn(t);
		LOWER_SPL(SPL_TIMER);
		RAISE_SPL(SPL_TOP);
	}
	RETURN;
}

void KTIMER_INIT(void)
{
	INIT_XLIST(&udata.timer_list);
	udata.timer_handler.tmp3 = KERNEL_AST_TMP3_NOREPOST;
	udata.timer_handler.fn = USER_TIMER;
	__barrier();
	udata.timer_available = 1;
	__barrier();
}

__COLD_ATTR__ sched_unsigned KERNEL$GET_SCHED_TICKS(void)
{
	return (sched_unsigned)KERNEL$GET_JIFFIES() << _FINE_TICKS_SHIFT_WITHOUT_TSC;
}

#define N_TIMERS	3

static void expire0(TIMER *t);
static void expire1(TIMER *t);
static void expire2(TIMER *t);

static struct itimerval itimers[N_TIMERS];

static struct timeval last[N_TIMERS];

static TIMER timers[N_TIMERS] = {
	{ &timers[0].list, &timers[0].list, expire0 },
	{ &timers[1].list, &timers[1].list, expire1 },
	{ &timers[2].list, &timers[2].list, expire2 },
};

int subtract_time(struct timeval *t1, struct timeval *t2);

static void start_timer(int which, int setlast)
{
	if (__likely(itimers[which].it_value.tv_sec != 0) || __likely(itimers[which].it_value.tv_usec != 0)) {
		u_jiffies_t j;
		if (setlast) gettimeofday(&last[which], NULL);
		KERNEL$DEL_TIMER(&timers[which]);
		TV_2_JIFFIES(&itimers[which].it_value, j);
		KERNEL$SET_TIMER(j, &timers[which]);
	}
}

static void commit_timer(int which)
{
	struct timeval t, t2;
	if (!itimers[which].it_value.tv_usec && __unlikely(!itimers[which].it_value.tv_sec)) return;
	if (__unlikely(gettimeofday(&t, NULL))) return;
	t2 = t;
	subtract_time(&t, &last[which]);
	last[which] = t2;
	if (subtract_time(&itimers[which].it_value, &t) <= 0) {
		itimers[which].it_value = itimers[which].it_interval;
		start_timer(which, 0);
		raise(which == ITIMER_REAL ? SIGALRM : which == ITIMER_VIRTUAL ? SIGVTALRM : SIGPROF);
	} else {
		start_timer(which, 0);
	}
}

static void expire0(TIMER *t)
{
	LOWER_SPL(SPL_TIMER);
	SET_TIMER_NEVER(t);
	commit_timer(0);
}

static void expire1(TIMER *t)
{
	LOWER_SPL(SPL_TIMER);
	SET_TIMER_NEVER(t);
	commit_timer(1);
}

static void expire2(TIMER *t)
{
	LOWER_SPL(SPL_TIMER);
	SET_TIMER_NEVER(t);
	commit_timer(2);
}

int getitimer(int which, struct itimerval *t)
{
	int spl;
	if (__unlikely((unsigned)which >= N_TIMERS)) {
		errno = EINVAL;
		return -1;
	}
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_TIMER), KERNEL$SPL))) KERNEL$SUICIDE("getitimer AT SPL %08X", KERNEL$SPL);
	spl = KERNEL$SPL;
	RAISE_SPL(SPL_TIMER);
	commit_timer(which);
	memcpy(t, &itimers[which], sizeof(struct itimerval));
	LOWER_SPLX(spl);
	return 0;
}

int setitimer(int which, const struct itimerval *value, struct itimerval *ovalue)
{
	int spl;
	if (__unlikely((unsigned)which >= N_TIMERS) || __unlikely(__tv_invalid(&value->it_interval)) || __unlikely(__tv_invalid(&value->it_value))) {
		errno = EINVAL;
		return -1;
	}
	spl = KERNEL$SPL;
	if (__unlikely(SPLX_BELOW(SPL_X(SPL_TIMER), spl))) KERNEL$SUICIDE("setitimer AT SPL %08X", KERNEL$SPL);
	RAISE_SPL(SPL_TIMER);
	commit_timer(which);
	if (ovalue) memcpy(ovalue, &itimers[which], sizeof(struct itimerval));
	memcpy(&itimers[which], value, sizeof(struct itimerval));
	start_timer(which, 1);
	LOWER_SPLX(spl);
	return 0;
}

__COLD_ATTR__ void KERNEL$UDELAY(unsigned u)
{
	struct timeval t, t2, t3, t4;
	if (__unlikely(!u)) return;
	t.tv_sec = u / 1000000;
	t.tv_usec = u % 1000000;
	if (__unlikely(gettimeofday(&t2, NULL))) return;
	repeat:
	if (__unlikely(gettimeofday(&t3, NULL))) return;
	t4 = t3;
	subtract_time(&t3, &t2);
	if (subtract_time(&t, &t3) <= 0) return;
	t2 = t4;
	goto repeat;
}

__COLD_ATTR__ void KERNEL$UDELAY_PREPARE(udelay_cookie_t *cookie)
{
	struct timeval t;
	gettimeofday(&t, NULL);
	*cookie = t.tv_usec;
}

__COLD_ATTR__ void KERNEL$UDELAY_WAIT(udelay_cookie_t *cookie, unsigned u)
{
	struct timeval t;
	gettimeofday(&t, NULL);
	t.tv_usec = (t.tv_usec + 1000000 - *cookie) % 1000000;
	if (u <= t.tv_usec)
		return;
	KERNEL$UDELAY(u - t.tv_usec);
}

int TIMER_DISABLE(void)
{
	return !__CMPXCHGC(&udata.timer_available, 1, 0);
}

void TIMER_ENABLE(int state)
{
	if (__likely(state)) CALL_AST(&udata.timer_handler);
}

void TIMER_FORK(void)
{
	int i;
	TIMER *t;
	while (__unlikely(!XLIST_EMPTY(&udata.timer_list))) {
		t = GET_STRUCT(udata.timer_list.next, TIMER, list);
		DEL_FROM_LIST(&t->list);
		VOID_LIST_ENTRY(&t->list);
	}
	for (i = 0; i < sizeof(itimers); i++) if (__unlikely(((char *)&itimers)[i])) {
		memset(&itimers, 0, sizeof(itimers));
		break;
	}
	for (i = 0; i < sizeof(last); i++) if (__unlikely(((char *)&last)[i])) {
		memset(&last, 0, sizeof(last));
		break;
	}
	KTIMER_INIT();
}
