#include <STDLIB.H>
#include <ERRNO.H>
#include <SPAD/LIBC.H>
#include <ARCH/BITOPS.H>
#include <ARCH/BARRIER.H>
#include <SPAD/TIMER.H>
#include <SPAD/SYNC.H>
#include <UNISTD.H>

#include "PTHREAD.H"
#include <PTHREAD.H>

int pthread_mutexattr_init(pthread_mutexattr_t *a)
{
	struct __pthread_mutexattr_s *as;
	if (__unlikely(!(as = malloc(sizeof(struct __pthread_mutexattr_s))))) return ENOMEM;
	a->__p = as;
	as->pshared = PTHREAD_PROCESS_PRIVATE;
	as->mtype = PTHREAD_MUTEX_DEFAULT;
	return 0;
}

int pthread_mutexattr_destroy(pthread_mutexattr_t *a)
{
	if (__unlikely(!a->__p)) KERNEL$SUICIDE("pthread_mutex_destroy: uninitialized attribute");
	free(a->__p);
	a->__p = NULL;
	return 0;
}

int pthread_mutexattr_getprioceiling(const pthread_mutexattr_t *a, int *p)
{
	return ENOSYS;
}

int pthread_mutexattr_setprioceiling(pthread_mutexattr_t *a, int p)
{
	return ENOSYS;
}

int pthread_mutexattr_getprotocol(const pthread_mutexattr_t *a, int *p)
{
	return ENOSYS;
}

int pthread_mutexattr_setprotocol(pthread_mutexattr_t *a, int p)
{
	return ENOSYS;
}

int pthread_mutexattr_getpshared(const pthread_mutexattr_t *a, int *s)
{
	*s = a->__p->pshared;
	return 0;
}

int pthread_mutexattr_setpshared(pthread_mutexattr_t *a, int s)
{
	if (__unlikely((unsigned)s > PTHREAD_PROCESS_SHARED)) return EINVAL;
	a->__p->pshared = s;
	return 0;
}

int pthread_mutexattr_gettype(const pthread_mutexattr_t *a, int *t)
{
	*t = a->__p->mtype;
	return 0;
}

int pthread_mutexattr_settype(pthread_mutexattr_t *a, int t)
{
	if (__unlikely((unsigned)t > PTHREAD_MUTEX_DEFAULT)) return EINVAL;
	a->__p->mtype = t;
	return 0;
}

int pthread_mutex_init(pthread_mutex_t *m, const pthread_mutexattr_t *a)
{
	m->__recurse_cnt = 0;
	m->__owner = NULL;
	if (__likely(!a) || __likely(a->__p->pshared == PTHREAD_PROCESS_PRIVATE)) {
		struct __pthread_mutex_s *as;
		if (__unlikely(!(as = malloc(sizeof(struct __pthread_mutex_s))))) return ENOMEM;
		MTX_INIT(&as->mtx, "PTHREAD$MUTEX_MTX");
		m->__p = as;
		m->__mtype = __unlikely(a != NULL) ? a->__p->mtype : PTHREAD_MUTEX_NORMAL;
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_DEFAULT)) m->__mtype = PTHREAD_MUTEX_NORMAL;
		__barrier();
		m->__pshared = PTHREAD_PROCESS_PRIVATE;
	} else {
		m->__mtype = a->__p->mtype;
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_DEFAULT)) m->__mtype = PTHREAD_MUTEX_NORMAL;
		m->__shared_val = 0;
		__barrier();
		m->__pshared = PTHREAD_PROCESS_SHARED;
	}
	return 0;
}

static int mutex_init(pthread_mutex_t *m)
{
	if (__likely(!__CMPXCHGI(&m->__pshared, _PTHREAD_PROCESS_CREATE, _PTHREAD_PROCESS_CREATING))) {
		return pthread_mutex_init(m, NULL);
	}
	while (__unlikely(m->__pshared == _PTHREAD_PROCESS_CREATING)) pth_sleep();
	if (__likely(m->__pshared == PTHREAD_PROCESS_PRIVATE) || __likely(m->__pshared == PTHREAD_PROCESS_SHARED)) return 0;
	KERNEL$SUICIDE("mutex_init: uninitialized mutex");
}

int pthread_mutex_destroy(pthread_mutex_t *m)
{
	if (__likely(m->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(!m->__p)) goto uni;
		free(m->__p);
	} else if (__likely(m->__pshared == PTHREAD_PROCESS_SHARED) || __likely(m->__pshared == _PTHREAD_PROCESS_CREATE)) {
	} else uni: KERNEL$SUICIDE("pthread_mutex_destroy: uninitialized mutex");
	m->__pshared = _PTHREAD_PROCESS_DESTROYED;
	return 0;
}

int pthread_mutex_lock(pthread_mutex_t *m)
{
	again:
	if (__likely(m->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			/*__debug_printf("%Ld: %p lock %p\n", time(NULL), pthread_self(), m);*/
			MTX_LOCK_SYNC(&m->__p->mtx);
			/*__debug_printf("%Ld: %p locked %p\n", time(NULL), pthread_self(), m);*/
		} else {
			pthread_t s = pthread_self();
			if (__unlikely(m->__owner == s)) {
				if (__unlikely(m->__mtype == PTHREAD_MUTEX_ERRORCHECK))
					return EDEADLK;
				m->__recurse_cnt++;
			} else {
				MTX_LOCK_SYNC(&m->__p->mtx);
				m->__owner = s;
				m->__recurse_cnt = 1;
			}
		}
	} else if (__likely(m->__pshared == PTHREAD_PROCESS_SHARED)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			while (__unlikely(__CMPXCHGI((int *)&m->__shared_val, 0, 1) != 0)) pth_sleep();
		} else {
			pthread_t s = pthread_self();
			set_uniq_id(s);
			if (__unlikely(m->__owner == s) && __likely(!memcmp(m->__owner_id, s->id, __PTHREAD_ID_LEN))) {
				if (__unlikely(m->__mtype == PTHREAD_MUTEX_ERRORCHECK))
					return EDEADLK;
				m->__recurse_cnt++;
			} else {
				while (__unlikely(__CMPXCHGI((int *)&m->__shared_val, 0, 1) != 0)) pth_sleep();
				m->__owner = s;
				memcpy(m->__owner_id, s->id, __PTHREAD_ID_LEN);
				m->__recurse_cnt = 1;
			}
		}
	} else {
		int v;
		if (__unlikely(v = mutex_init(m))) return v;
		goto again;
	}
	return 0;
}

int pthread_mutex_trylock(pthread_mutex_t *m)
{
	again:
	if (__likely(m->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			if (__unlikely(MTX_TRY_LOCK(&m->__p->mtx))) return EBUSY;
		} else {
			pthread_t s = pthread_self();
			if (__unlikely(m->__owner == s)) {
				if (__unlikely(m->__mtype == PTHREAD_MUTEX_ERRORCHECK))
					return EDEADLK;
				m->__recurse_cnt++;
			} else {
				if (__unlikely(MTX_TRY_LOCK(&m->__p->mtx))) return EBUSY;
				m->__owner = s;
				m->__recurse_cnt = 1;
			}
		}
	} else if (__likely(m->__pshared == PTHREAD_PROCESS_SHARED)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			if (__unlikely(__CMPXCHGI((int *)&m->__shared_val, 0, 1) != 0)) return EBUSY;
		} else {
			pthread_t s = pthread_self();
			set_uniq_id(s);
			if (__unlikely(m->__owner == s) && __likely(!memcmp(m->__owner_id, s->id, __PTHREAD_ID_LEN))) {
				if (__unlikely(m->__mtype == PTHREAD_MUTEX_ERRORCHECK))
					return EDEADLK;
				m->__recurse_cnt++;
			} else {
				if (__unlikely(__CMPXCHGI((int *)&m->__shared_val, 0, 1) != 0)) return EBUSY;
				m->__owner = s;
				memcpy(m->__owner_id, s->id, __PTHREAD_ID_LEN);
				m->__recurse_cnt = 1;
			}
		}
	} else {
		int v;
		if (__unlikely(v = mutex_init(m))) return v;
		goto again;
	}
	return 0;
}

struct lock_struct {
	IORQ io;
	TIMER timer;
	MTX *mtx;
	int timed;
};

static DECL_IOCALL(m_wait, SPL_TIMER + 1, IORQ)
{
	struct lock_struct *w = GET_STRUCT(RQ, struct lock_struct, io);
	MTX_LOCK(w->mtx, RQ, m_wait, RETURN);
	RQ->status = 0;
	RETURN_AST(RQ);
}

static void m_timer(TIMER *t)
{
	struct lock_struct *w;
	LOWER_SPL(SPL_TIMER);
	w = GET_STRUCT(t, struct lock_struct, timer);
	w->timed = 1;
	KERNEL$CIO(&w->io);
}

static void m_clean(void *w_)
{
	struct lock_struct *w = w_;
	RAISE_SPL(SPL_TIMER);
	if (!w->timed) KERNEL$DEL_TIMER(&w->timer);
	LOWER_SPL(SPL_ZERO);
	/* we don't need to take care about w.io, because longjmp already killed that */
}

int pthread_mutex_timedlock(pthread_mutex_t *m, const struct timespec *t)
{
	if (__unlikely(__ts_invalid(t))) return EINVAL;
	again:
	if (__likely(m->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			struct lock_struct w;
			INIT_TIMER(&w.timer);
			w.timer.fn = m_timer;
			w.timed = 0;
			RAISE_SPL(SPL_TIMER);
			set_timer(&w.timer, t);
			w.mtx = &m->__p->mtx;
			pthread_cleanup_push(m_clean, &w);
			SYNC_IO(&w.io, m_wait);
			pthread_cleanup_pop(1);
			if (__unlikely(w.timed)) {
				return ETIMEDOUT;
			}
		} else {
			pthread_t s = pthread_self();
			if (__unlikely(m->__owner == s)) {
				if (__unlikely(m->__mtype == PTHREAD_MUTEX_ERRORCHECK))
					return EDEADLK;
				m->__recurse_cnt++;
			} else {
				MTX_LOCK_SYNC(&m->__p->mtx);
				m->__owner = s;
				m->__recurse_cnt = 1;
			}
		}
	} else if (__likely(m->__pshared == PTHREAD_PROCESS_SHARED)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			while (__unlikely(__CMPXCHGI((int *)&m->__shared_val, 0, 1) != 0)) {
				if (__unlikely(time_after(t))) return ETIMEDOUT;
				pth_sleep();
			}
		} else {
			pthread_t s = pthread_self();
			set_uniq_id(s);
			if (__unlikely(m->__owner == s) && __likely(!memcmp(m->__owner_id, s->id, __PTHREAD_ID_LEN))) {
				if (__unlikely(m->__mtype == PTHREAD_MUTEX_ERRORCHECK))
					return EDEADLK;
				m->__recurse_cnt++;
			} else {
				while (__unlikely(__CMPXCHGI((int *)&m->__shared_val, 0, 1) != 0)) {
					if (__unlikely(time_after(t))) return ETIMEDOUT;
					pth_sleep();
				}
				m->__owner = s;
				memcpy(m->__owner_id, s->id, __PTHREAD_ID_LEN);
				m->__recurse_cnt = 1;
			}
		}
	} else {
		int v;
		if (__unlikely(v = mutex_init(m))) return v;
		goto again;
	}
	return 0;
}

/* may be called from asynchronous context */
int pthread_mutex_unlock(pthread_mutex_t *m)
{
	if (__likely(m->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			/*__debug_printf("%Ld: %p unlock %p\n", time(NULL), pthread_self(), m);*/
			if (__unlikely(MTX_UNLOCK(&m->__p->mtx))) if (__likely(KERNEL$SPL == SPL_X(SPL_ZERO))) KERNEL$THREAD_YIELD();
			/*__debug_printf("%Ld: %p unlocked %p\n", time(NULL), pthread_self(), m);*/
		} else {
			pthread_t s = pthread_self();
			if (__unlikely(m->__owner != s))
				return EPERM;
			if (__likely(!--m->__recurse_cnt)) {
				m->__owner = NULL;
				if (__unlikely(MTX_UNLOCK(&m->__p->mtx))) if (__likely(KERNEL$SPL == SPL_X(SPL_ZERO))) KERNEL$THREAD_YIELD();
			}
		}
	} else if (__likely(m->__pshared == PTHREAD_PROCESS_SHARED)) {
		if (__unlikely(m->__mtype == PTHREAD_MUTEX_NORMAL)) {
			m->__shared_val = 0;
		} else {
			pthread_t s = pthread_self();
			set_uniq_id(s);
			if (__unlikely(m->__owner != s) || __unlikely(memcmp(m->__owner_id, s->id, __PTHREAD_ID_LEN)))
				return EPERM;
			if (__likely(!--m->__recurse_cnt)) {
				m->__owner = NULL;
				m->__shared_val = 0;
			}
		}
	} else {
		KERNEL$SUICIDE("pthread_mutex_unlock: unlocking non-locked mutex");
	}
	return 0;
}

int pthread_mutex_getprioceiling(const pthread_mutex_t *m, int *p)
{
	return ENOSYS;
}

int pthread_mutex_setprioceiling(pthread_mutex_t *m, int p, int *op)
{
	return ENOSYS;
}

