#include <STDLIB.H>
#include <ERRNO.H>
#include <SPAD/LIBC.H>
#include <ARCH/BITOPS.H>
#include <SPAD/TIMER.H>
#include <SPAD/AC.H>

#include "PTHREAD.H"
#include <PTHREAD.H>

int pthread_rwlockattr_init(pthread_rwlockattr_t *a)
{
	struct __pthread_rwlockattr_s *as;
	if (__unlikely(!(as = malloc(sizeof(struct __pthread_rwlockattr_s))))) return ENOMEM;
	a->__p = as;
	as->pshared = PTHREAD_PROCESS_PRIVATE;
	return 0;
}

int pthread_rwlockattr_destroy(pthread_rwlockattr_t *a)
{
	if (__unlikely(!a->__p)) KERNEL$SUICIDE("pthread_rwlockattr_destroy: uninitialized attribute");
	free(a->__p);
	a->__p = NULL;
	return 0;
}

int pthread_rwlockattr_getpshared(const pthread_rwlockattr_t *a, int *s)
{
	*s = a->__p->pshared;
	return 0;
}

int pthread_rwlockattr_setpshared(pthread_rwlockattr_t *a, int s)
{
	if (__unlikely((unsigned)s > PTHREAD_PROCESS_SHARED)) return EINVAL;
	a->__p->pshared = s;
	return 0;
}

int pthread_rwlock_init(pthread_rwlock_t *r, const pthread_rwlockattr_t *a)
{
	if (__likely(!a) || __unlikely(a->__p->pshared == PTHREAD_PROCESS_PRIVATE)) {
		struct __pthread_rwlock_s *as;
		if (__unlikely(!(as = malloc(sizeof(struct __pthread_rwlock_s))))) return ENOMEM;
		r->__p = as;
		WQ_INIT(&as->wq, "PTHREAD$RWLOCK_WQ");
		r->__pshared = PTHREAD_PROCESS_PRIVATE;
	} else {
		r->__pshared = PTHREAD_PROCESS_SHARED;
		r->__shmem_lock = 0;
	}
	r->__shared_val = 0;
	r->__wr_blocked = 0;
	return 0;
}

static int rwlock_init(pthread_rwlock_t *r)
{
	if (__likely(!__CMPXCHGI(&r->__pshared, _PTHREAD_PROCESS_CREATE, _PTHREAD_PROCESS_CREATING))) {
		return pthread_rwlock_init(r, NULL);
	}
	while (__unlikely(r->__pshared == _PTHREAD_PROCESS_CREATING)) pth_sleep();
	if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE) || __likely(r->__pshared == PTHREAD_PROCESS_SHARED)) return 0;
	KERNEL$SUICIDE("rwlock_init: uninitialized rwlock");
}

int pthread_rwlock_destroy(pthread_rwlock_t *r)
{
	if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(!r->__p)) goto uni;
		free(r->__p);
	} else if (__likely(r->__pshared == PTHREAD_PROCESS_SHARED)) {
	} else uni: KERNEL$SUICIDE("pthread_rwlock_destroy: uninitialized rwlock");
	r->__pshared = _PTHREAD_PROCESS_DESTROYED;
	return 0;
}

static __finline__ void shmem_lock(pthread_rwlock_t *r)
{
	while (__unlikely(__CMPXCHGI((int *)&r->__shmem_lock, 0, 1) != 0)) pth_sleep();
}

static __finline__ void shmem_unlock(pthread_rwlock_t *r)
{
	r->__shmem_lock = 0;
}

int pthread_rwlock_rdlock(pthread_rwlock_t *r)
{
	RAISE_SPL(SPL_DEV);
	again:
	if (__unlikely(r->__pshared != PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(r->__pshared != PTHREAD_PROCESS_SHARED)) {
			int v;
			if (__unlikely(v = rwlock_init(r))) {
				LOWER_SPL(SPL_ZERO);
				return v;
			}
			goto again;
		}
		shmem_lock(r);
	}
	while (__unlikely(r->__shared_val < 0) || (__unlikely(r->__wr_blocked != 0) && r->__shared_val != 0)) {
		if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
			WQ_WAIT_SYNC(&r->__p->wq);
		} else {
			shmem_unlock(r);
			pth_sleep();
			shmem_lock(r);
		}
	}
	r->__shared_val++;
	shmem_unlock(r);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

int pthread_rwlock_tryrdlock(pthread_rwlock_t *r)
{
	RAISE_SPL(SPL_DEV);
	again:
	if (__unlikely(r->__pshared != PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(r->__pshared != PTHREAD_PROCESS_SHARED)) {
			int v;
			if (__unlikely(v = rwlock_init(r))) {
				LOWER_SPL(SPL_ZERO);
				return v;
			}
			goto again;
		}
		shmem_lock(r);
	}
	if (__unlikely(r->__shared_val < 0) || (__unlikely(r->__wr_blocked != 0) && r->__shared_val != 0)) {
		shmem_unlock(r);
		LOWER_SPL(SPL_ZERO);
		return EBUSY;
	}
	r->__shared_val++;
	shmem_unlock(r);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

struct rw_struct {
	TIMER timer;
	int timed;
	WQ *wq;
};

static void rw_timer(TIMER *t)
{
	struct rw_struct *w;
	LOWER_SPL(SPL_TIMER);
	w = GET_STRUCT(t, struct rw_struct, timer);
	w->timed = 1;
	WQ_WAKE_ALL_PL(w->wq);
}

static void rw_clean(void *w_)
{
	struct rw_struct *w = w_;
	RAISE_SPL(SPL_TIMER);
	if (__likely(!w->timed)) KERNEL$DEL_TIMER(&w->timer);
	LOWER_SPL(SPL_ZERO);
}

int pthread_rwlock_timedrdlock(pthread_rwlock_t *r, const struct timespec *t)
{
	union {
		struct rw_struct w;
	} u;
	if (__unlikely(__ts_invalid(t))) return EINVAL;
	RAISE_SPL(SPL_TIMER);
	again:
	if (__unlikely(r->__pshared != PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(r->__pshared != PTHREAD_PROCESS_SHARED)) {
			int v;
			if (__unlikely(v = rwlock_init(r))) {
				LOWER_SPL(SPL_ZERO);
				return v;
			}
			goto again;
		}
		shmem_lock(r);
	} else {
		u.w.wq = &r->__p->wq;
		u.w.timed = 0;
		INIT_TIMER(&u.w.timer);
		u.w.timer.fn = rw_timer;
		set_timer(&u.w.timer, t);
		pthread_cleanup_push(rw_clean, &u.w);
	}
	while (__unlikely(r->__shared_val < 0) || (__unlikely(r->__wr_blocked != 0) && r->__shared_val != 0)) {
		if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
			WQ_WAIT_SYNC(&r->__p->wq);
			if (__unlikely(u.w.timed)) {
				pthread_cleanup_pop(0);
				LOWER_SPL(SPL_ZERO);
				return ETIMEDOUT;
			}
		} else {
			shmem_unlock(r);
			if (__unlikely(time_after(t))) {
				LOWER_SPL(SPL_ZERO);
				return ETIMEDOUT;
			}
			pth_sleep();
			shmem_lock(r);
		}
	}
	r->__shared_val++;
	if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		pthread_cleanup_pop(1);
	} else {
		shmem_unlock(r);
		LOWER_SPL(SPL_ZERO);
	}
	return 0;
}

int pthread_rwlock_wrlock(pthread_rwlock_t *r)
{
	RAISE_SPL(SPL_DEV);
	again:
	if (__unlikely(r->__pshared != PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(r->__pshared != PTHREAD_PROCESS_SHARED)) {
			int v;
			if (__unlikely(v = rwlock_init(r))) {
				LOWER_SPL(SPL_ZERO);
				return v;
			}
			goto again;
		}
		shmem_lock(r);
	}
	while (__unlikely(r->__shared_val != 0)) {
		r->__wr_blocked++;
		if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
			WQ_WAIT_SYNC(&r->__p->wq);
		} else {
			shmem_unlock(r);
			pth_sleep();
			shmem_lock(r);
		}
		r->__wr_blocked--;
	}
	r->__shared_val = -1;
	shmem_unlock(r);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

int pthread_rwlock_trywrlock(pthread_rwlock_t *r)
{
	RAISE_SPL(SPL_DEV);
	again:
	if (__unlikely(r->__pshared != PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(r->__pshared != PTHREAD_PROCESS_SHARED)) {
			int v;
			if (__unlikely(v = rwlock_init(r))) {
				LOWER_SPL(SPL_ZERO);
				return v;
			}
			goto again;
		}
		shmem_lock(r);
	}
	if (__unlikely(r->__shared_val != 0)) {
		shmem_unlock(r);
		LOWER_SPL(SPL_ZERO);
		return EBUSY;
	}
	r->__shared_val = -1;
	shmem_unlock(r);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

int pthread_rwlock_timedwrlock(pthread_rwlock_t *r, const struct timespec *t)
{
	union {
		struct rw_struct w;
	} u;
	if (__unlikely(__ts_invalid(t))) return EINVAL;
	RAISE_SPL(SPL_TIMER);
	again:
	if (__unlikely(r->__pshared != PTHREAD_PROCESS_PRIVATE)) {
		if (__unlikely(r->__pshared != PTHREAD_PROCESS_SHARED)) {
			int v;
			if (__unlikely(v = rwlock_init(r))) {
				LOWER_SPL(SPL_ZERO);
				return v;
			}
			goto again;
		}
		shmem_lock(r);
	} else {
		u.w.wq = &r->__p->wq;
		u.w.timed = 0;
		INIT_TIMER(&u.w.timer);
		u.w.timer.fn = rw_timer;
		set_timer(&u.w.timer, t);
		pthread_cleanup_push(rw_clean, &u.w);
	}
	while (__unlikely(r->__shared_val != 0)) {
		r->__wr_blocked++;
		if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
			WQ_WAIT_SYNC(&r->__p->wq);
			if (__unlikely(u.w.timed)) {
				r->__wr_blocked--;
				pthread_cleanup_pop(0);
				LOWER_SPL(SPL_ZERO);
				return ETIMEDOUT;
			}
		} else {
			if (__unlikely(time_after(t))) {
				r->__wr_blocked--;
				shmem_unlock(r);
				LOWER_SPL(SPL_ZERO);
				return ETIMEDOUT;
			}
			shmem_unlock(r);
			pth_sleep();
			shmem_lock(r);
		}
		r->__wr_blocked--;
	}
	r->__shared_val = -1;
	if (__likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		pthread_cleanup_pop(1);
	} else {
		shmem_unlock(r);
		LOWER_SPL(SPL_ZERO);
	}
	return 0;
}

int pthread_rwlock_unlock(pthread_rwlock_t *r)
{
	RAISE_SPL(SPL_DEV);
	if (__unlikely(r->__pshared != PTHREAD_PROCESS_PRIVATE)) shmem_lock(r);
	if (__unlikely(!r->__shared_val)) {
		shmem_unlock(r);
		LOWER_SPL(SPL_ZERO);
		return EPERM;
	}
	if (__unlikely(r->__shared_val < 0)) {
		r->__shared_val = 0;
	} else {
		r->__shared_val--;
	}
	if (__likely(!r->__shared_val) && __likely(r->__pshared == PTHREAD_PROCESS_PRIVATE)) {
		if (r->__wr_blocked) WQ_WAKE_ONE(&r->__p->wq);
		else WQ_WAKE_ALL(&r->__p->wq);
	}
	shmem_unlock(r);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

