#include <STDLIB.H>
#include <VALUES.H>
#include <SPAD/LIBC.H>
#include <SPAD/READDIR.H>
#include <DIRENT.H>
#include <SPAD/SYNC.H>
#include <SYS/IOCTL.H>

#include "SHELL.H"

#define DIRFLAG_STDERR	0x01
#define DIRFLAG_A	0x02
#define DIRFLAG_W	0x04
#define DIRFLAG_B	0x08

static int FIND_COLUMNS(unsigned width, char **names, unsigned long n, unsigned intercolumn_space, unsigned entry_space, unsigned **sizes)
{
	unsigned cols;
	unsigned long min;
	unsigned long rows;
	unsigned long i;
	unsigned c;
	unsigned long rc;
	unsigned long s;
	if (__unlikely(width >= 0x10000)) width = 0x10000;
	width += intercolumn_space;
	entry_space += intercolumn_space;
	min = MAXULONG;
	for (i = 0; i < n; i++) {
		unsigned long sl = strlen(names[i]) + entry_space;
		if (sl && sl < min) min = sl;
	}
	cols = width / min;
	*sizes = __sync_malloc((cols + 1) * sizeof(unsigned));
	if (__unlikely(!*sizes)) return -1;
	less_cols:
	if (__unlikely(cols <= 1)) {
		**sizes = 1;
		return 1;
	}
	memset(*sizes, 0, cols * sizeof(unsigned));
	rows = (n + cols - 1) / cols;
	c = 0, rc = 0;
	for (i = 0; i < n; i++) {
		unsigned long l = strlen(names[i]) + entry_space;
		if (__unlikely(l > (*sizes)[c])) (*sizes)[c] = l;
		if (__unlikely(++rc == rows)) {
			rc = 0;
			c++;
		}
	}
	s = 0;
	for (c = 0; c < cols; c++) s += (*sizes)[c];
	if (__likely(s > width)) {
		cols--;
		goto less_cols;
	}
	return cols;
}

static void DIR_LIST(CTX *ctx, char *prefix, char **names, unsigned long n)
{
	size_t plen;
	if (__likely(prefix[0] == '.') && __likely(!prefix[1])) plen = 0;
	else plen = strlen(prefix);
	if (__unlikely(ctx->itmp1 & DIRFLAG_B)) {
		dirflag_b:
		while (n--) _printf("%s\n", *names++);
	} else if (__unlikely(ctx->itmp1 & DIRFLAG_W)) {
		unsigned *sizes;
		int cols;
		struct winsize ws;
		unsigned long rows, i;
		if (__unlikely(ioctl(KERNEL$STDOUT(), TIOCGWINSZ, &ws))) goto dirflag_b;
		cols = FIND_COLUMNS(ws.ws_col, names, n, 2, 0, &sizes);
		if (__unlikely(cols < 0)) {
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "DIR: OUT OF MEMORY");
			ctx->return_val = -ENOMEM;
			return;
		}
		rows = (n + cols - 1) / cols;
		for (i = 0; i < rows; i++) {
			unsigned j, pos, wpos;
			unsigned long p;
			for (j = 0, p = i, pos = 0, wpos = 0; j < cols && p < n; j++, p += rows) {
				_printf("%*s%s", wpos - pos, "", names[p]);
				pos = wpos + strlen(names[p]);
				wpos += sizes[j];
			}
			_printf("\n");
		}
		free(sizes);
	} else {
		while (n--) {
			char *n = *names;
			char *str;
			struct stat st;
			if (__unlikely(plen != 0)) {
				char *p;
				n = __sync_malloc(strlen(n) + plen + 2);
				p = mempcpy(n, prefix, plen);
				if (p[-1] != '/' && p[-1] != ':') *p++ = '/';
				strcpy(p, *names);
			}
			if (__likely(!stat(n, &st))) {
				if (!S_ISREG(st.st_mode)) {
					if (S_ISDIR(st.st_mode)) str = "<DIR>";
					else if (S_ISBLK(st.st_mode)) str = "<BLK>";
					else if (S_ISCHR(st.st_mode)) str = "<CHR>";
					else if (S_ISFIFO(st.st_mode)) str = "<PIPE>";
					else if (S_ISLNK(st.st_mode)) str = "<LINK>";
					else if (S_ISSOCK(st.st_mode)) str = "<SOCKET>";
					else str = "<TYPE?>";
					goto ps;
				}
				_printf("%10"__off_t_format"d %s\n", st.st_size, *names);
			} else if (errno != EINTR) {
				str = "<ERROR>";
				ps:
				_printf("%10s %s\n", str, *names);
			}
			if (__unlikely(n != *names)) free(n);
			names++;
		}
	}
}

static void DIR_DIR(CTX *ctx, char *name)
{
	READDIR_RQ rd;
	char **m;
	unsigned long i, j;
	rd.path = name;
	rd.flags = 0;
	rd.cwd = KERNEL$CWD();
	SYNC_IO_CANCELABLE(&rd, KERNEL$READDIR);
	if (__unlikely(rd.status == -ENOTDIR)) {
		DIR_LIST(ctx, ".", &name, 1);
		return;
	}
	if (__unlikely(rd.status < 0)) {
		if (rd.status != -EINTR) {
			if (!(ctx->itmp1 & DIRFLAG_STDERR)) _snprintf(ctx->return_msg, __MAX_STR_LEN, "DIR: %s (%s)", strerror(-rd.status), name);
			else _eprintf("DIR: %s (%s)\n", strerror(-rd.status), name);
		}
		ctx->return_val = rd.status;
		return;
	}
	m = __sync_malloc(rd.n_entries * sizeof(char *));
	if (__unlikely(!m)) {
		_snprintf(ctx->return_msg, __MAX_STR_LEN, "DIR: OUT OF MEMORY");
		ctx->return_val = -ENOMEM;
		return;
	}
	for (i = 0, j = 0; i < rd.n_entries; i++) {
		if (ctx->itmp1 & DIRFLAG_A || rd.entries[i]->d_name[0] != '.')
			m[j++] = rd.entries[i]->d_name;
	}
	DIR_LIST(ctx, name, m, j);
	free(m);
	KERNEL$FREE_READDIR(&rd);
}

struct dir_type {
	char *name;
	int type;
};

#define DIR_TYPE_FILE	1
#define DIR_TYPE_DIR	2

static int DIR_TYPE_COMPARE(const void *dt1_, const void *dt2_)
{
	const struct dir_type *dt1 = dt1_;
	const struct dir_type *dt2 = dt2_;
	int d1, d2;
	if (__unlikely(dt1->type - dt2->type)) return dt1->type - dt2->type;
	d1 = dt1->name[0] != '.';
	d2 = dt2->name[0] != '.';
	if (__unlikely(d1 - d2)) return d1 - d2;
	return strcmp(dt1->name, dt2->name);
}

void SHELL_DIR_COMMAND(CTX *ctx)
{
	const char * const *argv = (const char * const *)ctx->argv;
	int state = 0;
	unsigned n = 0;
	ctx->itmp1 = 0;
	static const struct __param_table params[4] = {
		"A", __PARAM_BOOL, DIRFLAG_A, DIRFLAG_A,
		"W", __PARAM_BOOL, DIRFLAG_W | DIRFLAG_B, DIRFLAG_W,
		"B", __PARAM_BOOL, DIRFLAG_W | DIRFLAG_B, DIRFLAG_B,
		NULL, 0, 0, 0,
	};
	void *vars[4];
	vars[0] = &ctx->itmp1;
	vars[1] = &ctx->itmp1;
	vars[2] = &ctx->itmp1;
	vars[3] = NULL;
	while (__unlikely(__parse_params(&argv, &state, params, vars, NULL, NULL, (const char **)&ctx->argv[n]))) n++;
	if (__likely(!n)) {
		DIR_DIR(ctx, ".");
	} else if (n == 1) {
		DIR_DIR(ctx, ctx->argv[0]);
	} else {
		struct dir_type *dt;
		unsigned i, j;
		ctx->itmp1 |= DIRFLAG_STDERR;
		if (__unlikely(n > MAXUINT / sizeof(struct dir_type)) || __unlikely(!(dt = __sync_malloc(n * sizeof(struct dir_type))))) {
			_snprintf(ctx->return_msg, __MAX_STR_LEN, "DIR: OUT OF MEMORY");
			ctx->return_val = -ENOMEM;
			return;
		}
		for (i = 0; i < n; i++) {
			struct stat st;
			dt[i].name = ctx->argv[i];
			if (!stat(ctx->argv[i], &st)) {
				dt[i].type = S_ISDIR(st.st_mode) ? DIR_TYPE_DIR : DIR_TYPE_FILE;
			} else {
				dt[i].type = -errno;
			}
		}
		qsort(dt, n, sizeof(struct dir_type), DIR_TYPE_COMPARE);
		for (i = 0; i < n && dt[i].type < 0; i++) {
			_eprintf("DIR: %s (%s)\n", strerror(ctx->return_val = -dt[i].type), dt[i].name);
		}
		j = i;
		for (; i < n && dt[i].type == DIR_TYPE_FILE; i++) ctx->argv[i - j] = dt[i].name;
		DIR_LIST(ctx, ".", ctx->argv, i - j);
		for (; i < n; i++) {
			_printf("%s%s\n", __likely(i) ? "\n" : "", dt[i].name);
			DIR_DIR(ctx, dt[i].name);
		}
		free(dt);
	}
}

