#include <UNISTD.H>
#include <FCNTL.H>
#include <STRING.H>
#include <TERMIOS.H>
#include <SYS/KD.H>
#include <SYS/IOCTL.H>
#include <SYS/TYPES.H>

#include <VGAKEYBOARD.H>

static void default_handler(int scancode, int press);

static __keyboard_handler handler;
static int orig_stdin_flags, stdin_flags;
struct termios orig_stdin_termios, stdin_termios;
static char translate_mask;

static char raw_keyboard = 0;

static unsigned long internal_key_array[(128 + 8 * sizeof(unsigned long) - 1) / (8 * sizeof(unsigned long))];

int keyboard_init(void)
{
	if (__likely(!raw_keyboard)) {
		if (__unlikely(ioctl(0, KDSKBMODE, K_MEDIUMRAW) < 0)) return -1;
		stdin_flags = fcntl(0, F_GETFL);
		if (__unlikely(stdin_flags == -1)) {
			ioctl(0, KDSKBMODE, K_XLATE);
			return -1;
		}
		orig_stdin_flags = stdin_flags;
		if (__unlikely(ioctl(0, TIOCGETA, &stdin_termios) == -1)) {
			ioctl(0, KDSKBMODE, K_XLATE);
			return -1;
		}
		memcpy(&orig_stdin_termios, &stdin_termios, sizeof(struct termios));
		memset(internal_key_array, 0, sizeof internal_key_array);
		raw_keyboard = 1;
	}
	handler = default_handler;
	translate_mask = 0;
	keyboard_clearstate();
	return 0;
}

int keyboard_init_return_fd(void)
{
	if (__unlikely(keyboard_init())) return -1;
	return 0;
}

void keyboard_close(void)
{
	if (__unlikely(!raw_keyboard)) return;
	fcntl(0, F_SETFL, orig_stdin_flags);
	ioctl(0, KDSKBMODE, K_XLATE);
	ioctl(0, TIOCSETA, &orig_stdin_termios);
	raw_keyboard = 0;
}

void keyboard_seteventhandler(__keyboard_handler hndl)
{
	handler = hndl;
}

void keyboard_setdefaulteventhandler(void)
{
	handler = default_handler;
}

static int keyboard_doevent(void);

int keyboard_update(void)
{
	if (__unlikely(!raw_keyboard)) return 0;
	if (__unlikely(!(stdin_flags & O_NONBLOCK))) {
		if (__unlikely(fcntl(0, F_SETFL, stdin_flags | O_NONBLOCK) == -1)) return 0;
		stdin_flags |= O_NONBLOCK;
	}
	return keyboard_doevent();
}

void keyboard_waitforupdate(void)
{
	if (__unlikely(!raw_keyboard)) return;
	if (__unlikely(stdin_flags & O_NONBLOCK)) {
		if (__unlikely(fcntl(0, F_SETFL, stdin_flags & ~O_NONBLOCK) == -1)) return;
		stdin_flags &= ~O_NONBLOCK;
	}
	keyboard_doevent();
}

static __finline__ void internal_handler(int scancode, int press)
{
	if (__unlikely(__BTC(internal_key_array, scancode) == press)) {
		__BC(internal_key_array, scancode);
	}
	handler(scancode, press);
}

static int keyboard_doevent(void)
{
	unsigned char keys[32];
	int r, i;
	int ret = 0;
	again:
	if (__likely((r = read(0, keys, handler == default_handler ? 1 : sizeof keys)) <= 0)) return ret;
	for (i = 0; i < r; i++) {
		internal_handler(keys[i] & 0x7f, (keys[i] >> 7) ^ 1);
	}
	if (__unlikely(r == sizeof keys)) {
		if (!(stdin_flags & O_NONBLOCK)) {
			if (__unlikely(fcntl(0, F_SETFL, stdin_flags | O_NONBLOCK) == -1)) return 0;
			stdin_flags |= O_NONBLOCK;
		}
		ret = 1;
		goto again;
	}
	return 1;
}

static char key_array[128];

static void default_handler(int scancode, int press)
{
	if (__unlikely(translate_mask & TRANSLATE_CURSORKEYS)) {
		switch (scancode) {
			case SCANCODE_CURSORBLOCKUP:
				scancode = SCANCODE_CURSORUP;
				break;
			case SCANCODE_CURSORBLOCKLEFT:
				scancode = SCANCODE_CURSORLEFT;
				break;
			case SCANCODE_CURSORBLOCKRIGHT:
				scancode = SCANCODE_CURSORRIGHT;
				break;
			case SCANCODE_CURSORBLOCKDOWN:
				scancode = SCANCODE_CURSORDOWN;
		}
	}
	if (__unlikely(translate_mask & TRANSLATE_DIAGONAL)) {
		switch (scancode) {
			case SCANCODE_CURSORUPLEFT:
				key_array[SCANCODE_CURSORUP] = press;
				key_array[SCANCODE_CURSORLEFT] = press;
				break;
			case SCANCODE_CURSORUPRIGHT:
				key_array[SCANCODE_CURSORUP] = press;
				key_array[SCANCODE_CURSORRIGHT] = press;
				break;
			case SCANCODE_CURSORDOWNLEFT:
				key_array[SCANCODE_CURSORDOWN] = press;
				key_array[SCANCODE_CURSORLEFT] = press;
				break;
			case SCANCODE_CURSORDOWNRIGHT:
				key_array[SCANCODE_CURSORDOWN] = press;
				key_array[SCANCODE_CURSORRIGHT] = press;
				break;
		}
	}
	if (__unlikely(translate_mask & TRANSLATE_KEYPADENTER)) {
		if (__unlikely(scancode == SCANCODE_KEYPADENTER))
			scancode = SCANCODE_ENTER;
	}
	key_array[scancode] = press;
}

char *keyboard_getstate(void)
{
	return key_array;
}

void keyboard_clearstate(void)
{
	memset(key_array, 0, 128);
}

int keyboard_keypressed(int scancode)
{
	if (__unlikely(!raw_keyboard)) return 0;
	if (__unlikely((unsigned)scancode >= 128)) return 0;
	return key_array[scancode];
}

void keyboard_translatekeys(int mask)
{
	int reprogram_keys;
	if (__unlikely(!raw_keyboard)) return;
	reprogram_keys = (translate_mask ^ mask) & (TRANSLATE_CURSORKEYS | TRANSLATE_DIAGONAL | TRANSLATE_KEYPADENTER);
	translate_mask = mask;
	if (__likely(reprogram_keys) && __likely(handler == default_handler)) {
		int i;
		keyboard_clearstate();
		for (i = 0; i < 128; i++) if (__unlikely(__BT(internal_key_array, i))) default_handler(i, 1);
	}
	if (!(mask & DONT_CATCH_CTRLC) != !!(stdin_termios.c_lflag & ISIG)) {
		stdin_termios.c_lflag ^= ISIG;
		ioctl(0, TIOCSETA, &stdin_termios);
	}
}
