#include <STDLIB.H>
#include <UNISTD.H>
#include <STRING.H>
#include <SYS/TYPES.H>
#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/TTY.H>
#include <SPAD/IOCTL.H>

#include <VGA.H>
#include <VGAMOUSE.H>
#include "VGA.H"

static int have_gfx_mouse = 0;

static int mouse_support = 0;

static int mouse_x = 0, mouse_y = 0, mouse_z = 0, mouse_rx = 0, mouse_ry = 0, mouse_rz = 0;
static int mouse_button = 0;
static int scale = 1, wrap = 0;
static int minx = 0, maxx = 32767, minrx = 0, maxrx = 32768;
static int miny = 0, maxy = 32767, minry = 0, maxry = 32768;
static int minz = 0, maxz = 32767, minrz = 0, maxrz = 32768;

static struct mouse_info mouse_info = { 3, 1, 2 };
static struct mouse_state last_mouse_state = { 0 };

int mouse_handle = -1;

void vga_setmousesupport(int s)
{
	mouse_support = s;
}

int vga_getmousetype(void)
{
	return MOUSE_GPM;
}

int mouse_init_return_fd(char *dev, int type, int samplerate)
{
	IOCTLRQ io;
	char *c, *d;
	if (__unlikely(mouse_handle >= 0)) goto ret_m;
	c = KERNEL$HANDLE_PATH(0);
	if (__unlikely(!c)) return -1;
	d = alloca(strlen(c) + 25);
	strcpy(stpcpy(d, c), "/^MOUSE=RAW/^MOUSESELECT");
	mouse_handle = open(d, O_RDONLY);
	if (__unlikely(mouse_handle < 0)) return mouse_handle;
	ret_m:
	io.h = mouse_handle;
	io.ioctl = IOCTL_TTY_MOUSE_INFO;
	io.param = 0;
	io.v.ptr = (unsigned long)&mouse_info;
	io.v.len = sizeof(struct mouse_info);
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	return mouse_handle;
}

int mouse_init(char *dev, int type, int samplerate)
{
	int r = mouse_init_return_fd(dev, type, samplerate);
	if (__likely(r >= 0)) r = 0;
	return r;
}

void mouse_close(void)
{
	if (__likely(mouse_handle >= 0)) close(mouse_handle), mouse_handle = -1;
}

void mouse_setmode(int gfx, int x, int y)
{
	if (!gfx) {
		IOCTLRQ io;
		io.h = 0;
		io.ioctl = IOCTL_TTY_GETMOUSEMODE;
		io.param = 0;
		io.v.ptr = 0;
		io.v.len = 0;
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&io, KERNEL$IOCTL);
		if (__likely(io.status == TTYF_MOUSE_RAW)) {
			CHHRQ ch;
			ch.h = 0;
			ch.option = "MOUSE";
			ch.value = "COPYPASTE";
			SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
		}
		have_gfx_mouse = 0;
	} else {
		IOCTLRQ io;
		CHHRQ ch;
		ch.h = 0;
		ch.option = "MOUSE";
		ch.value = "RAW";
		SYNC_IO(&ch, KERNEL$CHANGE_HANDLE);
		if (__likely(have_gfx_mouse = ch.status >= 0)) {
			if (mouse_support) {
				mouse_setxrange(0, x - 1);
				mouse_setyrange(0, y - 1);
				mouse_setwrap(MOUSE_NOWRAP);
				mouse_x = x >> 1;
				mouse_y = y >> 1;
			}
			mouse_init(NULL, 0, 0);
			io.h = 0;
			io.ioctl = IOCTL_TTY_MOUSE_EVENT;
			io.param = 0;
			io.v.ptr = (unsigned long)&last_mouse_state;
			io.v.len = sizeof(struct mouse_state);
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
		}
	}
}

typedef void (*_mouse_handler)(int button, int dx, int dy, int dz, int drx, int dry, int drz);
static void default_handler(int button, int dx, int dy, int dz, int drx, int dry, int drz);

static _mouse_handler mouse_handler = default_handler;

void mouse_seteventhandler(__mouse_handler handler)
{
	mouse_handler = handler;
}

void mouse_setdefaulteventhandler(void)
{
	mouse_seteventhandler(default_handler);
}

static void default_handler(int button, int dx, int dy, int dz, int drx, int dry, int drz)
{
	mouse_button = button;
	mouse_x += dx;
	mouse_y += dy;
	mouse_z += dz;
	while (__unlikely(mouse_x > maxx * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPX)) mouse_x -= (maxx - minx + 1) * scale;
		else mouse_x = maxx * scale;
	}
	while (__unlikely(mouse_x < minx * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPX)) mouse_x += (maxx - minx + 1) * scale;
		else mouse_x = minx * scale;
	}
	while (__unlikely(mouse_y > maxy * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPY)) mouse_y -= (maxy - miny + 1) * scale;
		else mouse_y = maxy * scale;
	}
	while (__unlikely(mouse_y < miny * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPY)) mouse_y += (maxy - miny + 1) * scale;
		else mouse_y = miny * scale;
	}
	while (__unlikely(mouse_z > maxz * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPZ)) mouse_z -= (maxz - minz + 1) * scale;
		else mouse_z = maxz * scale;
	}
	while (__unlikely(mouse_z < minz * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPZ)) mouse_z += (maxz - minz + 1) * scale;
		else mouse_z = minz * scale;
	}

	switch (wrap & MOUSE_ROT_COORDS) {
		case MOUSE_ROT_INFINITESIMAL:
			mouse_rx = drx;
			mouse_ry = dry;
			mouse_rz = drz;
			break;
		case MOUSE_ROT_RX_RY_RZ:
			mouse_rx += drx;
			mouse_ry += dry;
			mouse_rz += drz;
			break;
	}

	while (__unlikely(mouse_rx > maxrx * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPRX)) mouse_rx -= (maxrx - minrx + 1) * scale;
		else mouse_rx = maxrx * scale;
	}
	while (__unlikely(mouse_rx < minrx * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPRX)) mouse_rx += (maxrx - minrx + 1) * scale;
		else mouse_rx = minrx * scale;
	}
	while (__unlikely(mouse_ry > maxry * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPRY)) mouse_ry -= (maxry - minry + 1) * scale;
		else mouse_ry = maxry * scale;
	}
	while (__unlikely(mouse_ry < minry * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPRY)) mouse_ry += (maxry - minry + 1) * scale;
		else mouse_ry = minry * scale;
	}
	while (__unlikely(mouse_rz > maxrz * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPRY)) mouse_rz -= (maxrz - minrz + 1) * scale;
		else mouse_rz = maxrz * scale;
	}
	while (__unlikely(mouse_rz < minrz * scale)) {
		if (__unlikely(wrap & MOUSE_WRAPRY)) mouse_rz += (maxrz - minrz + 1) * scale;
		else mouse_rz = minrz * scale;
	}
}

void mouse_setposition(int x, int y)
{
	mouse_x = x * scale;
	mouse_y = y * scale;
}

void mouse_setposition_6d(int x, int y, int z, int rx, int ry, int rz, int dim_mask)
{
	if (dim_mask & MOUSE_XDIM) mouse_x = x * scale;
	if (dim_mask & MOUSE_YDIM) mouse_y = y * scale;
	if (dim_mask & MOUSE_ZDIM) mouse_z = z * scale;
	if (dim_mask & MOUSE_RXDIM) mouse_rx = rx * scale;
	if (dim_mask & MOUSE_RYDIM) mouse_ry = ry * scale;
	if (dim_mask & MOUSE_RZDIM) mouse_rz = rz * scale;
}

void mouse_setxrange(int x1, int x2)
{
	minx = x1;
	maxx = x2;
}

void mouse_setyrange(int y1, int y2)
{
	miny = y1;
	maxy = y2;
}

void mouse_setrange_6d(int x1, int x2, int y1, int y2, int z1, int z2, int rx1, int rx2, int ry1, int ry2, int rz1, int rz2, int dim_mask)
{
	if (dim_mask & MOUSE_XDIM) minx = x1, maxx = x2;
	if (dim_mask & MOUSE_YDIM) miny = y1, maxy = y2;
	if (dim_mask & MOUSE_ZDIM) minz = z1, maxz = z2;
	if (dim_mask & MOUSE_RXDIM) minrx = rx1, maxrx = rx2;
	if (dim_mask & MOUSE_RYDIM) minry = ry1, maxry = ry2;
	if (dim_mask & MOUSE_RZDIM) minrz = rz1, maxrz = rz2;
}

void mouse_setscale(int s)
{
	if (__unlikely(!s)) return;
	mouse_x = (mouse_x * s) / scale;
	mouse_y = (mouse_y * s) / scale;
	mouse_z = (mouse_z * s) / scale;
	mouse_rx = (mouse_rx * s) / scale;
	mouse_ry = (mouse_ry * s) / scale;
	mouse_rz = (mouse_rz * s) / scale;
	scale = s;
}

void mouse_setwrap(int w)
{
	wrap = w;
}

int mouse_getx(void)
{
	return mouse_x / scale;
}

int mouse_gety(void)
{
	return mouse_y / scale;
}

void mouse_getposition_6d(int *x, int *y, int *z, int *rx, int *ry, int *rz)
{
	if (x) *x = mouse_x / scale;
	if (y) *y = mouse_y / scale;
	if (z) *z = mouse_z / scale;
	if (rx) *rx = mouse_rx / scale;
	if (ry) *ry = mouse_ry / scale;
	if (rz) *rz = mouse_rz / scale;
}

int mouse_getbutton(void)
{
	return mouse_button;
}

int mouse_getcaps(struct MouseCaps *caps)
{
	caps->key = MOUSE_GOTCAPS;
	switch (mouse_info.buttons) {
		case 0: caps->buttons = 0; break;
		case 1: caps->buttons = MOUSE_LEFTBUTTON; break;
		case 2: caps->buttons = MOUSE_LEFTBUTTON | MOUSE_RIGHTBUTTON; break;
		case 3: caps->buttons = MOUSE_LEFTBUTTON | MOUSE_RIGHTBUTTON | MOUSE_MIDDLEBUTTON; break;
		case 4: caps->buttons = MOUSE_LEFTBUTTON | MOUSE_RIGHTBUTTON | MOUSE_MIDDLEBUTTON | MOUSE_FOURTHBUTTON; break;
		case 5: caps->buttons = MOUSE_LEFTBUTTON | MOUSE_RIGHTBUTTON | MOUSE_MIDDLEBUTTON | MOUSE_FOURTHBUTTON | MOUSE_FIFTHBUTTON; break;
		case 6: caps->buttons = MOUSE_LEFTBUTTON | MOUSE_RIGHTBUTTON | MOUSE_MIDDLEBUTTON | MOUSE_FOURTHBUTTON | MOUSE_FIFTHBUTTON | MOUSE_SIXTHBUTTON; break;
		default: caps->buttons = MOUSE_LEFTBUTTON | MOUSE_RIGHTBUTTON | MOUSE_MIDDLEBUTTON | MOUSE_FOURTHBUTTON | MOUSE_FIFTHBUTTON | MOUSE_SIXTHBUTTON | MOUSE_RESETBUTTON; break;
	}
	switch (mouse_info.axes) {
		case 0: caps->axes = 0; break;
		case 1: caps->axes = MOUSE_YDIM; break;
		case 2: caps->axes = MOUSE_XDIM | MOUSE_YDIM; break;
		default: caps->axes = MOUSE_XDIM | MOUSE_YDIM | MOUSE_ZDIM; break;
	}
	switch (mouse_info.wheels) {
		case 0: break;
		case 1: caps->axes |= MOUSE_RXDIM; break;
		case 2: caps->axes |= MOUSE_RXDIM | MOUSE_RYDIM; break;
		default: caps->axes |= MOUSE_RXDIM | MOUSE_RYDIM | MOUSE_RZDIM; break;
	}
	caps->info = 0;
	if (mouse_info.wheels > 0) caps->info |= MOUSE_INFO_WHEEL;
	caps->reserved0 = 0;
	caps->reserved1 = 0;
	return 0;
}

int ignore_1_mouse_event = 0;

int mouse_scaling = MOUSE_SCALED;

int mouse_update(void)
{
	struct mouse_state m;
	IOCTLRQ io;
	int n_events = 0;
	if (__unlikely(mouse_handle < 0)) return mouse_handle;
	if (__unlikely(!have_gfx_mouse)) return 0;
	next_event:
	io.h = 0;
	io.ioctl = IOCTL_TTY_MOUSE_EVENT;
	io.param = 0;
	io.v.ptr = (unsigned long)&m;
	io.v.len = sizeof(struct mouse_state);
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_SIGINTERRUPT(&io, KERNEL$IOCTL);
	if (__likely(io.status >= 0)) {
		unsigned b;
		int buttons;
		if (!memcmp(&last_mouse_state, &m, sizeof(struct mouse_state))) goto no_event;
		if (__unlikely(ignore_1_mouse_event)) {
			ignore_1_mouse_event = 0;
			last_mouse_state.x = m.x;
			last_mouse_state.y = m.y;
			last_mouse_state.z = m.z;
			last_mouse_state.rx = m.rx;
			last_mouse_state.ry = m.ry;
			last_mouse_state.rz = m.rz;
			last_mouse_state.wx = m.wx;
			last_mouse_state.wy = m.wy;
			last_mouse_state.wz = m.wz;
			last_mouse_state.buttons = 0;
		}
		n_events++;
		buttons = 0;
		if (__unlikely(m.buttons & 0x01)) buttons |= MOUSE_LEFTBUTTON;
		if (__unlikely(m.buttons & 0x02)) buttons |= MOUSE_RIGHTBUTTON;
		if (__unlikely(m.buttons & 0x04)) buttons |= MOUSE_MIDDLEBUTTON;
		if (__unlikely(m.buttons & 0x08)) buttons |= MOUSE_FOURTHBUTTON;
		if (__unlikely(m.buttons & 0x10)) buttons |= MOUSE_FIFTHBUTTON;
		if (__unlikely(m.buttons & 0x20)) buttons |= MOUSE_SIXTHBUTTON;
		if (__unlikely(m.buttons & 0x40)) buttons |= MOUSE_RESETBUTTON;
		if (__likely(mouse_scaling == MOUSE_SCALED)) {
			mouse_handler(buttons, m.x - last_mouse_state.x, m.y - last_mouse_state.y, m.z - last_mouse_state.z, m.wy - last_mouse_state.wy, m.wx - last_mouse_state.wx, m.wz - last_mouse_state.wz);
		} else {
			mouse_handler(buttons, m.rx - last_mouse_state.rx, m.ry - last_mouse_state.ry, m.rz - last_mouse_state.rz, m.wy - last_mouse_state.wy, m.wx - last_mouse_state.wx, m.wz - last_mouse_state.wz);
		}
		b = last_mouse_state.buttons ^ m.buttons;
		memcpy(&last_mouse_state, &m, sizeof(struct mouse_state));
			/* click may be lost otherwise --- default_handler does not save clicks in any way. It is expected that user handler will do it */
		if (__unlikely(b) && mouse_handler == default_handler) goto done_events;
		no_event:
		io.h = mouse_handle;
		io.ioctl = IOCTL_SELECT_READ;
		io.param = 0;
		io.v.ptr = 0;
		io.v.len = 0;
		io.v.vspace = &KERNEL$VIRTUAL;
		SYNC_IO(&io, KERNEL$IOCTL);
		if (__unlikely(!io.status)) goto next_event;
	}
	done_events:
	return n_events;
}

void mouse_waitforupdate(void)
{
	IOCTLRQ io;
	io.h = mouse_handle;
	io.ioctl = IOCTL_SELECT_READ;
	io.param = 1;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO_SIGINTERRUPT(&io, KERNEL$IOCTL);
	mouse_update();
}

int mouse_setscaling(int scaling)
{
	if (__unlikely((unsigned)scaling > MOUSE_SCALED)) return 1;
	mouse_scaling = scaling;
	return 0;
}


