#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/time.h>

#ifdef __SPAD__
#include "zlib.h"
#else
#include <zlib.h>
#endif

#ifdef __EMX__
#include <io.h>
#include <emx/syscalls.h>
#define strcasecmp stricmp
#define read __read
#define write __write
#define lseek __lseek
#define lstat __stat
#endif

#include "BIND_SYS.H"

#define MAGIC_1_VAL	0x12345678
#define MAGIC_1_POS	0x1a8
#define MAGIC_2_VAL	0x9abcdef0
#define MAGIC_2_POS	0x1ac

#define BUF_SIZE 4096	/* must be >= 4096 */
static char buffer[BUF_SIZE];
static char buffer_out[BUF_SIZE];
#define file	((BIND_HEADER *)(void *)buffer)

static __u32 chksum;
static __u32 partial_dword;
static unsigned char partial_count = 0;

static void reset_chksum(void)
{
	chksum = 0;
	partial_dword = 0;
	partial_count = 0;
}

static void set_le32(void *ptr, __u32 val)
{
	((char *)ptr)[0] = val & 0xff;
	((char *)ptr)[1] = (val >> 8) & 0xff;
	((char *)ptr)[2] = (val >> 16) & 0xff;
	((char *)ptr)[3] = (val >> 24) & 0xff;
}

static __u32 get_le32(void *ptr)
{
	return	(((unsigned char *)ptr)[0]) |
		(((unsigned char *)ptr)[1] << 8) |
		(((unsigned char *)ptr)[2] << 16) |
		(((unsigned char *)ptr)[3] << 24);
}

static int rread(int src, char *p, int l)
{
	int r;
	while (l > 0) {
		if ((r = read(src, p, l)) <= 0) {
			r ? perror("read") : fprintf(stderr, "could not read\n");
			close(src);
			return -1;
		}
		p += r, l -= r;
	}
	return 0;
}

static int rwrite(int dst, const char *p, int l)
{
	int w;
	int i;
	for (i = 0; i < l; i++) {
		partial_dword |= (unsigned char)p[i] << (partial_count++ * 8);
		if (partial_count == 4) {
			chksum += partial_dword;
			chksum += chksum < partial_dword;
			partial_dword = 0;
			partial_count = 0;
		}
	}
	while (l > 0) {
		if ((w = write(dst, p, l)) <= 0) {
			w ? perror("write") : fprintf(stderr, "could not write\n");
			close(dst);
			return -1;
		}
		p += w, l -= w;
	}
	return 0;
}

static off_t rlseek(int h, off_t off, int whence)
{
	off_t l;
	if ((l = lseek(h, off, whence)) == -1) {
		perror("lseek");
		close(h);
		return -1;
	}
	return l;
}

static int align(int dst, int al)
{
	off_t l;
	if ((l = rlseek(dst, 0, SEEK_CUR)) == -1) return -1;
	if (al > BUF_SIZE) {
		fprintf(stderr, "internal error: align %d\n", al);
		close(dst);
		return -1;
	}
	memset(buffer, 0, al);
	if (rwrite(dst, buffer, ((l + al - 1) & ~(al - 1)) - l)) return -1;
	return ((l + al - 1) & ~(al - 1));
}

static int copy_file(int dst, int src, int size, int method)
{
	int r;
	z_stream z;
	if (method == METHOD_ZLIB) {
		if (size == -1) {
			fprintf(stderr, "internal error, size not specified\n");
			goto close_2_ret;
		}
		z.next_in = NULL;
		z.avail_in = 0;
		z.zalloc = NULL;
		z.zfree = NULL;
		z.opaque = NULL;
		if ((r = deflateInit2(&z, 9, Z_DEFLATED, ZLIB_WINDOW_BITS, 9, Z_DEFAULT_STRATEGY)) != Z_OK) {
			fprintf(stderr, "zlib init error %d\n", r);
			close_2_ret:
			close(dst);
			close_src_ret:
			close(src);
			return -1;
		}
	}
	while ((r = read(src, buffer, BUF_SIZE))) {
		if (r <= 0) {
			perror("read");
			freez_close_2_ret:
			close(dst);
			freez_close_src_ret:
			if (method == METHOD_ZLIB) {
				deflateEnd(&z);
			}
			goto close_src_ret;
		}
		if (method == METHOD_ZLIB) {
			int d;
			z.next_in = (unsigned char *)buffer;
			z.avail_in = r;
			compress_again:
			z.next_out = (unsigned char *)buffer_out;
			z.avail_out = sizeof(buffer_out);
			d = deflate(&z, r == size ? Z_FINISH : Z_NO_FLUSH);
			if (d != Z_OK) {
				if (!(r == size && d == Z_STREAM_END)) {
					fprintf(stderr, "deflate error %d\n", d);
					goto freez_close_2_ret;
				}
			}
			if (rwrite(dst, buffer_out, (char *)z.next_out - buffer_out)) goto freez_close_src_ret;
			if (z.avail_in || (r == size && d != Z_STREAM_END)) goto compress_again;
		} else {
			if (rwrite(dst, buffer, r)) goto freez_close_src_ret;
		}
		if (size != -1) {
			size -= r;
			if (size < 0) {
				chg_size:
				fprintf(stderr, "file changed its size while copying\n");
				goto freez_close_2_ret;
			}
		}
	}
	if (size && size != -1) goto chg_size;
	if (method == METHOD_ZLIB) {
		int d = deflateEnd(&z);
		if (d != Z_OK) {
			fprintf(stderr, "zlib deinit error %d\n", d);
			goto close_2_ret;
		}
	}
	return 0;
}

static int is_ffile(int f)
{
	struct stat st;
	if (fstat(f, &st) == -1) return 0;
	return S_ISREG(st.st_mode);
}

static int is_file(char *name)
{
	struct stat st;
	if (lstat(name, &st) == -1) return 0;
	return S_ISREG(st.st_mode);
}

static void xunlink(char *name)
{
	if (is_file(name)) unlink(name);
}

static char *filename(char *path)
{
	char *fn = strrchr(path, '/');
	if (fn) return fn + 1;
	fn = strrchr(path, ':');
	if (fn) return fn + 1;
	return path;
}

static int filecmp(char *path, char *name)
{
	int r;
	char c = 0;	/* against warning */
	char *p1 = strchr(path, ' ');
	if (p1) c = *p1, *p1 = 0;
	r = strcasecmp(filename(path), filename(name));
	if (p1) *p1 = c;
	return r;
}

static int get_name(FILE *f, char *buffer, char *find)
{
	while (fgets(buffer, 256, f)) {
		char *p;
		if ((p = strchr(buffer, '\n'))) *p = 0;
		if ((p = strchr(buffer, '\r'))) *p = 0;
		if (!filecmp(buffer, find)) {
			rewind(f);
			return 0;
		}
	}
	return -1;
}

static int cmp(char **c1, char **c2)
{
	return filecmp(*c1, *c2);
}

int main(int argc, char *argv[])
{
	int method = METHOD_ZLIB;
	int o, f, fx;
	char **n;
	int l, i;
	off_t ldr;
	char *out_filename, *in_filename;
	char fn[256];
	FILE *fl, *tmpf;
	off_t size;
	while (argc > 3) {
		if (!strcmp(argv[1], "-u")) {
			method = METHOD_UNCOMPRESSED;
		} else {
			break;
		}
		argc--;
		argv++;
	}
	if (argc != 3) {
		fprintf(stderr, "Usage: BIND_SYS [-u] <file with list of files> <output file>\n");
		return 3;
	}
	in_filename = argv[1];
	out_filename = argv[2];
	if (!(fl = fopen(in_filename, "r"))) {
		perror("open");
		return 1;
	}
	if (get_name(fl, fn, "LOADER.BIN")) {
		fprintf(stderr, "%s doesn't contain LOADER.BIN\n", in_filename);
		fclose(fl);
		return 1;
	}
	if ((f = open(fn, O_RDONLY)) == -1) {
		perror("open LOADER.BIN");
		fclose(fl);
		return 1;
	}
/* open output filename later to catch error when user swaps arguments */
	if ((o = open(out_filename, O_CREAT | O_RDWR, 0664)) == -1) {
		perror("creat");
		close(f);
		fclose(fl);
		return 1;
	}
	if (copy_file(o, f, -1, METHOD_UNCOMPRESSED)) {
		xunlink(out_filename);
		fclose(fl);
		return 1;
	}
	close(f);
	if ((ldr = rlseek(o, 0, SEEK_CUR)) == -1) {
		xunlink(out_filename);
		fclose(fl);
		return 1;
	}
	reset_chksum();
	if (get_name(fl, fn, "FSD.BIN")) {
		fprintf(stderr, "%s doesn't contain FSD.BIN\n", in_filename);
		close(o);
		xunlink(out_filename);
		fclose(fl);
		return 1;
	}
	if ((f = open(fn, O_RDONLY)) == -1) {
		perror("open FSD.BIN");
		close(o);
		xunlink(out_filename);
		fclose(fl);
		return 1;
	}
	if (copy_file(o, f, -1, METHOD_UNCOMPRESSED)) {
		xunlink(out_filename);
		fclose(fl);
		return 1;
	}
	close(f);
	l = 0;
	n = NULL;	/* let the warning go away... */
	while (fgets(fn, 256, fl)) {
		char *p;
		if ((p = strchr(fn, '\n'))) *p = 0;
		if ((p = strchr(fn, '\r'))) *p = 0;
		if (!*fn || *fn == '#' || *fn == ';') continue;
		if (!filecmp(fn, "LOADER.BIN") ||
		    !filecmp(fn, "FSD.BIN")) continue;
		if (!l++) n = malloc(sizeof(char *));
		else n = realloc(n, l * sizeof(char *));
		if (!n || !(n[l - 1] = malloc(strlen(fn) + 1))) {
			fprintf(stderr, "out of memory\n");
			close(o);
			xunlink(out_filename);
			fclose(fl);
			return 1;
		}
		strcpy(n[l - 1], fn);
	}
	fclose(fl);
	qsort(n, l, sizeof(char *), (void *)cmp);
	tmpf = tmpfile();
	fprintf(tmpf, "LIB.:/FSD.BIN\n");
	fprintf(tmpf, "LIB.:/LOADER.BIN\n");
	for (i = 0; i < l + 1; i++) {
		int head_len;
		off_t s, file_start, file_end;
		int p;
		char *nnn, *nn;
		if (i == l) {
			/* Special case --- copy the list of files */
			fflush(tmpf);
			if (ferror(tmpf)) {
				fprintf(stderr, "temp file error\n");
				close(o);
				xunlink(out_filename);
				return 1;
			}
			nn = "SYSFILES.LST";
			f = dup(fileno(tmpf));
			if (f == -1) {
				perror("dup");
				close(o);
				xunlink(out_filename);
				return 1;
			}
			goto process_file;
		}
		nnn = n[i];
		nn = strchr(n[i], ' ');
		if (nn) {
			*nn = 0;
			nnn = nn + 1;
			nn = n[i];
			if (strchr(nn, '/') || strchr(nn, ':')) {
				fprintf(stderr, "filename %s contains path\n", nn);
				close(o);
				xunlink(out_filename);
				return 1;
			}
		} else {
			nn = filename(n[i]);
		}
		fprintf(tmpf, "%s%s\n", !strcasecmp(nn, "CONFIG.SYS") ? "MFS:/" : "LIB.:/", nn);
		if (strlen(nn) >= 256) {
			fprintf(stderr, "too long filename %s\n", nn);
			close(o);
			xunlink(out_filename);
			return 1;
		}
		if ((f = open(nnn, O_RDONLY)) == -1) {
			perror(nnn);
			close(o);
			xunlink(out_filename);
			return 1;
		}
		process_file:
		if ((p = align(o, BIND_ALIGN)) == -1) {
			xunlink(out_filename);
			return 1;
		}
		if ((s = rlseek(f, 0, SEEK_END)) == -1) {
			close(o);
			xunlink(out_filename);
			return 1;
		}
		if (p + s >= 16384 * 512 - 8192) {
			fprintf(stderr, "system is too big\n");
			close(o);
			close(f);
			xunlink(out_filename);
			return 1;
		}
		memset(buffer, 0, sizeof(BIND_HEADER) + 256);
		set_le32(&file->magic, BIND_MAGIC);
		file->method = method;
		file->namelen = strlen(nn);
		set_le32(&file->uncompressed_len, s);
		set_le32(&file->compressed_len, 0);
		strcpy(buffer + sizeof(BIND_HEADER), nn);
		head_len = (sizeof(BIND_HEADER) + file->namelen + (BIND_ALIGN - 1)) & ~(BIND_ALIGN - 1);
		if (rwrite(o, buffer, head_len)) {
			close(f);
			xunlink(out_filename);
			return 1;
		}
		if ((file_start = rlseek(o, 0, SEEK_CUR)) == -1) {
			close(f);
			xunlink(out_filename);
			return 1;
		}
		if ((rlseek(f, 0, SEEK_SET)) == -1) {
			close(o);
			xunlink(out_filename);
			return 1;
		}
		if (copy_file(o, f, s, method)) {
			xunlink(out_filename);
			return 1;
		}
		close(f);
		if ((file_end = rlseek(o, 0, SEEK_CUR)) == -1) {
			xunlink(out_filename);
			return 1;
		}
		if ((p = align(o, BIND_ALIGN)) == -1) {
			xunlink(out_filename);
			return 1;
		}
		if (rlseek(o, file_start - head_len + ((char *)&file->compressed_len - (char *)file), SEEK_SET) == -1) {
			xunlink(out_filename);
			return 1;
		}
		s = file_end - file_start;
		buffer[0] = s & 0xff;
		buffer[1] = (s >> 8) & 0xff;
		buffer[2] = (s >> 16) & 0xff;
		buffer[3] = (s >> 24) & 0xff;
		if (rwrite(o, buffer, 4)) {
			xunlink(out_filename);
			return 1;
		}
		if (rlseek(o, p, SEEK_SET) == -1) {
			xunlink(out_filename);
			return 1;
		}
	}
	memset(buffer, 0, sizeof(BIND_HEADER));
	set_le32(&file->magic, BIND_MAGIC);
	file->method = METHOD_END;
	if (rwrite(o, buffer, sizeof(BIND_HEADER))) {
		xunlink(out_filename);
		return 1;
	}
	if ((l = align(o, 4096)) == -1) {
		xunlink(out_filename);
		return 1;
	}
	size = l;
	l = (l - (ldr & ~511)) / 16;
		/* !!! FIXME: find out maximum loadable system size */
	/*
	if (l >= 65536) {
		fprintf(stderr, "system is too big (%d, max %d)\n", l * 16, 65536 * 16);
		close(o);
		xunlink(out_filename);
		return 2;
	}
	*/
	if (rlseek(o, ldr - 8, SEEK_SET) == -1) {
		xunlink(out_filename);
		return 1;
	}
	set_le32(buffer, size - ldr);
	set_le32(buffer + 4, chksum);
	if (rwrite(o, buffer, 8)) {
		xunlink(out_filename);
		return 1;
	}
	if (rlseek(o, 500, SEEK_SET) == -1) {
		xunlink(out_filename);
		return 1;
	}
	buffer[0] = l & 0xff;
	buffer[1] = (l >> 8) & 0xff;
	if (rwrite(o, buffer, 2)) {
		xunlink(out_filename);
		return 1;
	}
	if (l >= 65536) {
		if (rlseek(o, 0x206, SEEK_SET) == -1) {
			xunlink(out_filename);
			return 1;
		}
		if (rwrite(o, "\003", 1)) {
			xunlink(out_filename);
			return 1;
		}
	}
	l = (l + 31) / 32;
	if (rlseek(o, 0x200 + 38, SEEK_SET) == -1) {
		xunlink(out_filename);
		return 1;
	}
	buffer[0] = l & 0xff;
	buffer[1] = (l >> 8) & 0xff;
	if (rwrite(o, buffer, 2)) {
		xunlink(out_filename);
		return 1;
	}
#ifndef __SPAD__
	fx = open("/dev/urandom", O_RDONLY);
#else
	fx = open("SYS$RANDOM:/", O_RDONLY);
#endif
	if (fx == -1) {
		struct timeval tv;
		if (gettimeofday(&tv, NULL)) {
			perror("gettimeofday");
			close(o);
			xunlink(out_filename);
			return 1;
		}
		memcpy(buffer + 8, &tv.tv_sec, 4);
		memcpy(buffer + 12, &tv.tv_usec, 4);
	} else {
		if (rread(fx, buffer + 8, 8)) {
			close(o);
			xunlink(out_filename);
			return 1;
		}
		close(fx);
	}
	if (rlseek(o, MAGIC_1_POS, SEEK_SET) == -1) {
		xunlink(out_filename);
		return 1;
	}
	if (rread(o, buffer, 4)) {
		xunlink(out_filename);
		return 1;
	}
	if (get_le32(&buffer[0]) != MAGIC_1_VAL) {
		fprintf(stderr, "bad magic value at %d: %x\n", MAGIC_1_POS, (unsigned)get_le32(&buffer[0]));
		close(o);
		xunlink(out_filename);
		return 1;
	}
	if (rlseek(o, MAGIC_2_POS, SEEK_SET) == -1) {
		xunlink(out_filename);
		return 1;
	}
	if (rread(o, buffer + 4, 4)) {
		xunlink(out_filename);
		return 1;
	}
	if (*(unsigned *)&buffer[4] != MAGIC_2_VAL) {
		fprintf(stderr, "bad magic value at %d: %x\n", MAGIC_2_POS, *(unsigned *)&buffer[4]);
		close(o);
		xunlink(out_filename);
		return 1;
	}
	for (i = 0; i < 8; i++) buffer[i] ^= buffer[i + 8];
	if (rlseek(o, MAGIC_1_POS, SEEK_SET) == -1) {
		xunlink(out_filename);
		return 1;
	}
	if (rwrite(o, buffer, 4)) {
		xunlink(out_filename);
		return 1;
	}
	if (rlseek(o, MAGIC_2_POS, SEEK_SET) == -1) {
		xunlink(out_filename);
		return 1;
	}
	if (rwrite(o, buffer + 4, 4)) {
		xunlink(out_filename);
		return 1;
	}
	if (is_ffile(o)) {
		if (ftruncate(o, size)) {
			perror("ftruncate");
			close(o);
			xunlink(out_filename);
			return 1;
		}
	}
	if (fsync(o)) {
		perror("fsync");
		close(o);
		xunlink(out_filename);
		return 1;
	}
	if (close(o)) {
		perror("close");
		xunlink(out_filename);
		return 1;
	}
	return 0;
}
