#include <SPAD/DEV.H>
#include <SPAD/SYNC.H>
#include <SPAD/LIST.H>
#include <SPAD/HASH.H>
#include <SPAD/SYSLOG.H>
#include <STDLIB.H>

#include "DMAN.H"

XLIST_HEAD DEVICE_HASH[HASH_SIZE];

static DEVICE *GET_DEVICE(const char *name, char term, int create)
{
	DEVICE *d;
	int hash = 0;
	const char *e = name;
	quickcasehash(e, *e && *e != term, hash);
	hash &= HASH_SIZE - 1;
	XLIST_FOR_EACH(d, &DEVICE_HASH[hash], DEVICE, hash_list) {
		if (__likely(!__strcasexcmp(d->name, name, e))) return d;
	}
	if (__unlikely(!create)) return NULL;
	d = __sync_malloc(sizeof(DEVICE) + e - name);
	if (__unlikely(!d)) return NULL;
	memset(d, 0, sizeof(DEVICE));
	INIT_XLIST(&d->using_head);
	INIT_XLIST(&d->used_head);
	d->state = STATE_LOADED;
	d->flags = LNTE_NOT_LOADED;
	*(char *)mempcpy(d->name, name, e - name) = 0;
	__upcase(d->name);
	ADD_TO_XLIST(&DEVICE_HASH[hash], &d->hash_list);
	return d;
}

DEVICE *FIND_DEVICE(const char *name, const char **error)
{
	DEVICE *d;
	const char *e = strchr(name, ':');
	if ((e && e[1]) || strchr(name, '/')) {
		*error = "BAD DEVICE NAME";
		return NULL;
	}
	d = GET_DEVICE(name, ':', 0);
	if (!d) {
		*error = "DEVICE DOES NOT EXIST";
		return NULL;
	}
	if (d->flags == LNTE_PARENT) {
		*error = "DEVICE BELONGS TO PARENT PROCESS";
		return NULL;
	}
	if (d->flags == LNTE_NOT_LOADED) {
		*error = "DEVICE NOT LOADED";
		return NULL;
	}
	return d;
}

static void DROP_USING_DEPENDENCIES(DEVICE *d)
{
	while (__unlikely(!XLIST_EMPTY(&d->using_head))) {
		DEPENDENCY *dep = LIST_STRUCT(d->using_head.next, DEPENDENCY, using_list);
		DEL_FROM_LIST(&dep->used_list);
		DEL_FROM_LIST(&dep->using_list);
		free(dep);
	}
}

static void DROP_USED_DEPENDENCIES(DEVICE *d)
{
	while (__unlikely(!XLIST_EMPTY(&d->used_head))) {
		DEPENDENCY *dep = LIST_STRUCT(d->used_head.next, DEPENDENCY, used_list);
		DEL_FROM_LIST(&dep->used_list);
		DEL_FROM_LIST(&dep->using_list);
		free(dep);
	}
}

static void DROP_DEVICE(DEVICE *d)
{
	DROP_USING_DEPENDENCIES(d);
	DROP_USED_DEPENDENCIES(d);
	DEL_FROM_LIST(&d->hash_list);
	free(d);
}

static int LOAD_DEPENDENCIES(DEVICE *d)
{
	static char dep_name[__MAX_STR_LEN + 2];
	unsigned i;
	DROP_USING_DEPENDENCIES(d);
	for (i = 0; KERNEL$GET_DEPENDENCY(d->name, i, dep_name, sizeof dep_name) >= 0; i++) {
		DEPENDENCY *dep;
		DEVICE *dep_dev;
		char *p;
		if (dep_name[0] == ':') {
			if (!_strcasecmp(dep_name, ":DMAN_SKIP")) d->flags = LNTE_NOT_LOADED;
			if (!_strcasecmp(dep_name, ":DMAN_SKIP_LINKS")) d->skip_links = 1;
			continue;
		}
		p = strchr(dep_name, ':');
		if (__unlikely(!p)) {
			KERNEL$SYSLOG(__SYSLOG_SW_ERROR, "DMAN", "BAD DEPENDENCY FOR DEVICE %s: %s", d->name, dep_name);
			cont:
			continue;
		}
		*p = 0;
		XLIST_FOR_EACH(dep, &d->using_head, DEPENDENCY, using_list) {
			if (!_strcasecmp(dep_name, dep->used->name)) goto cont;
		}
		dep_dev = GET_DEVICE(dep_name, 0, 1);
		if (__unlikely(!dep_dev)) return -ENOMEM;
		dep = __sync_malloc(sizeof(DEPENDENCY));
		if (__unlikely(!dep)) return -ENOMEM;
		/*_printf("%s -> %s\n", d->name, dep_dev->name);*/
		dep->used = dep_dev;
		dep->using = d;
		ADD_TO_XLIST(&dep_dev->used_head, &dep->used_list);
		ADD_TO_XLIST(&d->using_head, &dep->using_list);
	}
	return 0;
}

static void PROCESS_SKIP_LINKS(void)
{
	DEVICE *d;
	DEPENDENCY *dep;
	int i;
	for (i = 0; i < HASH_SIZE; i++)	XLIST_FOR_EACH(d, &DEVICE_HASH[i], DEVICE, hash_list) {
		if (d->flags >= 0 && !(d->flags & LNTE_DEVICE)) {
			XLIST_FOR_EACH(dep, &d->using_head, DEPENDENCY, using_list) {
				if (dep->used->skip_links) {
					d->flags = LNTE_NOT_LOADED;
					break;
				}
			}
		}
	}
}

int BUILD_DEVICE_DATABASE(void)
{
	int r, i;
	union {
		LOGICAL_LIST_REQUEST llr;
	} u;
	for (i = 0; i < HASH_SIZE; i++)	INIT_XLIST(&DEVICE_HASH[i]);
	u.llr.prefix = "";
	SYNC_IO_CANCELABLE(&u.llr, KERNEL$LIST_LOGICALS);
	if (__unlikely(u.llr.status < 0)) return u.llr.status;

	for (i = 0; i < u.llr.n_entries; i++) {
		DEVICE *d = GET_DEVICE(u.llr.entries[i]->name, 0, 1);
		if (!d) {
			r = -ENOMEM;
			goto err;
		}
		d->flags = u.llr.entries[i]->flags;
		if ((r = LOAD_DEPENDENCIES(d))) goto err;
	}

	PROCESS_SKIP_LINKS();

	KERNEL$FREE_LOGICAL_LIST(&u.llr);
	return 0;

	err:
	KERNEL$FREE_LOGICAL_LIST(&u.llr);
	DROP_DEVICE_DATABASE();
	return -ENOMEM;
}

void DROP_DEVICE_DATABASE(void)
{
	int i;
	for (i = 0; i < HASH_SIZE; i++)	while (!XLIST_EMPTY(&DEVICE_HASH[i])) {
		DEVICE *d = LIST_STRUCT(DEVICE_HASH[i].next, DEVICE, hash_list);
		if (__unlikely(d->walk_flag))
			KERNEL$SUICIDE("DROP_DEVICE_DATABASE: WALK FLAG LEAKED FOR DEVICE %s: %d", d->name, d->walk_flag);
		DROP_DEVICE(d);
	}
}
