#ifndef __SSHD_OPS_H
#define __SSHD_OPS_H

#include <STRING.H>
#include <STDLIB.H>
#include <ENDIAN.H>
#include <SPAD/LIBC.H>
#include <OPENSSL/BN.H>

#include "SSHD.H"

#define MOD(x, d)	(__likely(!((d) & ((d) - 1))) ? (x) & ((d) - 1) : (x) % (d))
#define ROUNDUP(x, d)	(__likely(!((d) & ((d) - 1))) ? ((x) + (d) - 1) & ~((d) - 1) : ((x) + (d) - 1) - ((x) + (d) - 1) % (d))

/* ADD TO STRING */

static __finline__ void init_str(__u8 **str, unsigned *len)
{
	*str = NULL;
	*len = 0;
}

static __finline__ int add_bytes_to_str(__u8 **str, unsigned *len, const __u8 *data, unsigned datalen)
{
	if (__unlikely(!(*str = reallocf(*str, *len + datalen))))
		return -1;
	memcpy(*str + *len, data, datalen);
	*len += datalen;
	return 0;
}

static __finline__ int add_str_to_str(__u8 **str, unsigned *len, const char *ns)
{
	return add_bytes_to_str(str, len, (__u8 *)ns, strlen(ns));
}

static __finline__ int add_chr_to_str(__u8 **str, unsigned *len, __u8 chr)
{
	return add_bytes_to_str(str, len, (__u8 *)&chr, 1);
}

static __finline__ int add_int32_to_str(__u8 **str, unsigned *len, __u32 int32)
{
	int32 = __32CPU2BE(int32);
	return add_bytes_to_str(str, len, (__u8 *)&int32, sizeof int32);
}

static __finline__ int add_len_str_to_str(__u8 **str, unsigned *len, const __u8 *data, unsigned datalen)
{
	return add_int32_to_str(str, len, datalen) ||
	       add_bytes_to_str(str, len, data, datalen);
}

int add_bn_to_str(__u8 **s, unsigned *l, BIGNUM *bn);

/* ADD TO STRING AND SIGNAL ERROR ON CONNECTION */

static __finline__ int ssh_add_failure(struct ssh_connection *c)
{
	debug_fatal(c, "Can't realloc string");
	abort_ssh_connection(c);
	return -1;
}

static __finline__ int ssh_add_bytes_to_str(struct ssh_connection *c, __u8 **str, unsigned *len, const __u8 *data, unsigned datalen)
{
	if (__unlikely(add_bytes_to_str(str, len, data, datalen)))
		return ssh_add_failure(c);
	return 0;
}

static __finline__ int ssh_add_str_to_str(struct ssh_connection *c, __u8 **str, unsigned *len, const char *ns)
{
	if (__unlikely(add_str_to_str(str, len, ns)))
		return ssh_add_failure(c);
	return 0;
}

static __finline__ int ssh_add_chr_to_str(struct ssh_connection *c, __u8 **str, unsigned *len, __u8 chr)
{
	if (__unlikely(add_chr_to_str(str, len, chr)))
		return ssh_add_failure(c);
	return 0;
}
 
static __finline__ int ssh_add_int32_to_str(struct ssh_connection *c, __u8 **str, unsigned *len, __u32 int32)
{
	if (__unlikely(add_int32_to_str(str, len, int32)))
		return ssh_add_failure(c);
	return 0;
}

static __finline__ int ssh_add_len_str_to_str(struct ssh_connection *c, __u8 **str, unsigned *len, const __u8 *data, unsigned datalen)
{
	if (__unlikely(add_len_str_to_str(str, len, data, datalen)))
		return ssh_add_failure(c);
	return 0;
}

static __finline__ int ssh_add_bn_to_str(struct ssh_connection *c, __u8 **s, unsigned *l, BIGNUM *bn)
{
	if (__unlikely(add_bn_to_str(s, l, bn)))
		return ssh_add_failure(c);
	return 0;
}

/* ADD TO CONNECTION */

static __finline__ int ssh_add_bytes_to_connection(struct ssh_connection *c, const __u8 *data, unsigned datalen)
{
	return ssh_add_bytes_to_str(c, &c->out_stream, &c->out_stream_len, data, datalen);
}

static __finline__ int ssh_add_chr_to_connection(struct ssh_connection *c, __u8 chr)
{
	return ssh_add_chr_to_str(c, &c->out_stream, &c->out_stream_len, chr);
}

static __finline__ int ssh_add_int32_to_connection(struct ssh_connection *c, __u32 int32)
{
	return ssh_add_int32_to_str(c, &c->out_stream, &c->out_stream_len, int32);
}

static __finline__ int ssh_add_len_str_to_connection(struct ssh_connection *c, const __u8 *data, unsigned datalen)
{
	return ssh_add_len_str_to_str(c, &c->out_stream, &c->out_stream_len, data, datalen);
}

static __finline__ int ssh_add_bn_to_connection(struct ssh_connection *c, BIGNUM *bn)
{
	return ssh_add_bn_to_str(c, &c->out_stream, &c->out_stream_len,bn);
}

/* GET FROM STRING */

static __finline__ int get_chr_from_str(const __u8 *str, unsigned len, unsigned *ptr, __u8 *chr)
{
	if (__unlikely(*ptr >= len))
		return 1;
	*chr = str[(*ptr)++];
	return 0;
}

static __finline__ int get_int32_from_str(const __u8 *str, unsigned len, unsigned *ptr, __u32 *int32)
{
	if (__unlikely(*ptr + 4 > len))
		return 1;
	*int32 = (str[*ptr] << 24) |
		 (str[*ptr + 1] << 16) |
		 (str[*ptr + 2] << 8) |
		 (str[*ptr + 3]);
	*ptr += 4;
	return 0;
}

static __finline__ int get_len_str_from_str(const __u8 *str, unsigned len, unsigned *ptr, const __u8 **data, unsigned *datalen)
{
	__u32 dlen;
	if (__unlikely(get_int32_from_str(str, len, ptr, &dlen)))
		return 1;
	if (__unlikely(*ptr + dlen > len) || __unlikely(*ptr + dlen < dlen)) {
		*ptr -= 4;
		return 1;
	}
	*data = str + *ptr;
	*datalen = dlen;
	*ptr += dlen;
	return 0;
}

int get_bn_from_str(const __u8 *str, unsigned len, unsigned *ptr, BIGNUM *bn);

/* GET FROM CONNECTION */

static __finline__ int ssh_get_chr_from_connection(struct ssh_connection *c, unsigned *ptr, __u8 *byte)
{
	if (__unlikely(get_chr_from_str(c->in_stream + c->in_stream_start, c->in_stream_len, ptr, byte))) {
		ssh_get_next_packet(c);
		return 1;
	}
	return 0;
}

static __finline__ int ssh_get_int32_from_connection(struct ssh_connection *c, unsigned *ptr, __u32 *int32)
{
	if (__unlikely(get_int32_from_str(c->in_stream + c->in_stream_start, c->in_stream_len, ptr, int32))) {
		ssh_get_next_packet(c);
		return 1;
	}
	return 0;
}

static __finline__ int ssh_get_len_str_from_connection(struct ssh_connection *c, unsigned *ptr, const __u8 **start, unsigned *len)
{
	if (__unlikely(get_len_str_from_str(c->in_stream + c->in_stream_start, c->in_stream_len, ptr, start, len))) {
		ssh_get_next_packet(c);
		return 1;
	}
	return 0;
}

static __finline__ int ssh_get_bn_from_connection(struct ssh_connection *c, unsigned *ptr, BIGNUM *bn)
{
	int r = get_bn_from_str(c->in_stream + c->in_stream_start, c->in_stream_len, ptr, bn);
	if (__unlikely(r)) {
		if (__unlikely(r < 0)) {
			debug_fatal(c, "Error getting bignum");
			abort_ssh_connection(c);
			return 1;
		} else {
			ssh_get_next_packet(c);
			return 1;
		}
	}
	return 0;
}

static __finline__ void ssh_flush_stream(struct ssh_connection *c)
{
	c->in_stream_start += c->in_stream_len;
	c->in_stream_len = 0;
}

static __finline__ void ssh_got_data(struct ssh_connection *c, unsigned *ptr)
{
	if (__unlikely(*ptr > c->in_stream_len))
		KERNEL$SUICIDE("ssh_got_data: out of string: %u > %u", *ptr, c->in_stream_len);
	c->in_stream_start += *ptr;
	c->in_stream_len -= *ptr;
	*ptr = 0;
}


#endif
