/* !!! FIXME: improve fair policy for ip addresses */

#include <STDLIB.H>
#include <UNISTD.H>
#include <FCNTL.H>
#include <STRING.H>
#include <SYS/SOCKET.H>
#include <NETINET/IN.H>
#include <ARPA/INET.H>
#include <NETDB.H>
#include <SYS/UTSNAME.H>
#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <SPAD/DEV.H>
#include <SPAD/IOCTL.H>
#include <OPENSSL/BN.H>
#include <OPENSSL/DH.H>

#include "SSHD.H"
#include "STRUCT.H"

int port = -1;
char *default_user = "/USER/";
int ttyh;
int ssh_debug = 0;
char idstring[__MAX_STR_LEN];

DECL_LIST(in_sock_queue);
volatile int do_interrupt;

#define MSG_WARNING	1
#define MSG_ERROR	2
#define MSG_FATAL	3
#define MSG_INTERNAL	4

static void debug_msg(struct ssh_connection *c, int type, char *msg, va_list va)
{
	static char str[1024];
	if (__unlikely(ssh_debug) || __unlikely(type == MSG_INTERNAL)) {
		_snprintf(str, sizeof(str), "%s(%s:%d): ", __unlikely(type == MSG_INTERNAL) ? "Internal error" : __likely(type == MSG_FATAL) ? "Fatal error" : __unlikely(type == MSG_ERROR) ? "Error" : __likely(type == MSG_WARNING) ? "Warning" : "???", inet_ntoa(c->sin.sin_addr), ntohs(c->sin.sin_port));
		_vsnprintf(str + strlen(str), sizeof(str) - strlen(str), msg, va);
		__critical_printf("%s\n", str);
		va_end(va);
	}
}

void debug_warning(struct ssh_connection *c, char *msg, ...)
{
	va_list va;
	va_start(va, msg);
	debug_msg(c, MSG_WARNING, msg, va);
	va_end(va);
}

void debug_error(struct ssh_connection *c, char *msg, ...)
{
	va_list va;
	va_start(va, msg);
	debug_msg(c, MSG_ERROR, msg, va);
	va_end(va);
}

void debug_fatal(struct ssh_connection *c, char *msg, ...)
{
	va_list va;
	va_start(va, msg);
	debug_msg(c, MSG_FATAL, msg, va);
	va_end(va);
}

void debug_internal(struct ssh_connection *c, char *msg, ...)
{
	va_list va;
	va_start(va, msg);
	debug_msg(c, MSG_INTERNAL, msg, va);
	va_end(va);
}

static AST_STUB abort_closed;
static AST_STUB abort_tty;

void abort_ssh_connection(struct ssh_connection *c)
{
	int i;
	RAISE_SPL(SPL_BOTTOM);
	c->flags |= SSHD_NEED_ABORT;
	CHECKH("abort_ssh_connection");
#if __DEBUG >= 1
	if (__unlikely(c->outstanding < 0))
		KERNEL$SUICIDE("abort_ssh_connection: NEGATIVE outstanding %d", c->outstanding);
#endif
	if (c->outstanding) {
		KERNEL$CIO((IORQ *)(void *)&c->in_sock);
		KERNEL$CIO((IORQ *)(void *)&c->out_sock);
		KERNEL$CIO((IORQ *)(void *)&c->in_tty);
		KERNEL$CIO((IORQ *)(void *)&c->out_tty);
		LOWER_SPL(SPL_ZERO);
		return;
	}
	if (c->xflags & SSHD_ON_QUEUE) {
		DEL_FROM_LIST(&c->in_sock_list);
		c->xflags &= ~SSHD_ON_QUEUE;
	}
	LOWER_SPL(SPL_ZERO);
	for (i = 0; i < CHANNEL_HASH_SIZE; i++) while (__unlikely(!XLIST_EMPTY(&c->channel_hash[i]))) {
		struct ssh_channel *ch = LIST_STRUCT(c->channel_hash[i].next, struct ssh_channel, hash);
		free_channel(ch);
	}
		/* todo: destroy compress buffers */
	free(c->in_buffer);
	free(c->in_packet);
	free(c->in_stream);
	free(c->out_stream);
	free(c->out_buffer);

	free(c->session_id);
	free(c->client_version_string);
	free(c->client_kexinit_msg);
	free(c->server_kexinit_msg);

	free(c->iv_ctos);
	free(c->iv_stoc);
	free(c->key_ctos);
	free(c->key_stoc);
	free(c->mac_key_ctos);
	free(c->mac_key_stoc);

	if (c->dh) DH_free(c->dh);

	free(c->in_mac.key);
	free(c->out_mac.key);

	EVP_CIPHER_CTX_cleanup(&c->in_cipher.evp_ctx);
	EVP_CIPHER_CTX_cleanup(&c->out_cipher.evp_ctx);

	c->closerq.fn = abort_closed;
	c->closerq.h = c->in_sock.h;
	CALL_IORQ(&c->closerq, KERNEL$CLOSE);
}

static DECL_AST(abort_closed, SPL_BOTTOM, CLOSERQ)
{
	struct ssh_connection *c = GET_STRUCT(RQ, struct ssh_connection, closerq);
	if (!(c->flags & SSHD_HAS_TTY)) {
		free(c);
		RETURN;
	}
	c->out_tty.fn = abort_tty;
	c->out_tty.ioctl = IOCTL_TTYSRVR_CLOSETTY;
	c->out_tty.param = 0;
	c->out_tty.v.ptr = (unsigned long)c->tty_name;
	c->out_tty.v.len = TTYSTR_LEN;
	c->out_tty.v.vspace = &KERNEL$VIRTUAL;
	RETURN_IORQ(&c->out_tty, KERNEL$IOCTL);
}

static DECL_AST(abort_tty, SPL_BOTTOM, IOCTLRQ)
{
	struct ssh_connection *c = GET_STRUCT(RQ, struct ssh_connection, out_tty);
	free(c);
	RETURN;
}

static void new_connection(int ns, struct sockaddr_in *na, int nal)
{
	struct ssh_connection *c;
	int r;
	int i;
	if (__unlikely(fcntl(ns, F_SETFL, 0) == -1)) {
		close(ns);
		return;
	}
	if (__unlikely(nal != sizeof(struct sockaddr_in))) {
		close(ns);
		return;
	}
	if (__unlikely(!(c = calloc(1, sizeof(struct ssh_connection))))) {
		close(ns);
		return;
	}

	for (i = 0; i < CHANNEL_HASH_SIZE; i++) INIT_XLIST(&c->channel_hash[i]);

	memcpy(&c->sin, na, sizeof(struct sockaddr_in));

	c->flags = SSHD_SKIP_ID | SSHD_SEND_ID;

	c->in_sock.fn = ssh_in_sock;
	c->in_sock.h = ns;
	c->in_sock.v.vspace = &KERNEL$VIRTUAL;
	c->out_sock.fn = ssh_out_sock;
	c->out_sock.h = ns;
	c->out_sock.v.vspace = &KERNEL$VIRTUAL;
	c->in_tty.fn = ssh_in_tty;
	c->in_tty.h = ttyh;
	c->in_tty.v.vspace = &KERNEL$VIRTUAL;
	c->out_tty.fn = ssh_out_tty;
	c->out_tty.h = ttyh;
	c->out_tty.v.vspace = &KERNEL$VIRTUAL;

	EVP_CIPHER_CTX_init(&c->in_cipher.evp_ctx);
	EVP_CIPHER_CTX_init(&c->out_cipher.evp_ctx);

	c->in_packet_seq = 0;
	c->out_packet_seq = 0;

	if (__unlikely(r = setup_in_cipher(c, &cipher_none, NULL, (const __u8 *)"        "))) {
		abort_ssh_connection(c);
		return;
	}
	if (__unlikely(r = setup_out_cipher(c, &cipher_none, NULL, (const __u8 *)"        "))) {
		abort_ssh_connection(c);
		return;
	}
	if (__unlikely(r = setup_in_mac(c, &mac_none, NULL))) {
		abort_ssh_connection(c);
		return;
	}
	if (__unlikely(r = setup_out_mac(c, &mac_none, NULL))) {
		abort_ssh_connection(c);
		return;
	}
	if (__unlikely(r = setup_in_compress(c, &compress_none))) {
		abort_ssh_connection(c);
		return;
	}
	if (__unlikely(r = setup_out_compress(c, &compress_none))) {
		abort_ssh_connection(c);
		return;
	}

	c->in_afterkex_upcall = sshd_userauth;
	sshd_kexinit(c);
}

/* #include <KERNEL/UDATA.H> */

int main(int argc, const char * const argv[])
{
	const char * const *arg;
	int state;
	struct servent *serv;
	struct sockaddr_in la;
	int ls;
	struct utsname u;
	int r;
	IOCTLRQ io;
	static const struct __param_table params[4] = {
		"DEBUG", __PARAM_BOOL, ~0, 1,
		"PORT", __PARAM_INT, 0, IPPORT_MAX,
		"USER", __PARAM_STRING, 1, 256,
		NULL, 0, 0, 0,
	};
	static void * const vars[4] = {
		&ssh_debug,
		&port,
		&default_user,
		NULL,
	};
	if (__unlikely(((SSLeay() ^ OPENSSL_VERSION_NUMBER) & ~0xff0L) != 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "OPENSSL VERSION MISMATCH, BUILT AGAINS %lX, YOU HAVE %lX.", (unsigned long)OPENSSL_VERSION_NUMBER, (unsigned long)SSLeay());
		return -EINVAL;
	}
	/* *(int *)UPLACE(UDATA_STRUCT + OFF_UDATA_xcpt_available) = 0; */
	arg = argv;
	state = 0;
	if (__parse_params(&arg, &state, params, vars, NULL, NULL, NULL)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "SYNTAX ERROR");
		return -EBADSYN;
	}
	if (port == -1) {
		if (__unlikely(!(serv = getservbyname("ssh", "tcp")))) {
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT GET SSH SERVICE FROM ETC.:/SERVICES");
			return -ENETDOWN;
		}
		port = ntohs(serv->s_port);
	}
	if (__unlikely(r = sshd_load_server_keys())) return r;
	if ((ttyh = open("SYS$TTYSRVR:", O_RDWR)) == -1) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT OPEN SYS$TTYSRVR: %s", strerror(errno));
		return -errno;
	}
	if (__likely(!uname(&u)))
		_snprintf(idstring, __MAX_STR_LEN, "SSH-2.0-" SSHD_NAME " %s %s %s", u.sysname, u.release, u.machine);
	else
		_snprintf(idstring, __MAX_STR_LEN, "SSH-2.0-" SSHD_NAME " SPAD");
	if (__unlikely((ls = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT CREATE SOCKET: %s", strerror(errno));
		return -errno;
	}
	la.sin_family = AF_INET;
	la.sin_port = htons(port);
	la.sin_addr.s_addr = htonl(INADDR_ANY);
	if (__unlikely(bind(ls, (struct sockaddr *)(void *)&la, sizeof la))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT BIND SOCKET TO PORT %d: %s", port, strerror(errno));
		return -errno;
	}
	if (__unlikely(listen(ls, 100))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "COULD NOT LISTEN ON SOCKET: %s", strerror(errno));
		return -errno;
	}
	if (__unlikely(fcntl(ls, F_SETFL, O_NONBLOCK) == -1)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T SET SOCKET NON-BLOCKING");
		return -errno;
	}
	io.h = ttyh;
	io.ioctl = IOCTL_TTYSRVR_CLEAR;
	io.param = 0;
	io.v.ptr = 0;
	io.v.len = 0;
	io.v.vspace = &KERNEL$VIRTUAL;
	SYNC_IO(&io, KERNEL$IOCTL);
	if (__unlikely(io.status < 0)) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "CAN'T INIT SYS$TTYSRVR: %s", strerror(-io.status));
		return -io.status;
	}

	while (1) {
		struct sockaddr_in na;
		socklen_t nal = sizeof(na);
		int ns;
		ns = accept(ls, (struct sockaddr *)(void *)&na, &nal);
		if (__unlikely(ns == -1)) {
			if (__likely(errno == EWOULDBLOCK)) goto skip_accept;
			_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "ACCEPT ERROR: %s", strerror(errno));
			return -errno;
		} else {
			new_connection(ns, &na, nal);
		}
		skip_accept:
		RAISE_SPL(SPL_BOTTOM);
		if (__unlikely(LIST_EMPTY(&in_sock_queue))) {
			IOCTLRQ io;
			io.h = ls;
			io.ioctl = IOCTL_SELECT_READ;
			io.param = 1;
			io.v.ptr = 0;
			io.v.len = 0;
			io.v.vspace = &KERNEL$VIRTUAL;
			do_interrupt = 1;
			SYNC_IO_SIGINTERRUPT(&io, KERNEL$IOCTL);
			do_interrupt = 0;
			LOWER_SPL(SPL_ZERO);
		} else {
			struct ssh_connection *c = LIST_STRUCT(in_sock_queue.next, struct ssh_connection, in_sock_list);
#if __DEBUG >= 1
			if (__unlikely(!(c->xflags & SSHD_ON_QUEUE)))
				KERNEL$SUICIDE("SSHD: CONNECTION HAS NOT QUEUE FLAG");
#endif
			/*__debug_printf("getq: %x\n", c->xflags);*/
			if (__unlikely(c->xflags & SSHD_NEED_ABORT)) {
				c->xflags &= ~SSHD_ON_QUEUE;
				DEL_FROM_LIST(&c->in_sock_list);
				LOWER_SPL(SPL_ZERO);
				abort_ssh_connection(c);
			} else if (__unlikely(c->out_buffer_len >= MAX_STREAM_LENGTH)) {
				/*__debug_printf("throttling\n");*/
				/* throttle connection, do not allow external events to put more things into output queue */
				throttle:
				c->xflags &= ~SSHD_ON_QUEUE;
				DEL_FROM_LIST(&c->in_sock_list);
				LOWER_SPL(SPL_ZERO);
			} else if (__unlikely(c->xflags & SSHD_CALL_IN_UPCALL)) {
				c->xflags &= ~SSHD_CALL_IN_UPCALL;
				DEL_FROM_LIST(&c->in_sock_list);
				if (!(c->xflags & SSHD_CALL_MASK)) {
					c->xflags &= ~SSHD_ON_QUEUE;
				} else {
					ADD_TO_LIST_END(&in_sock_queue, &c->in_sock_list);
				}
				LOWER_SPL(SPL_ZERO);
				if (__unlikely(!c->in_upcall)) KERNEL$SUICIDE("SSHD: %s: IN_UPCALL ZERO", c->tty_name);
				c->in_upcall(c);
			} else if (__unlikely(c->in_afterkex_upcall != NULL)) {
				goto throttle;
			} else if (__unlikely(c->xflags & SSHD_CALL_OUT_TTY_UPCALL)) {
				c->xflags &= ~SSHD_CALL_OUT_TTY_UPCALL;
				DEL_FROM_LIST(&c->in_sock_list);
				if (!(c->xflags & SSHD_CALL_MASK)) {
					c->xflags &= ~SSHD_ON_QUEUE;
				} else {
					ADD_TO_LIST_END(&in_sock_queue, &c->in_sock_list);
				}
				LOWER_SPL(SPL_ZERO);
				if (__unlikely(!c->out_tty_upcall)) KERNEL$SUICIDE("SSHD: %s: OUT_UPCALL ZERO", c->tty_name);
				c->out_tty_upcall(c);
			} else if (__unlikely(c->xflags & SSHD_CALL_IN_TTY_UPCALL)) {
				c->xflags &= ~SSHD_CALL_IN_TTY_UPCALL;
				DEL_FROM_LIST(&c->in_sock_list);
				if (!(c->xflags & SSHD_CALL_MASK)) {
					c->xflags &= ~SSHD_ON_QUEUE;
				} else {
					ADD_TO_LIST_END(&in_sock_queue, &c->in_sock_list);
				}
				LOWER_SPL(SPL_ZERO);
				if (__unlikely(!c->in_tty_upcall)) KERNEL$SUICIDE("SSHD: %s: IN_TTY_UPCALL ZERO", c->tty_name);
				c->in_tty_upcall(c);
			} else {
				goto throttle;
			}
		}
	}
}
