#ifndef __SSHD_SSHD_H
#define __SSHD_SSHD_H

#include <SYS/TYPES.H>
#include <SPAD/DEV.H>
#include <SPAD/LIST.H>
#include <SPAD/IOCTL.H>
#include <NETINET/IN.H>
#include <OPENSSL/EVP.H>
#include <OPENSSL/BN.H>
#include <OPENSSL/DH.H>

#define SSHD_NAME		"SPAD_SSHD"

/*#define CHECKH(x)	KERNEL$HEAP_CHECK(x)*/
#define CHECKH(x)

struct ssh_connection;

struct kex_type {
	const char *name;
	void (*kex_fn)(struct ssh_connection *);
};

union p_key;

struct public_type {
	const char *name;
	union p_key *(*load_private_from_file)(const char *filename, char *err);
	union p_key *(*load_public_from_blob)(const __u8 *blob, unsigned len);
	int (*put_public_to_blob)(union p_key *key, __u8 **blob, unsigned *len);
	int (*sign)(union p_key *key, __u8 **signature, unsigned *signaturelen, const __u8 *data, unsigned datalen);
	int (*verify)(union p_key *key, const __u8 *signature, unsigned signaturelen, const __u8 *data, unsigned datalen);
	void (*destroy)(union p_key *key);
};

struct cipher_type {
	unsigned blocksize;
	unsigned keysize;
	const EVP_CIPHER *(*evp_cipher)(void);
	const char *name;
};

struct cipher {
	const struct cipher_type *type;
	EVP_CIPHER_CTX evp_ctx;
};

struct mac;

struct mac_type {
	unsigned size;
	unsigned keysize;
	void (*mac)(struct mac *mac, __u32 seq, __u8 *data, unsigned len, __u8 *md);
	const EVP_MD *(*mdfunc)(void);
	const char *name;
};

struct mac {
	const struct mac_type *type;
	const EVP_MD *md;
	__u8 *key;
};

struct compress;

struct compress_type {
	int (*inflate)(struct compress *ct, __u8 **cmp, unsigned *cmpl, __u8 **res, unsigned *resl);
	int (*deflate)(struct compress *ct, __u8 **data, unsigned *datal, __u8 **cmp, unsigned *cmpl);
	const char *name;
};

struct compress {
	const struct compress_type *type;
};

#define CHANNEL_HASH_SIZE	8

struct ssh_connection {
	LIST_ENTRY in_sock_list;
	int xflags;
	void (*in_upcall)(struct ssh_connection *);
	void (*in_afterkex_upcall)(struct ssh_connection *);
	void (*out_tty_upcall)(struct ssh_connection *);
	void (*in_tty_upcall)(struct ssh_connection *);

	__u8 *in_buffer;
	unsigned in_buffer_start;
	unsigned in_buffer_end;
	__u8 *in_packet;
	unsigned in_packet_len;
	__u8 *in_stream;
	unsigned in_stream_start;
	unsigned in_stream_len;

	__u8 *out_stream;
	unsigned out_stream_len;
	__u8 *out_buffer;
	unsigned out_buffer_len;
	__u8 *out_in_progress;

	struct cipher in_cipher;
	struct mac in_mac;
	__u32 in_packet_seq;
	struct compress in_compress;

	struct cipher out_cipher;
	struct mac out_mac;
	__u32 out_packet_seq;
	struct compress out_compress;

	int flags;

	SIORQ in_sock;
	SIORQ out_sock;
	IOCTLRQ in_tty;
	IOCTLRQ out_tty;
	int outstanding;

	XLIST_HEAD channel_hash[CHANNEL_HASH_SIZE];

	CLOSERQ closerq;
	struct sockaddr_in sin;

	char tty_name[TTYSTR_LEN];

	__u8 *session_id;
	unsigned session_id_len;

	char *client_version_string;
	char *client_prog_id;
	__u8 *client_kexinit_msg;
	unsigned client_kexinit_msg_len;
	__u8 *server_kexinit_msg;
	unsigned server_kexinit_msg_len;

	const struct kex_type *kex_type;
	const struct public_type *public_type;
	union p_key *server_key;
	const struct cipher_type *cipher_ctos;
	const struct cipher_type *cipher_stoc;
	const struct mac_type *mac_ctos;
	const struct mac_type *mac_stoc;
	const struct compress_type *compress_ctos;
	const struct compress_type *compress_stoc;

	__u8 *iv_ctos, *iv_stoc, *key_ctos, *key_stoc, *mac_key_ctos, *mac_key_stoc;

	DH *dh;

	int auth_count;
	__u32 chnum;

	struct ssh_channel *session_channel;
	char tty_type[33];
};

extern LIST_HEAD in_sock_queue;
extern volatile int do_interrupt;

#define SSHD_SKIP_ID			1
#define SSHD_SEND_ID			2
#define SSHD_HAS_TTY			4
#define SSHD_HAS_SESSION		8

#define SSHD_ON_QUEUE			1
#define SSHD_NEED_ABORT			2
#define SSHD_CALL_IN_UPCALL		4
#define SSHD_CALL_OUT_TTY_UPCALL	8
#define SSHD_CALL_IN_TTY_UPCALL		16

#define SSHD_CALL_MASK	(SSHD_CALL_IN_UPCALL | SSHD_CALL_OUT_TTY_UPCALL | SSHD_CALL_IN_TTY_UPCALL)

#define MIN_CHANNEL_IN_BUFFER		2048
#define MAX_CHANNEL_IN_BUFFER		(131072 - TTYSTR_LEN)
#define IN_WINDOW_ADJUST_DELAY		512
#define MAX_IN_WINDOW_ADJUST_DELAY	8192
#define CHANNEL_OUT_BUFFER		5120
#define OUT_WINDOW_DELAY_SHIFT		6

struct ssh_channel {
	LIST_ENTRY hash;
	struct ssh_connection *c;
	__u32 local_number;
	__u32 remote_number;
	unsigned in_buffer_start;
	unsigned in_buffer_end;
	unsigned in_buffer_size;
	unsigned in_window_adjust;
	unsigned out_buffer_start;
	unsigned out_buffer_end;
	unsigned out_window;
	unsigned out_maxwindow;
	unsigned out_maxpacket;
	int (*in_upcall)(struct ssh_channel *ch);
	int (*out_upcall)(struct ssh_channel *ch);
	int flags;
	__u8 ttyn2[TTYSTR_LEN];	/* get here via out_buffer - TTYSTR_LEN */
	__u8 out_buffer[CHANNEL_OUT_BUFFER];
	__u8 *in_buffer;
	__u8 ttyn1[TTYSTR_LEN];	/* get here via in_buffer - TTYSTR_LEN */
	__u8 in_buffer_default[MIN_CHANNEL_IN_BUFFER];
};

#define CH_SESSION		1
#define CH_HAS_SHELL		2
#define CH_CLOSE_SENT		4
#define CH_HAS_PTY		8
#define CH_EOF			16

/* SSHD.C */

extern char idstring[];
extern int ttyh;
extern char *default_user;

void debug_warning(struct ssh_connection *c, char *msg, ...);
void debug_error(struct ssh_connection *c, char *msg, ...);
void debug_fatal(struct ssh_connection *c, char *msg, ...);
void debug_internal(struct ssh_connection *c, char *msg, ...);
void abort_ssh_connection(struct ssh_connection *c);

/* SSHDBN.C */

int bn_2_string(BIGNUM *bn, __u8 **buf, unsigned *bytes);

/* SSHDPKT.C */

void ssh_read_more(struct ssh_connection *c);
void ssh_get_next_packet(struct ssh_connection *c);
extern AST_STUB ssh_in_sock;
extern AST_STUB ssh_out_sock;
extern AST_STUB ssh_in_tty;
extern AST_STUB ssh_out_tty;
int ssh_send_packet(struct ssh_connection *c, int hold);
int ssh_unknown_packet(struct ssh_connection *c, int allow_kex);
void ssh_send_disconnect(struct ssh_connection *c, __u32 code, char *string);

/* SSHDKEX.C */

extern const struct kex_type * const kex_types[];

void sshd_kexinit(struct ssh_connection *c);

/* SSHDPUB.C */

extern const struct public_type * const public_types[];
const struct public_type *get_public_type(const char *name);

/* SSHDSKEY.C */

union p_key *sshd_get_server_key(const char *name);
int sshd_load_server_keys(void);

/* SSHDCIPH.C */

extern const struct cipher_type * const cipher_types[];
extern const struct cipher_type cipher_none;

int setup_in_cipher(struct ssh_connection *c, const struct cipher_type *ct, const __u8 *key, const __u8 *iv);
int setup_out_cipher(struct ssh_connection *c, const struct cipher_type *ct, const __u8 *key, const __u8 *iv);

/* SSHDMAC.C */

extern const struct mac_type * const mac_types[];
extern const struct mac_type mac_none;

int setup_in_mac(struct ssh_connection *c, const struct mac_type *mt, __u8 *mac_key);
int setup_out_mac(struct ssh_connection *c, const struct mac_type *mt, __u8 *mac_key);


/* SSHDCMPR.C */

extern const struct compress_type * const compress_types[];
extern const struct compress_type compress_none;

int setup_in_compress(struct ssh_connection *c, const struct compress_type *ct);
int setup_out_compress(struct ssh_connection *c, const struct compress_type *ct);

/* SSHDAUTH.C */

void sshd_userauth(struct ssh_connection *c);

/* SSHDTTY.C */

void free_channel(struct ssh_channel *ch);
void sshd_main_loop(struct ssh_connection *c);

#endif
