#include <SPAD/DEV.H>
#include <SPAD/IOCTL.H>
#include <SPAD/SYNC.H>

#include "SSHD.H"
#include "OPS.H"
#include "STRUCT.H"

static int channel_send_data(struct ssh_channel *ch);
static int session_input(struct ssh_channel *ch);
static void sshd_tty_sent(struct ssh_connection *c);
static int session_output(struct ssh_channel *ch);
static void sshd_tty_received(struct ssh_connection *c);

#define hash(x)		((x) & (CHANNEL_HASH_SIZE - 1))

#define ENV_STR_LEN	4096

static struct ssh_channel *find_channel(struct ssh_connection *c, __u32 local_num)
{
	struct ssh_channel *ch;
	XLIST_FOR_EACH(ch, &c->channel_hash[hash(local_num)], struct ssh_channel, hash) if (__likely(ch->local_number == local_num)) return ch;
	return NULL;
}

static struct ssh_channel *allocate_channel(struct ssh_connection *c, __u32 remote_num, unsigned winsize, unsigned maxpacket)
{
	struct ssh_channel *ch;
	__u32 ln;
	if (__unlikely(!(ch = malloc(sizeof(struct ssh_channel))))) return NULL;
	ch->remote_number = remote_num;
	ch->out_window = winsize;
	ch->out_maxwindow = winsize;
	ch->out_maxpacket = maxpacket;
	ch->c = c;
	do ln = c->chnum++; while (__unlikely(find_channel(c, ln) != NULL));
	ch->local_number = ln;
	ch->in_buffer = ch->in_buffer_default;
	ch->in_buffer_size = MIN_CHANNEL_IN_BUFFER;
	ch->in_buffer_start = ch->in_buffer_end = 0;
	ch->in_window_adjust = 0;
	ch->out_buffer_start = ch->out_buffer_end = 0;
	ch->flags = 0;
	ch->in_upcall = ch->out_upcall = NULL;
	ADD_TO_XLIST(&c->channel_hash[hash(ln)], &ch->hash);
	return ch;
}

static void extend_channel(struct ssh_channel *ch)
{
	__u8 *b;
	if (__unlikely(ch->in_buffer != ch->in_buffer_default)) return;
	if (__unlikely(!(b = malloc(TTYSTR_LEN + MAX_CHANNEL_IN_BUFFER)))) return;
	memcpy(b, ch->in_buffer_default - TTYSTR_LEN, TTYSTR_LEN + ch->in_buffer_end);
	ch->in_buffer = b + TTYSTR_LEN;
	ch->in_buffer_size = MAX_CHANNEL_IN_BUFFER;
}

void free_channel(struct ssh_channel *ch)
{
	if (__likely(ch == ch->c->session_channel)) ch->c->session_channel = NULL;
	if (__unlikely(ch->in_buffer != ch->in_buffer_default)) free(ch->in_buffer - TTYSTR_LEN);
	DEL_FROM_LIST(&ch->hash);
	free(ch);
}

void sshd_main_loop(struct ssh_connection *c)
{
	unsigned sp;
	__u8 cmd;
	__u32 remote_num, winsize, maxpacket, local_num;
	const __u8 *name;
	unsigned name_len;
	__u8 want_reply;
	struct ssh_channel *ch;
	c->in_upcall = sshd_main_loop;
	again:
	sp = 0;
	CHECKH("sshd_main_loop");
	if (__unlikely(ssh_get_chr_from_connection(c, &sp, &cmd))) return;
	if (__likely(cmd == SSH_MSG_CHANNEL_WINDOW_ADJUST)) {
		__u32 val;
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &local_num))) return;
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &val))) return;
		ssh_got_data(c, &sp);
		if (__unlikely(!(ch = find_channel(c, local_num)))) {
			debug_error(c, "Window adjust on unknown channel %d", local_num);
			goto again;
		}
		if (__unlikely(!val)) goto again;
		if (__unlikely((__u32)(ch->out_window + val) < (__u32)ch->out_window)) ch->out_window = 0xffffffffU;
		else ch->out_window += val;
		if (__unlikely(channel_send_data(ch))) return;
		goto again;
	}
	if (__likely(cmd == SSH_MSG_CHANNEL_DATA) || __unlikely(cmd == SSH_MSG_CHANNEL_EXTENDED_DATA)) {
		const __u8 *data;
		unsigned data_len;
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &local_num))) return;
		if (__unlikely(cmd == SSH_MSG_CHANNEL_EXTENDED_DATA)) {
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, (void *)&KERNEL$LIST_END))) return;
		}
		if (__unlikely(ssh_get_len_str_from_connection(c, &sp, &data, &data_len))) return;
		/*__debug_printf("data input: \"%.*s\"\n", data_len, data);*/
		ssh_got_data(c, &sp);
		if (__unlikely(!(ch = find_channel(c, local_num)))) {
			debug_error(c, "Data on unknown channel %d", local_num);
			goto again;
		}
		if (__unlikely(ch->flags & CH_EOF)) {
			debug_error(c, "Data sent after EOF on channel %d", local_num);
			goto again;
		}
		aa:
		if (__unlikely(ch->in_buffer_end + data_len > MAX_CHANNEL_IN_BUFFER)) {
			debug_error(c, "Data over window, channel %d", local_num);
			data_len = MAX_CHANNEL_IN_BUFFER - ch->in_buffer_end;
			goto aa;
		}
		if (__unlikely(ch->in_buffer_end + data_len > ch->in_buffer_size)) {
			extend_channel(ch);
			if (__unlikely(ch->in_buffer_end + data_len > ch->in_buffer_size)) {
				debug_fatal(c, "Can't extend channel buffer");
				abort_ssh_connection(c);
				return;
			}
		}
		memcpy(ch->in_buffer + ch->in_buffer_end, data, data_len);
		ch->in_buffer_end += data_len;
		if (ch->in_upcall && __unlikely(ch->in_upcall(ch))) return;
		goto again;
	}
	if (__unlikely(cmd == SSH_MSG_CHANNEL_EOF)) {
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &local_num))) return;
		ssh_got_data(c, &sp);
		if (__unlikely(!(ch = find_channel(c, local_num)))) {
			debug_error(c, "EOF on unknown channel %d", local_num);
			goto again;
		}
		ch->flags |= CH_EOF;
		if (ch->in_upcall && __unlikely(ch->in_upcall(ch))) return;
		goto again;
	}
	if (__unlikely(cmd == SSH_MSG_CHANNEL_CLOSE)) {
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &local_num))) return;
		ssh_got_data(c, &sp);
		if (__unlikely(!(ch = find_channel(c, local_num)))) {
			debug_error(c, "Close on unknown channel %d", local_num);
			goto again;
		}
		if (!(ch->flags & CH_CLOSE_SENT)) {
			ch->flags |= CH_CLOSE_SENT;
			if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_CLOSE))) return;
			if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return;
			if (__unlikely(ssh_send_packet(c, 0))) return;
		}
		free_channel(ch);
		goto again;
	}
	if (__unlikely(cmd == SSH_MSG_GLOBAL_REQUEST)) {
		if (__unlikely(ssh_get_len_str_from_connection(c, &sp, &name, &name_len))) return;
		if (__unlikely(ssh_get_chr_from_connection(c, &sp, &want_reply))) return;
		ssh_got_data(c, &sp);
		ssh_flush_stream(c);
		if (want_reply) {
			if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_REQUEST_FAILURE))) return;
			if (__unlikely(ssh_send_packet(c, 0))) return;
		}
		goto again;
	}
	if (__unlikely(cmd == SSH_MSG_CHANNEL_OPEN)) {
		char *err;
		__u32 code;
		if (__unlikely(ssh_get_len_str_from_connection(c, &sp, &name, &name_len))) return;
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &remote_num))) return;
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &winsize))) return;
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &maxpacket))) return;
		if (__unlikely(!winsize) || __unlikely(!maxpacket)) {
			err = "INVALID PARAMETERS";
			code = SSH_OPEN_ADMINISTRATIVELY_PROHIBITED;
			goto channel_error;
		}
		if (__likely(name_len == 7) && __likely(!memcmp(name, "session", 7))) {
			ssh_got_data(c, &sp);
			if (__unlikely(c->flags & SSHD_HAS_SESSION)) {
				err = "ONLY ONE SESSION SUPPORTED";
				code = SSH_OPEN_RESOURCE_SHORTAGE;
				goto channel_error_noclr;
			}
			if (__unlikely(!(ch = allocate_channel(c, remote_num, winsize, maxpacket)))) {
				err = "UNABLE TO ALLOCATE CHANNEL";
				code = SSH_OPEN_RESOURCE_SHORTAGE;
				goto channel_error_noclr;
			}
			ch->flags = CH_SESSION;
			ch->in_upcall = session_input;
			ch->out_upcall = session_output;
			c->flags |= SSHD_HAS_SESSION;
			c->session_channel = ch;
			memcpy(ch->out_buffer - TTYSTR_LEN, c->tty_name, TTYSTR_LEN);
			if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_OPEN_CONFIRMATION))) return;
			if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return;
			if (__unlikely(ssh_add_int32_to_connection(c, ch->local_number))) return;
			if (__unlikely(ssh_add_int32_to_connection(c, MAX_CHANNEL_IN_BUFFER))) return;
			if (__unlikely(ssh_add_int32_to_connection(c, MAX_STREAM_LENGTH - 256))) return;
			if (__unlikely(ssh_send_packet(c, 0))) return;
			if (__unlikely(session_output(ch))) return;
			goto again;
		}
		err = "THIS CHANNEL TYPE IS NOT SUPPORTED";
		code = SSH_OPEN_UNKNOWN_CHANNEL_TYPE;
		channel_error:
		ssh_got_data(c, &sp);
		ssh_flush_stream(c);
		channel_error_noclr:
		if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_OPEN_FAILURE))) return;
		if (__unlikely(ssh_add_int32_to_connection(c, remote_num))) return;
		if (__unlikely(ssh_add_int32_to_connection(c, code))) return;
		if (__unlikely(ssh_add_len_str_to_connection(c, (__u8 *)err, strlen(err)))) return;
		if (__unlikely(ssh_add_len_str_to_connection(c, (__u8 *)"en", 2))) return;
		if (__unlikely(ssh_send_packet(c, 0))) return;
		goto again;
	}
	if (__unlikely(cmd == SSH_MSG_CHANNEL_REQUEST)) {
#define M_SHELL		0
#define M_EXEC		1
		int mode;
		const __u8 *command;
		unsigned command_len;
		if (__unlikely(ssh_get_int32_from_connection(c, &sp, &local_num))) return;
		if (__unlikely(ssh_get_len_str_from_connection(c, &sp, &name, &name_len))) return;
		if (__unlikely(ssh_get_chr_from_connection(c, &sp, &want_reply))) return;
		if (__unlikely(!(ch = find_channel(c, local_num)))) {
			ssh_got_data(c, &sp);
			ssh_flush_stream(c);
			debug_error(c, "Request on unknown channel %d", local_num);
			goto again;
		}
		if (name_len == 7 && !memcmp(name, "pty-req", 7) && __likely(ch->flags & CH_SESSION)) {
			IOCTLRQ io;
			const __u8 *tty_type;
			unsigned tty_type_len;
			__u32 width, height;
			if (__unlikely(ssh_get_len_str_from_connection(c, &sp, &tty_type, &tty_type_len))) return;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, &width))) return;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, &height))) return;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, (void *)&KERNEL$LIST_END))) return;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, (void *)&KERNEL$LIST_END))) return;
			if (__unlikely(ssh_get_len_str_from_connection(c, &sp, (void *)&KERNEL$LIST_END, (void *)&KERNEL$LIST_END))) return;
			ssh_got_data(c, &sp);
			io.h = ttyh;
			io.ioctl = IOCTL_TTYSRVR_WINSIZE;
			io.param = (width << 16) + height;
			io.v.ptr = (unsigned long)c->tty_name;
			io.v.len = TTYSTR_LEN;
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
			if (__unlikely(io.status < 0)) {
				debug_warning(c, "Can't set terminal size: %ld", io.status);
			}
			_snprintf(c->tty_type, sizeof c->tty_type, "%.*s", (int)tty_type_len, tty_type);
			ch->flags |= CH_HAS_PTY;
			goto ch_request_succ;
		}
		if (name_len == 13 && !memcmp(name, "window-change", 13) && __likely(ch->flags & CH_SESSION)) {
			IOCTLRQ io;
			__u32 width, height;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, &width))) return;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, &height))) return;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, (void *)&KERNEL$LIST_END))) return;
			if (__unlikely(ssh_get_int32_from_connection(c, &sp, (void *)&KERNEL$LIST_END))) return;
			ssh_got_data(c, &sp);
			io.h = ttyh;
			io.ioctl = IOCTL_TTYSRVR_WINSIZE;
			io.param = (width << 16) + height;
			io.v.ptr = (unsigned long)c->tty_name;
			io.v.len = TTYSTR_LEN;
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
			if (__unlikely(io.status < 0)) {
				debug_warning(c, "Can't set terminal size: %ld", io.status);
			}
			goto ch_request_succ;
		}
		if (name_len == 5 && !memcmp(name, "shell", 4) && !(ch->flags & CH_HAS_SHELL)) {
			char *str, *p;
			IOCTLRQ io;
			ssh_got_data(c, &sp);
			mode = M_SHELL;
			command = NULL, command_len = 0;
			spawn_ses:
			if (__unlikely(!(ch->flags & CH_HAS_PTY))) {
				io.h = ttyh;
				io.ioctl = IOCTL_TTYSRVR_RAW;
				io.param = 0;
				io.v.ptr = (unsigned long)c->tty_name;
				io.v.len = TTYSTR_LEN;
				io.v.vspace = &KERNEL$VIRTUAL;
				SYNC_IO(&io, KERNEL$IOCTL);
				if (__unlikely(io.status < 0)) {
					debug_warning(c, "Can't set raw terminal mode: %ld", io.status);
				}
			} else {
				int nd = 1;
				setsockopt(c->out_sock.h, SOL_TCP, TCP_NODELAY, &nd, sizeof nd);
			}
			if (__unlikely(!(str = malloc(TTYSTR_LEN + ENV_STR_LEN)))) {
				debug_fatal(c, "Can't alloc env data");
				abort_ssh_connection(c);
				return;
			}
			memcpy(str, c->tty_name, TTYSTR_LEN);
			p = str + TTYSTR_LEN;
#define room	(str + TTYSTR_LEN + ENV_STR_LEN - p)
			if (c->tty_type[0]) _snprintf(p, room, "TERMINAL %s\n", c->tty_type), p = strchr(p, 0);
			if (__unlikely(command_len)) _snprintf(p, room, "EXEC %.*s\n", (int)command_len, command), p = strchr(p, 0);
			else if (__unlikely(!_memcasecmp(c->client_prog_id, "WinSCP", 6))) _snprintf(p, room, "EXEC SH /SET:KERNEL$ABS_LN_OPEN=1\n"), p = strchr(p, 0);
			_snprintf(p, room, "DONE\n"), p = strchr(p, 0);
			if (__unlikely(room <= 1)) {
				lng:
				free(str);
				debug_fatal(c, "Env buffer overflow");
				ssh_send_disconnect(c, SSH_DISCONNECT_TOO_MANY_CONNECTIONS, "ENV BUFFER OVERFLOW (TOO LONG COMMAND LINE)");
			}
#undef room
			io.h = ttyh;
			io.ioctl = IOCTL_TTYSRVR_SENDCHR;
			io.param = 1;
			io.v.ptr = (unsigned long)str;
			io.v.len = p - str;
			io.v.vspace = &KERNEL$VIRTUAL;
			SYNC_IO(&io, KERNEL$IOCTL);
			if (__unlikely(io.status < 0)) {
				free(str);
				debug_fatal(c, "Can't send env buffer: %ld", io.status);
				abort_ssh_connection(c);
				return;
			}
			if (__unlikely(io.status != p - str - TTYSTR_LEN)) goto lng;
			free(str);
			ch->flags |= CH_HAS_SHELL;
			goto ch_request_succ;
		}
		if (name_len == 4 && !memcmp(name, "exec", 4)) {
			if (__unlikely(ssh_get_len_str_from_connection(c, &sp, &command, &command_len))) return;
			ssh_got_data(c, &sp);
			/*__debug_printf("exec: \"%.*s\"\n", command_len, command);*/
			mode = M_EXEC;
			goto spawn_ses;
		}
		ssh_got_data(c, &sp);
		ssh_flush_stream(c);
		if (!want_reply) goto again;
		if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_FAILURE))) return;
		if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return;
		if (__unlikely(ssh_send_packet(c, 0))) return;
		goto again;
		ch_request_succ:
		if (!want_reply) goto again;
		if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_SUCCESS))) return;
		if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return;
		if (__unlikely(ssh_send_packet(c, 0))) return;
		goto again;
	}
	if (ssh_unknown_packet(c, 1)) return;
	goto again;
}

static int channel_send_window_adjust(struct ssh_channel *ch, int hold)
{
	struct ssh_connection *c = ch->c;
	if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_WINDOW_ADJUST))) return -1;
	if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return -1;
	if (__unlikely(ssh_add_int32_to_connection(c, ch->in_window_adjust))) return -1;
	if (__unlikely(ssh_send_packet(c, hold))) return -1;
	ch->in_window_adjust = 0;
	return 0;
}

static int channel_fold(struct ssh_channel *ch)
{
	struct ssh_connection *c;
	if (__unlikely(!ch->in_buffer_start)) return 0;
	c = ch->c;
	ch->in_window_adjust += ch->in_buffer_start;
	if (__unlikely(ch->in_window_adjust >= MAX_IN_WINDOW_ADJUST_DELAY)) {
		if (__unlikely(channel_send_window_adjust(ch, 0))) return -1;
	}
	memmove(ch->in_buffer, ch->in_buffer + ch->in_buffer_start, ch->in_buffer_end - ch->in_buffer_start);
	ch->in_buffer_end -= ch->in_buffer_start;
	ch->in_buffer_start = 0;
	return 0;
}

static int channel_send_data(struct ssh_channel *ch)
{
	struct ssh_connection *c;
	unsigned l;
	again:
	l = ch->out_buffer_end - ch->out_buffer_start;
	if (__unlikely(!l)) {
		upcall:
		if (ch->out_upcall) return ch->out_upcall(ch);
		return 0;
	}
	if (__unlikely(ch->out_window < ch->out_maxwindow >> OUT_WINDOW_DELAY_SHIFT)) goto upcall;
	if (__unlikely(l > ch->out_window)) l = ch->out_window;
	if (__unlikely(l > ch->out_maxpacket)) l = ch->out_maxpacket;
	if (__unlikely(l > MAX_PACKET_LENGTH - 1024)) l = MAX_PACKET_LENGTH - 1024;
	if (__unlikely(!l)) {
		if (!ch->out_buffer_start && ch->out_buffer_end == CHANNEL_OUT_BUFFER) return 0;
		goto upcall;
	}
	if (__unlikely(ch->in_window_adjust >= IN_WINDOW_ADJUST_DELAY)) {
		if (__unlikely(channel_send_window_adjust(ch, 1))) return -1;
	}
	c = ch->c;
	if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_DATA))) return -1;
	if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return -1;
	if (__unlikely(ssh_add_len_str_to_connection(c, ch->out_buffer + ch->out_buffer_start, l))) return -1;
	if (__unlikely(ssh_send_packet(c, 0))) return -1;
	/*__debug_printf("send: \"%.*s\"\n", l, ch->out_buffer + ch->out_buffer_start);*/
	ch->out_buffer_start += l;
	ch->out_window -= l;
	goto again;
}

static int session_input(struct ssh_channel *ch)
{
	struct ssh_connection *c = ch->c;
	if (__unlikely(ch->in_buffer_start == ch->in_buffer_end)) {
		if (__unlikely(ch->flags & CH_EOF)) {
			IOCTLRQ io;
			io.h = ttyh;
			io.ioctl = IOCTL_TTYSRVR_EOF;
			io.param = 0;
			io.v.ptr = (unsigned long)c->tty_name;
			io.v.len = TTYSTR_LEN;
			io.v.vspace = &KERNEL$VIRTUAL;
		 	SYNC_IO(&io, KERNEL$IOCTL);
		}
		return 0;
	}
	ch->in_upcall = NULL;
	/*__debug_printf("recv: \"%.*s\"\n", ch->in_buffer_end - ch->in_buffer_start, ch->in_buffer + ch->in_buffer_start);*/
	c->out_tty_upcall = sshd_tty_sent;
	c->out_tty.ioctl = IOCTL_TTYSRVR_SENDCHR;
	c->out_tty.param = 0;
	c->out_tty.v.ptr = (unsigned long)(ch->in_buffer + ch->in_buffer_start - TTYSTR_LEN);
	c->out_tty.v.len = ch->in_buffer_end - ch->in_buffer_start + TTYSTR_LEN;
	memcpy((void *)(unsigned long)c->out_tty.v.ptr, c->tty_name, TTYSTR_LEN);
	RAISE_SPL(SPL_BOTTOM);
	c->outstanding++;
	CALL_IORQ(&c->out_tty, KERNEL$IOCTL);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

static void sshd_tty_sent(struct ssh_connection *c)
{
	struct ssh_channel *ch;
	ch = c->session_channel;
	if (__unlikely(!ch)) return;
	if (__unlikely(c->out_tty.status <= 0)) {
		debug_fatal(c, "Write to terminal failed: %ld", c->out_tty.status);
		abort_ssh_connection(c);
		return;
	}
	/*{
		int i;
		__debug_printf("put to tty: \"");
		for (i = 0; i < c->out_tty.status; i++) __debug_printf("%c", ch->in_buffer[ch->in_buffer_start + i]);
		__debug_printf("\"\n");
	}*/
	ch->in_buffer_start += c->out_tty.status;
	if (__likely(ch->in_buffer_start == ch->in_buffer_end) ||
	    __unlikely(ch->in_buffer_start >= ch->in_buffer_size - ch->in_buffer_end)) {
	    	if (__unlikely(channel_fold(ch))) return;
	}
	ch->in_upcall = session_input;
	session_input(ch);
}

static int session_output(struct ssh_channel *ch)
{
	struct ssh_connection *c = ch->c;
	if (__unlikely(ch->out_buffer_start != ch->out_buffer_end)) return 0;
	ch->out_buffer_start = ch->out_buffer_end = 0;
	ch->out_upcall = NULL;
	c->in_tty_upcall = sshd_tty_received;
	c->in_tty.ioctl = IOCTL_TTYSRVR_RCVCHR;
	c->in_tty.param = 0;
	c->in_tty.v.ptr = (unsigned long)(ch->out_buffer - TTYSTR_LEN);
	c->in_tty.v.len = TTYSTR_LEN + CHANNEL_OUT_BUFFER;
	RAISE_SPL(SPL_BOTTOM);
	c->outstanding++;
	CALL_IORQ(&c->in_tty, KERNEL$IOCTL);
	LOWER_SPL(SPL_ZERO);
	return 0;
}

static void sshd_tty_received(struct ssh_connection *c)
{
	struct ssh_channel *ch;
	ch = c->session_channel;
	if (__unlikely(!ch)) return;
	if (__unlikely(c->in_tty.status <= 0)) {
		ch->flags |= CH_CLOSE_SENT;
		if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_REQUEST))) return;
		if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return;
		if (__unlikely(ssh_add_len_str_to_connection(c, (__u8 *)"exit-status", 11))) return;
		if (__unlikely(ssh_add_chr_to_connection(c, 0))) return;
		if (__unlikely(ssh_add_int32_to_connection(c, 0))) return;
		if (__unlikely(ssh_send_packet(c, 1))) return;
		if (__unlikely(ssh_add_chr_to_connection(c, SSH_MSG_CHANNEL_CLOSE))) return;
		if (__unlikely(ssh_add_int32_to_connection(c, ch->remote_number))) return;
		if (__unlikely(ssh_send_packet(c, 0))) return;
		return;
	}
	/*{
		int x = open("ssh.log", O_CREAT | O_APPEND | O_WRONLY);
		write(x, ch->out_buffer, c->in_tty.status);
		close(x);
	}*/
	ch->out_buffer_end += c->in_tty.status;
	ch->out_upcall = session_output;
	channel_send_data(ch);
}

