#include <SPAD/LIBC.H>
#include <SPAD/SYNC.H>
#include <ARCH/IO.H>
#include <SPAD/CMOS.H>
#include <SPAD/PCI.H>
#include <SPAD/VM.H>
#include <SPAD/DL.H>
#include <STRING.H>

static const struct __param_table general_params[1] = {
	NULL, 0, 0, 0,
};

static void * const general_vars[1] = {
	NULL,
};

static pci_id_t pci_id;
static __s8 pci_valid = 0;

static __u64 fn_inb(__u64 addr, int *err) { return io_inb(addr); }

static __u64 fn_inw(__u64 addr, int *err) { return io_inw(addr); }

static __u64 fn_inl(__u64 addr, int *err) { return io_inl(addr); }

static void fn_outb(__u64 addr, __u64 data, int *err) { io_outb(addr, data); }

static void fn_outw(__u64 addr, __u64 data, int *err) { io_outw(addr, data); }

static void fn_outl(__u64 addr, __u64 data, int *err) { io_outl(addr, data); }

static __u64 fn_mminb(__u64 addr, int *err)
{
	__u8 val;
	__u8 *p = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(addr, 1, PAT_UC);
	if (__unlikely(__IS_ERR(p))) {
		*err = __PTR_ERR(p);
		return 0;
	}
	val = mmio_inb(p);
	KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(p, 1);
	return val;
}

static __u64 fn_mminw(__u64 addr, int *err)
{
	__u16 val;
	__u8 *p = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(addr, 2, PAT_UC);
	if (__unlikely(__IS_ERR(p))) {
		*err = __PTR_ERR(p);
		return 0;
	}
	val = mmio_inw(p);
	KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(p, 2);
	return val;
}

static __u64 fn_mminl(__u64 addr, int *err)
{
	__u32 val;
	__u8 *p = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(addr, 4, PAT_UC);
	if (__unlikely(__IS_ERR(p))) {
		*err = __PTR_ERR(p);
		return 0;
	}
	val = mmio_inl(p);
	KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(p, 4);
	return val;
}

static void fn_mmoutb(__u64 addr, __u64 data, int *err)
{
	__u8 *p = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(addr, 1, PAT_UC);
	if (__unlikely(__IS_ERR(p))) {
		*err = __PTR_ERR(p);
		return;
	}
	mmio_outb(p, data);
	KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(p, 1);
}

static void fn_mmoutw(__u64 addr, __u64 data, int *err)
{
	__u8 *p = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(addr, 2, PAT_UC);
	if (__unlikely(__IS_ERR(p))) {
		*err = __PTR_ERR(p);
		return;
	}
	mmio_outw(p, data);
	KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(p, 2);
}

static void fn_mmoutl(__u64 addr, __u64 data, int *err)
{
	__u8 *p = KERNEL$MAP_PHYSICAL_REGION_LONG_TERM(addr, 4, PAT_UC);
	if (__unlikely(__IS_ERR(p))) {
		*err = __PTR_ERR(p);
		return;
	}
	mmio_outl(p, data);
	KERNEL$UNMAP_PHYSICAL_REGION_LONG_TERM(p, 4);
}

static __u64 fn_msrin(__u64 addr, int *err)
{
	__u64 data;
	*err = KERNEL$READ_MSR(addr, &data);
	return data;
}

static void fn_msrout(__u64 addr, __u64 data, int *err)
{
	*err = KERNEL$WRITE_MSR(addr, data);
}

static __u64 fn_cmosin(__u64 addr, int *err)
{
	int val = CMOS$READ(addr);
	if (__unlikely(val < 0)) *err = val;
	return val;
}

static void fn_cmosout(__u64 addr, __u64 data, int *err)
{
	*err = CMOS$WRITE(addr, data);
}

static __u64 fn_pciinb(__u64 addr, int *err)
{
	if (__unlikely(pci_valid <= 0)) { *err = !pci_valid ? -EBADSYN : -EOPNOTSUPP; return 0; }
	return PCI$READ_CONFIG_BYTE(pci_id, addr);
}

static __u64 fn_pciinw(__u64 addr, int *err)
{
	if (__unlikely(pci_valid <= 0)) { *err = !pci_valid ? -EBADSYN : -EOPNOTSUPP; return 0; }
	return PCI$READ_CONFIG_WORD(pci_id, addr);
}

static __u64 fn_pciinl(__u64 addr, int *err)
{
	if (__unlikely(pci_valid <= 0)) { *err = !pci_valid ? -EBADSYN : -EOPNOTSUPP; return 0; }
	return PCI$READ_CONFIG_DWORD(pci_id, addr);
}

static void fn_pcioutb(__u64 addr, __u64 data, int *err)
{
	if (__unlikely(pci_valid <= 0)) { *err = !pci_valid ? -EBADSYN : -EOPNOTSUPP; return; }
	PCI$WRITE_CONFIG_BYTE(pci_id, addr, data);
}

static void fn_pcioutw(__u64 addr, __u64 data, int *err)
{
	if (__unlikely(pci_valid <= 0)) { *err = !pci_valid ? -EBADSYN : -EOPNOTSUPP; return; }
	PCI$WRITE_CONFIG_WORD(pci_id, addr, data);
}

static void fn_pcioutl(__u64 addr, __u64 data, int *err)
{
	if (__unlikely(pci_valid <= 0)) { *err = !pci_valid ? -EBADSYN : -EOPNOTSUPP; return; }
	PCI$WRITE_CONFIG_DWORD(pci_id, addr, data);
}

static const struct {
	__u8 data_bits;
	__u8 increment;
	__u64 addr_max;
	char *in_str;
	__u64 (*in)(__u64, int *);
	char *out_str;
	void (*out)(__u64, __u64, int *);
} iotype[] = {
	8, 1, IO_SPACE_LIMIT, "INB", fn_inb, "OUTB", fn_outb,
	16, 2, IO_SPACE_LIMIT, "INW", fn_inw, "OUTW", fn_outw,
	32, 4, IO_SPACE_LIMIT, "INL", fn_inl, "OUTL", fn_outl,
	8, 1, (__p_addr)-1, "MMINB", fn_mminb, "MMOUTB", fn_mmoutb,
	16, 2, (__p_addr)-2, "MMINW", fn_mminw, "MMOUTW", fn_mmoutw,
	32, 4, (__p_addr)-4, "MMINL", fn_mminl, "MMOUTL", fn_mmoutl,
	64, 1, 0xFFFFFFFF, "MSRIN", fn_msrin, "MSROUT", fn_msrout,
	8, 1, 255, "CMOSIN", fn_cmosin, "CMOSOUT", fn_cmosout,
	8, 1, 255, "PCIINB", fn_pciinb, "PCIOUTB", fn_pcioutb,
	16, 2, 254, "PCIINW", fn_pciinw, "PCIOUTW", fn_pcioutw,
	32, 4, 252, "PCIINL", fn_pciinl, "PCIOUTL", fn_pcioutl,
	0, 0, 0, NULL, NULL, NULL, NULL,
};

int main(int argc, const char * const argv[])
{
	int state = 0;
	const char * const *arg = argv;
	const char *opt, *optend, *val;
	if (__unlikely(FEATURE_TEST(FEATURE_USERSPACE))) {
		_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "IO: AVAILABLE ONLY IN KERNEL MODE");
		return -EUSER;
	}
	while (__parse_params(&arg, &state, general_params, general_vars, &opt, &optend, &val)) {
		int i;
		int err = 0;
		if (__unlikely(!__strcasexcmp("PCI", opt, optend))) {
			int order;
			pci_id_t id_mask;
			if (!pci_valid) {
				DLLZRQ lz;
				lz.handle = KERNEL$DLRQ();
				lz.caller = NULL;
				strcpy(lz.modname, "PCI");
				SYNC_IO_CANCELABLE(&lz, KERNEL$DL_LOAD_LAZY);
				if (__unlikely(lz.status < 0)) {
					if (!pci_valid) _printf("CAN'T LINK PCI\n");
					pci_valid = -1;
				} else {
					pci_valid = 1;
				}
			}
			if (__unlikely(pci_valid <= 0)) goto done;
			order = -1;
			id_mask = 0;
			if (__unlikely(PCI$PARSE_PARAMS(opt, optend, val, &pci_id, &id_mask, &order))) goto bads;
			if (__unlikely(order != -1) || __unlikely(id_mask != (pci_id_t)~0)) goto bads;
			goto done;
		}
		for (i = 0; iotype[i].data_bits; i++) {
			__u64 addr;
			__u64 data;
			if (__likely(iotype[i].in_str != NULL) && !__strcasexcmp(iotype[i].in_str, opt, optend)) {
				if (__unlikely(__get_64_number(val, strchr(val, 0), 0, (__s64 *)&addr))) goto bads;
				if (__unlikely(addr > iotype[i].addr_max)) goto bads;
				data = iotype[i].in(addr, &err);
				_printf("%s(#%"__64_format"X)", iotype[i].in_str, addr);
				if (__unlikely(err)) {
					_printf(" ! %s\n", strerror(-err));
				} else {
					_printf(" = #%0*"__64_format"X\n", (iotype[i].data_bits + 3) >> 2, data);
				}
				goto done;
			} else if (__likely(iotype[i].out_str != NULL) && !__strcasexcmp(iotype[i].out_str, opt, optend)) {
				char *split = strchr(val, ',');
				if (__unlikely(!split)) goto bads;
				if (__unlikely(__get_64_number(val, split, 0, (__s64 *)&addr))) goto bads;
				if (__unlikely(addr > iotype[i].addr_max)) goto bads;
				if (__unlikely(__get_64_number(split + 1, strchr(split, 0), 0, (__s64 *)&data))) goto bads;
				if (__unlikely((data & (__u64)-2 << (iotype[i].data_bits - 1)) != 0)) goto bads;
				_printf("%s(#%"__64_format"X) <- #%0*"__64_format"X", iotype[i].out_str, addr, (iotype[i].data_bits + 3) >> 2, data);
				iotype[i].out(addr, data, &err);
				if (__unlikely(err)) {
					_printf(" ! %s\n", strerror(-err));
				} else {
					_printf("\n");
				}
				goto done;
			}
		}
		goto bads;
		done:;
	}
	return 0;

	bads:
	_snprintf(KERNEL$ERROR_MSG(), __MAX_STR_LEN, "IO: SYNTAX ERROR");
	return -EBADSYN;
}
